/*
   Project: Cartotheque

   Copyright (C) 2005 Stefan Urbanek

   Author : Stefan Urbanek
   Created: 2005-01-27
   License: GNU LGPL 2.1
*/

#import "Cartotheque.h"

#import <Foundation/NSAttributedString.h>
#import <Foundation/NSException.h>
#import <Foundation/NSNotification.h>
#import <Foundation/NSString.h>

#import "Card.h"
#import "CardRepository.h"

NSString *CardCreatedNotification = @"CardCreatedNotification";
NSString *CardEditedNotification = @"CardEditedNotification";

@interface Cartotheque(CartothequePrivate)
- (Card *)_cardWithIdentifier:(id)identifier;
- (Card *)_fetchCardWithIdentifier:(id)identifier;
@end

@implementation Cartotheque
- initWithRepository:(CardRepository *)aRepository
{
    self = [super init];
    
    repository = RETAIN(aRepository);
    
    return self;
}
- (Card *)homeCard
{
    return [self _cardWithIdentifier:[repository homeCard]];
}

- (Card *)createCard
{
    NSAttributedString *contents;
    Card               *card;
    id                  ident;

    //NSLog(@"1 create in repository");
    ident = [repository createCard];
    
    //NSLog(@"2 alloc card");
    card = [[Card alloc] initWithRepository:repository
                                 identifier:ident];

    //NSLog(@"3 set contents");
    /* FIXME: use some default context */
    contents = [[NSAttributedString alloc] init];
    [card setContents:contents];
    AUTORELEASE(contents);
    //NSLog(@"4 done");

    /* FIXME: post card that was created as user info */
    [[NSNotificationCenter defaultCenter]
                          postNotificationName:CardCreatedNotification
                                        object:self];
    
    return AUTORELEASE(card);
}
- (void)deleteCards:(NSArray *)cards
{
    NSEnumerator    *enumerator;
    Card            *card;

    /* FIXME: use card deleted notification, as this is misleading
                we just reuse it here temporarily */
    enumerator = [cards objectEnumerator];
    
    while( (card = [enumerator nextObject]) )
    {
        [self deleteCard:card];
    }
    [[NSNotificationCenter defaultCenter]
                          postNotificationName:CardCreatedNotification
                                        object:self];
}
- (void)deleteCard:(Card *)card
{
    [fetchedCards removeObjectForKey:[card identifier]];
    [changedCards removeObject:card];
    [repository deleteCard:[card identifier]];
    [card invalidate];
}
- (NSArray *)allCards
{
    NSMutableArray *array;
    NSEnumerator   *enumerator;
    id              identifier;
    
    array = [NSMutableArray array];
    
    enumerator = [[repository allCards] objectEnumerator];
    
    while( (identifier = [enumerator nextObject]) )
    {
        [array addObject:[self _cardWithIdentifier:identifier]];
    }
    
    return [NSArray arrayWithArray:array];
}
- (Card *)_cardWithIdentifier:(id)identifier
{
    Card *card;
    card = [fetchedCards objectForKey:identifier];
    if(!card)
    {
        card = [self _fetchCardWithIdentifier:identifier];
    }
    return card;
}
- (Card *)_fetchCardWithIdentifier:(id)identifier
{
    Card *card;
    
    card = [[Card alloc] initWithRepository:repository
                                 identifier:identifier];
    if(!fetchedCards)
    {
        fetchedCards = [[NSMutableDictionary alloc] init];
    }
    
    [fetchedCards setObject:AUTORELEASE(card) forKey:identifier];
    
    return card;
}
- (void)commitChanges
{
    NSEnumerator *enumerator;
    Card         *card;
    id            ident;
    NSLog(@"    Commiting changes to %i cards", [changedCards count]);
    enumerator = [changedCards objectEnumerator];
    while( (card = [enumerator nextObject]) )
    {
        ident = [card identifier];

        [repository setContents:[card contents] forCard:ident];
        [repository setInfo:[card info] forCard:ident];
    }

    [changedCards removeAllObjects];
}

- (void)touchCard:(Card *)card
{
    if(card)
    {
        if(!changedCards)
        {
            changedCards = [[NSMutableSet alloc] init];
        }
        [changedCards addObject:card];
    }
    else
    {
        NSLog(@"Warning: nil card to touch. Ignoring.");
    }
}
@end
/* BEGIN Generated by DevelKit */

@implementation Cartotheque (DKGeneratedMethods)
/* Accessor methods */

- (CardRepository *)repository
{
    return repository;
}
- (void)dealloc
{
    RELEASE(changedCards);

    [super dealloc];
}

/* Encoding methods */
- (void)encodeWithCoder:(NSCoder *)coder
{
    // NSLog(@"Encoding class %@", [self className]);
    if ( [coder allowsKeyedCoding] )
    {
        [coder encodeObject:changedCards forKey:@"changedCards"];
        [coder encodeObject:repository forKey:@"repository"];
    }
    else
    {
        [coder encodeValueOfObjCType: @encode(NSMutableSet *) at: &changedCards];
        [coder encodeValueOfObjCType: @encode(CardRepository *) at: &repository];
    }
}
- initWithCoder:(NSCoder *)decoder
{
    self = [super init];
    if ( [decoder allowsKeyedCoding] )
    {
        changedCards = [decoder decodeObjectForKey:@"changedCards"];
        repository = [decoder decodeObjectForKey:@"repository"];
    }
    else
    {
        [decoder decodeValueOfObjCType: @encode(NSMutableSet *) at: &changedCards];
        [decoder decodeValueOfObjCType: @encode(CardRepository *) at: &repository];
    }

    RETAIN(changedCards);

    return self;
}

@end
/* END Generated by DevelKit */
