/*
   Project: Cartotheque

   Copyright (C) 2005 Stefan Urbanek

   Author : Stefan Urbanek
   Created: 2005-Feb
   License: GNU LGPL 2.1
*/

#import "CardMatrix.h"

#import "Card.h"
#import "Cartotheque.h"

#import <Foundation/NSNotification.h>
#import <AppKit/NSColor.h>
#import <AppKit/NSGraphics.h>
#import <AppKit/NSBezierPath.h>
#import <AppKit/NSCell.h>
#import <AppKit/NSMatrix.h>
#import <AppKit/NSScrollView.h>
#import <AppKit/NSStringDrawing.h>
#import <AppKit/PSOperators.h>

@interface CardCell:NSCell
@end

@implementation CardCell
- initWithCard:(Card *)card
{
    self = [super init];
    
    [self setRepresentedObject:card];
    
    return self;
}
- (void)drawInteriorWithFrame:(NSRect)cellFrame
                       inView:(NSView *)controlView
{
    NSRect        rect;
    NSRect        textRect;
    
#define FACTOR 0.75 /* or whatever */
    PSgsave();
    PStranslate(cellFrame.origin.x,cellFrame.origin.y);
    PSscale(FACTOR,FACTOR);
    rect = NSMakeRect(0,0,cellFrame.size.width/FACTOR,
                          cellFrame.size.height/FACTOR);

    rect = NSInsetRect(rect, 4,4);

    [[NSColor whiteColor] set];
    [NSBezierPath fillRect:rect];
    [[NSColor blackColor] set];
    [NSBezierPath strokeRect:rect];

    textRect = NSInsetRect(rect, 4,4);

    /* FIXME: this works only for NSString/NSAttributeString card contents */
    [[[self representedObject] contents] drawInRect: textRect];

    PSgrestore();

    if([self isHighlighted])
    {
        NSColor *color;
        color = [NSColor selectedControlColor];
        color = [color colorUsingColorSpaceName:NSDeviceRGBColorSpace];
        color = [color colorWithAlphaComponent:0.5];

        [color set];
        [NSBezierPath fillRect:cellFrame];
    }
}
@end

@interface CardMatrix(Private)
- (void)_initSubviews;
@end

@implementation CardMatrix
- initWithFrame:(NSRect)rect
{
    self = [super initWithFrame:rect];

    [self _initSubviews];

    return self;
}
- (void)awakeFromNib
{
    [self _initSubviews];
}
- (void)_initSubviews
{
    scrollView = [[NSScrollView alloc] initWithFrame:[self bounds]];
    matrix = [[NSMatrix alloc] initWithFrame:NSMakeRect(0,0,0,0)];

    [matrix setTarget:self];
    [matrix setAction:@selector(cardSelectionChanged:)];
    [matrix setDoubleAction:@selector(cardDoubleclicked:)];
    [matrix setMode:NSListModeMatrix];
    [matrix setAllowsEmptySelection:YES];
    
    [scrollView setBorderType:NSBezelBorder];
    [scrollView setDocumentView:matrix];
    [scrollView setAutoresizingMask:[self autoresizingMask]];
    [scrollView setHasVerticalScroller:YES];
    [scrollView setHasHorizontalScroller:NO];

    [self addSubview:scrollView];
    cardFrameSize = NSMakeSize(80.0, 60.0);
    cardSpacing = NSMakeSize(4.0, 4.0);

    [self setPostsFrameChangedNotifications:YES];
}
- (void)dealloc
{
    /* FIXME: release subviews */
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    RELEASE(cartotheque);
    RELEASE(delegate);
    [super dealloc];
}
- (void)setDelegate:(id)anObject
{
    ASSIGN(delegate, anObject);
}
- (void)setCartotheque:(Cartotheque *)cards
{
    NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];

    ASSIGN(cartotheque, cards);

    [nc removeObserver:self];

    [nc addObserver:self 
           selector:@selector(cartothequeCardsChanged:)
               name:CardCreatedNotification
             object:cartotheque];

    [nc addObserver:self 
           selector:@selector(frameDidChange:)
               name:NSViewFrameDidChangeNotification
             object:self];

    /* FIXME: use some other, probably private, method */
    [self reloadCards];
}
- (void)cardSelectionChanged:(id)sender
{
    Card *card;
    
    card = [[matrix selectedCell] representedObject];
    
    /* FIXME: add array of selected objetcs */
    //[delegate cardMatrix:self didChangeSelection:nil];
}
- (void)cardDoubleclicked:(id)sender
{
    Card *card;
    
    card = [[matrix selectedCell] representedObject];
    
    if([delegate respondsToSelector:@selector(cardMatrix:openCard:)])
    {
        [delegate cardMatrix:self openCard:card];
    }
}
- (void)reloadCards
{
    NSMutableArray *cells;
    NSEnumerator   *enumerator;
    CardCell       *cell;
    NSSize          size;
    Card           *card;
    int             i;
    int             columns;
    
    /* remove all matrix cells */
    while([matrix numberOfRows] > 0)
    {
        [matrix removeRow:0];
    }
    size = [scrollView contentSize];
    columns = size.width / (cardFrameSize.width + cardSpacing.width);
    
    enumerator = [[cartotheque allCards] objectEnumerator];
    
    cells = [NSMutableArray array];

    i = 0;
    
    while( (card = [enumerator nextObject]) )
    {
        cell = [[CardCell alloc] initWithCard:card];
        [cells addObject:AUTORELEASE(cell)];

        i = i + 1;
        if(i >= columns)
        {
            [matrix addRowWithCells:cells];
            [cells removeAllObjects];
            i = 0;
        }
    }
    if([cells count] > 0)
    {
        [matrix addRowWithCells:cells];
    }
    [matrix setCellSize:cardFrameSize];
    // [matrix setIntercellSpacing:cardSpacing];
    [matrix sizeToCells];

    /* FIXME: is this really needed? */
    [matrix setNeedsDisplay:YES];
    [scrollView setNeedsDisplay:YES];
        
    [self setNeedsDisplay:YES];
}
- (NSArray *)selectedCards
{
    NSMutableArray *array;
    NSEnumerator   *enumerator;
    CardCell       *cell;    
    Card           *card;
    
    array = [NSMutableArray array];
    enumerator = [[matrix selectedCells] objectEnumerator];

    while( (cell = [enumerator nextObject]) )
    {
        card = [cell representedObject];
        [array addObject:card];
    }
    
    return [NSArray arrayWithArray:array];
}

- (void)setCardFrameSize:(NSSize)size
{
    cardFrameSize = size;
}
- (void)cartothequeCardsChanged:(NSNotification *)notification
{
    [self reloadCards];
    [self setNeedsDisplay:YES];
}
- (void)frameDidChange:(NSNotification *)notification
{
    [self reloadCards];
    [self setNeedsDisplay:YES];
}
@end
