/*
   Project: Cartotheque

   Copyright (C) 2005 Stefan Urbanek

   Author : Stefan Urbanek
   Created: 2005-01-27
   License: GNU LGPL 2.1
*/

#include "CardRepository.h"

#import <Foundation/NSFileManager.h>
#import <Foundation/NSArchiver.h>
#import <Foundation/NSAttributedString.h>

@interface CardRepository(PrivateMethods)
- (NSString *)_fileForCard:(NSNumber *)card;
- (NSString *)_infoFileForCard:(NSNumber *)card;
@end

@implementation CardRepository
+ (BOOL)createCardRepositoryAtPath:(NSString *)aPath
{
    NSFileManager *manager = [NSFileManager defaultManager];
    BOOL           isDir = NO;

    if(![manager fileExistsAtPath:aPath isDirectory:&isDir])
    {
        if(![manager createDirectoryAtPath:aPath attributes:nil])
        {
            NSLog(@"Unable to create repository at path '%@'", aPath);
            return NO;
        }
        else
        {
            NSMutableDictionary *dict;
            NSString            *file;

            dict = [NSMutableDictionary dictionary];
            [dict setObject:[NSNumber numberWithInt:1] forKey:@"nextCardID"];
            [dict setObject:[NSNumber numberWithInt:0] forKey:@"homeCardID"];
            [dict setObject:[NSArray arrayWithObject:[NSNumber numberWithInt:0]]
                        forKey:@"allCards"];

            file = [aPath stringByAppendingPathComponent:@"Info.plist"];
            [dict writeToFile:file atomically:YES];
        }
    }
    else
    {
        NSLog(@"Repository at path '%@' already exists", aPath);
    }


    return YES;
}

- initWithPath:(NSString *)aPath
{
    NSString      *file;
    NSDictionary  *dict;
    BOOL           isDir = NO;
    NSArray       *array;
    
    self = [super init];
    
    path = RETAIN(aPath);

    allCards = [[NSMutableSet alloc] init];

    fileManager = [NSFileManager defaultManager];
    
    if(![fileManager fileExistsAtPath:aPath isDirectory:&isDir])
    {
        NSLog(@"Card repository at path %@ does not exists", path);
        [self dealloc];
        return nil;
    }

    /* read info */
    file = [path stringByAppendingPathComponent:@"Info.plist"];
    if([fileManager fileExistsAtPath:aPath isDirectory:NULL])
    {
        dict = [NSDictionary dictionaryWithContentsOfFile:file];

        nextCardID = [[dict objectForKey:@"nextCardID"] intValue];
        homeCardID = [[dict objectForKey:@"homeCardID"] intValue];
        array = [dict objectForKey:@"allCards"];        
        [allCards addObjectsFromArray:array];
    }


    return self;
}

- (void)dealloc
{
    RELEASE(path);
    [super dealloc];
}

- (id)homeCard
{
    return [NSNumber numberWithInt:homeCardID];
}

- (void)setHomeCard:(id)card
{
    homeCardID = [card intValue];
}

- (id)createCard
{
    NSMutableDictionary *dict = [NSMutableDictionary dictionary];
    NSNumber            *identifier;
    
    NSLog(@"Creating new card with id %i", nextCardID);
    identifier = [NSNumber numberWithInt:nextCardID];
    nextCardID++;

    /* Set default card information */    
    [dict setObject:[NSDate date] forKey:@"creationDate"];
    
    [self setInfo:dict forCard:identifier];

    [allCards addObject:identifier];

    [self commitChanges];
    
    return identifier;
}

- (void)deleteCard:(id)card
{
    NSString *file = [self _fileForCard:card];

    /* FIXME: make sure that the home card is not removed */

    [allCards removeObject:card];

    if(![fileManager removeFileAtPath:file handler:nil])
    {
        NSLog(@"Unable to remove card file %@", file);
    }

    file = [self _infoFileForCard:card];
    if(![fileManager removeFileAtPath:file handler:nil])
    {
        NSLog(@"Unable to remove info file %@", file);
    }

    [self commitChanges];
}

- (NSString *)_fileForCard:(NSNumber *)card
{
    NSString *file;

    file = [path stringByAppendingPathComponent:[card stringValue]];

    return file;
}

- (NSString *)_infoFileForCard:(NSNumber *)card
{
    NSString *file;
    file = [path stringByAppendingPathComponent:[card stringValue]];
    file = [file stringByAppendingPathExtension:@"plist"];
    return file;
}

- (NSData *)contentsForCard:(id)card
{
    NSString *cardFile = [self _fileForCard:card];
    id contents = nil;
    
    if([fileManager fileExistsAtPath:cardFile isDirectory:NULL])
    {
        NSLog(@"Reading contents %@", cardFile);
        contents = [NSUnarchiver unarchiveObjectWithFile:cardFile];

        /* FIXME */
        if(!contents)
        {
            NSLog(@"No contents at '%@'", cardFile);
            return AUTORELEASE([[NSAttributedString alloc] init]);
        }
    }
    else
    {
        NSLog(@"No contents for card %@", cardFile);
        return nil;
    }
    return contents;
}

- (void)setContents:(NSData *)data forCard:(id)card
{
    NSString *cardFile = [self _fileForCard:card];

    [NSArchiver archiveRootObject:data toFile:cardFile];
}
- (NSDictionary *)infoForCard:(id)card
{
    NSString *cardFile = [self _infoFileForCard:card];
    
    if([fileManager fileExistsAtPath:cardFile isDirectory:NULL])
    {
        return [NSDictionary dictionaryWithContentsOfFile:cardFile];
    }
    else
    {
        NSLog(@"No contents for card %@", cardFile);
        return [NSDictionary dictionary];
    }
}

- (void)setInfo:(NSDictionary *)info forCard:(id)card
{
    NSString *cardFile = [self _infoFileForCard:card];

    [info writeToFile:cardFile atomically:YES];
}
- (NSArray *)allCards
{
    return [allCards allObjects];
}
- (void)commitChanges
{
    NSMutableDictionary *dict;
    NSString            *file;

    NSLog(@"Synchronize");
    
    dict = [NSMutableDictionary dictionary];
    [dict setObject:[NSNumber numberWithInt:nextCardID] forKey:@"nextCardID"];
    [dict setObject:[NSNumber numberWithInt:homeCardID] forKey:@"homeCardID"];
    [dict setObject:[allCards allObjects] forKey:@"allCards"];

    NSLog(@"SYNC CARDS %@", allCards);
    file = [path stringByAppendingPathComponent:@"Info.plist"];
    [dict writeToFile:file atomically:YES];
}
@end

/* BEGIN Generated by DevelKit */

@implementation CardRepository (DKGeneratedMethods)
/* Accessor methods */

- (void)dealloc
{

    [super dealloc];
}

/* Encoding methods */
- (void)encodeWithCoder:(NSCoder *)coder
{
    // NSLog(@"Encoding class %@", [self className]);
    if ( [coder allowsKeyedCoding] )
    {
        [coder encodeInt:homeCardID forKey:@"homeCardID"];
        [coder encodeObject:fileManager forKey:@"fileManager"];
        [coder encodeInt:nextCardID forKey:@"nextCardID"];
        [coder encodeObject:path forKey:@"path"];
    }
    else
    {
        [coder encodeValueOfObjCType: @encode(unsigned int) at: &homeCardID];
        [coder encodeValueOfObjCType: @encode(NSFileManager *) at: &fileManager];
        [coder encodeValueOfObjCType: @encode(unsigned int) at: &nextCardID];
        [coder encodeValueOfObjCType: @encode(NSString *) at: &path];
    }
}
- initWithCoder:(NSCoder *)decoder
{
    self = [super init];
    if ( [decoder allowsKeyedCoding] )
    {
        homeCardID = [decoder decodeIntForKey:@"homeCardID"];
        fileManager = [decoder decodeObjectForKey:@"fileManager"];
        nextCardID = [decoder decodeIntForKey:@"nextCardID"];
        path = [decoder decodeObjectForKey:@"path"];
    }
    else
    {
        [decoder decodeValueOfObjCType: @encode(unsigned int) at: &homeCardID];
        [decoder decodeValueOfObjCType: @encode(NSFileManager *) at: &fileManager];
        [decoder decodeValueOfObjCType: @encode(unsigned int) at: &nextCardID];
        [decoder decodeValueOfObjCType: @encode(NSString *) at: &path];
    }


    return self;
}

@end
/* END Generated by DevelKit */
