#ifndef _COMPLEX_H_
#define _COMPLEX_H_


#include "Object.h"
#include <iostream>
#include "ObjectPool.h"
#include "ObjectParser.h"
#include "typetraits.h"
#include <complex>
#include "binio.h"
#include "net_types.h"

using namespace std;



/**
   Base class for Complex<T> numbers.
   \author Dominic Letourneau
   \date 18/02/2004
*/
template <class T>
class Complex : public complex<T>, public Object {

 public:

  ///You can always get the type of the Complex<T> by using typename Complex<T>::basicType.
  typedef complex<T> basicType;
  
  ///Default constructor
  Complex() : complex<T>() {}
  
  ///Constructor with a complex<T>
  Complex(const complex<T> &val) : complex<T>(val) {}

  ///Constructor with a Complex<T>
  Complex(const Complex<T> &val) : complex<T>(val) {}

  ///Constructor with a NetCType<complex<T> >
  Complex(const NetCType<complex<T> > &obj) : complex<T>(obj.val()) {}

  /**
     Formatted output in the FlowDesigner format<br>
     <b>Format : </b> \<Complex\<T\> <i> complex<T> value</i> \>
     \param out the output stream
  */
  void printOn(ostream &out) const {
    out << "<"<<className()<<" ";
    out << *((complex<T>*) this);
    out << ">"<<endl;
  }
  
  /**
     Formatted input in the FlowDesigner format<br>
     <b>Format : </b> \<Complex\<T\> <i> complex<T> value</i> \>
     \param in the input stream
  */
  void readFrom(istream &in) {
    complex<T> value;
    in >> value;
    *this = value;
    char ch;
    in >> ch;
    if (ch != '>') {
      throw new GeneralException(string("Error reading ") + className() + string(" > expected"), __FILE__, __LINE__);
    }
  }
  
  /**
      Binary output in the FlowDesigner format<br>
      <b>Format : </b> {Complex\<T\> |<i>complex<T> value</i> }
      \param out the output stream
  */
  void serialize(ostream &out) const {
    out << "{" << className() << " |";
    BinIO::write(out, (complex<T>*) this, 1);
    out << " }";
  }
  
  /**
      Binary input in the FlowDesigner format<br>
      <b>Format : </b> {Complex\<T\> |<i>complex<T> value</i> }
      \param in the input stream
  */
  void unserialize(istream &in) { 
    complex<T> value;
    BinIO::read(in, &value, 1);
    *this = value;    
    char ch;
    in >> ch;
    if (ch != '}') {
       throw new GeneralException(string("Error reading ") + className() + string(" } expected"), __FILE__, __LINE__);
    }    
  }
  
  /**
     Standard formatted output for Complex<T>
     \param out the output stream
  */
  void prettyPrint(ostream &out) const { 
    out << *((complex<T>*) this);
  }
  
  /**
     Returns the complex<T> wrapped value
     \return complex<T> the complex<T> value wrapped
  */
  complex<T>& val() const {return *((complex<T>*) this);}
  
  
  /**
     Allocate a Complex<T> value from the pool
     \return Complex<T>* the value from the pool
  */
  static Complex<T> *alloc()  {return ObjectPool<Complex<T> >::alloc();}
  
  /**
     Allocat a Complet<T> value from the pool, copying from another Complex<T> value
     \param obj The value to be copied
     \return Complex<T>* a copy of obj
  */
  static Complex<T> *alloc(const Complex<T> &obj)  
  {
    Complex<T> *ret = ObjectPool<Complex<T> >::alloc();
    *ret = obj;
    return ret;
  }

  /**
     Destroy any Complex<T> from memory. This function is used by the ObjectPool<Complex<T> > class.
  */
  void destroy() {ObjectPool<Complex<T> >::release(this);}

  /**
     clone a Complex<T>
  */
  virtual ObjectRef clone() {
    return ObjectRef(Complex<T>::alloc(*(complex<T>*) this));
  }

};

/**
   operator >> for Complex<T>
   \param in input stream
   \param value the Complex<T> value to print
   \return istream the input stream
*/
template <class T> 
istream &operator >> (istream &in, Complex<T> &value) { 
  
  char ch;
  in >> ch;
  
  string expected = ObjectGetClassName<Complex<T> >();
  
  if (ch == '<') {
    string type;
    in >> type;
    
    if (expected != type) {
      throw new ParsingException ("Parser expected type " + expected + " and got " + type);
    }

    //reading object
    value.readFrom(in);
  }
  else if (ch == '{') {
    string type;
    in >> type;

    if (expected != type) {
      throw new ParsingException ("Parser expected type " + expected + " and got " + type);
    }
    
    //reading dummy spaces
    char dummy;      
    do {
      in >> dummy;
    } while(dummy != '|');

    value.unserialize(in);

  } else {
    throw new ParsingException ("Parser expected < or { while parsing type " + expected);
  }

  return in;
}

_DEF_OBJECT_TYPE(Complex<float>)
_DEF_OBJECT_TYPE(Complex<double>)

#endif
