/**********************************************************************
 *
 *   FreeDoko a Doppelkopf-Game
 *    
 *   Copyright (C) 2001-2006  by Diether Knof and Borg Enders
 *
 *   This program is free software; you can redistribute it and/or 
 *   modify it under the terms of the GNU General Public License as 
 *   published by the Free Software Foundation; either version 2 of 
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details. 
 *   You can find this license in the file 'gpl.txt'.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *   MA  02111-1307  USA
 *
 *  Contact:
 *    Diether Knof dknof@gmx.de
 *    Borg Enders  borg@borgsoft.de
 *
 **********************************************************************/

#ifndef CLASS_GAME_HEADER
#define CLASS_GAME_HEADER

#include "reservation.h"
#include "../card/card.h"
class Player;
class Human;
class Trick;
class HandCard;
class HandCards;
class Hand;

class Gameplay;

class Game {
  friend class Party;
  friend class GameSummary;
  friend class Ai;
  friend class WVirtualGames;
  friend class WMonteCarlo;
  friend class Gametree;
  public:
  class AnnouncementWithTrickno {
    public:
      AnnouncementWithTrickno() :
	announcement(ANNOUNCEMENT::NOANNOUNCEMENT),
      trickno(UINT_MAX)
      { }
      AnnouncementWithTrickno(Announcement const announcement,
			      unsigned const trickno) :
	announcement(announcement),
      trickno(trickno)
      { }
      operator Announcement const&() const
      { return this->announcement; }
      Announcement announcement;
      unsigned trickno;
  }; // class AnnouncementWithTrickno

  public:
  // constructor
  Game(class Party& party, Player& startplayer);
  // constructor
  Game(Game const& game, vector<Player*>& players);
  // destructor
  ~Game();

  // checks the data
  bool self_check() const;

  // output of the game
  ostream& write(ostream& ostr) const;
  // output of the tricks
  ostream& write_tricks(ostream& ostr = cout) const;

  // whether this is a virtual game
  bool isvirtual() const;
  // whether this game is finished
  bool is_finished() const;

  // the rule of the game/party
  class Rule const& rule() const;


  // the gametype
  GameType type() const;
  // set the gametype
  void set_type(GameType const type);
  // number of cards of the given tcolor
  unsigned numberof(Card::TColor const& tcolor) const;



  // players

  private:
  // all players
  vector<Player*>& players();
  public:
  // number of players
  unsigned playerno() const;
  unsigned number_of_players() const;
  // number of human players
  unsigned humanno() const;
  // number of the player
  unsigned no_of_player(Player const& player) const;
  // the human player (if there is exactly one)
  Human* human_player();
  Human const* human_player() const;
  // the player
  Player const& player(unsigned const& p) const;
  // the player
  Player& player(unsigned const& p);
  // the following player
  Player const& player_following(Player const& player) const;
  // the following player
  Player& player_following(Player& player);
  // the previous player
  Player const& player_previous(Player const& player) const;
  // for iterating over the players
  vector<Player*>::iterator players_begin();
  // for iterating over the players
  vector<Player*>::iterator players_end();
  // for iterating over the players
  vector<Player*>::const_iterator players_begin() const;
  // for iterating over the players
  vector<Player*>::const_iterator players_end() const;
  // sort the hand of the players
  void players_hands_sort();
  // mix the hand of the players
  void players_hands_mix();
  // the startplayer
  Player const& startplayer() const;
  // the player whose turn it is
  Player const& player_current() const;
  // the player whose turn it is
  Player& player_current();


  // whether it exists a soloplayer
  bool exists_soloplayer() const;
  // the soloplayer
  Player const& soloplayer() const;
  // the soloplayer
  Player& soloplayer();
  /// the player woh takes the poverty
  Player const& povertyPartner() const;


  // the reservation (not for the players, because of the swines)

  // the reservation of the player
  Reservation const& reservation(Player const& player) const;



  // poverty


  // number of shifted cards / returned trumps
  unsigned poverty_cardno() const;
  // shifted cards
  HandCards const* poverty_cards() const;


  // infos of the gameplay

  // number of different cards of the color
  unsigned different_cards_no(Card::TColor const& tcolor) const;
  // the number of trumps
  unsigned trumps_no() const;
  // the number of played trumps
  unsigned trumps_played_no() const;
  // the number of the played cards
  unsigned played_cards_no() const;
  // the number of the card played
  unsigned played_no(Card const& card) const;
  // the trumpcolor
  Card::Color trumpcolor() const;
  // whether the team has got a trick
  bool hastrick(TEAM::Team const& team) const;
  // whether the player has got a trick
  bool hastrick(Player const& player) const;
  // the number of tricks the player has got
  unsigned numberoftricks_of_player(Player const& player) const;
  // the number of tricks the player til the trick 'trickno' has got
  unsigned numberoftricks_of_player(Player const& player,
				    unsigned trickno) const;
  // the number of tricks the team has got
  unsigned numberoftricks_of_team(Team const& team) const;
  // the number of points the player has got
  unsigned points_of_player(Player const& player) const;
  // the number of points the player til the trick 'trickno' has got
  unsigned points_of_player(Player const& player,
			    unsigned trickno) const;
  // the points, the team has got
  unsigned points_of_team(Team const& team) const;
  // the winnerteam
  Team winnerteam() const;


  // announcements

  // the announcement of the team
  AnnouncementWithTrickno announcement_of_team(Team const& team) const;
  // the announcement of the player
  AnnouncementWithTrickno const& announcement_of_player(Player const& player) const;
  // all announcements of the player
  vector<AnnouncementWithTrickno> const& announcements_of_player(Player const& player) const;
  // set the announcement of the player
  void announcement_make(Announcement announcement,
			 Player const& player);
  // test, whether the announcement is valid
  bool announcement_valid(Announcement const& announcement,
			  Player const& player) const;
  // whether this is the last possibility to announce the announcement
  bool last_chance_to_announce(Announcement const& announcement,
			       Player const& player) const;


  // tricks

  private:
  // all tricks
  vector<Trick*>& tricks();
  public:
  // the number of tricks in the game
  unsigned trickno() const;
  // the trick nmber 'i'
  Trick const& trick(unsigned const& i) const;
  // the current trick
  Trick const& trick_current() const;
  // the current trick
  Trick& trick_current();
  // the number of the current trick
  unsigned trick_current_no() const;
  // the number of the current trick
  unsigned real_trick_current_no() const;
  // the number of remaining tricks
  unsigned tricks_remaining_no() const;
  // the number of remaining tricks
  unsigned real_tricks_remaining_no() const;
  // whether this is the last trick
  bool is_last_trick() const;


  // teaminfo (known to all

  // the global known team of the player
  Team teaminfo(Player const& player) const;
  // the known team for all humans
  Team teaminfo_for_humans(Player const& player) const;
  // set the teaminfo of 'player' to 'team'
  void set_teaminfo(Player const& player, Team const& team);

  // set the teaminfo for the gamestart
  void teaminfo_set_at_gamestart();
  // update the team info
  void teaminfo_update();
  // update the team info for the humans
  void human_teaminfo_update();


  // team (real team)

  // the team of the player
  Team team(Player const& player) const;



  // swines and hyperswines

  // the player who has swines
  Player const* swines_owner() const;
  // whether the player can announce swines
  bool swines_announcement_valid(Player const& player) const;
  // announce swines
  bool swines_announce(Player& player);
  // the player who has hyperswines
  Player const* hyperswines_owner() const;
  // whether the player can announce swines
  bool hyperswines_announcement_valid(Player const& player) const;
  // announce hyperswines
  bool hyperswines_announce(Player& player);
  private:
  // test swines from the reservations
  void test_swines_from_reservations();
  public:


  // gameplay

  // play the game
  void play();
  private:
  // the party, the game is part of
  class Party& party();

  // request announcements from the players
  void announcements_request();
  // check, whether a genscher is played
  void check_for_genscher(HandCard const& played_card);
  // check, whether the partner is now found
  void determine_marriage();

  // init the game
  void init();
  // make the shifting of the poverty cards
  void poverty_shift();
  // distribute the cards
  void distributecards();
  protected:
  // the next player has to play a card
  void nextplayer();
  // evaluate the full trick
  void evaluatetrick();
  private:
  // finish the game
  void finish();

  private:
  // party, the game belongs to
  PRIV_VAR_P_R(class Party, party);

  // the gameplay saved as actions
  PRIV_VAR_P_R(Gameplay, gameplay);

  // players in this game 
  PRIV_VAR_R(vector<Player*>, players);
  // the startplayer of the game
  PRIV_VAR_CP(Player, startplayer);
  // the player whose turn it is
  PRIV_VAR_P(Player, player_current);

  // Seed for the distribution of the cards
  PRIV_VAR_RW(unsigned, seed);
  // type of the game
  PRIV_VAR(GameType, type);
  // player, who is playing a solo (NULL, if no one)
  PRIV_VAR_P(Player, soloplayer);
  // which trick selects the team
  PRIV_VAR_R(MarriageSelector, marriage_selector);
  // in which trick the marriage has been determined
  PRIV_VAR_R(unsigned, marriage_determination_trickno);

  // the shifted cards in the poverty;
  PRIV_VAR_P(HandCards, poverty_cards);
  // the number of shifted cards
  PRIV_VAR(unsigned, poverty_cardno);
  // whether the poverty cards were shifted
  PRIV_VAR_R(bool, poverty_shifted);


  // reservations of the players
  PRIV_VAR(vector<Reservation>, reservation);

  // tricks in this game
  PRIV_VAR_R(vector<Trick*>, tricks);

  // Team of the players (information, known to all)
  PRIV_VAR(vector<Team>, teaminfo);
  // Team of the players (information, known to all humans)
  PRIV_VAR(vector<Team>, human_teaminfo);
  // Announcement of the players (information, known to all)
  PRIV_VAR(vector<vector<AnnouncementWithTrickno> >, announcement);
  PRIV_VAR_R(AnnouncementWithTrickno, announcement_replied_on);

  //card::is... 

  // the player who has two fox and has made the reservation (for game)
  PRIV_VAR_P(Player, swines_owner);
  // the player, who has caught the first fox
  PRIV_VAR_CP_PR(Player, first_fox_catcher);
  // whether the swines has been announced
  PRIV_VAR_R(bool, swines_announced);

  // the player who has two trump nines and has made the reservation (for game)
  PRIV_VAR_P(Player, hyperswines_owner);
  // whether the swines has been announced
  PRIV_VAR_R(bool, hyperswines_announced);

  // whether the game is finished
  PRIV_VAR_R(bool, finished);

  private: // not to be used  
  Game();
  Game(Game const& game);
  Game& operator=(Game const& g);
}; // class Game

// output of the game
ostream& operator<<(ostream& ostr, Game const& game);

// comparison
bool operator==(Game const& game1, Game const& game2);
bool operator!=(Game const& game1, Game const& game2);

#endif // #ifndef CLASS_GAME_HEADER
