/**********************************************************************
 *
 *   FreeDoko a Doppelkopf-Game
 *    
 *   Copyright (C) 2001-2006  by Diether Knof and Borg Enders
 *
 *   This program is free software; you can redistribute it and/or 
 *   modify it under the terms of the GNU General Public License as 
 *   published by the Free Software Foundation; either version 2 of 
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details. 
 *   You can find this license in the file 'gpl.txt'.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *   MA  02111-1307  USA
 *
 *  Contact:
 *    Diether Knof dknof@gmx.de
 *    Borg Enders  borg@borgsoft.de
 *
 **********************************************************************/

#include "constants.h"

#ifdef USE_UI_TEXT

#include "settings.h"

#include "../../misc/language.h"
#include "../../misc/setting.h"
#include "../../utils/string.h"

namespace UI_TEXT_NS {

  /**
   **
   ** constructor
   **
   ** @param	ui	pointer to the ui
   **
   ** @return	-
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  UI_Text::Settings::Settings(UI_Text* const ui) :
    ui(ui)
    {
      DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_INIT,
		    "UI_Text::Settings::Settings(ui)");

      DEBUG_RETURNING(VOID,
		      INFO_UI_SETTINGS && INFO_UI_INIT,
		      "UI_Text::Settings::Settings(ui)");
    } // UI_Text::Settings::Settings(UI_Text* const ui)

  /**
   **
   ** destructor
   **
   ** @param	-
   **
   ** @return	-
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  UI_Text::Settings::~Settings()
  {
    DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_INIT,
		  "UI_Text::Settings::~Settings()");

    DEBUG_RETURNING(VOID,
		    INFO_UI_SETTINGS && INFO_UI_INIT,
		    "UI_Text::Settings::~Settings()");
  } // UI_Text::Settings::~Settings()

  /**
   **
   ** Enter the settings subsystem.
   ** The subsystem can be left with 'leave'
   **
   ** @param	-
   **
   ** @return	-
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   ** @todo	all
   **
   **/
  void
  UI_Text::Settings::enter()
  {
    DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_SUBSYSTEM,
		  "UI_Text::Settings::enter()");



    DEBUG_RETURNING(VOID,
		    INFO_UI_SETTINGS && INFO_UI_SUBSYSTEM,
		    "UI_Text::Settings::enter()");
  } // void UI_Text::Settings::enter()

  /**
   **
   ** interprets the line read by the ui
   **
   ** @param	-
   **
   ** @return	whether the line was interpreted
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  bool
    UI_Text::Settings::interpret_line()
    {
      DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_EVENT,
		    "UI_Text::Settings::interpret_line()");

      // whether the line has been interpreted
      bool interpreted = false;

      if (this->ui->iskeyword("settings")) {
	interpreted = this->interpret_command("show");
      } else if ((this->ui->first_iskeyword("setting"))
		 || (this->ui->first_iskeyword("settings"))) {
	string name = this->ui->line;
	DK::Utils::String::word_first_remove_with_blanks(name);
	interpreted = this->interpret_command(name);
      } // if (this->ui->iskeyword(...)

      DEBUG_RETURNING(interpreted,
		      INFO_UI_SETTINGS && INFO_UI_EVENT,
		      "UI_Text::Settings::interpret_line()");
    } // bool UI_Text::Settings::interpret_line()

  /**
   **
   ** interprets the given command
   **
   ** @param	command	command that shall be interpreted
   **
   ** @return	whether the command was interpreted
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  bool
    UI_Text::Settings::interpret_command(string const& command)
    {
      DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_EVENT,
		    "UI_Text::Settings::interpret_(command)");

      // whether the line has been interpreted
      bool interpreted = false;

      if (this->ui->iskeyword(command, "show")) {
	this->show();
	interpreted = true;
      } else if (this->ui->iskeyword(command, "load")) {
	// load the settings
	::setting.load();
	interpreted = true;
      } else if (this->ui->iskeyword(command, "save")) {
	// save the settings
	if (::setting.save())
	  this->ui->ostr() << ::language("Settings saved.");
	interpreted = true;
      } else {
	// test for a type

	if (command.find('=') == string::npos) {
	  int const type = ::setting.type_translated_or_normal(command);
	  if (type != -1) {
	    this->show(type);
	    interpreted = true;
	  } // if (type != -1)
	} else { // if !(command.find('=') == string::npos)
	  // a setting shall be changed
	  string name = command;
	  string value = string(name, name.find('=') + 1);
	  name.erase(name.find('='));
	  DK::Utils::String::remove_blanks(name);
	  DK::Utils::String::remove_blanks(value);
	  COUT << "name = " << name << endl;
	  COUT << "value = " << value << endl;
	  int const type = ::setting.type_translated_or_normal(name);
	  if (type != -1) {
	    ::setting.set(name, value);
	    interpreted = true;
	  } // if (type != -1)
	} // if !(command.find('=') == string::npos)
      } // if (this->ui->iskeyword(...)

      DEBUG_RETURNING(interpreted,
		      INFO_UI_SETTINGS && INFO_UI_EVENT,
		      "UI_Text::Settings::interpret_(command)");
    } // bool UI_Text::Settings::interpret_command(string const& command)

  /**
   **
   ** show the settings
   **
   ** @param	-
   **
   ** @return	-
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  void
    UI_Text::Settings::show() const
    {
      DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_SHOW,
		    "UI_Text::Settings::show()");

      this->ui->ostr() << ::language("Settings") << ":\n"
	<< "{\n";
      for (int t = Setting::BOOL_FIRST; t <= Setting::BOOL_LAST; t++)
	this->show(t);
      for (int t = Setting::UNSIGNED_FIRST; t <= Setting::UNSIGNED_LAST; t++)
	this->show(t);
      for (int t = Setting::STRING_FIRST; t <= Setting::STRING_LAST; t++)
	this->show(t);
      for (int t = Setting::CARDS_ORDER_FIRST; t <= Setting::CARDS_ORDER_LAST; t++)
	this->show(t);

      DEBUG_RETURNING(VOID,
		      INFO_UI_SETTINGS && INFO_UI_SHOW,
		      "UI_Text::Settings::show()");
    } // void UI_Text::Settings::show() const

  /**
   **
   ** show the given setting
   **
   ** @param	type	type to show
   **
   ** @return	-
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  void
    UI_Text::Settings::show(int const type) const
    {
      DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_SHOW,
		    "UI_Text::Settings::show(type)");

      if ((type >= Setting::BOOL_FIRST)
	  && (type <= Setting::BOOL_LAST))
	this->ui->ostr() << "  "
	  << setw(32) << ::language(::name(static_cast<Setting::TypeBool>(type)))
	  << " = " << ::setting(static_cast<Setting::TypeBool>(type))
	  << '\n';
      else if ((type >= Setting::UNSIGNED_FIRST)
	       && (type <= Setting::UNSIGNED_LAST))
	this->ui->ostr() << "  "
	  << setw(32) << ::language(::name(static_cast<Setting::TypeUnsigned>(type)))
	  << " = " << ::setting(static_cast<Setting::TypeUnsigned>(type))
	  << '\n';
      else if ((type >= Setting::STRING_FIRST)
	       && (type <= Setting::STRING_LAST))
	this->ui->ostr() << "  "
	  << setw(32) << ::language(::name(static_cast<Setting::TypeString>(type)))
	  << " = " << ::setting(static_cast<Setting::TypeString>(type))
	  << '\n';
      else if ((type >= Setting::CARDS_ORDER_FIRST)
	       && (type <= Setting::CARDS_ORDER_LAST))
	this->ui->ostr() << "  "
	  << setw(32)
	  << ::language(::name(static_cast<Setting::TypeCardsOrder>(type)))
	  << " = " << ::setting(static_cast<Setting::TypeCardsOrder>(type))
	  << '\n';
      else
	this->ui->ostr() << ::language("Setting")
	  << ": "
	  << "type '%itype%' not found."
	  << '\n';

      DEBUG_RETURNING(VOID,
		      INFO_UI_SETTINGS && INFO_UI_SHOW,
		      "UI_Text::Settings::show(type)");
    } // void UI_Text::Settings::show(int const type) const

  /**
   **
   ** show the given setting
   **
   ** @param	name	name of the type
   **
   ** @return	-
   **
   ** @version	0.5.4
   **
   ** @author	Diether Knof
   **
   **/
  void
    UI_Text::Settings::show(string const& name) const
    {
      DEBUG_CALLING(INFO_UI_SETTINGS && INFO_UI_SHOW,
		    "UI_Text::Settings::show(name)");

      int const type = ::setting.type_translated_or_normal(name);

      if (type != -1)
	this->show(type);
      else
	this->ui->ostr() << ::language("Setting '%sname%' not found.",
				       name.c_str())
	  << '\n';

      DEBUG_RETURNING(VOID,
		      INFO_UI_SETTINGS && INFO_UI_SHOW,
		      "UI_Text::Settings::show(name)");
    } // void UI_Text::Settings::show(string const& name) const

} // namespace UI_TEXT_NS

#endif // #ifdef USE_UI_TEXT
