/**********************************************************************
 *
 *   FreeDoko a Doppelkopf-Game
 *    
 *   Copyright (C) 2001-2006  by Diether Knof and Borg Enders
 *
 *   This program is free software; you can redistribute it and/or 
 *   modify it under the terms of the GNU General Public License as 
 *   published by the Free Software Foundation; either version 2 of 
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details. 
 *   You can find this license in the file 'gpl.txt'.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *   MA  02111-1307  USA
 *
 *  Contact:
 *    Diether Knof dknof@gmx.de
 *    Borg Enders  borg@borgsoft.de
 *
 *********************************************************************/

#include "constants.h"

#ifdef USE_UI_GTKMM

#include "gametype_info.h"
#include "table.h"
#include "ui.h"
#include "translations.h"
#include "icons.h"
#include "main_window.h"
#include "menu.h"

#include "../../party/rule.h"
#include "../../game/game.h"
#include "../../player/player.h"
#include "../../misc/setting.h"

#include "../../utils/string.h"

#include <gtkmm/box.h>
#include <gtkmm/image.h>
#include <gtkmm/label.h>

namespace UI_GTKMM_NS {

  /**
   **
   ** Constructor
   **
   ** @param	table	the table
   **
   **
   ** @return	-
   ** 
   ** @author	Diether Knof
   **
   ** @version	0.6.1
   **
   **/
  GametypeInfo::GametypeInfo(Table* const table) :
    GameinfoDialog(table),
    icon(NULL),
    text(NULL),
    swines_announce_button(NULL),
    hyperswines_announce_button(NULL)
  {
    return ;
  } // GametypeInfo::GametypeInfo(Table* const table)

  /**
   **
   ** Destruktor
   **
   ** @param	-
   **
   ** @return	-
   ** 
   ** @author	Diether Knof
   **
   ** @version	0.6.1
   **
   **/
  GametypeInfo::~GametypeInfo()
  {
    return ;
  } // GametypeInfo::~GametypeInfo()

  /**
   **
   ** init the dialog
   **
   ** @param	-
   **
   ** @return	-
   ** 
   ** @author	Diether Knof
   **
   ** @version	0.6.1
   **
   **/
  void
    GametypeInfo::init()
    {
      this->GameinfoDialog::init();

      this->ui->translations->add(*this, ::translation("~gametype"));

      this->icon = this->ui->icons->new_managed_image();
      this->text = Gtk::manage(new Gtk::Label("Gametype"));

      Gtk::VBox* vbox = Gtk::manage(new Gtk::VBox(false, 2 ex));
      vbox->set_border_width(2 ex);

      vbox->add(*(this->icon));
      vbox->add(*(this->text));

      this->get_vbox()->add(*vbox);

      this->swines_announce_button
	= Gtk::manage(new Gtk::Button("announce swines"));
      this->ui->translations->add(*this->swines_announce_button,
				  ::translation("swines"));
      this->get_action_area()->add(*this->swines_announce_button);
      this->get_action_area()->reorder_child(*this->swines_announce_button,
					     0);

      this->hyperswines_announce_button
	= Gtk::manage(new Gtk::Button("announce hyperswines"));
      this->ui->translations->add(*this->hyperswines_announce_button,
				  ::translation("hyperswines"));
      this->get_action_area()->add(*this->hyperswines_announce_button);
      this->get_action_area()->reorder_child(*this->hyperswines_announce_button,
					     1);


      this->swines_announce_button->signal_clicked().connect(sigc::bind<unsigned const>(sigc::mem_fun(*this->ui->main_window->menu, &Menu::swines_announcement_event), UINT_MAX));
      this->swines_announce_button->signal_clicked().connect(sigc::mem_fun(*this, &GameinfoDialog::close));
      this->hyperswines_announce_button->signal_clicked().connect(sigc::bind<unsigned const>(sigc::mem_fun(*this->ui->main_window->menu, &Menu::hyperswines_announcement_event), UINT_MAX));
      this->hyperswines_announce_button->signal_clicked().connect(sigc::mem_fun(*this, &GameinfoDialog::close));

      this->show_all_children();

      this->language_update();

      return ;
    } // void GametypeInfo::init()

  /**
   **
   ** show the window
   **
   ** @param	-
   **
   ** @return	-
   ** 
   ** @author	Diether Knof
   **
   ** @version	0.6.1
   **
   **/
  void
    GametypeInfo::show_information()
    {
      Game const& game = this->ui->game();

      if (!this->is_realized())
	this->realize();

      // If a color solo is played the human player has got to have a chance
      // to announce swines and hyperswines.
      this->swines_announce_button->hide();
      this->hyperswines_announce_button->hide();
      if (GAMETYPE::is_color_solo(game.type())) {
	for (vector<Player*>::const_iterator p = game.players_begin();
	     p != game.players_end();
	     p++) {
	  if (**p == game.soloplayer())
	    continue;
	  if ((*p)->type() == Player::HUMAN) {
	    if (game.rule()(Rule::SWINES_ANNOUNCEMENT_BEGIN)
		&& game.swines_announcement_valid(**p))
	      this->swines_announce_button->show();
	    if (game.rule()(Rule::HYPERSWINES_ANNOUNCEMENT_BEGIN)
		&& game.hyperswines_announcement_valid(**p))
	      this->hyperswines_announce_button->show();
	  } // if (((*p)->type() == Player::HUMAN)
	} // for (p \in game.players)
      } // if (GAMETYPE::is_color_solo(game.type())) {

      this->language_update();
      this->ui->translations->change(*this, ::translation(game.type()));
      this->ui->icons->change_managed(this->icon, game.type());

      this->GameinfoDialog::show_information(::setting(Setting::SHOW_GAMETYPE_WINDOW),
					     ::setting(Setting::CLOSE_GAMETYPE_WINDOW_AUTOMATICALLY),
					     ::setting(Setting::GAMETYPE_WINDOW_CLOSE_DELAY));

      this->ui->icons->change_managed(this->icon, Icons::NONE);

      return ;
    } // void GametypeInfo::show_information()

    /**
     **
     ** the name of 'player' has changed
     **
     ** @param	player	the player whose name has changed
     **
     ** @return	-
     ** 
     ** @author	Diether Knof
     **
     ** @version	0.6.1
     **
     **/
    void
      GametypeInfo::name_changed(Player const& player)
      {
	if ((::game_status != GAMESTATUS::GAME_RESERVATION)
	    && (::game_status != GAMESTATUS::GAME_REDISTRIBUTE))
	  return ;

	if (this->ui->game().type() != GAMETYPE::NORMAL)
	  if (player.is_same(this->ui->game().soloplayer()))
	    this->language_update();

	return ;
      } // void GametypeInfo::name_changed(Player const& player)

    /**
     **
     ** update the language
     **
     ** @param	-
     **
     ** @return	-
     ** 
     ** @author	Diether Knof
     **
     ** @version	0.6.1
     **
     **/
    void
      GametypeInfo::language_update()
      {
	if ((::game_status != GAMESTATUS::GAME_RESERVATION)
	    && (::game_status != GAMESTATUS::GAME_REDISTRIBUTE))
	  if (this->ui->game().type() != GAMETYPE::NORMAL)
	    return ;

	if (!this->is_realized())
	  return ;

	Game const& game = this->ui->game();

	switch (this->ui->game().type()) {
	case GAMETYPE::NORMAL:
	  this->text->set_text(::translation(game.type()).translation());
	  break;
	case GAMETYPE::THROWN_NINES:
	  this->text->set_text(::translation("%splayer% has %tnumber% nines.",
					     game.soloplayer().name(),
					     ::translation(DK::Utils::String::to_string(game.soloplayer().hand().numberofnines()))
					    ).translation()
			      );
	  break;
	case GAMETYPE::THROWN_KINGS:
	  this->text->set_text(::translation("%splayer% has %tnumber% kings.",
					     game.soloplayer().name(),
					     ::translation(DK::Utils::String::to_string(game.soloplayer().hand().numberofkings()))
					    ).translation()
			      );
	  break;
	case GAMETYPE::THROWN_NINES_AND_KINGS:
	  this->text->set_text(::translation("%splayer% has %tnumber% nines and kings.",
					     game.soloplayer().name(),
					     ::translation(DK::Utils::String::to_string(game.soloplayer().hand().numberofnines() + game.soloplayer().hand().numberofkings()))
					    ).translation()
			      );
	  break;
	case GAMETYPE::FOX_HIGHEST_TRUMP:
	  this->text->set_text(::translation("%splayer% has %tgametype%.",
					     game.soloplayer().name(),
					     ::translation(game.type())
					    ).translation()
			      );
	  break;
	case GAMETYPE::POVERTY:
	  // error
	  this->text->set_text(::translation("%splayer% has a %tgametype%.",
					     game.soloplayer().name(),
					     ::translation(game.type())
					    ).translation()
			      );
	  if (::game_status != GAMESTATUS::GAME_REDISTRIBUTE) {
	    for (vector<Player*>::const_iterator player
		 = game.players_begin();
		 player != game.players_end();
		 player++) {
	      if ((**player != game.soloplayer())
		  && ((*player)->team() == TEAM::RE)) {
		string text = this->text->get_text();
#ifdef POSTPHONED
		text +=  ("        \n        "
			  + ::translation("Partner is %splayer%.",
					  (*player)->name()));
#endif
		if (game.soloplayer().hand().numberoftrumps() == 1)
		  text +=  ("        \n        "
			    + ::translation("%splayer% has returned %utrumpno% trump.",
					    (*player)->name(),
					    game.soloplayer().hand().numberoftrumps()
					   ).translation());
		else
		  text +=  ("        \n        "
			    + ::translation("%splayer% has returned %utrumpno% trumps.",
					    (*player)->name(),
					    game.soloplayer().hand().numberoftrumps()
					   ).translation());
		this->text->set_text(text);

	      } // if (player is the partner)
	    } // for (player)
	  } // if (::game_status != GAME_REDISTRIBUTE)
	  break;
	case GAMETYPE::GENSCHER:
	  // cannot happen
	  break;
	case GAMETYPE::MARRIAGE:
	case GAMETYPE::MARRIAGE_SOLO:
	  this->text->set_text(::translation("%splayer% has a %tgametype%: %tselector%.",
					     game.soloplayer().name(),
					     ::translation(game.type()),
					     ::translation(game.marriage_selector())
					    ).translation()
			      );
	  break;
	case GAMETYPE::MARRIAGE_SILENT:
	  // cannot happen
	  break;
	case GAMETYPE::SOLO_MEATLESS:
	case GAMETYPE::SOLO_JACK:
	case GAMETYPE::SOLO_QUEEN:
	case GAMETYPE::SOLO_KING:
	case GAMETYPE::SOLO_QUEEN_JACK:
	case GAMETYPE::SOLO_KING_JACK:
	case GAMETYPE::SOLO_KING_QUEEN:
	case GAMETYPE::SOLO_KOEHLER:
	case GAMETYPE::SOLO_CLUB:
	case GAMETYPE::SOLO_SPADE:
	case GAMETYPE::SOLO_HEART:
	case GAMETYPE::SOLO_DIAMOND:
	  this->text->set_text(::translation("%splayer% plays a %tgametype%.",
					     game.soloplayer().name(),
					     ::translation(game.type())
					    ).translation()
			      );
	  break;
	} // switch (game.type())

	this->GameinfoDialog::language_update();

	return ;
      } // void GametypeInfo::language_update()
    } // namespace UI_GTKMM_NS

#endif // #ifdef USE_UI_GTKMM
