#include "BCobject.h"

/*
    Glasteroids, a asteroids type game.
    Copyright (C) 1999 Matt Cohen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


BCobject::BCobject ( void )
{
    listNumber = glGenLists ( 1 );

    //printf("Entered BCobject\n");
    //cube = NULL;
    
    active = 0;
    
    radius = 0.0;
    
    translate = 0;
    xLoc = 0.0;
    yLoc = 0.0;
    zLoc = 0.0;
    
    xSpeed = 0.0;
    ySpeed = 0.0;
    zSpeed = 0.0;
    
    rotate = 0;
    xRot = 0.0;
    yRot = 0.0;
    zRot = 0.0;
    
    xRotSpeed = 0.0;
    yRotSpeed = 0.0;
    
    mass = 0.0;
    oneOverM = 0.0;
    
    color = 0;
    colorRed   = 0.0;
    colorGreen = 0.0;
    colorBlue  = 0.0;
}

BCobject::~BCobject ( void )
{
    glDeleteLists ( listNumber, 1 );
    //delete cube;
}


void BCobject::CreateGeometric ( enum BcObjectArgs object, ... )
{
    float red = 1.0;
    float green = 1.0;
    float blue = 1.0;
    int readingArgs = 1;
    int addColor = 0;
    GLdouble size;
    va_list ap;
    
    // Begin variable arguments
    va_start ( ap, object );
    
    // Continue while reading arguements
    while ( readingArgs )
    {
	switch ( va_arg ( ap, enum BcObjectArgs ) )
	{
	    case BcNULL:
		readingArgs = 0;
		break;
	    case BcSIZE:
		size = (GLdouble)va_arg ( ap, double );
		break;
	    case BcCOLOR:
		addColor = 1;
		red = (float)va_arg ( ap, double );
		green = (float)va_arg ( ap, double );
		blue = (float)va_arg ( ap, double );
		break;
	    default:
		cout << "Warning!  Incorrect usage of BCworld::AddBCObject().\n";
		readingArgs = 0;
		break;
	}
    }
    
    radius = (double)size;
    // Compute object mass
    mass = (4.0/3.0)*M_PI*size*size*size;
    oneOverM = 1.0/mass;
    
    glNewList ( listNumber, GL_COMPILE );
    
    if ( addColor )
	glColor3f ( red, green, blue );
    
    switch ( object )
    {
	case BcSPHERE:
	    glutSolidSphere ( size, 10, 10 );
	    break;
	case BcTEAPOT:
	    glutSolidTeapot ( size );
	    break;
	case BcTORUS:
	    cout << "BCtorus not yet implemented\n";
	    break;
	case BcCUBE:
	    glutSolidCube ( size );
	    break;
	case BcCONE:
	    cout << "BCcone not yet implemented\n";
	    break;
	default:
	    cout << "Unknown object type\n";
	    break;
    }
    
    glEndList ( );
}


void BCobject::CreateIsosurface ( unsigned char *data, MarchingCubes *mcubes, int xmax, int ymax, int zmax, ... )
{
    unsigned char *cubeIndex;
    int x, y, z;
    int xval, yval, zval;
    int addColor = 0;
    float scaleX = 1.0;
    float scaleY = 1.0;
    float scaleZ = 1.0;
    float red = 1.0;
    float green = 1.0;
    float blue = 1.0;
    int readingArgs = 1;
    va_list ap;

    cube = mcubes;
    
    // Begin variable arguments
    va_start ( ap, zmax );
    
    // Continue while reading arguements
    while ( readingArgs )
    {
	switch ( va_arg ( ap, enum BcObjectArgs ) )
	{
	    case BcNULL:
		readingArgs = 0;
		break;
	    case BcSCALE:
		scaleX = (float)va_arg ( ap, double );
		scaleY = (float)va_arg ( ap, double );
		scaleZ = (float)va_arg ( ap, double );
		break;
	    case BcCOLOR:
		addColor = 1;
		red = (float)va_arg ( ap, double );
		green = (float)va_arg ( ap, double );
		blue = (float)va_arg ( ap, double );
		break;
	    default:
		cout << "Warning!  Incorrect usage of BCobject::CreateIsosurface().\n";
		readingArgs = 0;
		break;
	}
    }
    
    // End of variable list
    va_end ( ap );

    xval = xmax - 1;
    yval = ymax - 1;
    zval = zmax - 1; 
    
    cubeIndex = new unsigned char[yval*yval*zval];
    
    memset ( cubeIndex, 0, xval*yval*zval );

    for ( y = 0; y < yval; y++ )
	for ( z = 0; z < zval; z++ )
	    for ( x = 0; x < xval; x++ )
	    {
		if ( data[(y  )*ymax*xmax + (z  )*xmax + (x  )] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT1;

		if ( data[(y  )*ymax*xmax + (z  )*xmax + (x+1)] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT2;

		if ( data[(y  )*ymax*xmax + (z+1)*xmax + (x+1)] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT3;

		if ( data[(y  )*ymax*xmax + (z+1)*xmax + (x  )] )
                    cubeIndex[y*zval*xval + z*xval + x] |= BIT4;

		if ( data[(y+1)*ymax*xmax + (z  )*xmax + (x  )] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT5;

		if ( data[(y+1)*ymax*xmax + (z  )*xmax + (x+1)] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT6;

		if ( data[(y+1)*ymax*xmax + (z+1)*xmax + (x+1)] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT7;

		if ( data[(y+1)*ymax*xmax + (z+1)*xmax + (x  )] )
		    cubeIndex[y*zval*xval + z*xval + x] |= BIT8;
	    }

    BuildObject ( cubeIndex, 
		  xval, yval, zval, 
		  scaleX, scaleY, scaleZ,
		  addColor, red, green, blue );
    
    delete cubeIndex;
}


void BCobject::BuildObject ( unsigned char *cubeIndex,
                             int xmax, int ymax, int zmax,
                             float scaleX, float scaleY, float scaleZ,
                             int addColor, float red, float green, float blue )
{
    int l, m, val;
    float fx, fy, fz, tx, ty, tz;
    int numTris = 0;
    unsigned char cell;
    float nx, ny, nz, cx, cy, cz;
    int special;
    int count = 0;
    int x, y, z;
    float centerX, centerY, centerZ;
    Cell_Triangle_t tris[30];
    double total_radius = 0.0;
    double temp_radius;
    int radius_count = 0;

    centerX = (float)xmax/2.0;
    centerY = (float)ymax/2.0;
    centerZ = (float)zmax/2.0;
    
    glNewList ( listNumber, GL_COMPILE );
    
    if ( addColor )
    {
	glColor3f ( red, green, blue );
    }
    
    glBegin(GL_TRIANGLES);	
    
    for ( y = 0; y < ymax; y++ )
	for ( z = 0; z < zmax; z++ )
	    for ( x = 0; x < xmax; x++ )
	    {
		special = GetCubePolygons ( cubeIndex, tris, &numTris, 
					    x, y, z, xmax, ymax, zmax );
		
		//cout << "numTris = " << numTris << endl;
		//cout << "Building triangle #" << count << endl;
		
		for ( l = 0; l < numTris; l++ )
		    for ( m = 0; m < 3; m++ )
		    {
			if (m == 0) val = tris[l].a;
			else if (m == 1) val = tris[l].b;
			else if (m == 2) val = tris[l].c;
			
			// Translate x, y, and z so the 
			// object is centered at the origin
			cx = x - centerX;
			cy = y - centerY;
			cz = z - centerZ;
			
			switch ( val )
			{
			    case 0:  fx = cx+0.5; fy = cy;     fz = cz;     break;
			    case 1:  fx = cx+1.0; fy = cy;     fz = cz+0.5; break;
			    case 2:  fx = cx+0.5; fy = cy;     fz = cz+1.0; break;
			    case 3:  fx = cx;     fy = cy;     fz = cz+0.5; break;
			    case 4:  fx = cx+0.5; fy = cy+1.0; fz = cz;     break;
			    case 5:  fx = cx+1.0; fy = cy+1.0; fz = cz+0.5; break;
			    case 6:  fx = cx+0.5; fy = cy+1.0; fz = cz+1.0; break;
			    case 7:  fx = cx;     fy = cy+1.0; fz = cz+0.5; break;
			    case 8:  fx = cx;     fy = cy+0.5; fz = cz;     break;
			    case 9:  fx = cx+1.0; fy = cy+0.5; fz = cz;     break;
			    case 10: fx = cx+1.0; fy = cy+0.5; fz = cz+1.0; break;
			    case 11: fx = cx;     fy = cy+0.5; fz = cz+1.0; break;
			}
			
			tx = ( fx / (float)xmax );
			ty = ( fy / (float)ymax );
			tz = ( fz / (float)zmax );

#ifdef GL_TEXTURE_3D_EXT
			glTexCoord3f ( tx,ty,tz );
#else	
			glTexCoord2f ( tx,ty );
#endif
			
			fx *= scaleX;
			fy *= scaleY;
			fz *= scaleZ;
			
			// Compute the objects radius
			temp_radius = sqrt ( fx*fx + fy*fy + fz*fz );
			total_radius += temp_radius;
			radius_count++;

			cell = cubeIndex[y*zmax*xmax + z*xmax + x]; 
			
			cube->FindNormalForMidpoint ( cubeIndex, val, cell,
                                                     special, x, y, z,
                                                     xmax, ymax, zmax,
                                                     &nx, &ny, &nz );
			
			//cout << "x = " << x << "\ny = " << y 
			//     << "\nz = " << z << endl;
			
			//cout << "nx = " << nx << "\nny = " 
			//     << ny << "\nnz = " << nz << endl << endl;
			

			glNormal3f ( nx, ny, nz );
			glVertex3f ( fx, fy, fz );

                    }
		
		count++;
	    }
    
    glEnd ( );   

    radius = total_radius/(double)radius_count;
    mass = (4.0/3.0)*M_PI*radius*radius*radius;
    oneOverM = 1.0/mass;
    

    
    //cout << "Object #" << NumObjects << " has " << count << " triangles\n";
    glEndList ( );
}


int BCobject::GetCubePolygons ( unsigned char *cubeIndex, 
                                Cell_Triangle_t *tris, 
                                int *numTriangles, int x, int y, int z,
                                int xmax, int ymax, int zmax )
{
    int num_tris, i, retval = 0;
    unsigned char cube0, cube1, cube2, cube3, cube4, cube5, cube6;
    
    cube0 = cubeIndex[(y  )*zmax*xmax + (z  )*xmax + (x  )];
    
    /* find the number of triangles in the cube */
    num_tris = 0;
    for (i = 0; cube->triTable[cube0][i] != -1; i +=3 )
    {
	tris[num_tris].a = cube->triTable[cube0][i];
	tris[num_tris].b = cube->triTable[cube0][i+1];
	tris[num_tris].c = cube->triTable[cube0][i+2];
	
	num_tris++;
    }	
    
    if ( cube->specialCases[cube0] )
    {
	if ( x+1 < xmax ) cube1 = cubeIndex[(y  )*zmax*xmax + (z  )*xmax + (x+1)];
	else              cube1 = 0;
	
	if ( x-1 > 0 )    cube2 = cubeIndex[(y  )*zmax*xmax + (z  )*xmax + (x-1)];
	else              cube2 = 0;
	
	if ( y-1 > 0 )    cube3 = cubeIndex[(y  )*zmax*xmax + (z-1)*xmax + (x  )];
	else              cube3 = 0;  
	
	if ( y+1 < ymax ) cube4 = cubeIndex[(y  )*zmax*xmax + (z+1)*xmax + (x  )];
	else              cube4 = 0;
	
	if ( z+1 < zmax ) cube5 = cubeIndex[(y+1)*zmax*xmax + (z  )*xmax + (x  )];
	else              cube5 = 0;
	
	if ( z-1 > 0 )    cube6 = cubeIndex[(y-1)*zmax*xmax + (z  )*xmax + (x  )];
	else              cube6 = 0;
	
	retval = cube->BuildTrianglesForSpecialCube ( cube0, cube1, cube2, cube3, 
                                                     cube4, cube5, cube6, tris, 
                                                     &num_tris );
    }
    
    *numTriangles = num_tris;
    
    return ( retval );
}
