#include "BCship.h"

/*
    Glasteroids, a asteroids type game.
    Copyright (C) 1999 Matt Cohen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


BCship::BCship ( float w, float h, float r, float m )
{
    int i;
    worldWidth = w;
    worldHeight = h;
    
    objData.active = 1;
    objData.radius = r;
    objData.mass = m;
    objData.oneOverM = 1.0/m;
    objData.yLoc = SHIP_BASELINE;
    
    xBulletMax = worldWidth/2.0 + 10.0;
    xBulletMin = -xBulletMax;
    yBulletMax = worldHeight/2.0 + 10.0;
    yBulletMin = -yBulletMax;

    hullRotX = 0.0;
    hullRotY = 0.0;
    hullRotZ = 0.0;
    
    hullRotXspeed = 0.0;
    hullRotYspeed = 0.0;
    hullRotZspeed = 0.0;
    
    ringsRotX = 0.0;
    ringsRotY = 0.0;
    ringsRotZ = 0.0;
    
    ringsRotXspeed = 0.0;
    ringsRotYspeed = RING_SPEED;
    ringsRotZspeed = 0.0;

    joyLeft = 0;
    joyRight = 0;
    joyDown = 0;
    joyUp = 0;

    keyLeft = 0;
    keyRight = 0;
    keyDown = 0;
    keyUp = 0;

    shieldButtonPressed = 0;

    triggerRight = 0;
    triggerLeft = 0;
    shieldUp = 0;

    right = 0;
    left = 0;
    up = 0;
    down = 0;
    
    // Initialize bullet stuff
    numBullets = MAX_BULLETS;
    numActiveBullets = 0;
    bulletSpeed = BULLET_SPEED;
    bulletRadius = BULLET_RADIUS;
    bulletReloadTime = BULLET_RELOAD_TIME;
    bulletMass = BULLET_MASS;
    fire = 0;
    fireButton = 0;
    
    life = 100.0;
    shieldLife = 100.0;
    startExplosion = 1;
    exploding = 0;
    numShipParts = 5;
    baseLine = SHIP_BASELINE;
    
    BuildShipParts  ( );
    BuildShipHull   ( );
    BuildPowerCore  ( );
    BuildShipRings  ( );
    BuildShipShield ( );
    BuildBullets    ( );

    numParticleSystems = 0;
    for ( i = 0; i < MAX_BULLET_PARTICLE_SYSTEMS; i++ )
    {
        ParticleSystem[i].active = 0;
    }

    SetPowerCoreColor ( 1.0, 1.0, 1.0 );

}

BCship::~BCship ( void )
{ }


void BCship::ResetShip ( void )
{
    life = 100.0;
    shieldLife = 100.0;
    startExplosion = 1;

    objData.xLoc = 0.0;
    objData.yLoc = SHIP_BASELINE;

    objData.xSpeed = 0.0;
    objData.ySpeed = 0.0;

    ringsRotX = 0.0;
    ringsRotY = 0.0;
    ringsRotZ = 0.0;
}


void BCship::SetPowerCoreColor ( float r, float g, float b )
{
    glow_emission[0] = r;
    glow_emission[1] = g;
    glow_emission[2] = b;
    glow_emission[3] = 1.0;
}

void BCship::SetBulletColor ( float r, float g, float b )
{
    bullet_color[0] = r;
    bullet_color[1] = g;
    bullet_color[2] = b;
}


int BCship::DrawShip ( void )
{
    static GLfloat metal_ambient[] = { 0.05, 0.05, 0.05, 1.0 };
    static GLfloat metal_diffuse[] = { 0.5, 0.5, 0.5, 1.0 };
    static GLfloat metal_specular[] = { 0.7, 0.7, 0.7, 1.0 };

    static GLfloat glow_ambient[] = { 0.1745, 0.01175, 0.01175, 1.0 };
    static GLfloat glow_diffuse[] = { 0.61424, 0.04136, 0.04136, 1.0 };
    static GLfloat glow_specular[] = { 0.727811, 0.626959, 0.626959, 1.0 };
   
    if ( life <= 0.0 )
        return ( 0 );   

    glDisable ( GL_COLOR_MATERIAL );
    
    glPushMatrix ( );
    
    // move entire ship
    glTranslatef ( objData.xLoc, objData.yLoc, 0.0 );

    glPushAttrib ( GL_LIGHTING_BIT );

    glMaterialfv ( GL_FRONT, GL_AMBIENT, metal_ambient );
    glMaterialfv ( GL_FRONT, GL_DIFFUSE, metal_diffuse );
    glMaterialfv ( GL_FRONT, GL_SPECULAR, metal_specular );
    glMaterialf ( GL_FRONT, GL_SHININESS, 10.0 );
    

    // Rotate the hull
    glPushMatrix ( );
    glRotatef ( hullRotY, 0.0, 1.0, 0.0 );
				
    glCallList ( hull );

    glPushAttrib ( GL_LIGHTING_BIT );

    glMaterialfv ( GL_FRONT, GL_AMBIENT, glow_ambient );
    glMaterialfv ( GL_FRONT, GL_DIFFUSE, glow_diffuse );
    glMaterialfv ( GL_FRONT, GL_SPECULAR, glow_specular );
    glMaterialfv ( GL_FRONT, GL_EMISSION, glow_emission );
    glMaterialf ( GL_FRONT, GL_SHININESS, 10.0 );
    glCallList ( powerCore );

    glPopAttrib ( );
			
    glPopMatrix ( );   

    // Rotate the rings
    glPushMatrix ( );
    glRotatef ( ringsRotZ, 0.0, 0.0, 1.0 );
    glRotatef ( ringsRotY, 0.0, 1.0, 0.0 );
    glCallList ( rings );
    glPopMatrix ( );   

    glPopAttrib ( );

   
    if ( shieldUp )
    {
        DrawShipShield ( );
    }

    glPopMatrix ( );
    
    glEnable ( GL_COLOR_MATERIAL );
    
    return ( 1 );
}


void BCship::DrawShipShield ( void )
{
    static GLfloat ambient[] = { 0.05f, 0.0f, 0.05f, 0.51f };
    static GLfloat diffuse[] = { 0.4f, 0.0f, 0.4f, 0.51f };
    static GLfloat specular[] = { 1.0f, 0.0f, 1.0f, 0.51f };

    glEnable ( GL_BLEND );

    glPushAttrib ( GL_LIGHTING_BIT );
    
    glMaterialfv ( GL_FRONT, GL_AMBIENT, ambient );
    glMaterialfv ( GL_FRONT, GL_DIFFUSE, diffuse );
    glMaterialfv ( GL_FRONT, GL_SPECULAR, specular );
    glMaterialf ( GL_FRONT, GL_SHININESS, 20.0 );
    
    glCallList ( shield );

    glPopAttrib ( );

    glDisable ( GL_BLEND);
}



    
void BCship::DrawShipExploding ( void )
{
    static GLfloat metal_ambient[] = { 0.05, 0.05, 0.05, 1.0 };
    static GLfloat metal_diffuse[] = { 0.5, 0.5, 0.5, 1.0 };
    static GLfloat metal_specular[] = { 0.7, 0.7, 0.7, 1.0 };

    static GLfloat glow_ambient[] = { 0.1745, 0.01175, 0.01175, 1.0 };
    static GLfloat glow_diffuse[] = { 0.61424, 0.04136, 0.04136, 1.0 };
    static GLfloat glow_specular[] = { 0.727811, 0.626959, 0.626959, 1.0 };
   
    static int i;

    glDisable ( GL_COLOR_MATERIAL );

    glPushAttrib ( GL_LIGHTING_BIT );

    glMaterialfv ( GL_FRONT, GL_AMBIENT, metal_ambient );
    glMaterialfv ( GL_FRONT, GL_DIFFUSE, metal_diffuse );
    glMaterialfv ( GL_FRONT, GL_SPECULAR, metal_specular );
    glMaterialf ( GL_FRONT, GL_SHININESS, 10.0 );
    
    for ( i = 0; i < numShipParts - 1; i++ )
    {
        if ( !part[i].active )
            continue;
	
        glPushMatrix ( );
	
        glTranslatef ( part[i].xLoc, 
                       part[i].yLoc,
                       part[i].zLoc );
	
        glRotatef ( part[i].xRot, 1.0, 0.0, 0.0 );
        glRotatef ( part[i].yRot, 0.0, 1.0, 0.0 );
        glRotatef ( part[i].zRot, 0.0, 0.0, 1.0 );

        glCallList ( part[i].listNumber );
	
        glPopMatrix ( );
    }

    glPopAttrib ( );

    // Draw Powercore
    glPushAttrib ( GL_LIGHTING_BIT );
    glMaterialfv ( GL_FRONT, GL_AMBIENT, glow_ambient );
    glMaterialfv ( GL_FRONT, GL_DIFFUSE, glow_diffuse );
    glMaterialfv ( GL_FRONT, GL_SPECULAR, glow_specular );
    glMaterialfv ( GL_FRONT, GL_EMISSION, glow_emission );
    glMaterialf ( GL_FRONT, GL_SHININESS, 10.0 );
			
    glPushMatrix ( );
	
    glTranslatef ( part[numShipParts - 1].xLoc, 
                   part[numShipParts - 1].yLoc,
                   part[numShipParts - 1].zLoc );
	
    glRotatef ( part[numShipParts - 1].xRot, 1.0, 0.0, 0.0 );
    glRotatef ( part[numShipParts - 1].yRot, 0.0, 1.0, 0.0 );
    glRotatef ( part[numShipParts - 1].zRot, 0.0, 0.0, 1.0 );

    glCallList ( part[numShipParts - 1].listNumber );
	
    glPopMatrix ( );	
    glPopAttrib ( );

    glEnable ( GL_COLOR_MATERIAL );
}

void BCship::ExplodeBullet ( int num )
{
    static int i, p;

    Bullet[num].active = 0;
    numActiveBullets --;

    p = ActiveParticleSystem ( );

    if ( p == -1 )
    {
        cout << "Not enough particle systems!\n";
        return;
    }

    ParticleSystem[p].active = 1;
    ParticleSystem[p].life = 0.0;

    for ( i = 0; i < MAX_PARTICLES; i++ )
    {
        ParticleSystem[p].particle[i].x = Bullet[num].xLoc;
        ParticleSystem[p].particle[i].y = Bullet[num].yLoc;

	float theta = ((float)(rand()%360))/(2*M_PI);
	float speed = (float)(rand()%45-10);
	
	ParticleSystem[p].particle[i].vx = sin(theta)*speed;
	ParticleSystem[p].particle[i].vy = cos(theta)*speed;


        //ParticleSystem[p].particle[i].vx = (float)(rand()%30 - 15);
        //ParticleSystem[p].particle[i].vy = (float)(rand()%30 - 15);
    }
}


int BCship::ActiveParticleSystem ( void )
{
    static int i;

    for ( i = 0; i < MAX_BULLET_PARTICLE_SYSTEMS; i++ )
    {
        if ( !ParticleSystem[i].active )
        {
            numParticleSystems ++;
            return ( i );
        }
    }

    return ( -1 );
}


void BCship::UpdateBulletExplosions ( double time )
{
    static int i, j;

    for ( i = 0; i < MAX_BULLET_PARTICLE_SYSTEMS; i++ )
    {
        if ( ParticleSystem[i].active )
        {
            ParticleSystem[i].life += time;

            if ( ParticleSystem[i].life > BULLET_EXPLOSION_LENGTH )
            {
                ParticleSystem[i].active = 0;
                numParticleSystems --;
                continue;
            }

            ParticleSystem[i].color 
                = 1.0 - ParticleSystem[i].life/BULLET_EXPLOSION_LENGTH;
	
            for ( j = 0; j < MAX_PARTICLES; j++ )
            {
                ParticleSystem[i].particle[j].x 
                    += ParticleSystem[i].particle[j].vx*time;
                ParticleSystem[i].particle[j].y 
                    += ParticleSystem[i].particle[j].vy*time;
            }
        }
    }
}


void BCship::DrawBulletExplosions ( void )
{
    static int i, j;

    glDisable ( GL_LIGHTING );
    glBegin ( GL_POINTS );

    for ( i = 0; i < MAX_BULLET_PARTICLE_SYSTEMS; i++ )
    {
        if ( ParticleSystem[i].active )
        {
            glColor3f ( bullet_color[0]*ParticleSystem[i].color, 
                        bullet_color[1]*ParticleSystem[i].color, 
                        bullet_color[2]*ParticleSystem[i].color );

 
            for ( j = 0; j < MAX_PARTICLES; j++ )
            {
                glVertex3f ( ParticleSystem[i].particle[j].x,	
                             ParticleSystem[i].particle[j].y, 0.0 );	
            }

 
        }
    }

    glEnd ( );
    glEnable ( GL_LIGHTING );
}


void BCship::DrawBullets ( void )
{
    static int i;

    if ( numParticleSystems )
        DrawBulletExplosions ( );

    if ( !numActiveBullets )
    {
        return;
    }
    
    for ( i = 0; i < numBullets; i++ )
    {
        if ( Bullet[i].active )
        {
            glColor3f ( bullet_color[0],
                        bullet_color[1],
                        bullet_color[2] );

            glPushMatrix ( );
            glTranslatef ( Bullet[i].xLoc,
                           Bullet[i].yLoc,
                           0.0 );

            glCallList ( Bullet[i].listNumber );
            glPopMatrix ( );
        }
    }
}


void BCship::UpdateShipExplosion ( float time )
{
    static int i;

    if ( startExplosion )
    {

        for ( i = 0; i < numShipParts; i++ )
        {
            part[i].xRotSpeed = (float)(rand() % 100 + 50 );
            part[i].yRotSpeed = (float)(rand() % 100 + 50 );
            part[i].zRotSpeed = (float)(rand() % 100 + 50 );
	
            part[i].xSpeed = (float)(rand()%50 - 25 );
            part[i].ySpeed = (float)(rand()%50 - 25 );
	
            part[i].xLoc = objData.xLoc;
            part[i].yLoc = objData.yLoc;

            part[i].active = 1;
        }

        startExplosion = 0;
    }

    for ( i = 0; i < numShipParts; i++ )
    {
        if ( !part[i].active )
            continue;
	
        // Translation
        part[i].xLoc += part[i].xSpeed * time; 
        part[i].yLoc += part[i].ySpeed * time; 
        part[i].zLoc += part[i].zSpeed * time;
	
        /*if ( part[i].xLoc + part[i].radius > worldWidth/2.0 )
          {	
          part[i].xLoc = worldWidth/2.0 - part[i].radius;
          part[i].xSpeed = -part[i].xSpeed;
          }
          else if ( part[i].xLoc - part[i].radius < -worldWidth/2.0 )
          {
          part[i].xLoc = -worldWidth/2.0 + part[i].radius;
          part[i].xSpeed = -part[i].xSpeed;
          }

          if ( part[i].yLoc + part[i].radius > worldHeight/2.0 )
          {	
          part[i].yLoc = worldHeight/2.0 - part[i].radius;
          part[i].ySpeed = -part[i].ySpeed;
          }
          else if ( part[i].yLoc - part[i].radius < -worldHeight/2.0 )
          {
          part[i].yLoc = -worldHeight/2.0 + part[i].radius;
          part[i].ySpeed = -part[i].ySpeed;
          }*/

        if ( part[i].xLoc > xBulletMax )
        {	
            part[i].active = 0;
        }
        else if ( part[i].xLoc < xBulletMin )
        {
            part[i].active = 0;
        }

        if ( part[i].yLoc > yBulletMax )
        {	
            part[i].active = 0;
        }
        else if ( part[i].yLoc < yBulletMin )
        {
            part[i].active = 0;
        }

        // Rotation
        part[i].xRot += part[i].xRotSpeed * time;
        if ( part[i].xRot > 360.0 )
            part[i].xRot -= 360.0;
        else if ( part[i].xRot < -360.0 )
            part[i].xRot += 360.0;
	    
        part[i].yRot += part[i].yRotSpeed * time;
        if ( part[i].yRot > 360.0 )
            part[i].yRot -= 360.0;
        else if ( part[i].yRot < -360.0 )
            part[i].yRot += 360.0;
	    
        part[i].zRot += part[i].zRotSpeed * time;
        if ( part[i].zRot > 360.0 )
            part[i].zRot -= 360.0;
        else if ( part[i].zRot < -360.0 )
            part[i].zRot += 360.0;
    }
}


void BCship::UpdateBullets ( float time )
{
    static int i;
    static float reload = 0.0;

    reload += time;

    if ( numParticleSystems )
        UpdateBulletExplosions ( time );

    if ( !numActiveBullets && !fire )
    {
        return;
    }

    // Advance bullets
    for ( i = 0; i < numBullets; i++ )
    {
        if ( Bullet[i].active )
        {
            Bullet[i].xLoc += Bullet[i].xSpeed * time;
            Bullet[i].yLoc += Bullet[i].ySpeed * time;
	 
            Bullet[i].xRot += Bullet[i].xRotSpeed * time;
            Bullet[i].yRot += Bullet[i].yRotSpeed * time;
            Bullet[i].zRot += Bullet[i].zRotSpeed * time;

            // Check if bullets are in bounds
            if ( Bullet[i].xLoc > xBulletMax ||
                 Bullet[i].xLoc < xBulletMin ||
                 Bullet[i].yLoc > yBulletMax ||
                 Bullet[i].yLoc < yBulletMin    )
            {
                Bullet[i].active = 0;
                numActiveBullets--;
            }
        }
    }

    // Check if a new bullet was fired
    if ( fire )
    {
	if ( numActiveBullets < numBullets && life > 0.0)
	{
            if ( reload > bulletReloadTime )
            {
                ActivateNewBullet ( );
                numActiveBullets++;
                reload = 0.0;
            }
	}

	//fire = 0;
    }
}


void BCship::ActivateNewBullet ( void )
{
    static int i;
    static int bulletNum;
    static float radians;
    static float xStart, yStart;

    //cout << "Activating new bullet\n";

    for ( i = 0; i < numBullets; i++ )
    {
	if ( !Bullet[i].active )
	{
	    bulletNum = i;
	    break;
	}
    }

    radians = ringsRotZ * ( M_PI/180.0 );

    xStart = - ( sin ( radians ) * (objData.radius+bulletRadius*2.0) );
    yStart =     cos ( radians ) * (objData.radius+bulletRadius*2.0);
    
    Bullet[bulletNum].active = 1;
    Bullet[bulletNum].xLoc = objData.xLoc + xStart;
    Bullet[bulletNum].yLoc = objData.yLoc + yStart;
    
    Bullet[bulletNum].xSpeed = -(sin ( radians ) * bulletSpeed);
    Bullet[bulletNum].ySpeed = cos ( radians ) * bulletSpeed;
}


int BCship::UpdateShip ( float time )
{
    if ( life <= 0.0 )
        return ( 0 );   

    // Update the rings (spinning)
    ringsRotY += ringsRotYspeed * time;
    if ( ringsRotY > 360.0 )
	ringsRotY -= 360.0;
    else if ( ringsRotY < 0.0 )
	ringsRotY += 360.0;
    
    // Update the direction the cannon is pointing
    ringsRotZ += ringsRotZspeed * time;
    
    // Move the ship along the x axis
    objData.xLoc += objData.xSpeed * time;
    objData.yLoc += objData.ySpeed * time;
    
    if ( !keyUp && !keyDown && !joyUp && !joyDown )
        if ( objData.yLoc - 4.0 > -worldHeight/2.0 ) 
        {
            objData.ySpeed -= 30.0 * time;
        }
    
    if ( objData.yLoc + 4.0 > worldHeight/2.0 )
    {
        objData.yLoc = worldHeight/2.0 - 4.0;
        objData.ySpeed = -objData.ySpeed;
    }
    else if ( objData.yLoc - 4.0 < -worldHeight/2.0 )
    {
        objData.yLoc = -worldHeight/2.0 + 4.0;
        if ( !keyUp && !keyDown && !joyUp && !joyDown )
            objData.ySpeed *= 0.5;
        objData.ySpeed = -objData.ySpeed;
    }

    // make sure ship stays in bounds
    if ( objData.xLoc + 4.0 > worldWidth/2.0 )
    {
        objData.xLoc = worldWidth/2.0 - 4.0;
        objData.xSpeed = -objData.xSpeed;
    }
    else if ( objData.xLoc - 4.0 < -worldWidth/2.0 )
    {
        objData.xLoc = -worldWidth/2.0 + 4.0;
        objData.xSpeed = -objData.xSpeed;
    }

    // Check for keyboard or joystick input
    if ( right || left || keyRight || keyLeft || joyRight || joyLeft )
    {
	hullRotY = (float)(right-left) * 20.0;
	
	if ( !joyRight && !keyRight && right > 0 )
	    right--;
	
	if ( !joyLeft && !keyLeft && left > 0 )
	    left--;
	
	if ( ( joyRight || keyRight ) && right < 3 )
	    right++;
	
	if ( ( joyLeft || keyLeft ) && left < 3 )
	    left++;
    }
    
    return ( 1 );
}


void BCship::UpdateShield ( int s_up )
{
    if ( s_up )
    {
        shieldButtonPressed = 1;
        objData.radius = 6.0;
    }
    else 
    {
        shieldButtonPressed = 0;
        objData.radius = 4.0;
    }
}


void BCship::UpdateShieldBar ( double time )
{
    if ( shieldButtonPressed )
    {	
        if ( shieldLife > time * 50.0 )
        {
            shieldLife -= time * 50.0;
            shieldUp = 1;
        }
        else
        {
            shieldLife = 0.0;
            shieldUp = 0;
        }
    }
    else
    {
        if ( shieldLife + time * 10.0 < 100.0 )
            shieldLife += time * 10.0;
        else
            shieldLife = 100.0;

        shieldUp = 0;
    }
}


void BCship::DrawLifeBars ( void )
{
    static GLUquadricObj *life_bar, *shield_bar;
    static int first_call = 1;
    
    if ( first_call )
    {
        life_bar = gluNewQuadric ( );
        shield_bar = gluNewQuadric ( );
        first_call = 0;
    }
       
    if ( life <= 0.0 )
        return;
    
    if ( life > 67.0 )
        glColor3f ( 0.0, 1.0, 0.0 );
    else if ( life > 33.0 ) 
        glColor3f ( 1.0, 1.0, 0.0 );
    else
        glColor3f ( 1.0, 0.0, 0.0 );

    glPushMatrix ( );

    glTranslatef ( -3.0, 35.0, 20.0 );
    glRotatef ( 90.0, 0.0, 1.0, 0.0 );
    gluCylinder ( life_bar, 0.75, 0.75,(GLdouble)life*0.5, 10, 10 );

    glPopMatrix ( );
 
    glColor3f ( 1.0, 0.0, 1.0 );
 
    glPushMatrix ( );

    glTranslatef ( -3.0, 33.0, 20.0 );
    glRotatef ( 90.0, 0.0, 1.0, 0.0 );
    gluCylinder ( shield_bar, 0.75, 0.75,(GLdouble)shieldLife*0.5, 10, 10 );

    glPopMatrix ( );
}

    
void BCship::BuildBullets ( void )
{
    int i;
    
    for ( i = 0; i < numBullets; i++ )
    {
        Bullet[i].CreateGeometric ( BcSPHERE,
                                    BcSIZE, bulletRadius,
                                    NULL );
	
        Bullet[i].type = REGULAR_BULLET;
        Bullet[i].radius = bulletRadius;
        Bullet[i].mass = bulletMass;
        Bullet[i].oneOverM = 1.0/bulletMass;
    }
}


// Build Ship Hull
void BCship::BuildShipParts ( void )
{
    GLUquadricObj *cylinder = gluNewQuadric ( );

    // Build Inside sphere
    part[0].listNumber = glGenLists ( 1 );  
    part[0].radius = 2.5;
    part[0].mass = 100.0;
    part[0].oneOverM = 0.05;
    part[0].rotate = 1;
    part[0].translate = 1;

    glNewList ( part[0].listNumber, GL_COMPILE );
    glutSolidSphere ( 2.5, 10, 10 );
    glEndList ( );

    // Build first outter ring
    part[1].listNumber = glGenLists ( 1 );
    part[1].radius = 4.0;
    part[1].mass = 100.0;
    part[1].oneOverM = 0.05;
    part[1].rotate = 1;
    part[1].translate = 1;
  
    glNewList ( part[1].listNumber, GL_COMPILE );
    glutSolidTorus ( 0.5, 3.5, 15, 15 );   
    glEndList ( );
 
    // Build second outter ring
    part[2].listNumber = glGenLists ( 1 );
    part[2].radius = 4.0;
    part[2].mass = 100.0;
    part[2].oneOverM = 0.05;
    part[2].rotate = 1;
    part[2].translate = 1;

    glNewList ( part[2].listNumber, GL_COMPILE );
    glutSolidTorus ( 0.5, 3.5, 15, 15 );
    glEndList ( );   
	
    // Build cannon
    part[3].listNumber = glGenLists ( 1 );
    part[3].radius = 1.0;
    part[3].mass = 60.0;
    part[3].oneOverM = 0.1;
    part[3].rotate = 1;
    part[3].translate = 1;

    glNewList ( part[3].listNumber, GL_COMPILE );  
    gluCylinder ( cylinder, 0.75, 0.5, 1.5, 10, 10 );
    glEndList ( );

    // Build inside torus
    part[4].listNumber = glGenLists ( 1 );
    part[4].radius = 2.3;
    part[4].mass = 60.0;
    part[4].oneOverM = 0.1;
    part[4].rotate = 1;
    part[4].translate = 1;

    glNewList ( part[4].listNumber, GL_COMPILE );

    glDisable ( GL_COLOR_MATERIAL );
    glutSolidTorus ( 0.5, 2.25, 15, 15 );
    glEndList ( );
}


// Build Ship Hull
void BCship::BuildShipHull ( void )
{
    hull = glGenLists ( 1 );

    glNewList ( hull, GL_COMPILE );

    glutSolidSphere ( 2.5, 10, 10 );
            
    glEndList ( );
}


void BCship::BuildPowerCore ( void )
{
    powerCore = glGenLists ( 1 );
    
    glNewList ( powerCore, GL_COMPILE );
    
    glutSolidTorus ( 0.5, 2.25, 15, 15 );   

    glEndList ( );
}


// Build Ship Rings
void BCship::BuildShipRings ( void )
{
    GLUquadricObj *cylinder = gluNewQuadric ( );
    rings = glGenLists ( 1 );

    glNewList ( rings, GL_COMPILE );

    glutSolidTorus ( 0.5, 3.5, 15, 15 );

    glPushMatrix ( );
    glRotatef ( 90.0, 0.0, 1.0, 0.0 );
    glutSolidTorus ( 0.5, 3.5, 15, 15 );
    glPopMatrix ( );

    glPushMatrix ( );
    glTranslatef ( 0.0, 3.5, 0.0 );
    glRotatef ( -90.0, 1.0, 0.0, 0.0 );
    gluCylinder ( cylinder, 0.75, 0.5, 1.5, 10, 10 );
    glPopMatrix ( );

    glEndList ( );
}


// Build Ship Rings
void BCship::BuildShipShield ( void )
{
    shield = glGenLists ( 1 );

    glNewList ( shield, GL_COMPILE );

    glutSolidSphere ( 6.0, 15, 15 );
    
    glEndList ( );
}
