///////////////////////////////////////////////////////////////////////////////
//  Bobcat.cxx
//
//  OpenGL game engine.
///////////////////////////////////////////////////////////////////////////////
#include "Bobcat.h"

/*
    Glasteroids, a asteroids type game.
    Copyright (C) 1999 Matt Cohen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/



BCworld internalWorld;


BCworld *BCgetWorld ( void )
{
    return ( &internalWorld );
}


BCworld::BCworld ( void )
{
    //numLargeAsteroids = NUM_LARGE_ASTEROIDS;
    //numMediumAsteroids = numLargeAsteroids * 4;
    //numSmallAsteroids = numMediumAsteroids * 4;
    numSmallAsteroids = 30;
    numMediumAsteroids = 30;
    numLargeAsteroids = 30;
 
    numAsteroids = numLargeAsteroids+numMediumAsteroids+numSmallAsteroids;

    numActiveAsteroids = 0;
    numActiveSmallAsteroids = 0;
    numActiveMediumAsteroids = 0;
    numActiveLargeAsteroids = 0;

    State = BcSTARTUP;

    elasticity = 1.0;

    Ship = NULL;

    worldWidth = 100;

    srand ( time ( NULL ) );

    InitializeCollisionTest ( );

    showData = 0;
    texturing = 1;
    smoothShading = 1;

    Score = 0;

    clock ( );
}


BCworld::~BCworld ( void )
{ }

void BCworld::InitWorld ( int argc, char *argv[], char *title, 
			  int width, int height, float worldDimension )
{   
    float scale;
    int i;
    
    screenSizeX = width;
    screenSizeY = height;
    
    scale = (float)width/(float)height;
    
    worldWidth = worldDimension;
    worldHeight = worldDimension/scale;
    
    asteroidXmax = worldWidth/2.0 + 30.0;
    asteroidXmin = -asteroidXmax;
    asteroidYmax = worldHeight/2.0 + 30.0;
    asteroidYmin = -asteroidYmax;
 
    glutInit ( &argc, argv );
    glutInitDisplayMode ( GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGBA );
    glutInitWindowSize ( width, height );
    glutCreateWindow ( title );
    
    glutReshapeFunc ( InternalReshape );
    
    glutDisplayFunc ( InternalDisplay );
    glutVisibilityFunc ( InternalVisible );
    glutKeyboardFunc ( InternalKeyboard );
    glutKeyboardUpFunc ( InternalKeyboardUp );
    glutSpecialFunc ( InternalSpecial );
    glutSpecialUpFunc( InternalSpecialUp );
    glutJoystickFunc ( InternalJoystick, 100 );
    
    glutIdleFunc ( InternalIdle );

    font = txfLoadFont ( "gameText.txf" );

    Menu = new BCmenu ( font );

    InitTextures ( );    
    InitOpenGL ( );

    LoadSplashScreen ( "textures/splash.raw" );
    
    numActiveDust = 0;
    for ( i = 0; i < MAX_ASTEROID_DUST_SYSTEMS; i++ )
        Dust[i].active = 0;

    InitializeStars ( );

    Ship = new BCship ( worldWidth, worldHeight, 4.0, 400.0 );
    Ship->SetPowerCoreColor ( 0.0, 1.0, 0.0 );
    Ship->SetBulletColor ( 0.0, 1.0, 0.0 );

    Enemy = new BCenemy ( worldWidth, worldHeight, 4.0, 1000.0 );
}


void BCworld::BeginWorld ( void )
{
    glutMainLoop ( );
}


void BCworld::InitOpenGL ( void )
{
    GLfloat lmodel_ambient[] = {0.2, 0.2, 0.2, 1.0};
    GLfloat local_view[] = {0.0};
 
    // lighting stuff
    GLfloat     light_position[]  = {1.0f, 1.0f, 1.0f, 0.0f};
    GLfloat     light_diffuse[]   = {1.0f, 1.0f, 1.0f, 0.0f};
    GLfloat     light_ambient[]   = {0.2f, 0.2f, 0.2f, 0.0f};
    
    glEnable ( GL_AUTO_NORMAL );
    glEnable ( GL_NORMALIZE );

    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
    glLightModelfv(GL_LIGHT_MODEL_LOCAL_VIEWER, local_view);

    // Enable the lights
    glEnable ( GL_LIGHTING );
    
    // LIGHT0 Setup (directional light)
    glLightfv (GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv (GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv (GL_LIGHT0, GL_POSITION, light_position);
    
    glEnable ( GL_LIGHT0 );

    glEnable ( GL_COLOR_MATERIAL );
    glColorMaterial ( GL_FRONT, GL_AMBIENT_AND_DIFFUSE );
    glBlendFunc ( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    
    glShadeModel ( GL_SMOOTH );
    glEnable ( GL_DEPTH_TEST );

    glEnable ( GL_CULL_FACE );
    glFrontFace ( GL_CCW );

    glEnable(GL_ALPHA_TEST);
    glAlphaFunc(GL_GEQUAL, 0.5);
    
    txfEstablishTexture ( font, 0, GL_TRUE );
}


void BCworld::AddDirectionalLight ( float x, float y, float z )
{
    // lighting stuff
    GLfloat     light_position[4];
    GLfloat     light_diffuse[]   = {1.0, 1.0, 1.0, 1.0};
    GLfloat     light_ambient[]   = {0.2f, 0.2f, 0.2f, 1.0f};
    
    light_position[0] = ( GLfloat ) x;
    light_position[1] = ( GLfloat ) y;
    light_position[2] = ( GLfloat ) z;
    light_position[3] = 0.0;
    
    // LIGHT0 Setup (directional light)
    glLightfv (GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv (GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv (GL_LIGHT0, GL_POSITION, light_position);
    
    glEnable ( GL_LIGHT0 );
}


void BCworld::Restart ( void )
{
    int i;

    Score = 0;
    Ship->ResetShip ( );

    numActiveAsteroids = 0;
    for ( i = 0; i < numAsteroids; i++ )
        Asteroid[i]->active = 0;

    Enemy->ResetShip ( );
}


void BCworld::Build2dTexture ( void )
{
    int xLoc, yLoc, radius, i, x, y, count;
    int distance;
    int shade;
    
    texture = new unsigned char[TEXTURE_SIZE*TEXTURE_SIZE];
    memset ( texture, 200, TEXTURE_SIZE*TEXTURE_SIZE );
    
    count = rand ( ) % 30 + 10;

    for ( i = 0; i < count; i++ )
    {
        xLoc = rand ( ) % TEXTURE_SIZE;
        yLoc = rand ( ) % TEXTURE_SIZE;

        shade = rand ( ) % 200;
	
        radius = rand ( ) % (TEXTURE_SIZE / 8) + 1;
	
        //cout << "Making hole:  shade = " << shade << "  radius = " 
        //	 << radius << "  xLoc = " << xLoc << "  yLoc = " << yLoc << endl;

        for ( y = yLoc - (radius + 2); y < yLoc + radius+2; y++ )
            for ( x = xLoc - (radius + 2); x < xLoc + radius+2; x++ )
            {
                distance = (int) (sqrt ( (x-xLoc)*(x-xLoc) + (y-yLoc)*(y-yLoc) ) - 0.5);
		    
                if ( distance < radius )
                {
                    if ( x >= 0 && x < TEXTURE_SIZE && 
                         y >= 0 && y < TEXTURE_SIZE ) 
                        texture[y*TEXTURE_SIZE + x] = shade;
                }
            }
    }
}


void BCworld::Build3dTexture ( void )
{
    int xLoc, yLoc, zLoc, radius, i, x, y, z, count;
    float distance;
    int shade;
    
    texture = new unsigned char[TEXTURE_SIZE*TEXTURE_SIZE*TEXTURE_SIZE];
    memset ( texture, 200, TEXTURE_SIZE*TEXTURE_SIZE*TEXTURE_SIZE );
    
    count = rand ( ) % 1000 + 50;


    for ( i = 0; i < count; i++ )
    {
        xLoc = rand ( ) % TEXTURE_SIZE;
        yLoc = rand ( ) % TEXTURE_SIZE;
        zLoc = rand ( ) % TEXTURE_SIZE;

        shade = rand ( ) % 256;
	
        radius = rand ( ) % ( ( TEXTURE_SIZE + TEXTURE_SIZE + TEXTURE_SIZE ) / 12 );
	
        for ( y = yLoc - (radius + 2); y < yLoc + radius+2; y++ )
            for ( z = zLoc - (radius + 2); z < zLoc + radius+2; z++ )
                for ( x = xLoc - (radius + 2); x < xLoc + radius+2; x++ )
                {
                    distance = sqrt ( (x-xLoc)*(x-xLoc) + 
                                      (y-yLoc)*(y-yLoc) + 
                                      (z-zLoc)*(z-zLoc) );
		    
                    if ( distance < (float) radius )
                    {
                        if ( x >= 0 && x < TEXTURE_SIZE && 
                             y >= 0 && y < TEXTURE_SIZE && 
                             z >= 0 && z < TEXTURE_SIZE ) 
                            texture[y*TEXTURE_SIZE*TEXTURE_SIZE + z*TEXTURE_SIZE + x] = shade;
                    }
                }
    }
}


void BCworld::MakeRandomSurface ( unsigned char *data,
				  int x, int y, int z, int pad )
{
    unsigned char *temp;
    int i, j, k, m;
    int count;
    int p = 2;

    count = (int)((float)(x+y+z)/6.0) - pad;
    //count = 4;

    temp = new unsigned char[x*y*z];
    memset ( data, 0, x*y*z );
    
    data[(x*y*z)/2] = 1;
    memcpy ( temp, data, x*y*z );

    for ( m = 0; m < count; m++ )
    {

	for ( i = pad; i < z-pad; i++ )
            for ( j = pad; j < y-pad; j++ )
                for ( k = pad; k < x-pad; k++ )
                {
                    if ( data[i*y*x+j*x+k] )
                    {
                        //cout << "Found point at x = " << k << " y = " << j << " z = " << i << "========================" << endl;
                        if ( !data[(i+1)*y*x + (j  )*x + (k  )] )
                            if ( !(rand()%p) )
                                temp[(i+1)*y*x + (j  )*x + (k  )] = 1;

                        if ( !data[(i-1)*y*x + (j  )*x + (k  )] )
                            if ( !(rand()%p) )
                                temp[(i-1)*y*x + (j  )*x + (k  )] = 1;

                        if ( !data[(i  )*y*x + (j+1)*x + (k  )] )
                            if ( !(rand()%p) )
                                temp[(i  )*y*x + (j+1)*x + (k  )] = 1;

                        if ( !data[(i  )*y*x + (j-1)*x + (k  )] )
                            if ( !(rand()%p) )
                                temp[(i  )*y*x + (j-1)*x + (k  )] = 1;

                        if ( !data[(i  )*y*x + (j  )*x + (k+1)] )
                            if ( !(rand()%p) )
                                temp[(i  )*y*x + (j  )*x + (k+1)] = 1;

                        if ( !data[(i  )*y*x + (j  )*x + (k-1)] )
                            if ( !(rand()%p) )
                                temp[(i  )*y*x + (j  )*x + (k-1)] = 1;
                    }
                }	
 	memcpy ( data, temp, x*y*z );
    }
    delete []temp;
}	


void BCworld::BuildAsteroids ( void )
{
    int i;
    float red, green, blue, size;
    unsigned char *temp;
    int x = 7, y = 7, z = 7;
    int pad = 2;
    int count = 0;

    //printf("Entered BuildAsteroids\n");

    LargeAsteroid = new BCobject[numLargeAsteroids];
    MediumAsteroid = new BCobject[numMediumAsteroids];
    SmallAsteroid = new BCobject[numSmallAsteroids];

    Asteroid = new BCobject*[numLargeAsteroids+numMediumAsteroids+numSmallAsteroids];

    //printf("made room for %d asteroids\n",numLargeAsteroids+numMediumAsteroids+numSmallAsteroids);

    red = 0.6;
    green = 0.4;
    blue = 0.3;
	
    temp = new unsigned char[x*y*z];

    for ( i = 0; i < numSmallAsteroids; i++ )
    {
        MakeRandomSurface ( temp, x, y, z, pad );

        //size = (float)(rand ( ) % 2 + 2 );
        size = 2.0;

        SmallAsteroid[i].CreateIsosurface ( temp, &mcubes, x, y, z,
                                            BcCOLOR, (double)red, (double)green, (double)blue,
                                            BcSCALE, (double)size, (double)size, (double)size,
                                            NULL );

        SmallAsteroid[i].type = SMALL_ASTEROID;

        Asteroid[count] = &SmallAsteroid[i];
        count++;
    }

    for ( i = 0; i < numMediumAsteroids; i++ )
    {
        MakeRandomSurface ( temp, x, y, z, pad );

        //size = (float)(rand ( ) % 2 + 2 );
        size = 4.0;

        MediumAsteroid[i].CreateIsosurface ( temp, &mcubes, x, y, z,
                                             BcCOLOR, (double)red, (double)green, (double)blue,
                                             BcSCALE, (double)size, (double)size, (double)size,
                                             NULL );

        MediumAsteroid[i].type = MEDIUM_ASTEROID;

        Asteroid[count] = &MediumAsteroid[i];
        count++;
    }

    for ( i = 0; i < numLargeAsteroids; i++ )
    {

        MakeRandomSurface ( temp, x, y, z, pad );

        //size = (float)(rand ( ) % 2 + 4 );
        size = 6.0;

        LargeAsteroid[i].CreateIsosurface ( temp, &mcubes, x, y, z,
                                            BcCOLOR, (double)red, (double)green, (double)blue,
                                            BcSCALE, (double)size, (double)size, (double)size,
                                            NULL );

        LargeAsteroid[i].type = LARGE_ASTEROID;

        Asteroid[count] = &LargeAsteroid[i];
        count++;
    }

    delete []temp;

    //printf("Built the small asteroids\n");

}


double BCworld::GetCurrentFrameTime ( void )
{
    static float currentTime;
    static float lastTime = 0.0;
    static double total;
    static double *queue;
    static int next = 0;
    static int first_time = 1;
    static int queue_size = 10;
    static double queue_default = 0.0001;

    if ( first_time )
    {
	int i;

	queue = new double[queue_size];

	for ( i = 0; i < queue_size; i++ )
        {
	    queue[i] = queue_default;
        }

	total = queue_default * (double) queue_size;
      
	first_time = 0;
    }

    currentTime = ((float)clock())/(float)CLOCKS_PER_SEC;

    total -= queue[next];
    queue[next] = currentTime - lastTime;
    total += queue[next];

    next++;
    next = next%queue_size;

    lastTime = currentTime;

    //printf ( "Frame time: %f\n", total/(double)queue_size );

    return ( total/(double)queue_size);
}


void BCworld::DrawNormal ( void )
{
    static double deltaTime;
 
    deltaTime = GetCurrentFrameTime ( );

    glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT  );
     
    if ( showData )
        DrawData ( deltaTime );
    
    DrawScore ( );
    
    UpdateStars ( deltaTime );
    DrawStars ( );
 
    UpdateAsteroids ( deltaTime );
    CheckAsteroidsForCollisions ( );

    if ( Ship->UpdateShip ( deltaTime ) )
        CheckShipForCollisionWithAsteroids ( );
  
    Ship->UpdateBullets ( deltaTime );
    CheckBulletsForCollisions ( );
	
    if ( Enemy->UpdateEnemyShip ( Ship, deltaTime ) )
    {
        CheckEnemyShipForCollisionWithAsteroids ( );
        CheckEnemyShipForCollisionWithBullets ( );
    }

    CheckEnemyBulletsForCollisions ( );

    Ship->DrawBullets ( );	

    CheckBothShipsForCollision ( );

    if ( !Ship->DrawShip ( ) )
    {	
        State = BcGAME_DEAD_SHIP;
        Menu->SetCurrentMenu ( BcMENU_GAME_OVER );
        Ship->UpdateShipExplosion ( deltaTime );
        CheckShipPartsForCollisions ( );
        Ship->DrawShipExploding ( );
    }
  
    if ( !Enemy->DrawEnemyShip ( ) )
    {
        Enemy->UpdateEnemyShipExplosion ( deltaTime );
        CheckEnemyShipPartsForCollisions ( );
        Enemy->DrawEnemyShipExploding ( );
    }

    DrawAsteroids ( );

    Ship->UpdateShieldBar ( deltaTime );
    Ship->DrawLifeBars ( );
    
    //DrawTestSquare ( );

    glFlush ( );
    glutSwapBuffers( );
}


void BCworld::DrawGameOver ( void )
{
    static double deltaTime;
 
    deltaTime = GetCurrentFrameTime ( );

    glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT  );

    if ( showData )
        DrawData ( deltaTime );

    UpdateStars ( deltaTime );
    DrawStars ( );
 
    UpdateAsteroids ( deltaTime );
    CheckAsteroidsForCollisions ( );

    if ( Enemy->UpdateEnemyShip ( Ship, deltaTime ) )
    {
        CheckEnemyShipForCollisionWithAsteroids ( );
        CheckEnemyShipForCollisionWithBullets ( );
    }

    CheckEnemyBulletsForCollisions ( );

    if ( Ship->UpdateShip ( deltaTime ) )
        CheckShipForCollisionWithAsteroids ( );

    if ( !Enemy->DrawEnemyShip ( ) )
    {
        Enemy->UpdateEnemyShipExplosion ( deltaTime );
        CheckEnemyShipPartsForCollisions ( );
        Enemy->DrawEnemyShipExploding ( );
    }
	
    Ship->UpdateBullets ( deltaTime );
    CheckBulletsForCollisions ( );
    Ship->DrawBullets ( );

    Ship->UpdateShipExplosion ( deltaTime );
    CheckShipPartsForCollisions ( );
    Ship->DrawShipExploding ( );
      
    DrawAsteroids ( );

    DrawScore ( );

    Menu->DrawMenu ( );

    glFlush ( );
    glutSwapBuffers( );
}


void BCworld::DrawGameMenu ( void )
{
    static double deltaTime;
 
    // Keep updating the time
    deltaTime = GetCurrentFrameTime ( );

    glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT  );

    UpdateStars ( deltaTime );
    DrawStars ( );
 
    Ship->DrawBullets ( );

    if ( !Ship->DrawShip ( ) )	
        Ship->DrawShipExploding ( );

    if ( !Enemy->DrawEnemyShip ( ) )
    {
        Enemy->DrawEnemyShipExploding ( );
    }

    DrawAsteroids ( );

    Ship->DrawLifeBars ( );
    
    DrawScore ( );

    if(texturing)
      DrawSplashScreen ( deltaTime );
    
    Menu->DrawMenu ( );

    glFlush ( );
    glutSwapBuffers( );
}


void BCworld::DrawTestSquare ( void )
{
    if ( texturing )
    {
        glEnable ( GL_TEXTURE_2D );
        glBindTexture ( GL_TEXTURE_2D, asteroid_texture );
    }

    glColor3f ( 1.0, 1.0, 0.0 ); 
    glBegin ( GL_QUADS );

    glTexCoord2f ( 1.0, 0.0 );  glVertex3f (  20.0, -20.0, 20.0 );
    glTexCoord2f ( 1.0, 1.0 );  glVertex3f (  20.0,  20.0, 20.0 );
    glTexCoord2f ( 0.0, 1.0 );  glVertex3f ( -20.0,  20.0, 20.0 );
    glTexCoord2f ( 0.0, 0.0 );  glVertex3f ( -20.0, -20.0, 20.0 );
    
    glEnd ( );

    if ( texturing )
        glDisable ( GL_TEXTURE_2D );
}


void BCworld::UpdateScore ( int i )
{
  if(Score / 10000 < (Score+i) / 10000)
    Ship->life+=10.0;

  if ( Ship->life > 0.0 )
    Score += i;
}


void BCworld::DrawScore ( void )
{
    static char scoreString[64];

    sprintf ( scoreString, "%d", Score );

    glDisable ( GL_LIGHTING );
    glEnable ( GL_TEXTURE_2D );

    glPushMatrix();
    glColor3f(0.0, 0.0, 1.0);
    glTranslatef ( -47.0, 32.5, 20.0 );
    glScalef( 0.125, 0.125, 0.125);
    txfRenderFancyString(font, scoreString, (int) strlen(scoreString));
    glPopMatrix();

    glDisable ( GL_TEXTURE_2D );
    glEnable ( GL_LIGHTING );
}


void BCworld::DrawData ( double timePerFrame )
{
    static char fpsString[64];
    static char textureString[64];
    static char shadeString[64];
    static int count = 0;
    static double ave;

    count++;

    if ( !(count%10) ) 
    {
        if ( texturing )
            sprintf ( textureString, "Texturing: ON" );
        else
            sprintf ( textureString, "Texturing: OFF" );

        if ( smoothShading )
            sprintf ( shadeString, "Shading: SMOOTH" );
        else
            sprintf ( shadeString, "Shading: FLAT" ); 

        ave = 1.0/timePerFrame;
        sprintf ( fpsString, "FPS: %3.2f", ave );
    }

    glDisable ( GL_LIGHTING );
    glEnable ( GL_TEXTURE_2D );

    glPushMatrix();
    glColor3f(1.0, 0.0, 0.0);
    glTranslatef ( 23.0, -27.0, 20.0 );
    glScalef( 0.075, 0.075, 0.075);
    txfRenderFancyString(font, textureString, (int) strlen(textureString));
    glPopMatrix();

    glPushMatrix();
    glColor3f(1.0, 0.0, 0.0);
    glTranslatef ( 23.0, -31.0, 20.0 );
    glScalef( 0.075, 0.075, 0.075);
    txfRenderFancyString(font, shadeString, (int) strlen(shadeString));
    glPopMatrix();

    glPushMatrix();
    glColor3f(1.0, 0.0, 0.0);
    glTranslatef ( 23.0, -35.0, 20.0 );
    glScalef( 0.075, 0.075, 0.075);
    txfRenderFancyString(font, fpsString, (int) strlen(fpsString));
    glPopMatrix();

    glDisable ( GL_TEXTURE_2D );
    glEnable ( GL_LIGHTING );
}


void BCworld::DrawAsteroids ( void )
{
    int i;

    if ( numActiveDust )
        DrawAsteroidDust ( );

    if ( texturing )
    {
#ifdef GL_TEXTURE_3D_EXT
        glEnable ( GL_TEXTURE_3D_EXT );
        glBindTexture ( GL_TEXTURE_3D_EXT, asteroid_texture );
#else
        glEnable ( GL_TEXTURE_2D );
        glBindTexture ( GL_TEXTURE_2D, asteroid_texture );
#endif
    }

    for ( i = 0; i < numAsteroids; i++ )
    {
	if ( !Asteroid[i]->active )
	    continue;
	
	glPushMatrix ( );
	
	glTranslatef ( Asteroid[i]->xLoc, 
		       Asteroid[i]->yLoc,
		       Asteroid[i]->zLoc );
	
	glRotatef ( Asteroid[i]->xRot, 1.0, 0.0, 0.0 );
	glRotatef ( Asteroid[i]->yRot, 0.0, 1.0, 0.0 );
	glRotatef ( Asteroid[i]->zRot, 0.0, 0.0, 1.0 );
	
	glCallList ( Asteroid[i]->listNumber );
	
	glPopMatrix ( );
    }

    if ( texturing )
    {
#ifdef GL_TEXTURE_3D_EXT
        glDisable ( GL_TEXTURE_3D_EXT );
#else
        glDisable ( GL_TEXTURE_2D );
#endif
    }

}


void BCworld::UpdateAsteroidDust ( double time )
{
    static int i, j;

    for ( i = 0; i < MAX_ASTEROID_DUST_SYSTEMS; i++ )
    {
        if ( Dust[i].active )
        {
            Dust[i].life += time;

            if ( Dust[i].life > ASTEROID_DUST_LENGTH )
            {
                Dust[i].active = 0;
                numActiveDust --;
                continue;
            }

            Dust[i].color 
                = 1.0 - Dust[i].life/ASTEROID_DUST_LENGTH;
	
            for ( j = 0; j < MAX_PARTICLES; j++ )
            {
                Dust[i].particle[j].x 
                    += Dust[i].particle[j].vx * time;
                Dust[i].particle[j].y 
                    += Dust[i].particle[j].vy * time;
            }
        }
    }
}


void BCworld::DrawAsteroidDust ( void )
{
    static int i, j;

    glDisable ( GL_LIGHTING );
    glBegin ( GL_POINTS );

    for ( i = 0; i < MAX_ASTEROID_DUST_SYSTEMS; i++ )
    {
        if ( Dust[i].active )
        {
            glColor3f ( Dust[i].color * 0.6, 
                        Dust[i].color * 0.4, 
                        Dust[i].color * 0.3 );


            for ( j = 0; j < MAX_PARTICLES; j++ )
            {
                glVertex3f ( Dust[i].particle[j].x,	
                             Dust[i].particle[j].y, 0.0 );	
            }

        }
    }
    
    glEnd ( );
    glEnable ( GL_LIGHTING );
}


void BCworld::InitializeStars ( void )
{
    int i;

    for ( i = 0; i < NUM_STARS; i++ )
    {
        Star[i].y = (float)(rand()%100) - 50.0;
        Star[i].x = (float)(rand()%100) - 50.0;
        Star[i].vy = (float)(-(rand()%5 + 5) );
    }
}


void BCworld::UpdateStars ( double time )
{
    static int i;

    for ( i = 0; i < NUM_STARS; i++ )
    {
        Star[i].y += Star[i].vy * time;

        if ( Star[i].y < asteroidYmin )
        {
            Star[i].y = asteroidYmax;
            Star[i].x = (float)(rand()%100) - 50.0;
            Star[i].vy = (float)(-(rand()%10 + 5) );
        }
    }
}


void BCworld::DrawStars ( void )
{
    static int i;

    glDisable ( GL_LIGHTING );

    glColor3f ( 1.0, 1.0, 1.0 );

    glBegin ( GL_POINTS );

    for ( i = 0; i < NUM_STARS; i++ )
    {
        glVertex3f ( Star[i].x,	
                     Star[i].y, -20.0 );	
    }

    glEnd ( );

    glEnable ( GL_LIGHTING );
}


void BCworld::BreakAsteroid ( int num, int hit)
{
    static float x, y, vx, vy;
    static int m, n, i, p;

    p = ActivateAsteroidDust ( );
    if ( p == -1 )
    {
        cout << "Not enough particle systems!\n";
    }
    else
    {
        Dust[p].active = 1;
        Dust[p].life = 0.0;

        for ( i = 0; i < MAX_PARTICLES; i++ )
        {
            Dust[p].particle[i].x = Asteroid[num]->xLoc;
            Dust[p].particle[i].y = Asteroid[num]->yLoc;

            float theta = ((float)(rand()%360))/(2*M_PI);
            float speed = (float)(rand()%45-10);

            Dust[p].particle[i].vx = sin(theta)*speed;
            Dust[p].particle[i].vy = cos(theta)*speed;
            
            
            //Dust[p].particle[i].vx = (float)(rand()%30 - 15);
            //Dust[p].particle[i].vy = (float)(rand()%30 - 15);
        }
    }

    Asteroid[num]->active = 0;
    numActiveAsteroids--;

    if ( Asteroid[num]->type == SMALL_ASTEROID )
    {
        if ( hit )
            UpdateScore ( 100 );
    }
    else if ( Asteroid[num]->type == MEDIUM_ASTEROID )
    {
        x = Asteroid[num]->xLoc;
        y = Asteroid[num]->yLoc;
        vx = Asteroid[num]->xSpeed;
        vy = Asteroid[num]->ySpeed;

        m = ActivateNewAsteroid ( SMALL_ASTEROID );
        Asteroid[m]->xLoc = x + 1.5;
        Asteroid[m]->yLoc = y;
        Asteroid[m]->xSpeed = (float)(rand()%5 + 5) + vx;
        Asteroid[m]->ySpeed = vy;
		
        n = ActivateNewAsteroid ( SMALL_ASTEROID );
        Asteroid[n]->xLoc = x - 1.5;
        Asteroid[n]->yLoc = y;
        Asteroid[n]->xSpeed = (float)(-(rand()%5 + 5)) + vx;
        Asteroid[n]->ySpeed = vy;

        colliding[Asteroid[m]->listNumber][Asteroid[n]->listNumber] = 1;
        colliding[Asteroid[n]->listNumber][Asteroid[m]->listNumber] = 1;

        if ( hit )
            UpdateScore ( 50 );
    }
    else // LARGE_ASTEROID
    {
        x = Asteroid[num]->xLoc;
        y = Asteroid[num]->yLoc;
        vx = Asteroid[num]->xSpeed;
        vy = Asteroid[num]->ySpeed;

        m = ActivateNewAsteroid ( MEDIUM_ASTEROID );
        Asteroid[m]->xLoc = x + 3.0;
        Asteroid[m]->yLoc = y;
        Asteroid[m]->xSpeed = (float)(rand()%5 + 5) + vx;
        Asteroid[m]->ySpeed =  vy;

        n = ActivateNewAsteroid ( MEDIUM_ASTEROID );
        Asteroid[n]->xLoc = x - 3.0;
        Asteroid[n]->yLoc = y;
        Asteroid[n]->xSpeed = (float)(-(rand()%5 + 5)) + vx;
        Asteroid[n]->ySpeed = vy;

        colliding[Asteroid[m]->listNumber][Asteroid[n]->listNumber] = 1;
        colliding[Asteroid[n]->listNumber][Asteroid[m]->listNumber] = 1;

        if ( hit )
            UpdateScore ( 25 );
    }
}



void BCworld::UpdateAsteroids ( double time )
{
    static int i;
    
    if ( numActiveDust )
        UpdateAsteroidDust ( time );

    for ( i = 0; i < numAsteroids; i++ )
    {
        if ( !Asteroid[i]->active )
            continue;
	
        Asteroid[i]->xLoc += Asteroid[i]->xSpeed * time; 
        Asteroid[i]->yLoc += Asteroid[i]->ySpeed * time; 
        Asteroid[i]->zLoc += Asteroid[i]->zSpeed * time;
			
        if ( Asteroid[i]->xLoc > asteroidXmax )
        {
            Asteroid[i]->active = 0;
            numActiveAsteroids--;
            UpdateScore ( 1 );
        }
        else if ( Asteroid[i]->xLoc < asteroidXmin )
        {
            Asteroid[i]->active = 0;
            numActiveAsteroids--;
            UpdateScore ( 1 );
        }
		
        if ( Asteroid[i]->yLoc > asteroidYmax )
        {
            Asteroid[i]->active = 0;
            numActiveAsteroids--;
            UpdateScore ( 1 );
        }
        else if ( Asteroid[i]->yLoc < asteroidYmin )
        {
            Asteroid[i]->active = 0;
            numActiveAsteroids--;
            UpdateScore ( 1 );
        }

        Asteroid[i]->xRot += Asteroid[i]->xRotSpeed * time;
        if ( Asteroid[i]->xRot > 360.0 )
            Asteroid[i]->xRot -= 360.0;
        else if ( Asteroid[i]->xRot < -360.0 )
            Asteroid[i]->xRot += 360.0;
	    
        Asteroid[i]->yRot += Asteroid[i]->yRotSpeed * time;
        if ( Asteroid[i]->yRot > 360.0 )
            Asteroid[i]->yRot -= 360.0;
        else if ( Asteroid[i]->yRot < -360.0 )
            Asteroid[i]->yRot += 360.0;
		    
        Asteroid[i]->zRot += Asteroid[i]->zRotSpeed * time;
        if ( Asteroid[i]->zRot > 360.0 )
            Asteroid[i]->zRot -= 360.0;
        else if ( Asteroid[i]->zRot < -360.0 )
            Asteroid[i]->zRot += 360.0;
    }

    while ( numActiveAsteroids < MAX_NUM_ACTIVE_ASTEROIDS )
        ActivateNewAsteroid ( ALL_ASTEROIDS );
}
	

int BCworld::ActivateNewAsteroid ( int size )
{
    static int asteroidNum;
    int found_asteroid = 0;
   
    if ( size == SMALL_ASTEROID )
    {
        while ( !found_asteroid )
        {
            asteroidNum = rand()%numSmallAsteroids;

            if ( !SmallAsteroid[asteroidNum].active )
            {
                found_asteroid = 1;
            }
        }
    }
    else if ( size == MEDIUM_ASTEROID )
    {
        while ( !found_asteroid )
        {
            asteroidNum = rand()%numMediumAsteroids;

            if ( !MediumAsteroid[asteroidNum].active )
            {
                found_asteroid = 1;
                asteroidNum = numSmallAsteroids + asteroidNum;
            }
        }
    }
    else if ( size == LARGE_ASTEROID )
    {
        while ( !found_asteroid )
        {
            asteroidNum = rand()%numLargeAsteroids;

            if ( !LargeAsteroid[asteroidNum].active )
            {
                found_asteroid = 1;
                asteroidNum = numSmallAsteroids + numMediumAsteroids + asteroidNum;
            }
        }
    }
    else // if ( size == ALL_ASTEROIDS )
    {
        while ( !found_asteroid )
        {
            asteroidNum = rand()%numAsteroids;

            if ( !Asteroid[asteroidNum]->active )
                found_asteroid = 1;
        }
    }

    Asteroid[asteroidNum]->xRotSpeed = (float)(rand() % 50);
    Asteroid[asteroidNum]->yRotSpeed = (float)(rand() % 50);
    Asteroid[asteroidNum]->zRotSpeed = (float)(rand() % 50);
	
    Asteroid[asteroidNum]->xSpeed = (float)(rand()%60 - 30);
    Asteroid[asteroidNum]->ySpeed = (float)(rand()%60 - 60);
	
    Asteroid[asteroidNum]->xLoc = (float)(rand()%100) - 50.0;
    Asteroid[asteroidNum]->yLoc = asteroidYmax;
	
    Asteroid[asteroidNum]->active = 1;
    Asteroid[asteroidNum]->translate = 1;
    Asteroid[asteroidNum]->rotate = 1;

    numActiveAsteroids++;

    return ( asteroidNum );
}


void BCworld::CheckBothShipsForCollision ( void )
{
    static float vx, vy;
 
    if ( CollisionTest ( &Ship->objData, &Enemy->Ship->objData, &vx, &vy ) )
    { 
        ProcessObjectCollision ( &Ship->objData, &Enemy->Ship->objData, vx, vy );

        if ( !Ship->shieldUp )
            Ship->life -= ( Enemy->Ship->objData.mass * 0.04 );
    }
}


void BCworld::CheckAsteroidsForCollisions ( void )
{
    static int i, j;
    static float vx, vy;
    
    for ( i = 0; i < numAsteroids; i++ )
    {
	for ( j = i+1; j < numAsteroids; j++ )
	{
	    if ( CollisionTest ( Asteroid[i], Asteroid[j], &vx, &vy ) )
	    {
		ProcessObjectCollision ( Asteroid[i], Asteroid[j], vx, vy );
	    }
	}
    }
}


void BCworld::CheckShipForCollisionWithAsteroids ( void )
{
    static int i;
    static float vx, vy;
    
    for ( i = 0; i < numAsteroids; i++ )
    {
        if ( CollisionTest ( &Ship->objData, Asteroid[i], &vx, &vy ) )
        {
            ProcessObjectCollision ( &Ship->objData, Asteroid[i], vx, vy );

            if ( !Ship->shieldUp )
                Ship->life -= ( Asteroid[i]->mass * 0.04 );

            break;  // Only one collision at a time
        }
    }
}


void BCworld::CheckEnemyShipForCollisionWithAsteroids ( void )
{
    static int i;
    static float vx, vy;
    
    for ( i = 0; i < numAsteroids; i++ )
    {
        if ( CollisionTest ( &Enemy->Ship->objData, Asteroid[i], &vx, &vy ) )
        {
            ProcessObjectCollision ( &Enemy->Ship->objData, Asteroid[i], vx, vy );

            // Doesn't hurt enemy Ship
            break;  // Only one collision at a time
        }
    }
}

void BCworld::CheckEnemyShipForCollisionWithBullets ( void )
{
    static int i;
    static float vx, vy;

    // Check for collisions with ship bullets
    for ( i = 0; i < Ship->numBullets; i++ )
    {
        if ( CollisionTest ( &Ship->Bullet[i], &Enemy->Ship->objData, &vx, &vy ) )
        {
            ProcessObjectCollision ( &Ship->Bullet[i], &Enemy->Ship->objData, vx, vy );
  
            if ( !Enemy->Ship->shieldUp )
                Enemy->Ship->life -= 34.0;  // Takes 3 hits

            if ( Enemy->Ship->life < 0.0 )
                UpdateScore ( 1000 );
        }
    }

    // Check for collisions with enemy ship bullets
    for ( i = 0; i < Ship->numBullets; i++ )
    {
        if ( CollisionTest ( &Enemy->Ship->Bullet[i], &Enemy->Ship->objData, &vx, &vy ) )
        {
            ProcessObjectCollision ( &Enemy->Ship->Bullet[i], &Enemy->Ship->objData, vx, vy );
  
            // Do nothing
        }
    }
}



int BCworld::ActivateAsteroidDust ( void )
{
    static int i;

    for ( i = 0; i < MAX_ASTEROID_DUST_SYSTEMS; i++ )
    {
        if ( !Dust[i].active )
        {
            numActiveDust ++;
            return ( i );
        }
    }

    return ( -1 );
}



void BCworld::CheckBulletsForCollisions ( void )
{
    static int i, j;
    static float vx, vy;
    
    // Check for collisions with bullets
    for ( i = 0; i < Ship->numBullets; i++ )
	for ( j = i+1; j < Ship->numBullets; j++ )
	{
	    if ( CollisionTest ( &Ship->Bullet[i], &Ship->Bullet[j], &vx, &vy ) )
	    {
                ProcessObjectCollision ( &Ship->Bullet[i], &Ship->Bullet[j], vx, vy );
	    }	
	}
    
    // Check for collisions with asteroids
    for ( i = 0; i < Ship->numBullets; i++ )
	for ( j = 0; j < numAsteroids; j++ )
	{
	    if ( CollisionTest ( &Ship->Bullet[i], Asteroid[j], &vx, &vy ) )
	    {
                ProcessObjectCollision ( &Ship->Bullet[i], Asteroid[j], vx, vy );
			
                Ship->ExplodeBullet ( i );

                BreakAsteroid ( j, 1 );

                //printf("Breaking asteroid\n");
	    }	
	}
    
    // Check for collisions with ship
    for ( i = 0; i < Ship->numBullets; i++ )
    {
	if ( CollisionTest ( &Ship->Bullet[i], &Ship->objData, &vx, &vy ) )
	{
	    ProcessObjectCollision ( &Ship->Bullet[i], &Ship->objData, vx, vy );
            //if ( !Ship->shieldUp )
            //    Ship->life -= 5.0;
	}
    }
}


void BCworld::CheckEnemyBulletsForCollisions ( void )
{
    static int i, j;
    static float vx, vy;
    
    // Check for collisions with enemy bullets
    for ( i = 0; i < Enemy->Ship->numBullets; i++ )
	for ( j = i+1; j < Enemy->Ship->numBullets; j++ )
	{
	    if ( CollisionTest ( &Enemy->Ship->Bullet[i], &Enemy->Ship->Bullet[j], &vx, &vy ) )
	    {
                ProcessObjectCollision ( &Enemy->Ship->Bullet[i], &Enemy->Ship->Bullet[j], vx, vy );
	    }	
	}

    // Check for collisions with ship bullets
    for ( i = 0; i < Enemy->Ship->numBullets; i++ )
	for ( j = 0; j < Ship->numBullets; j++ )
	{
	    if ( CollisionTest ( &Enemy->Ship->Bullet[i], &Ship->Bullet[j], &vx, &vy ) )
	    {
                ProcessObjectCollision ( &Enemy->Ship->Bullet[i], &Ship->Bullet[j], vx, vy );
	    }	
	}
    
    // Check for collisions with asteroids
    for ( i = 0; i < Enemy->Ship->numBullets; i++ )
	for ( j = 0; j < numAsteroids; j++ )
	{
	    if ( CollisionTest ( &Enemy->Ship->Bullet[i], Asteroid[j], &vx, &vy ) )
	    {
                ProcessObjectCollision ( &Enemy->Ship->Bullet[i], Asteroid[j], vx, vy );
			
                Enemy->Ship->ExplodeBullet ( i );

                BreakAsteroid ( j, 0 );
	    }	
	}
    
    // Check for collisions with ship
    for ( i = 0; i < Enemy->Ship->numBullets; i++ )
    {
	if ( CollisionTest ( &Enemy->Ship->Bullet[i], &Ship->objData, &vx, &vy ) )
	{
	    ProcessObjectCollision ( &Enemy->Ship->Bullet[i], &Ship->objData, vx, vy );

            if ( !Ship->shieldUp )
                Ship->life -= ( Enemy->Ship->Bullet[i].mass * 0.04 );
	}
    }

    // Check for collisions with enemy ship
    for ( i = 0; i < Enemy->Ship->numBullets; i++ )
    {
	if ( CollisionTest ( &Enemy->Ship->Bullet[i], &Enemy->Ship->objData, &vx, &vy ) )
	{
	    ProcessObjectCollision ( &Enemy->Ship->Bullet[i], &Enemy->Ship->objData, vx, vy );
            // Do Nothing
	}
    }
}


void BCworld::CheckShipPartsForCollisions ( void )
{
    static int i, j;
    static float vx, vy;
    
    // Check for collisions with asteroids
    for ( i = 0; i < Ship->numShipParts; i++ )
        for ( j = 0; j < numAsteroids; j++ )
        {
            if ( CollisionTest ( &Ship->part[i], Asteroid[j], &vx, &vy ) )
            {
                ProcessObjectCollision ( &Ship->part[i], Asteroid[j], vx, vy );
            }	
        }
}


void BCworld::CheckEnemyShipPartsForCollisions ( void )
{
    static int i, j;
    static float vx, vy;
    
    // Check for collisions with bullets
    for ( i = 0; i < Enemy->Ship->numShipParts; i++ )
        for ( j = 0; j < Ship->numBullets; j++ )
        {
            if ( CollisionTest ( &Enemy->Ship->part[i], &Ship->Bullet[j], &vx, &vy ) )
            {
                ProcessObjectCollision ( &Enemy->Ship->part[i], &Ship->Bullet[j], vx, vy );
            }	
        }

    // Check for collisions with ship
    for ( i = 0; i < Enemy->Ship->numShipParts; i++ )
    {
        if ( CollisionTest ( &Enemy->Ship->part[i], &Ship->objData, &vx, &vy ) )
        {
            ProcessObjectCollision ( &Enemy->Ship->part[i], 
                                     &Ship->objData, vx, vy );

            if ( !Ship->shieldUp )
                Ship->life -= ( Enemy->Ship->part[i].mass * 0.04 );

        }	
    }	

    // Check for collisions with asteroids
    for ( i = 0; i < Enemy->Ship->numShipParts; i++ )
        for ( j = 0; j < numAsteroids; j++ )
        {
            if ( CollisionTest ( &Enemy->Ship->part[i], Asteroid[j], &vx, &vy ) )
            {
                ProcessObjectCollision ( &Enemy->Ship->part[i], Asteroid[j], vx, vy );
            }	
        }
}


void BCworld::InitTextures ( void )
{

#ifdef GL_TEXTURE_3D_EXT  // Try for 3D texturing

    Build3dTexture ( );

    glGenTextures ( 1, &asteroid_texture );
    glBindTexture ( GL_TEXTURE_3D_EXT, asteroid_texture );

    glTexParameteri ( GL_TEXTURE_3D_EXT, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
    glTexParameteri ( GL_TEXTURE_3D_EXT, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    glTexImage3DEXT ( GL_TEXTURE_3D_EXT, 0, GL_LUMINANCE, 
                      TEXTURE_SIZE, TEXTURE_SIZE, TEXTURE_SIZE,
                      0, GL_LUMINANCE, GL_UNSIGNED_BYTE, texture );

#else   // Otherwise use 2D texturing

    Build2dTexture ( );

    glGenTextures ( 1, &asteroid_texture );
    glBindTexture ( GL_TEXTURE_2D, asteroid_texture );
    
    glPixelStorei ( GL_UNPACK_ALIGNMENT, 1 );

    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
    glTexParameteri ( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    glTexImage2D ( GL_TEXTURE_2D, 0, GL_LUMINANCE, 
                   TEXTURE_SIZE, TEXTURE_SIZE,
                   0, GL_LUMINANCE, GL_UNSIGNED_BYTE, texture );
#endif

}


void BCworld::AddEngineFunction ( BCengineCB e )
{
    UserEngine = e;
}


void BCworld::AddShip ( void )
{
    //Ship = new BCship ( worldWidth, worldHeight, 4.0, 400.0 );
    //Enemy = new BCenemy ( worldWidth, worldHeight, 4.0, 400.0 );
}


void BCworld::MenuItemSelected ( void )
{
    int m, i;

    m = Menu->GetCurrentMenu ( );
    i = Menu->GetCurrentItem ( );

    if ( m == BcMENU_MAIN )
    {
        switch ( i )
        {
            case BcMENU_NEW:
                Restart ( );
                State = BcGAME_IN_PROGRESS;
                break;

            case BcMENU_RESUME:
                if ( Ship->life > 0.0 )
                    State = BcGAME_IN_PROGRESS;
                break;

            case BcMENU_HIGH_SCORES:
                Menu->SetCurrentMenu ( BcMENU_HIGH_SCORES );
                break;

            case BcMENU_DIRECTIONS:
                Menu->SetCurrentMenu ( BcMENU_DIRECTIONS );
                break;

            case BcMENU_QUIT:
                State = BcEXITING_GAME;
                break;

            default:
                break;
        }
    }
    else if ( m == BcMENU_HIGH_SCORES )
    {
        Menu->SetCurrentMenu ( BcMENU_MAIN );
    }
    else if ( m == BcMENU_DIRECTIONS )
    {
        Menu->SetCurrentMenu ( BcMENU_MAIN );
    }
}
