///////////////////////////////////////////////////////////////////////////////
// MarchingCubes.cxx
//
// This file contains routines for the marching cubes algorithm.  It is
// an attempt to address some of the holes that can be created when certain
// cubes are neighboring.  
//
// Here is a representation of the standard cube being used in this
// implementation of the marching cubes algorithm:
//
//              4 ---------- 4 ---------- 5 
//             /|                        /|
//            7 |                       5 |
//           /  |                      /  | 
//          7 ---------- 6 ---------- 6   | 
//          |   |                     |   | 
//          |   8                     |   9       y
//          |   |                     |   |       |   
//          |   |                     |   |       |  
//          |   |                     |   |       | 
//         11   |                    10   |       |
//          |   |                     |   |       /-----x
//          |   0 ---------- 0 -------|-- 1      /
//          |  /                      |  /      /
//          | 3                       | 1      z
//          |/                        |/
//          3 ---------- 2 ---------- 2 
//       (x,y,z)
//      reference 
//        point             
//
//  Below is a list of how I numbered neighboring cubes:
//
//  Right Neighbor  = 1
//  Left Neighbor   = 2
//  Front Neighbor  = 3
//  Back Neighbor   = 4
//  Top Neighbor    = 5
//  Bottom Neighbor = 6
//
//
//  MTC 3/1/99
///////////////////////////////////////////////////////////////////////////////
#include "MarchingCubes.h"


/*
    Glasteroids, a asteroids type game.
    Copyright (C) 1999 Matt Cohen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

int _triTable[256][16] = {
    {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 8, 3, 9, 8, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 2, 10, 0, 2, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 8, 3, 2, 10, 8, 10, 9, 8, -1, -1, -1, -1, -1, -1, -1},
    {3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 11, 2, 8, 11, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 11, 2, 1, 9, 11, 9, 8, 11, -1, -1, -1, -1, -1, -1, -1},
    {3, 10, 1, 11, 10, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 10, 1, 0, 8, 10, 8, 11, 10, -1, -1, -1, -1, -1, -1, -1},
    {3, 9, 0, 3, 11, 9, 11, 10, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 3, 0, 7, 3, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 1, 9, 4, 7, 1, 7, 3, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 4, 7, 3, 0, 4, 1, 2, 10, -1, -1, -1, -1, -1, -1, -1},
    {9, 2, 10, 9, 0, 2, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 9, 2, 9, 7, 2, 7, 3, 7, 9, 4, -1, -1, -1, -1},
    {8, 4, 7, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 4, 7, 11, 2, 4, 2, 0, 4, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 1, 8, 4, 7, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
    {4, 7, 11, 9, 4, 11, 9, 11, 2, 9, 2, 1, -1, -1, -1, -1},
    {3, 10, 1, 3, 11, 10, 7, 8, 4, -1, -1, -1, -1, -1, -1, -1},
    {1, 11, 10, 1, 4, 11, 1, 0, 4, 7, 11, 4, -1, -1, -1, -1},
    {4, 7, 8, 9, 0, 11, 9, 11, 10, 11, 0, 3, -1, -1, -1, -1},
    {4, 7, 11, 4, 11, 9, 9, 11, 10, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 4, 1, 5, 0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 5, 4, 8, 3, 5, 3, 1, 5, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 1, 2, 10, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
    {5, 2, 10, 5, 4, 2, 4, 0, 2, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 5, 3, 2, 5, 3, 5, 4, 3, 4, 8, -1, -1, -1, -1},
    {9, 5, 4, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 11, 2, 0, 8, 11, 4, 9, 5, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 4, 0, 1, 5, 2, 3, 11, -1, -1, -1, -1, -1, -1, -1},
    {2, 1, 5, 2, 5, 8, 2, 8, 11, 4, 8, 5, -1, -1, -1, -1},
    {10, 3, 11, 10, 1, 3, 9, 5, 4, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 5, 0, 8, 1, 8, 10, 1, 8, 11, 10, -1, -1, -1, -1},
    {5, 4, 0, 5, 0, 11, 5, 11, 10, 11, 0, 3, -1, -1, -1, -1},
    {5, 4, 8, 5, 8, 10, 10, 8, 11, -1, -1, -1, -1, -1, -1, -1},
    {9, 7, 8, 5, 7, 9, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 3, 0, 9, 5, 3, 5, 7, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 8, 0, 1, 7, 1, 5, 7, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 7, 8, 9, 5, 7, 10, 1, 2, -1, -1, -1, -1, -1, -1, -1},
    {10, 1, 2, 9, 5, 0, 5, 3, 0, 5, 7, 3, -1, -1, -1, -1},
    {8, 0, 2, 8, 2, 5, 8, 5, 7, 10, 5, 2, -1, -1, -1, -1},
    {2, 10, 5, 2, 5, 3, 3, 5, 7, -1, -1, -1, -1, -1, -1, -1},
    {7, 9, 5, 7, 8, 9, 3, 11, 2, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 7, 9, 7, 2, 9, 2, 0, 2, 7, 11, -1, -1, -1, -1},
    {2, 3, 11, 0, 1, 8, 1, 7, 8, 1, 5, 7, -1, -1, -1, -1},
    {11, 2, 1, 11, 1, 7, 7, 1, 5, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 8, 8, 5, 7, 10, 1, 3, 10, 3, 11, -1, -1, -1, -1},
    {5, 7, 0, 5, 0, 9, 7, 11, 0, 1, 0, 10, 11, 10, 0, -1},
    {11, 10, 0, 11, 0, 3, 10, 5, 0, 8, 0, 7, 5, 7, 0, -1},
    {11, 10, 5, 7, 11, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 1, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 8, 3, 1, 9, 8, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 5, 2, 6, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 5, 1, 2, 6, 3, 0, 8, -1, -1, -1, -1, -1, -1, -1},
    {9, 6, 5, 9, 0, 6, 0, 2, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 9, 8, 5, 8, 2, 5, 2, 6, 3, 2, 8, -1, -1, -1, -1},
    {2, 3, 11, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 0, 8, 11, 2, 0, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 1, 9, 2, 9, 11, 2, 9, 8, 11, -1, -1, -1, -1},
    {6, 3, 11, 6, 5, 3, 5, 1, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 11, 0, 11, 5, 0, 5, 1, 5, 11, 6, -1, -1, -1, -1},
    {3, 11, 6, 0, 3, 6, 0, 6, 5, 0, 5, 9, -1, -1, -1, -1},
    {6, 5, 9, 6, 9, 11, 11, 9, 8, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 3, 0, 4, 7, 3, 6, 5, 10, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 5, 10, 6, 8, 4, 7, -1, -1, -1, -1, -1, -1, -1},
    {10, 6, 5, 1, 9, 7, 1, 7, 3, 7, 9, 4, -1, -1, -1, -1},
    {6, 1, 2, 6, 5, 1, 4, 7, 8, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 5, 5, 2, 6, 3, 0, 4, 3, 4, 7, -1, -1, -1, -1},
    {8, 4, 7, 9, 0, 5, 0, 6, 5, 0, 2, 6, -1, -1, -1, -1},
    {7, 3, 9, 7, 9, 4, 3, 2, 9, 5, 9, 6, 2, 6, 9, -1},
    {3, 11, 2, 7, 8, 4, 10, 6, 5, -1, -1, -1, -1, -1, -1, -1},
    {5, 10, 6, 4, 7, 2, 4, 2, 0, 2, 7, 11, -1, -1, -1, -1},
    {0, 1, 9, 4, 7, 8, 2, 3, 11, 5, 10, 6, -1, -1, -1, -1},
    {9, 2, 1, 9, 11, 2, 9, 4, 11, 7, 11, 4, 5, 10, 6, -1},
    {8, 4, 7, 3, 11, 5, 3, 5, 1, 5, 11, 6, -1, -1, -1, -1},
    {5, 1, 11, 5, 11, 6, 1, 0, 11, 7, 11, 4, 0, 4, 11, -1},
    {0, 5, 9, 0, 6, 5, 0, 3, 6, 11, 6, 3, 8, 4, 7, -1},
    {6, 5, 9, 6, 9, 11, 4, 7, 9, 7, 11, 9, -1, -1, -1, -1},
    {10, 4, 9, 6, 4, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 10, 6, 4, 9, 10, 0, 8, 3, -1, -1, -1, -1, -1, -1, -1},
    {10, 0, 1, 10, 6, 0, 6, 4, 0, -1, -1, -1, -1, -1, -1, -1},
    {8, 3, 1, 8, 1, 6, 8, 6, 4, 6, 1, 10, -1, -1, -1, -1},
    {1, 4, 9, 1, 2, 4, 2, 6, 4, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 1, 2, 9, 2, 4, 9, 2, 6, 4, -1, -1, -1, -1},
    {0, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 3, 2, 8, 2, 4, 4, 2, 6, -1, -1, -1, -1, -1, -1, -1},
    {10, 4, 9, 10, 6, 4, 11, 2, 3, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 2, 2, 8, 11, 4, 9, 10, 4, 10, 6, -1, -1, -1, -1},
    {3, 11, 2, 0, 1, 6, 0, 6, 4, 6, 1, 10, -1, -1, -1, -1},
    {6, 4, 1, 6, 1, 10, 4, 8, 1, 2, 1, 11, 8, 11, 1, -1},
    {9, 6, 4, 9, 3, 6, 9, 1, 3, 11, 6, 3, -1, -1, -1, -1},
    {8, 11, 1, 8, 1, 0, 11, 6, 1, 9, 1, 4, 6, 4, 1, -1},
    {3, 11, 6, 3, 6, 0, 0, 6, 4, -1, -1, -1, -1, -1, -1, -1},
    {6, 4, 8, 11, 6, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 10, 6, 7, 8, 10, 8, 9, 10, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 3, 0, 10, 7, 0, 9, 10, 6, 7, 10, -1, -1, -1, -1},
    {10, 6, 7, 1, 10, 7, 1, 7, 8, 1, 8, 0, -1, -1, -1, -1},
    {10, 6, 7, 10, 7, 1, 1, 7, 3, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 6, 1, 6, 8, 1, 8, 9, 8, 6, 7, -1, -1, -1, -1},
    {2, 6, 9, 2, 9, 1, 6, 7, 9, 0, 9, 3, 7, 3, 9, -1},
    {7, 8, 0, 7, 0, 6, 6, 0, 2, -1, -1, -1, -1, -1, -1, -1},
    {7, 3, 2, 6, 7, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 11, 10, 6, 8, 10, 8, 9, 8, 6, 7, -1, -1, -1, -1},
    {2, 0, 7, 2, 7, 11, 0, 9, 7, 6, 7, 10, 9, 10, 7, -1},
    {1, 8, 0, 1, 7, 8, 1, 10, 7, 6, 7, 10, 2, 3, 11, -1},
    {11, 2, 1, 11, 1, 7, 10, 6, 1, 6, 7, 1, -1, -1, -1, -1},
    {8, 9, 6, 8, 6, 7, 9, 1, 6, 11, 6, 3, 1, 3, 6, -1},
    {0, 9, 1, 11, 6, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 8, 0, 7, 0, 6, 3, 11, 0, 11, 6, 0, -1, -1, -1, -1},
    {7, 11, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 8, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 9, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 1, 9, 8, 3, 1, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
    {10, 1, 2, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 3, 0, 8, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
    {2, 9, 0, 2, 10, 9, 6, 11, 7, -1, -1, -1, -1, -1, -1, -1},
    {6, 11, 7, 2, 10, 3, 10, 8, 3, 10, 9, 8, -1, -1, -1, -1},
    {7, 2, 3, 6, 2, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {7, 0, 8, 7, 6, 0, 6, 2, 0, -1, -1, -1, -1, -1, -1, -1},
    {2, 7, 6, 2, 3, 7, 0, 1, 9, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 2, 1, 8, 6, 1, 9, 8, 8, 7, 6, -1, -1, -1, -1},
    {10, 7, 6, 10, 1, 7, 1, 3, 7, -1, -1, -1, -1, -1, -1, -1},
    {10, 7, 6, 1, 7, 10, 1, 8, 7, 1, 0, 8, -1, -1, -1, -1},
    {0, 3, 7, 0, 7, 10, 0, 10, 9, 6, 10, 7, -1, -1, -1, -1},
    {7, 6, 10, 7, 10, 8, 8, 10, 9, -1, -1, -1, -1, -1, -1, -1},
    {6, 8, 4, 11, 8, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 6, 11, 3, 0, 6, 0, 4, 6, -1, -1, -1, -1, -1, -1, -1},
    {8, 6, 11, 8, 4, 6, 9, 0, 1, -1, -1, -1, -1, -1, -1, -1},
    {9, 4, 6, 9, 6, 3, 9, 3, 1, 11, 3, 6, -1, -1, -1, -1},
    {6, 8, 4, 6, 11, 8, 2, 10, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 3, 0, 11, 0, 6, 11, 0, 4, 6, -1, -1, -1, -1},
    {4, 11, 8, 4, 6, 11, 0, 2, 9, 2, 10, 9, -1, -1, -1, -1},
    {10, 9, 3, 10, 3, 2, 9, 4, 3, 11, 3, 6, 4, 6, 3, -1},
    {8, 2, 3, 8, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 2, 4, 6, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 0, 2, 3, 4, 2, 4, 6, 4, 3, 8, -1, -1, -1, -1},
    {1, 9, 4, 1, 4, 2, 2, 4, 6, -1, -1, -1, -1, -1, -1, -1},
    {8, 1, 3, 8, 6, 1, 8, 4, 6, 6, 10, 1, -1, -1, -1, -1},
    {10, 1, 0, 10, 0, 6, 6, 0, 4, -1, -1, -1, -1, -1, -1, -1},
    {4, 6, 3, 4, 3, 8, 6, 10, 3, 0, 3, 9, 10, 9, 3, -1},
    {10, 9, 4, 6, 10, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 5, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 4, 9, 5, 11, 7, 6, -1, -1, -1, -1, -1, -1, -1},
    {5, 0, 1, 5, 4, 0, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
    {11, 7, 6, 8, 3, 4, 3, 5, 4, 3, 1, 5, -1, -1, -1, -1},
    {9, 5, 4, 10, 1, 2, 7, 6, 11, -1, -1, -1, -1, -1, -1, -1},
    {6, 11, 7, 1, 2, 10, 0, 8, 3, 4, 9, 5, -1, -1, -1, -1},
    {7, 6, 11, 5, 4, 10, 4, 2, 10, 4, 0, 2, -1, -1, -1, -1},
    {3, 4, 8, 3, 5, 4, 3, 2, 5, 10, 5, 2, 11, 7, 6, -1},
    {7, 2, 3, 7, 6, 2, 5, 4, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 5, 4, 0, 8, 6, 0, 6, 2, 6, 8, 7, -1, -1, -1, -1},
    {3, 6, 2, 3, 7, 6, 1, 5, 0, 5, 4, 0, -1, -1, -1, -1},
    {6, 2, 8, 6, 8, 7, 2, 1, 8, 4, 8, 5, 1, 5, 8, -1},
    {9, 5, 4, 10, 1, 6, 1, 7, 6, 1, 3, 7, -1, -1, -1, -1},
    {1, 6, 10, 1, 7, 6, 1, 0, 7, 8, 7, 0, 9, 5, 4, -1},
    {4, 0, 10, 4, 10, 5, 0, 3, 10, 6, 10, 7, 3, 7, 10, -1},
    {7, 6, 10, 7, 10, 8, 5, 4, 10, 4, 8, 10, -1, -1, -1, -1},
    {6, 9, 5, 6, 11, 9, 11, 8, 9, -1, -1, -1, -1, -1, -1, -1},
    {3, 6, 11, 0, 6, 3, 0, 5, 6, 0, 9, 5, -1, -1, -1, -1},
    {0, 11, 8, 0, 5, 11, 0, 1, 5, 5, 6, 11, -1, -1, -1, -1},
    {6, 11, 3, 6, 3, 5, 5, 3, 1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 9, 5, 11, 9, 11, 8, 11, 5, 6, -1, -1, -1, -1},
    {0, 11, 3, 0, 6, 11, 0, 9, 6, 5, 6, 9, 1, 2, 10, -1},
    {11, 8, 5, 11, 5, 6, 8, 0, 5, 10, 5, 2, 0, 2, 5, -1},
    {6, 11, 3, 6, 3, 5, 2, 10, 3, 10, 5, 3, -1, -1, -1, -1},
    {5, 8, 9, 5, 2, 8, 5, 6, 2, 3, 8, 2, -1, -1, -1, -1},
    {9, 5, 6, 9, 6, 0, 0, 6, 2, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 8, 1, 8, 0, 5, 6, 8, 3, 8, 2, 6, 2, 8, -1},
    {1, 5, 6, 2, 1, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 6, 1, 6, 10, 3, 8, 6, 5, 6, 9, 8, 9, 6, -1},
    {10, 1, 0, 10, 0, 6, 9, 5, 0, 5, 6, 0, -1, -1, -1, -1},
    {0, 3, 8, 5, 6, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {10, 5, 6, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 5, 10, 7, 5, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {11, 5, 10, 11, 7, 5, 8, 3, 0, -1, -1, -1, -1, -1, -1, -1},
    {5, 11, 7, 5, 10, 11, 1, 9, 0, -1, -1, -1, -1, -1, -1, -1},
    {10, 7, 5, 10, 11, 7, 9, 8, 1, 8, 3, 1, -1, -1, -1, -1},
    {11, 1, 2, 11, 7, 1, 7, 5, 1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 1, 2, 7, 1, 7, 5, 7, 2, 11, -1, -1, -1, -1},
    {9, 7, 5, 9, 2, 7, 9, 0, 2, 2, 11, 7, -1, -1, -1, -1},
    {7, 5, 2, 7, 2, 11, 5, 9, 2, 3, 2, 8, 9, 8, 2, -1},
    {2, 5, 10, 2, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1},
    {8, 2, 0, 8, 5, 2, 8, 7, 5, 10, 2, 5, -1, -1, -1, -1},
    {9, 0, 1, 5, 10, 3, 5, 3, 7, 3, 10, 2, -1, -1, -1, -1},
    {9, 8, 2, 9, 2, 1, 8, 7, 2, 10, 2, 5, 7, 5, 2, -1},
    {1, 3, 5, 3, 7, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 7, 0, 7, 1, 1, 7, 5, -1, -1, -1, -1, -1, -1, -1},
    {9, 0, 3, 9, 3, 5, 5, 3, 7, -1, -1, -1, -1, -1, -1, -1},
    {9, 8, 7, 5, 9, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {5, 8, 4, 5, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1},
    {5, 0, 4, 5, 11, 0, 5, 10, 11, 11, 3, 0, -1, -1, -1, -1},
    {0, 1, 9, 8, 4, 10, 8, 10, 11, 10, 4, 5, -1, -1, -1, -1},
    {10, 11, 4, 10, 4, 5, 11, 3, 4, 9, 4, 1, 3, 1, 4, -1},
    {2, 5, 1, 2, 8, 5, 2, 11, 8, 4, 5, 8, -1, -1, -1, -1},
    {0, 4, 11, 0, 11, 3, 4, 5, 11, 2, 11, 1, 5, 1, 11, -1},
    {0, 2, 5, 0, 5, 9, 2, 11, 5, 4, 5, 8, 11, 8, 5, -1},
    {9, 4, 5, 2, 11, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 5, 10, 3, 5, 2, 3, 4, 5, 3, 8, 4, -1, -1, -1, -1},
    {5, 10, 2, 5, 2, 4, 4, 2, 0, -1, -1, -1, -1, -1, -1, -1},
    {3, 10, 2, 3, 5, 10, 3, 8, 5, 4, 5, 8, 0, 1, 9, -1},
    {5, 10, 2, 5, 2, 4, 1, 9, 2, 9, 4, 2, -1, -1, -1, -1},
    {8, 4, 5, 8, 5, 3, 3, 5, 1, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 5, 1, 0, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {8, 4, 5, 8, 5, 3, 9, 0, 5, 0, 3, 5, -1, -1, -1, -1},
    {9, 4, 5, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 11, 7, 4, 9, 11, 9, 10, 11, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 3, 4, 9, 7, 9, 11, 7, 9, 10, 11, -1, -1, -1, -1},
    {1, 10, 11, 1, 11, 4, 1, 4, 0, 7, 4, 11, -1, -1, -1, -1},
    {3, 1, 4, 3, 4, 8, 1, 10, 4, 7, 4, 11, 10, 11, 4, -1},
    {4, 11, 7, 9, 11, 4, 9, 2, 11, 9, 1, 2, -1, -1, -1, -1},
    {9, 7, 4, 9, 11, 7, 9, 1, 11, 2, 11, 1, 0, 8, 3, -1},
    {11, 7, 4, 11, 4, 2, 2, 4, 0, -1, -1, -1, -1, -1, -1, -1},
    {11, 7, 4, 11, 4, 2, 8, 3, 4, 3, 2, 4, -1, -1, -1, -1},
    {2, 9, 10, 2, 7, 9, 2, 3, 7, 7, 4, 9, -1, -1, -1, -1},
    {9, 10, 7, 9, 7, 4, 10, 2, 7, 8, 7, 0, 2, 0, 7, -1},
    {3, 7, 10, 3, 10, 2, 7, 4, 10, 1, 10, 0, 4, 0, 10, -1},
    {1, 10, 2, 8, 7, 4, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 1, 4, 1, 7, 7, 1, 3, -1, -1, -1, -1, -1, -1, -1},
    {4, 9, 1, 4, 1, 7, 0, 8, 1, 8, 7, 1, -1, -1, -1, -1},
    {4, 0, 3, 7, 4, 3, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {4, 8, 7, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {9, 10, 8, 10, 11, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 9, 3, 9, 11, 11, 9, 10, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 10, 0, 10, 8, 8, 10, 11, -1, -1, -1, -1, -1, -1, -1},
    {3, 1, 10, 11, 3, 10, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 11, 1, 11, 9, 9, 11, 8, -1, -1, -1, -1, -1, -1, -1},
    {3, 0, 9, 3, 9, 11, 1, 2, 9, 2, 11, 9, -1, -1, -1, -1},
    {0, 2, 11, 8, 0, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {3, 2, 11, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 8, 2, 8, 10, 10, 8, 9, -1, -1, -1, -1, -1, -1, -1},
    {9, 10, 2, 0, 9, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 3, 8, 2, 8, 10, 0, 1, 8, 1, 10, 8, -1, -1, -1, -1},
    {1, 10, 2, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 8, 9, 1, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 9, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 3, 8, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}};

unsigned char _specialCases[256] = {
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
    0, 0, 1, 0, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0,
    0, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 1, 1, 0, 0,
    0, 0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 0,
    0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0,
    1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 0, 0,
    0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
    0, 1, 0, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 0,
    0, 0, 1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0,
    1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 0, 0,
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 1, 1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };


// Constructor
MarchingCubes::MarchingCubes ( void )
{
    FillCubeTriangleArrays ( );
    ConstructMidpointNormalArray ( );
}

// Destructor
MarchingCubes::~MarchingCubes ( void )
{
    DestroyMidpointNormalArray ( );
}

/* ============================================================================
   Function:     construct_midpoint_normal_array

   Purpose:      Fills in midpoint normal array and midpoint locations array.

   Parmeters:    None.

   Returned:     1 if arrays are allocated correctly, 0 otherwise.
		 
   MTC 2/22/99
=============================================================================*/
int MarchingCubes::ConstructMidpointNormalArray ( void )
{
    int i, j;

    midpointLoc = new XYZ[12];
    if ( midpointLoc == NULL )
    {
        cout << "Error getting memory for midpoint location array\n";
	return ( 0 );
    }

    FillMidpointLocationArray ( );

    midNormals = new float **[12];
    if ( midNormals == NULL ) 
    {
        cout << "Error getting memory for midpoint array\n";
	return ( 0 );
    }

    for ( i = 0; i < 12; i++ )
    {
        midNormals[i] = new float *[256];
        if ( midNormals[i] == NULL ) 
        {
	    cout << "Error getting memory for midpoint array\n";
	    return ( 0 );
	}
    }

    for ( i = 0; i < 12; i++ )
    {
	for ( j = 0; j < 256; j++ )
	{
	    midNormals[i][j] = new float[3];
	    if ( midNormals[i][j] == NULL )
	    {
	        cout << "Error getting memory for midpoint array\n";
		return ( 0 );
	    }  
	}
    }

    FillMidpointNormalArray ( );

    return ( 1 );
}


/* ============================================================================
   Function:     destroy_midpoint_normal_array

   Purpose:      Frees midpoint normal array and midpoint locations array.

   Parmeters:    None.

   Returned:     None.
		 
   MTC 2/22/99
==============================================================================*/
void MarchingCubes::DestroyMidpointNormalArray ( void )
{
    int i, j;

    for ( i = 0; i < 12; i++ )
    {
	for ( j = 0; j < 256; j++ )
	{
	    delete midNormals[i][j];
	}
    }

    for ( i = 0; i < 12; i++ )
    {
        delete midNormals[i];
    }
    
    delete midNormals;
    delete midpointLoc;
}


void MarchingCubes::FillCubeTriangleArrays ( void )
{
    int i, j;

    for ( i = 0; i < 256; i++ )
    {
        specialCases[i] = _specialCases[i];

        for ( j = 0; j < 16; j++ )
            triTable[i][j] = _triTable[i][j];
    }
}


/* ============================================================================
   Function:     fill_midpoint_location_array

   Purpose:      Fills the midpoint location array.

   Parmeters:    None.

   Returned:     None.
		 
   MTC 2/22/99
==============================================================================*/
void MarchingCubes::FillMidpointLocationArray ( void )
{
    midpointLoc[0].x = 0.5;
    midpointLoc[0].y = 0.0;
    midpointLoc[0].z = 0.0;

    midpointLoc[1].x = 1.0;
    midpointLoc[1].y = 0.0;
    midpointLoc[1].z = 0.5;          //     y
    //     |
    midpointLoc[2].x = 0.5;          //     |
    midpointLoc[2].y = 0.0;          //     /----x
    midpointLoc[2].z = 1.0;          //    /
    //   z
    midpointLoc[3].x = 0.0;
    midpointLoc[3].y = 0.0;
    midpointLoc[3].z = 0.5;

    midpointLoc[4].x = 0.5;
    midpointLoc[4].y = 1.0;
    midpointLoc[4].z = 0.0;

    midpointLoc[5].x = 1.0;
    midpointLoc[5].y = 1.0;
    midpointLoc[5].z = 0.5;

    midpointLoc[6].x = 0.5;
    midpointLoc[6].y = 1.0;
    midpointLoc[6].z = 1.0;

    midpointLoc[7].x = 0.0;
    midpointLoc[7].y = 1.0;
    midpointLoc[7].z = 0.5;

    midpointLoc[8].x = 0.0;
    midpointLoc[8].y = 0.5;
    midpointLoc[8].z = 0.0;

    midpointLoc[9].x = 1.0;
    midpointLoc[9].y = 0.5;
    midpointLoc[9].z = 0.0;

    midpointLoc[10].x = 1.0;
    midpointLoc[10].y = 0.5;
    midpointLoc[10].z = 1.0;

    midpointLoc[11].x = 0.0;
    midpointLoc[11].y = 0.5;
    midpointLoc[11].z = 1.0;
}  


/* ============================================================================
   Function:     fill_midpoint_normal_array

   Purpose:      Fills the midpoint normal array.

   Parmeters:    None.

   Returned:     None.
		 
   MTC 2/22/99
==============================================================================*/
void MarchingCubes::FillMidpointNormalArray ( void )
{
    int i, j;
    float x, y, z;

    for ( i = 0; i < 12; i++ )
    {
	for ( j = 0; j < 256; j++ )
        {
	    FindMidpointNormalSum ( i, j, &x, &y, &z );
	    midNormals[i][j][0] = x;
	    midNormals[i][j][1] = y;
	    midNormals[i][j][2] = z;
	}
    }
}


/* ===========================================================================
   Function:     find_midpoint_normal_sum

   Purpose:      Finds the sum of the normals to a midpoint in a cube.

   Parmeters:    midnum - midpoint number
                 cube_val - Index value of the cube.

   Returned:     norm_x, norm_y, norm_z - Normal sum
		 
   MTC 2/22/99
============================================================================*/
void MarchingCubes::FindMidpointNormalSum ( int midnum, int cube_val, 
					    float *norm_x, 
					    float *norm_y, float *norm_z )
{
    int i, j;
    float x, y, z;
    float previous[10][3];
    int count = 0;
    int add_to_normal;

    *norm_x = 0.0;
    *norm_y = 0.0;
    *norm_z = 0.0;

    for ( i = 0; triTable[cube_val][i] != -1; i += 3 ) 
    {
        /* Only count triangles using the specified midpoint */
        if ( ( triTable[cube_val][i  ] == midnum ) ||
	     ( triTable[cube_val][i+1] == midnum ) ||
	     ( triTable[cube_val][i+2] == midnum ) )
	{
	    FindNormalVector ( midpointLoc[triTable[cube_val][i  ]].x, 
			       midpointLoc[triTable[cube_val][i  ]].y,
			       midpointLoc[triTable[cube_val][i  ]].z,
				
			       midpointLoc[triTable[cube_val][i+1]].x, 
			       midpointLoc[triTable[cube_val][i+1]].y,
			       midpointLoc[triTable[cube_val][i+1]].z,
				  
			       midpointLoc[triTable[cube_val][i+2]].x, 
			       midpointLoc[triTable[cube_val][i+2]].y,
			       midpointLoc[triTable[cube_val][i+2]].z,
				  
			       &x, &y, &z);
	     
	    /* Check if this normal has already been added to the sum */
	    add_to_normal= 1;
	    for ( j = 0; j < count; j++ )
	    {
		if ( x == previous[j][0] && y == previous[j][1] 
		     && z == previous[j][2] )
		{
		    add_to_normal = 0;
		}
	    }

	    if ( add_to_normal )
	    {
		previous[count][0] = x;
		previous[count][1] = y;
		previous[count][2] = z;
		count++;
		 
		*norm_x += x;
		*norm_y += y;
		*norm_z += z;
	    }
	}
    }
}    


/* ==========================================================================
   Function:     find_normal_for_midpoint

   Purpose:      Given a cube and a midpoint on the cube, this function finds
                 the vertex normal for the midpoint.  

		 How it works:
		 1. Determine the index value of the four surrounding cubes.
		    The cube1 value is passed in and the other three are 
		    calculated.
		 2. Using the index of the four cubes, get the sum of the 
		    normals
		    for the cube by using the array found in normals.h
		 3. Sum up the sums for the cubes and find the unit normal.

   Parmeters:    *data - The array of data representing the cubes in the 
                         isosurface.
                 midpoint - The midpoint of the cube we are looking at.
		 cube1_val - Index value of cube1.
		 x, y, z - Reference point for cube1.
		 xsize, ysize, zsize - max value in data array.
		 *normx, *normy, *normz - return values for he normal.

   Returned:     1 for success, 0 otherwise.
		 
   MTC 2/22/99
============================================================================*/
int MarchingCubes::FindNormalForMidpoint ( unsigned char *data, int midpoint, 
					   int cube1_val, int special_case, 
					   int x, int y, int z, int xsize, 
					   int ysize, int zsize, float *normx,
					   float *normy, float *normz )
{
    int cube2_val = 0;
    int cube3_val = 0;
    int cube4_val = 0;

    int temp;
    int neighbor1, neighbor2, neighbor3, neighbor4, neighbor5, neighbor6;
    int retval;

    temp = y;
    y = z;
    z = temp;

    *normx = 0.0;
    *normy = 0.0;
    *normz = 0.0;

    if ( special_case )
    {
	neighbor1 = special_case & 1;
	neighbor2 = special_case & 2;
	neighbor3 = special_case & 4;	
	neighbor4 = special_case & 8;	
	neighbor5 = special_case & 16;	
	neighbor6 = special_case & 32;	
    }

    switch ( midpoint )
    {
	case 0:
	{	
	    if ( y + 1 >= ysize || z - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y+1)*xsize + (x  )];
	    cube3_val = data[(z-1)*ysize*xsize + (y  )*xsize + (x  )];
	    cube4_val = data[(z-1)*ysize*xsize + (y+1)*xsize + (x  )];
	  
	    *normx += midNormals[0][cube1_val][0];
	    *normy += midNormals[0][cube1_val][1];
	    *normz += midNormals[0][cube1_val][2];
	  
	    *normx += midNormals[2][cube2_val][0];
	    *normy += midNormals[2][cube2_val][1];
	    *normz += midNormals[2][cube2_val][2];

	    *normx += midNormals[4][cube3_val][0];
	    *normy += midNormals[4][cube3_val][1];
	    *normz += midNormals[4][cube3_val][2];
	  
	    *normx += midNormals[6][cube4_val][0];
	    *normy += midNormals[6][cube4_val][1];
	    *normz += midNormals[6][cube4_val][2];

	    if ( special_case )
	    {
		if ( neighbor4 )
		    *normz += 1.0;
		if ( neighbor6 )
		    *normy += 1.0;
	    }
	  
	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 1:
	{	
	    if ( x + 1 >= xsize || y + 1 >= ysize || z - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y  )*xsize + (x+1)];
	    cube3_val = data[(z-1)*ysize*xsize + (y  )*xsize + (x  )];
	    cube4_val = data[(z-1)*ysize*xsize + (y  )*xsize + (x+1)];
	  
	    *normx += midNormals[1][cube1_val][0];
	    *normy += midNormals[1][cube1_val][1];
	    *normz += midNormals[1][cube1_val][2];

	    *normx += midNormals[3][cube2_val][0];
	    *normy += midNormals[3][cube2_val][1];
	    *normz += midNormals[3][cube2_val][2];

	    *normx += midNormals[5][cube3_val][0];
	    *normy += midNormals[5][cube3_val][1];
	    *normz += midNormals[5][cube3_val][2];
	  
	    *normx += midNormals[7][cube4_val][0];
	    *normy += midNormals[7][cube4_val][1];
	    *normz += midNormals[7][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor1 )
		    *normx -= 1.0;
		if ( neighbor6 )
		    *normy += 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 2:
	{	
	    if ( y - 1 < 0 || z - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y-1)*xsize + (x  )];
	    cube3_val = data[(z-1)*ysize*xsize + (y-1)*xsize + (x  )];
	    cube4_val = data[(z-1)*ysize*xsize + (y  )*xsize + (x  )];
        
	    *normx += midNormals[2][cube1_val][0];
	    *normy += midNormals[2][cube1_val][1];
	    *normz += midNormals[2][cube1_val][2];

	    *normx += midNormals[0][cube2_val][0];
	    *normy += midNormals[0][cube2_val][1];
	    *normz += midNormals[0][cube2_val][2];

	    *normx += midNormals[4][cube3_val][0];
	    *normy += midNormals[4][cube3_val][1];
	    *normz += midNormals[4][cube3_val][2];
	  
	    *normx += midNormals[6][cube4_val][0];
	    *normy += midNormals[6][cube4_val][1];
	    *normz += midNormals[6][cube4_val][2];

	    if ( special_case )
	    {
		if ( neighbor3 )
		    *normz -= 1.0;
		if ( neighbor6 )
		    *normy += 1.0;
	    }
	  
	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 3:
	{	
	    if ( z - 1 < 0 || x - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y  )*xsize + (x-1)];
	    cube3_val = data[(z-1)*ysize*xsize + (y  )*xsize + (x-1)];
	    cube4_val = data[(z-1)*ysize*xsize + (y  )*xsize + (x  )];
	  
	    *normx += midNormals[3][cube1_val][0];
	    *normy += midNormals[3][cube1_val][1];
	    *normz += midNormals[3][cube1_val][2];

	    *normx += midNormals[1][cube2_val][0];
	    *normy += midNormals[1][cube2_val][1];
	    *normz += midNormals[1][cube2_val][2];

	    *normx += midNormals[5][cube3_val][0];
	    *normy += midNormals[5][cube3_val][1];
	    *normz += midNormals[5][cube3_val][2];
	  
	    *normx += midNormals[7][cube4_val][0];
	    *normy += midNormals[7][cube4_val][1];
	    *normz += midNormals[7][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor2 )
		    *normx += 1.0;
		if ( neighbor6 )
		    *normy += 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 4:
	{	
	    if ( y + 1 >= ysize || z + 1 >= zsize )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z+1)*ysize*xsize + (y  )*xsize + (x  )];
	    cube3_val = data[(z+1)*ysize*xsize + (y+1)*xsize + (x  )];
	    cube4_val = data[(z  )*ysize*xsize + (y+1)*xsize + (x  )];
	  
	    *normx += midNormals[4][cube1_val][0];
	    *normy += midNormals[4][cube1_val][1];
	    *normz += midNormals[4][cube1_val][2];

	    *normx += midNormals[0][cube2_val][0];
	    *normy += midNormals[0][cube2_val][1];
	    *normz += midNormals[0][cube2_val][2];

	    *normx += midNormals[2][cube3_val][0];
	    *normy += midNormals[2][cube3_val][1];
	    *normz += midNormals[2][cube3_val][2];
	  
	    *normx += midNormals[6][cube4_val][0];
	    *normy += midNormals[6][cube4_val][1];
	    *normz += midNormals[6][cube4_val][2];

	    if ( special_case )
	    {
		if ( neighbor4 )
		    *normz += 1.0;
		if ( neighbor5 )
		    *normy -= 1.0;
	    }
	  
	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 5:
	{	
	    if ( x + 1 >= xsize || z + 1 >= zsize )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z+1)*ysize*xsize + (y  )*xsize + (x  )];
	    cube3_val = data[(z+1)*ysize*xsize + (y  )*xsize + (x+1)];
	    cube4_val = data[(z  )*ysize*xsize + (y  )*xsize + (x+1)];
	  
	    *normx += midNormals[5][cube1_val][0];
	    *normy += midNormals[5][cube1_val][1];
	    *normz += midNormals[5][cube1_val][2];
	  
	    *normx += midNormals[1][cube2_val][0];
	    *normy += midNormals[1][cube2_val][1];
	    *normz += midNormals[1][cube2_val][2];

	    *normx += midNormals[3][cube3_val][0];
	    *normy += midNormals[3][cube3_val][1];
	    *normz += midNormals[3][cube3_val][2];
	  
	    *normx += midNormals[7][cube4_val][0];
	    *normy += midNormals[7][cube4_val][1];
	    *normz += midNormals[7][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor1 )
		    *normx -= 1.0;
		if ( neighbor5 )
		    *normy -= 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 6:
	{	
	    if ( z + 1 >= zsize || y - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z+1)*ysize*xsize + (y  )*xsize + (x  )];
	    cube3_val = data[(z  )*ysize*xsize + (y-1)*xsize + (x  )];
	    cube4_val = data[(z+1)*ysize*xsize + (y-1)*xsize + (x  )];
	  
	    *normx += midNormals[6][cube1_val][0];
	    *normy += midNormals[6][cube1_val][1];
	    *normz += midNormals[6][cube1_val][2];
	  
	    *normx += midNormals[2][cube2_val][0];
	    *normy += midNormals[2][cube2_val][1];
	    *normz += midNormals[2][cube2_val][2];

	    *normx += midNormals[4][cube3_val][0];
	    *normy += midNormals[4][cube3_val][1];
	    *normz += midNormals[4][cube3_val][2];
	  
	    *normx += midNormals[0][cube4_val][0];
	    *normy += midNormals[0][cube4_val][1];
	    *normz += midNormals[0][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor3 )
		    *normz -= 1.0;
		if ( neighbor5 )
		    *normy -= 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 7:
	{	
	    if ( z + 1 >= zsize || x - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z+1)*ysize*xsize + (y  )*xsize + (x  )];
	    cube3_val = data[(z  )*ysize*xsize + (y  )*xsize + (x-1)];
	    cube4_val = data[(z+1)*ysize*xsize + (y  )*xsize + (x-1)];
	  
	    *normx += midNormals[7][cube1_val][0];
	    *normy += midNormals[7][cube1_val][1];
	    *normz += midNormals[7][cube1_val][2];
	  
	    *normx += midNormals[3][cube2_val][0];
	    *normy += midNormals[3][cube2_val][1];
	    *normz += midNormals[3][cube2_val][2];

	    *normx += midNormals[5][cube3_val][0];
	    *normy += midNormals[5][cube3_val][1];
	    *normz += midNormals[5][cube3_val][2];
	  
	    *normx += midNormals[1][cube4_val][0];
	    *normy += midNormals[1][cube4_val][1];
	    *normz += midNormals[1][cube4_val][2];

	    if ( special_case )
	    {
		if ( neighbor2 )
		    *normx += 1.0;
		if ( neighbor5 )
		    *normy -= 1.0;
	    }	  

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 8:
	{	
	    if ( y + 1 >= ysize || x - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y  )*xsize + (x-1)];
	    cube3_val = data[(z  )*ysize*xsize + (y+1)*xsize + (x  )];
	    cube4_val = data[(z  )*ysize*xsize + (y+1)*xsize + (x-1)];
	  
	    *normx += midNormals[8][cube1_val][0];
	    *normy += midNormals[8][cube1_val][1];
	    *normz += midNormals[8][cube1_val][2];

	    *normx += midNormals[9][cube2_val][0];
	    *normy += midNormals[9][cube2_val][1];
	    *normz += midNormals[9][cube2_val][2];

	    *normx += midNormals[11][cube3_val][0];
	    *normy += midNormals[11][cube3_val][1];
	    *normz += midNormals[11][cube3_val][2];
	  
	    *normx += midNormals[10][cube4_val][0];
	    *normy += midNormals[10][cube4_val][1];
	    *normz += midNormals[10][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor4 )
		    *normz += 1.0;
		if ( neighbor2 )
		    *normx += 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 9:
	{	
	    if ( x + 1 >= xsize || y + 1 >= ysize )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y  )*xsize + (x+1)];
	    cube3_val = data[(z  )*ysize*xsize + (y+1)*xsize + (x+1)];
	    cube4_val = data[(z  )*ysize*xsize + (y+1)*xsize + (x  )];
	  
	    *normx += midNormals[9][cube1_val][0];
	    *normy += midNormals[9][cube1_val][1];
	    *normz += midNormals[9][cube1_val][2];

	    *normx += midNormals[8][cube2_val][0];
	    *normy += midNormals[8][cube2_val][1];
	    *normz += midNormals[8][cube2_val][2];

	    *normx += midNormals[11][cube3_val][0];
	    *normy += midNormals[11][cube3_val][1];
	    *normz += midNormals[11][cube3_val][2];
	  
	    *normx += midNormals[10][cube4_val][0];
	    *normy += midNormals[10][cube4_val][1];
	    *normz += midNormals[10][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor4 )
		    *normz += 1.0;
		if ( neighbor1 )
		    *normx -= 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 10:
	{	
	    if ( x + 1 >= xsize || y - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y-1)*xsize + (x  )];
	    cube3_val = data[(z  )*ysize*xsize + (y  )*xsize + (x+1)];
	    cube4_val = data[(z  )*ysize*xsize + (y-1)*xsize + (x+1)];
	  
	    *normx += midNormals[10][cube1_val][0];
	    *normy += midNormals[10][cube1_val][1];
	    *normz += midNormals[10][cube1_val][2];

	    *normx += midNormals[9][cube2_val][0];
	    *normy += midNormals[9][cube2_val][1];
	    *normz += midNormals[9][cube2_val][2];

	    *normx += midNormals[11][cube3_val][0];
	    *normy += midNormals[11][cube3_val][1];
	    *normz += midNormals[11][cube3_val][2];
	  
	    *normx += midNormals[8][cube4_val][0];
	    *normy += midNormals[8][cube4_val][1];
	    *normz += midNormals[8][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor3 )
		    *normz -= 1.0;
		if ( neighbor1 )
		    *normx -= 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	case 11:
	{	
	    if ( x - 1 < 0 || y - 1 < 0 )
	    {
		retval = 0;
		break;
	    }

	    cube2_val = data[(z  )*ysize*xsize + (y-1)*xsize + (x  )];
	    cube3_val = data[(z  )*ysize*xsize + (y-1)*xsize + (x-1)];
	    cube4_val = data[(z  )*ysize*xsize + (y  )*xsize + (x-1)];
	  
	    *normx += midNormals[11][cube1_val][0];
	    *normy += midNormals[11][cube1_val][1];
	    *normz += midNormals[11][cube1_val][2];

	    *normx += midNormals[8][cube2_val][0];
	    *normy += midNormals[8][cube2_val][1];
	    *normz += midNormals[8][cube2_val][2];

	    *normx += midNormals[9][cube3_val][0];
	    *normy += midNormals[9][cube3_val][1];
	    *normz += midNormals[9][cube3_val][2];
	  
	    *normx += midNormals[10][cube4_val][0];
	    *normy += midNormals[10][cube4_val][1];
	    *normz += midNormals[10][cube4_val][2];
	  
	    if ( special_case )
	    {
		if ( neighbor2 )
		    *normx += 1.0;
		if ( neighbor3 )
		    *normz -= 1.0;
	    }

	    MakeUnitNormal ( normx, normy, normz );
	    retval = 1;
	    break;
	}
	default:
	{
	    cout << "Invalid midpoint (%d)! Returning.\n";
	    retval = 0;
	    break;
	}
    }

    return ( retval );
}


/* Takes a normal of any length and makes it have length 1 */
void MarchingCubes::MakeUnitNormal ( float *x, float *y, float *z )
{
    float mag;

    mag = (float) sqrt ( (*x)*(*x) + (*y)*(*y) + (*z)*(*z) );

    if ( (int)(mag+0.5) == 0 )
    {
	*x = 0.0;
	*y = 0.0;
	*z = 0.0;
    }
    else
    {
	*x = (*x)/mag;
	*y = (*y)/mag;
	*z = (*z)/mag;
    }
}


/* Finds normal to a triangle... not unit normal */
void MarchingCubes::FindNormalVector ( float x1, float y1, float z1,
				       float x2, float y2, float z2,
				       float x3, float y3, float z3,
				       float *normx, float *normy, 
				       float *normz )
{
    float ux, uy, uz, vx, vy, vz, wx, wy, wz;
    /*u=p1-p2   v=p3-p2    w=u x v     wnorm=||w||   */

    ux = x1-x2;
    uy = y1-y2;           /*p1-p2*/
    uz = z1-z2;

    vx = x3-x2;
    vy = y3-y2;           /*p3-p2*/
    vz = z3-z2;

    wx = uy*vz-uz*vy;
    wy = uz*vx-ux*vz;     /*u cross v*/
    wz = ux*vy-uy*vx;

    *normx = -wx;
    *normy = -wy;
    *normz = -wz;
}


/*=============================================================================
  Function:    build_triangles_for_special_cube

  Purpose:     Determines if a 'special case' cube has any neighbors that cause
               the cube to require extra triangles to close holes.

  Parameters:  cube_val - value of the 'special case' cube.
               neighbor[1-6] - values of the neighboring cubes.
	       tris - Array of triangles.
	       num_triangles - Current number of triangles in the cube.

  Returned:    Extra triangles are returned through tris parameter.
               In the return variable, a value is sent depending on which 
	       neighbors were causing holes.
               
  MTC 3/3/99
  ===========================================================================*/
int MarchingCubes::BuildTrianglesForSpecialCube ( int cube_val, 
						  int neighbor1, 
						  int neighbor2, 
						  int neighbor3, 
						  int neighbor4, 
						  int neighbor5, 
						  int neighbor6, 
						  Cell_Triangle_t *tris, 
						  int *num_triangles )
{
    int retval = 0;
    int i, num_tris;

    num_tris = *num_triangles;

    InitExtraTriangles ( );

    if ( CheckCubeForExtraTriangles ( cube_val, neighbor1, 1 ) ) 
        retval |= BIT1;
    if ( CheckCubeForExtraTriangles ( cube_val, neighbor2, 2 ) ) 
        retval |= BIT2;
    if ( CheckCubeForExtraTriangles ( cube_val, neighbor3, 3 ) ) 
        retval |= BIT3;
    if ( CheckCubeForExtraTriangles ( cube_val, neighbor4, 4 ) ) 
        retval |= BIT4;
    if ( CheckCubeForExtraTriangles ( cube_val, neighbor5, 5 ) ) 
        retval |= BIT5;
    if ( CheckCubeForExtraTriangles ( cube_val, neighbor6, 6 ) ) 
        retval |= BIT6;

    if ( retval )
    {
	for ( i = 0; i < extra.num_added; i++ )
        {
	    tris[num_tris].a = extra.triangles[i  ];
	    tris[num_tris].b = extra.triangles[i+1];
	    tris[num_tris].c = extra.triangles[i+2];

	    num_tris++;
	}
    }

    *num_triangles = num_tris;

    return ( retval );
}


/*============================================================================
  Function:    init_extra_triangles

  Purpose:     Initializes the global structure variable extra.

  Parameters:  None.

  Returned:    None.
               
  MTC 3/3/99
  ===========================================================================*/
void MarchingCubes::InitExtraTriangles ( void )
{
    int i;

    for ( i = 0; i < MAX_EXTRA_TRIS*3; i++ )
    {
	extra.triangles[i] = -1;
    }

    extra.num_added = 0;
}


/*============================================================================
  Function:    AddTriangles

  Purpose:     Adds triangles to the global triangle structure bases on which 
               side it receives.

  Parameters:  side - Which side to add the extra triangles to.

  Returned:    None.
               
  MTC 3/3/99
  ===========================================================================*/
void MarchingCubes::AddTriangles ( int side )
{
    switch ( side )
    {
        case 1:  /* Right Side */
	{
            extra.triangles[extra.num_added  ] = 10; 
            extra.triangles[extra.num_added+1] = 5;  
            extra.triangles[extra.num_added+2] = 9;  extra.num_added++;
            extra.triangles[extra.num_added  ] = 10; 
            extra.triangles[extra.num_added+1] = 9;  
            extra.triangles[extra.num_added+2] = 1;  extra.num_added++;
            break;
	}
        case 2:  /* Left Side */
	{
            extra.triangles[extra.num_added  ] = 8;  
            extra.triangles[extra.num_added+1] = 7;  
            extra.triangles[extra.num_added+2] = 11; extra.num_added++;
            extra.triangles[extra.num_added  ] = 8;  
            extra.triangles[extra.num_added+1] = 11; 
            extra.triangles[extra.num_added+2] = 2;  extra.num_added++;
            break;
	}
        case 3:  /* Front Side */
	{
            extra.triangles[extra.num_added  ] = 11; 
            extra.triangles[extra.num_added+1] = 6;  
            extra.triangles[extra.num_added+2] = 10; extra.num_added++;
            extra.triangles[extra.num_added  ] = 11; 
            extra.triangles[extra.num_added+1] = 10; 
            extra.triangles[extra.num_added+2] = 2;  extra.num_added++;
            break;
	}
        case 4:  /* Back Side */
	{
            extra.triangles[extra.num_added  ] = 9;  
            extra.triangles[extra.num_added+1] = 4;  
            extra.triangles[extra.num_added+2] = 8;  extra.num_added++;
            extra.triangles[extra.num_added  ] = 9;  
            extra.triangles[extra.num_added+1] = 8;  
            extra.triangles[extra.num_added+2] = 0;  extra.num_added++;
            break;
	}
        case 5:  /* Top Side */
	{
            extra.triangles[extra.num_added  ] = 7;  
            extra.triangles[extra.num_added+1] = 4;  
            extra.triangles[extra.num_added+2] = 5;  extra.num_added++;
            extra.triangles[extra.num_added  ] = 7;  
            extra.triangles[extra.num_added+1] = 5;  
            extra.triangles[extra.num_added+2] = 6;  extra.num_added++;
            break;
	}
        case 6:  /* Bottom Side */
	{
            extra.triangles[extra.num_added  ] = 1;  
            extra.triangles[extra.num_added+1] = 0;  
            extra.triangles[extra.num_added+2] = 3;  extra.num_added++;
            extra.triangles[extra.num_added  ] = 1;  
            extra.triangles[extra.num_added+1] = 3;  
            extra.triangles[extra.num_added+2] = 2;  extra.num_added++;
            break;
	}
        default:
	{
            cout << "Invalide cube face: " << side << " (must be 1-6 only)\n!"; 
            break;
	}
    }
}


/*============================================================================
  Function:    CheckCubeForExtraTriangles

  Purpose:     Given the cube_val, neighbor_val, and which side to look at, 
               this function determines whether a hole will be formed.  If a 
	       hole will be formed, triangles are added to the cube to fill 
	       it in.

  Parameters:  cube_val - value of cube in question.
               neighbor_val - value of the neighbor being considered.
	       side - represents which side the neighbor is on.

  Returned:    1 if triangles were added, 0 if not.
               
  MTC 3/3/99

  Note:  This function was generated using the code found in 
         gen_neighbor_cases.c
  ===========================================================================*/
int MarchingCubes::CheckCubeForExtraTriangles ( int cube_val, 
						int neighbor_val, int side )
{
    int retval = 0;

    switch ( cube_val )
    {
	case 5:
	{
	    switch ( side )
	    {
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 10:
	{
	    switch ( side )
	    {
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 18:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 21:
	{
	    switch ( side )
	    {
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 22:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 24:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 26:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 28:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 30:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 33:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 36:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 37:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 41:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 42:
	{
	    switch ( side )
	    {
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 44:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 45:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 52:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 53:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 56:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 58:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 60:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 66:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 67:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 69:
	{
	    switch ( side )
	    {
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 72:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 73:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 74:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 75:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 80:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 81:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 82:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 83:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 84:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 85:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 86:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 88:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 89:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 90:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 92:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 117:
			case 181:
			case 213:
			case 229:
			case 245:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 97:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 101:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 104:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 105:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 106:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 120:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 61:
			case 173:
			case 181:
			case 188:
			case 189:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 109:
			case 173:
			case 229:
			case 233:
			case 237:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 129:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 131:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 132:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 133:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 134:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 135:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 138:
	{
	    switch ( side )
	    {
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 146:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 148:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 149:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 150:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 154:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 160:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 161:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 162:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 163:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 164:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 165:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 166:
	{
	    switch ( side )
	    {
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 168:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 169:
	{
	    switch ( side )
	    {
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 170:
	{
	    switch ( side )
	    {
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 172:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 5:
		{
		    switch ( neighbor_val )
		    {
			case 122:
			case 186:
			case 218:
			case 234:
			case 250:
			    AddTriangles ( 5 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 180:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 62:
			case 94:
			case 122:
			case 124:
			case 126:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 3:
		{
		    switch ( neighbor_val )
		    {
			case 94:
			case 158:
			case 214:
			case 218:
			case 222:
			    AddTriangles ( 3 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 193:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 194:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 195:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 197:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 87:
			case 91:
			case 93:
			case 94:
			case 95:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 202:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 6:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 171:
			case 173:
			case 174:
			case 175:
			    AddTriangles ( 6 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 210:
	{
	    switch ( side )
	    {
		case 1:
		{
		    switch ( neighbor_val )
		    {
			case 167:
			case 199:
			case 227:
			case 229:
			case 231:
			    AddTriangles ( 1 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 151:
			case 167:
			case 181:
			case 182:
			case 183:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	case 225:
	{
	    switch ( side )
	    {
		case 2:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 203:
			case 211:
			case 218:
			case 219:
			    AddTriangles ( 2 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		case 4:
		{
		    switch ( neighbor_val )
		    {
			case 91:
			case 107:
			case 121:
			case 122:
			case 123:
			    AddTriangles ( 4 );
			    retval = 1;
			    break;
			default:
			    break;
		    }
		}
		default:
		{
		    break;
		}
	    }
	}
	break;
	default:
	{
	    cout << "Invalid cube number (" << cube_val << ")! Returning.\n";
	    break;
	}
    }

    return ( retval );
}
