#include "icegradient_theme.h"
#include <gmodule.h>

/* Theme functions to export */
void theme_init (GtkThemeEngine * engine);
void theme_exit (void);

/* Exported vtable from th_draw */

extern GtkStyleClass icegradient_default_class;

/* internals */

/* external theme functions called */

static struct {
  gchar *name;
  guint token;
} theme_symbols[] = {
  {
  "background", TOKEN_BACKGROUND}
  , {
  "gradient", TOKEN_GRADIENT}
  , {
  "quadratic_gradient", TOKEN_QUADRATIC_GRADIENT}
  , {
  "diagonal_gradient", TOKEN_DIAGONAL_GRADIENT}
  , {
  "edge_gradient", TOKEN_EDGE_GRADIENT}
  , {
  "corner_gradient", TOKEN_CORNER_GRADIENT}
  , {
  "clean", TOKEN_CLEAN}
  , {
  "north_east", TOKEN_NORTH_EAST}
  , {
  "north_west", TOKEN_NORTH_WEST}
  , {
  "south_east", TOKEN_SOUTH_EAST}
  , {
  "south_west", TOKEN_SOUTH_WEST}
  , {
  "horizontal", TOKEN_HORIZONTAL}
  , {
  "vertical", TOKEN_VERTICAL}
  , {
  "automatic", TOKEN_AUTOMATIC}
  , {
  "decoration", TOKEN_DECORATION}
  , {
  "slash", TOKEN_SLASH}
  , {
  "back_slash", TOKEN_BACK_SLASH}
  , {
  "double_slash", TOKEN_DOUBLE_SLASH}
  , {
  "dot", TOKEN_DOT}
  , {
  "name", TOKEN_NAME}
  , {
  "detail", TOKEN_DETAIL}
  , {
  "edge_size", TOKEN_EDGE_SIZE}
  , {
  "corner_size", TOKEN_CORNER_SIZE}
  , {
  "none", TOKEN_NONE}
};

static guint n_theme_symbols =
  sizeof (theme_symbols) / sizeof (theme_symbols[0]);


static void
theme_detail_ref (Detail * data)
{
  data->refcount++;
}


static void
theme_detail_unref (Detail * data)
{
  data->refcount--;
  if (data->refcount == 0) {
    if (data->detail_name) {
      g_free (data->detail_name);
    }
    g_free (data);
  }
}


static void
theme_data_ref (ThemeData * theme_data)
{
  theme_data->refcount++;
}


static void
h_theme_detail_unref (gpointer key, gpointer value, gpointer user_data)
{
  theme_detail_unref ((Detail *) value);
}


static void
theme_data_unref (ThemeData * theme_data)
{
  theme_data->refcount--;
  if (theme_data->refcount == 0) {
    g_hash_table_foreach (theme_data->detail_list,
			  (GHFunc) h_theme_detail_unref, NULL);
    g_hash_table_destroy (theme_data->detail_list);
    g_free (theme_data);
  }
}


static inline guint
theme_parse_detail_aspect (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_peek_next_token (scanner);
  switch (token) {
  case G_TOKEN_COMMA:
    g_scanner_get_next_token (scanner);
    break;
  case G_TOKEN_RIGHT_CURLY:
    return G_TOKEN_NONE;
  default:
    return G_TOKEN_COMMA;
  }

  token = g_scanner_get_next_token (scanner);
  switch (token) {
  case TOKEN_HORIZONTAL:
    detail->aspect = HORIZONTAL;
    break;
  case TOKEN_VERTICAL:
    detail->aspect = VERTICAL;
    break;
  case TOKEN_AUTOMATIC:
    detail->aspect = AUTOMATIC;
    break;
  default:
    return TOKEN_VERTICAL;
  }

  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail_diagonal_aspect (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_peek_next_token (scanner);
  switch (token) {
  case G_TOKEN_COMMA:
    g_scanner_get_next_token (scanner);
    break;
  case G_TOKEN_RIGHT_CURLY:
    return G_TOKEN_NONE;
  default:
    return G_TOKEN_COMMA;
  }

  token = g_scanner_get_next_token (scanner);
  switch (token) {
  case TOKEN_NORTH_EAST:
    detail->aspect = NORTH_EAST;
    break;
  case TOKEN_NORTH_WEST:
    detail->aspect = NORTH_WEST;
    break;
  case TOKEN_SOUTH_EAST:
    detail->aspect = SOUTH_EAST;
    break;
  case TOKEN_SOUTH_WEST:
    detail->aspect = SOUTH_WEST;
    break;
  default:
    return TOKEN_NORTH_EAST;
  }

  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail_background (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_get_next_token (scanner);
  if (token != TOKEN_BACKGROUND)
    return TOKEN_BACKGROUND;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_LEFT_CURLY)
    return G_TOKEN_LEFT_CURLY;

  token = g_scanner_get_next_token (scanner);
  switch (token) {
  case TOKEN_GRADIENT:
    detail->background = GRADIENT;
    token = theme_parse_detail_aspect (scanner, detail);
    break;
  case TOKEN_QUADRATIC_GRADIENT:
    detail->background = QUADRATIC_GRADIENT;
    token = theme_parse_detail_aspect (scanner, detail);
    break;
  case TOKEN_EDGE_GRADIENT:
    detail->background = EDGE_GRADIENT;
    token = theme_parse_detail_aspect (scanner, detail);
    break;
  case TOKEN_CORNER_GRADIENT:
    detail->background = CORNER_GRADIENT;
    token = G_TOKEN_NONE;
    break;
  case TOKEN_DIAGONAL_GRADIENT:
    detail->background = DIAGONAL_GRADIENT;
    token = theme_parse_detail_diagonal_aspect (scanner, detail);
    break;
  case TOKEN_CLEAN:
    detail->background = CLEAN;
    token = G_TOKEN_NONE;
    break;
  default:
    return TOKEN_GRADIENT;
  }

  if (token != G_TOKEN_NONE) {
    return token;
  }

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_RIGHT_CURLY)
    return G_TOKEN_RIGHT_CURLY;

  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail_decoration (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_get_next_token (scanner);
  if (token != TOKEN_DECORATION)
    return TOKEN_DECORATION;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token (scanner);
  switch (token) {
  case TOKEN_SLASH:
    detail->decoration = SLASH;
    break;
  case TOKEN_BACK_SLASH:
    detail->decoration = BACK_SLASH;
    break;
  case TOKEN_DOUBLE_SLASH:
    detail->decoration = DOUBLE_SLASH;
    break;
  case TOKEN_DOT:
    detail->decoration = DOT;
    break;
  case TOKEN_NONE:
    detail->decoration = NONE;
    break;
  default:
    return TOKEN_SLASH;
  }

  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail_name (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_get_next_token (scanner);
  if (token != TOKEN_NAME)
    return TOKEN_NAME;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_STRING)
    return G_TOKEN_STRING;

  detail->detail_name = g_strdup (scanner->value.v_string);
  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail_edge_size (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_get_next_token (scanner);
  if (token != TOKEN_EDGE_SIZE)
    return TOKEN_EDGE_SIZE;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_INT)
    return G_TOKEN_INT;

  detail->edge_size = (guint) scanner->value.v_int;
  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail_corner_size (GScanner * scanner, Detail * detail)
{
  guint token;

  token = g_scanner_get_next_token (scanner);
  if (token != TOKEN_CORNER_SIZE)
    return TOKEN_CORNER_SIZE;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_INT)
    return G_TOKEN_INT;

  detail->corner_size = (guint) scanner->value.v_int;
  return G_TOKEN_NONE;
}


static inline guint
theme_parse_detail (GScanner * scanner, ThemeData * theme_data)
{
  guint token;
  Detail *detail;

  token = g_scanner_get_next_token (scanner);
  if (token != TOKEN_DETAIL)
    return TOKEN_DETAIL;

  token = g_scanner_get_next_token (scanner);
  if (token != G_TOKEN_LEFT_CURLY)
    return G_TOKEN_LEFT_CURLY;

  detail = g_new0 (Detail, 1);
  detail->refcount = 1;
  detail->background = CLEAN;
  detail->decoration = NONE;
  detail->diagonal_aspect = NORTH_EAST;
  detail->aspect = AUTOMATIC;
  detail->edge_size = 7;
  detail->corner_size = 5;

  token = g_scanner_peek_next_token (scanner);
  while (token != G_TOKEN_RIGHT_CURLY) {
    switch (token) {
    case TOKEN_NAME:
      token = theme_parse_detail_name (scanner, detail);
      break;
    case TOKEN_BACKGROUND:
      token = theme_parse_detail_background (scanner, detail);
      break;
    case TOKEN_DECORATION:
      token = theme_parse_detail_decoration (scanner, detail);
      break;
    case TOKEN_EDGE_SIZE:
      token = theme_parse_detail_edge_size (scanner, detail);
      break;
    case TOKEN_CORNER_SIZE:
      token = theme_parse_detail_corner_size (scanner, detail);
      break;
    default:
      g_scanner_get_next_token (scanner);
      token = G_TOKEN_RIGHT_CURLY;
      break;
    }

    if (token != G_TOKEN_NONE) {
      g_free (detail);
      return token;
    }
    token = g_scanner_peek_next_token (scanner);
  }

  g_scanner_get_next_token (scanner);
  token = G_TOKEN_NONE;
  g_hash_table_insert (theme_data->detail_list, detail->detail_name, detail);

  return token;
}


guint theme_parse_rc_style (GScanner * scanner, GtkRcStyle * rc_style)
{
  static GQuark scope_id = 0;
  ThemeData *theme_data;
  guint old_scope;
  guint token;
  guint i;

  /* Set up a new scope in this scanner. */

  if (!scope_id)
    scope_id = g_quark_from_string ("theme_engine");

  /* If we bail out due to errors, we *don't* reset the scope, so the
   * error messaging code can make sense of our tokens.
   */
  old_scope = g_scanner_set_scope (scanner, scope_id);

  /* Now check if we already added our symbols to this scope
   * (in some previous call to theme_parse_rc_style for the
   * same scanner.
   */

  if (!g_scanner_lookup_symbol (scanner, theme_symbols[0].name)) {
    g_scanner_freeze_symbol_table (scanner);
    for (i = 0; i < n_theme_symbols; i++) {
      g_scanner_scope_add_symbol (scanner, scope_id,
				  theme_symbols[i].name,
				  GINT_TO_POINTER (theme_symbols[i].token));
    }
    g_scanner_thaw_symbol_table (scanner);
  }

  /* We're ready to go, now parse the top level */

  theme_data = g_new0 (ThemeData, 1);
  theme_data->refcount = 1;
  theme_data->detail_list = g_hash_table_new (g_str_hash, g_str_equal);

  token = g_scanner_peek_next_token (scanner);
  while (token != G_TOKEN_RIGHT_CURLY) {
    token = theme_parse_detail (scanner, theme_data);
    if (token != G_TOKEN_NONE) {
      g_free (theme_data);
      return token;
    }
    token = g_scanner_peek_next_token (scanner);
  }

  g_scanner_get_next_token (scanner);

  rc_style->engine_data = theme_data;
  g_scanner_set_scope (scanner, old_scope);

  return G_TOKEN_NONE;
}


static void
theme_detail_merger (gpointer key, Detail * detail, GHashTable * htable)
{
  if (!g_hash_table_lookup (htable, key)) {
    g_hash_table_insert (htable, key, detail);
    theme_detail_ref (detail);
  }
}


void
theme_merge_rc_style (GtkRcStyle * dest, GtkRcStyle * src)
{
  ThemeData *src_data = src->engine_data;
  ThemeData *dest_data = dest->engine_data;

  if (!dest_data) {
    dest_data = g_new0 (ThemeData, 1);
    dest_data->refcount = 1;
    dest_data->detail_list = g_hash_table_new (g_str_hash, g_str_equal);
    dest->engine_data = dest_data;
  }

  g_hash_table_foreach (src_data->detail_list, (GHFunc) theme_detail_merger,
			dest_data->detail_list);
}


void
theme_rc_style_to_style (GtkStyle * style, GtkRcStyle * rc_style)
{
  ThemeData *data = rc_style->engine_data;

  style->klass = &icegradient_default_class;
  style->engine_data = data;
  theme_data_ref (data);
}


static void
theme_detail_dupper (gpointer key, Detail * detail, GHashTable * htable)
{
  g_hash_table_insert (htable, key, detail);
  theme_detail_ref (detail);
}


void
theme_duplicate_style (GtkStyle * dest, GtkStyle * src)
{
  ThemeData *src_data = src->engine_data;
  ThemeData *dest_data;

  dest_data = g_new (ThemeData, 1);
  dest_data->refcount = 1;
  dest_data->detail_list = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_foreach (src_data->detail_list, (GHFunc) theme_detail_dupper,
			dest_data->detail_list);

  dest->klass = &icegradient_default_class;
  dest->engine_data = dest_data;
}


void
theme_realize_style (GtkStyle * style)
{
}


void
theme_unrealize_style (GtkStyle * style)
{
}


void
theme_destroy_rc_style (GtkRcStyle * rc_style)
{
  theme_data_unref (rc_style->engine_data);
}


void
theme_destroy_style (GtkStyle * style)
{
  theme_data_unref (style->engine_data);
}


void
theme_set_background (GtkStyle * style,
		      GdkWindow * window, GtkStateType state_type)
{
  GdkPixmap *pixmap;
  gint parent_relative;
  ThemeData *data;

  g_return_if_fail (style != NULL);
  g_return_if_fail (window != NULL);

  if (style->bg_pixmap[state_type]) {
    if (style->bg_pixmap[state_type] == (GdkPixmap *) GDK_PARENT_RELATIVE) {
      pixmap = NULL;
      parent_relative = TRUE;
    } else {
      pixmap = style->bg_pixmap[state_type];
      parent_relative = FALSE;
    }

    data = style->engine_data;
/*    data->background = PIXMAP; */
    gdk_window_set_back_pixmap (window, pixmap, parent_relative);
  } else {
    gdk_window_set_background (window, &style->bg[state_type]);
  }
}


void
theme_init (GtkThemeEngine * engine)
{
  GtkRangeClass *rangeclass;

  engine->parse_rc_style = theme_parse_rc_style;
  engine->merge_rc_style = theme_merge_rc_style;
  engine->rc_style_to_style = theme_rc_style_to_style;
  engine->duplicate_style = theme_duplicate_style;
  engine->realize_style = theme_realize_style;
  engine->unrealize_style = theme_unrealize_style;
  engine->destroy_rc_style = theme_destroy_rc_style;
  engine->destroy_style = theme_destroy_style;
  engine->set_background = theme_set_background;

  rangeclass = (GtkRangeClass *) gtk_type_class (gtk_range_get_type ());
  rangeclass->slider_width = DEFAULT_SLIDER_WIDTH;
  rangeclass->stepper_size = DEFAULT_SLIDER_WIDTH;
  rangeclass->min_slider_size = DEFAULT_MIN_SLIDER_SIZE;
}


void
theme_exit (void)
{
}


/* The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
*/
G_MODULE_EXPORT const gchar *g_module_check_init (GModule * module);

const gchar *
g_module_check_init (GModule * module)
{
  return gtk_check_version (GTK_MAJOR_VERSION,
			    GTK_MINOR_VERSION,
			    GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}
