/* $Id: icon.c,v 10.1 92/10/06 23:03:18 ca Exp $ */
/*
 * MaRS Maryland Routing Simulator
 * Copyright (c) 1991 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Eric Bull
 *          Systems Design and Analysis Group
 *          Department of Computer Science 
 *          University of Maryland at College Park. 
 */
#include <Xm/Xm.h>
#include <Xm/Label.h>

#include "xm-icon.h"
#include "xm-util.h"
#include "xm-network.h"
#include "comptypes.h"


/* This file contains functions for creating/destroying, and manipulating component icons.

Widget CreateIcon(Widget parent, char *name, int type, *x, *y);      Create an icon.
void DestroyIcon(Widget icon);                                       Destroy an icon.
void DrawIcon(Widget icon);                                          Draw (manage) an icon.
void EraseIcon(Widget icon);                                         Erase (unmanage) an icon.
void MoveIcon(Widget icon, int *new_x, int *new_y);                  Move and icon.
void SelectComponentIcon(MComponent *comp);                          Select the component's icon.
void UnselectComponentIcon(MComponent *comp);                        Unselect the component's icon.
void ChangeComponentIconName(MComponent *comp, char *new_name);      Change the name drawn in the component's icon.

void GetIconDimensions(Widget icon, int *return_w, *return_h)         Returns the dimensions of the icon.
void TranslateIconCoords(Widget icon, Position x, y, *rel_x, *rel_y)  Returns the root relative coordinates of the icon.

*/

/********************************************************************************************************************************************/
void GetIconDimensions(Widget icon, int *return_w, int *return_h) { /* Return the width and height of the icon. */
   Dimension w,h;
   XtVaGetValues(icon,XmNwidth, &w, XmNheight, &h, NULL);
   *return_w=(int)w;
   *return_h=(int)h;
}

/********************************************************************************************************************************************/
void TranslateIconCoords(Widget icon, Position x, Position y, Position *rel_x, Position *rel_y) {
/* Translate the icon's position to root relative coordinates.  The icon's position is really the center of the icon. */
  Dimension w,h;

  XtVaGetValues(icon,XmNwidth, &w, XmNheight, &h, NULL);
  XtTranslateCoords(icon,w/2+x,h/2+y,rel_x,rel_y);
}

/*********************************************** Widget Drawing Functions **************************************************************/
void UnselectComponentIcon(MComponent *comp) { /* background=component_color; foreground=fore_color. */
  Pixel fore_color,back_color;
  Dimension bw;

  TRACE("Unselect Icon");
  
  if (comp->type==TYPE_GROUP) 
     back_color=the_environment.group_color;
  else if (comp->type==TYPE_METER)
     back_color=the_environment.meter_color;
  else 
     back_color=the_environment.component_color[comp->type];

  fore_color=the_environment.fore_color;
  
  XtVaSetValues(comp->icon,
		XmNbackground, back_color,
		XmNforeground, fore_color,
		NULL);

}


/********************************************************************************************************************************************/
void SelectComponentIcon(MComponent *comp) { /* background=fore_color; foreground=component_color. */
  Pixel fore_color,back_color;
  Dimension bw;

  TRACE("Select Icon");

  if (comp->type == TYPE_GROUP)
     fore_color=the_environment.group_color;
  else if (comp->type == TYPE_METER)
     fore_color=the_environment.meter_color;
  else 
     fore_color=the_environment.component_color[comp->type];

  back_color=the_environment.fore_color;
 
  XtVaSetValues(comp->icon,
		XmNbackground, back_color,
		XmNforeground, fore_color,
		NULL);
}    


/*********************************************************************************************************************************************/
/* This routine creates an icon of the given type as a child of the given parent.  The icon's position is given by the arguments x and y.
   If the given position is not accepted, the new position will be returned in x and y. */

Widget CreateIcon(Widget parent,char *name,int type,int *x, int *y)
{
  Widget icon;
  Dimension w,h;
  char *icon_name;
  char buf[40];
  XmString str;
  Pixel fore_color, back_color;
  Position newx,newy;

  TRACE("CreateIcon");

  if (type == TYPE_GROUP) {
     back_color=the_environment.group_color;
     icon_name="GROUP";
  }
  else if (type == TYPE_METER) {
     back_color=the_environment.meter_color;
     icon_name="METER";
  }
  else {
     back_color=the_environment.component_color[type];
     icon_name=component_types[type].typename;
  }

  fore_color=the_environment.fore_color;

  str=XmStringCreateSimple((name==NULL)? icon_name : name);
  icon=XtVaCreateWidget("component_icon",xmLabelWidgetClass,parent,
			XmNlabelString, str,
			XmNbackground, back_color,
			XmNforeground, fore_color,
			NULL);
  XmStringFree(str);
  
/* Position icon so that it is centered on the given coordinates. */
  XtVaGetValues(icon,XmNwidth, &w, XmNheight, &h, NULL);
  if (*x-w/2<0) {
     *x += -(*x-w/2);
  }
  if (*y-h/2<0) {
     *y += -(*y-h/2);
  }

  XtVaSetValues(icon, XmNx, *x-w/2, XmNy, *y-h/2, NULL);
  return(icon);
}

/****************************************************************************************************************************/
void DestroyIcon(Widget icon) {

  TRACE("DestroyIcon");

  XtDestroyWidget(icon);
}

/*****************************************************************************************************************************/
void ChangeComponentIconName(MComponent *comp, char *new_name) {
  XmString str;
  Dimension w,h;
  int diff_x=0;
  int diff_y=0;

  TRACE("ChangeComponentIconName");

  str=XmStringCreateSimple(new_name);
  XtVaSetValues(comp->icon,XmNlabelString,str,NULL);

  /* Recenter the icon. */
  XtVaGetValues(comp->icon,
		XmNwidth, &w,
		XmNheight, &h,
		NULL);


  if (comp->x-w/2<0) {
     diff_x=-(comp->x-w/2);
  }
  if (comp->y-h/2<0) {
     diff_y=-(comp->y-h/2);
  }
  if (diff_x!=0 || diff_y!=0) {
     MoveComponent(comp, diff_x, diff_y);
  }
  SetWidgetCoordinates(comp->icon, comp->x-w/2, comp->y-h/2);
  XmStringFree(str);
}

/*****************************************************************************************************************************/
void DrawIcon(Widget icon) { 
  TRACE("DrawIcon");

  XtManageChild(icon);
}

/******************************************************************************************************************************/
void EraseIcon(Widget icon) {

  TRACE("EraseIcon");

  XtUnmanageChild(icon);
}

/******************************************************************************************************************************/
/* Moves the icon to the new coordinates.  If the new coordinates push the icon's label off the edge of the network, then
   the icon is moved to the closest position that keeps it's label entirely within the network window.  The true icon
   coordinates are returned in new_x and new_y. */
void MoveIcon(Widget icon, int *new_x, int *new_y) {
  Dimension w,h;
  int x,y;

  TRACE("MoveIcon");

  /* Center icon on new_x, new_y. */
  XtVaGetValues(icon,XmNwidth, &w, XmNheight, &h, NULL);
  
  x=*new_x-w/2;
  y=*new_y-h/2;

  if (x < 0) {
     x=0;
     *new_x=w/2;
  }
  if (y < 0) {
     y=0;
     *new_y=h/2;
  }

  SetWidgetCoordinates(icon, x, y);
}







