/* $Id: info_w.c,v 10.1 92/10/06 23:03:20 ca Exp $ */
/*
 * MaRS Maryland Routing Simulator
 * Copyright (c) 1991 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Eric Bull
 *          Systems Design and Analysis Group
 *          Department of Computer Science 
 *          University of Maryland at College Park. 
 */
#include <Xm/Xm.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/SeparatoG.h>
#include <Xm/Form.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/MessageB.h>
#include <Xm/Frame.h>

#include "sim.h"
#include "simx.h"
#include "xm-info_w.h"
#include "xm-util.h"
#include "xm-network.h"
#include "list.h"
#include "xm-comp_trans.h"
#include "xm-param_w.h"
#include "comptypes.h"
#include "xm-main_w.h"

/* This file contains routines associated with component information windows and group information windows.
   The following routines are used for opening and closing information windows:

   Opening and closing component information windows:
      void OpenComponentInformationWindow(MComponent *comp, int given_x, given_y, valid_coords);
      void CloseComponentInformationWindow(MComponent *comp);

   Opening and closing group information windows:
      void OpenGroupInformationWindow(MComponent *comp, int given_x, given_y, valid_coords);
      void CloseGroupInformationWindow(MComponent *comp);

   void CloseInformationWindow(MComponent *comp);   Close any information window (group or component).


   The following action routines for mars.componentTranslations are defined:

      open_info_w_tn()         Open an information window.
      close_info_w_tn()        Close an information window.
      open_close_info_w_tn()   Open an information window if not already open, else close the window.

*/
    
typedef struct CompInfoWinData { 
  MComponent *comp;
  Widget dialog_widget;  /* We will need this widget to close the dialog. */
  int cancel_changes;   /* This is set before calling CloseInformationWindow, to indicate whether or not 
			   changes should be accepted or canceled. */
  void *name;  /* The name_input textfield will keep a pointer to this as user data. */
  unsigned short initial_hide_settings;
} CompInfoWinData;           

typedef struct GroupInfoWinData { 
  MComponent *comp;
  Widget dialog_widget; /* Widgets we will need to access later. */
  int cancel_changes;   /* This is set before calling CloseInformationWindow, to indicate whether or not 
			   changes should be accepted or canceled. */

  Widget name_textf;    
  String current_name;  /* Check this to see if the name in the textfield is actually a different name. */
  String initial_name;  /* Backup in case changes are canceled. */
  Widget open_close_button;
  unsigned short group_open;   
} GroupInfoWinData;           


Widget BuildComponentInformationWindow(CompInfoWinData *data);

void comp_info_children_toggle_cb(Widget w, CompInfoWinData *window_data, XmToggleButtonCallbackStruct *cbs);
void comp_info_meter_toggle_cb(Widget w, CompInfoWinData *window_data, XmToggleButtonCallbackStruct *cbs);
void comp_info_ok_action(Widget w, CompInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void comp_info_apply_action(Widget w, CompInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void comp_info_cancel_action(Widget w, CompInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void comp_confirm_kill_cb(Widget w, CompInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void comp_info_kill_action(Widget w, CompInfoWinData *window_data, XmAnyCallbackStruct *cbs);

Widget BuildGroupInformationWindow(GroupInfoWinData *data);

void group_info_ok_action(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void group_info_apply_action(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void group_info_cancel_action(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void group_confirm_kill_cb(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void group_info_kill_action(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void group_info_ungroup_action(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);
void group_info_open_action(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs);

void initialize_open_close_button(Widget w, GroupInfoWinData *window_data);

/*======================================== Action Routines for mars.componentTranslations  ===================================================*/
void close_info_w_tn(w,event,args,num_args)
     Widget w;
     XEvent *event;
     String *args;
     int *num_args;
{
  MComponent *comp;
  comp=GetComponentOfIcon(w);

  if (CompInfoWindowOpen(comp)) {
     if (comp->type==TYPE_GROUP) CloseGroupInformationWindow(comp);
     else if (comp->type==TYPE_METER); /* do nothing */
     else CloseComponentInformationWindow(comp);
  }
}
   
/**********************************************************************************************************************************************/
void open_info_w_tn(w,event,args,num_args)
     Widget w;
     XEvent *event;
     String *args;
     int *num_args;
{
  MComponent *comp;

  TRACE("open_info_w_cb");

  comp=GetComponentOfIcon(w);
  
  if (!CompInfoWindowOpen(comp)) {
    if (comp->type==TYPE_GROUP) OpenGroupInformationWindow(comp,0,0,FALSE);
    else if (comp->type==TYPE_METER); /* do nothing */
    else OpenComponentInformationWindow(comp,0,0,FALSE);
  }
  else
     RaiseWidget(((InfoWinData *)comp->info_window)->dialog_widget);
}
   
/**********************************************************************************************************************************************/
void open_close_info_w_tn(w,event,args,num_args)  /* Open the window if not open, else close the window. */
     Widget w;
     XEvent *event;
     String *args;
     int *num_args;
{
  MComponent *comp;

  TRACE("open_info_w_cb");

  comp=GetComponentOfIcon(w);
  
  if (!CompInfoWindowOpen(comp)) {
    if (comp->type==TYPE_GROUP) OpenGroupInformationWindow(comp,0,0,FALSE);
    else if (comp->type==TYPE_METER); /* do nothing */
    else OpenComponentInformationWindow(comp,0,0,FALSE);
  }
  else
     if (comp->type!=TYPE_METER) 
	CloseInformationWindow(comp);
}

/**********************************************************************************************************************************************/
void CloseInformationWindow(MComponent *comp) {  /* Close any information window (Group or Component) */
  if (comp->type==TYPE_GROUP) CloseGroupInformationWindow(comp);
  else CloseComponentInformationWindow(comp);
}

/*=============================================================================================================================================
                                              COMPONONENT INFORMATION WINDOW FUNCTIONS
 ==============================================================================================================================================*/
void OpenComponentInformationWindow(MComponent *comp, int given_x, int given_y, int valid_coords) {
  Widget comp_info_w;
  CompInfoWinData *window_data;
  
  if (CompInfoWindowOpen(comp)) {
     WARNING("Attempting to open an already open window");
     return;
  }
  window_data=(CompInfoWinData *)sim_malloc(sizeof(CompInfoWinData));
  if (window_data==NULL) return;
  window_data->comp=comp;
  
  window_data->initial_hide_settings=0;
  window_data->initial_hide_settings|=(CompChildrenHidden(comp))? HIDE_CHILDREN : 0;
  window_data->initial_hide_settings|=(CompMetersHidden(comp))? HIDE_METERS : 0;

  window_data->dialog_widget=comp_info_w=BuildComponentInformationWindow(window_data);
  window_data->cancel_changes=0; /* By default, don't cancel the changes. */

/* Set the shell position. */
  if (valid_coords)  {
     if (given_x<0) 
	given_x=0;	
     if (given_y<0) 
	given_y=0;
     SetWidgetCoordinates(comp_info_w, given_x, given_y);
  }
  else {
     SetWidgetCoordinates(comp_info_w, comp->x, comp->y);
  }
   
  window_data->comp->info_window=window_data;
  
  l_traverse(comp->children,SaveVisibility); 

  XtManageChild(comp_info_w);
  RaiseWidget(comp_info_w);
  SetStatusInfoWindowOpen(comp);
}

/**********************************************************************************************************************************************/
void CloseComponentInformationWindow(MComponent *comp) 
{
   CompInfoWinData *window_data=comp->info_window;
   COMPONENT *scomponent=comp->scomponent;
   ParameterTextfield *p_textf=window_data->name;

   if (!CompInfoWindowOpen(comp)) {
      WARNING("Attempting to close an already closed window");
      return;
   }
   
   if (window_data->cancel_changes) { /* Cancel the changes. */

      restore_initial_parameter_value(p_textf, scomponent);  /* Restore the component's name. */
  
      l_traverse(comp->children,RestoreVisibility);          /* Restore visibility of all children. */

      /* Restore the hide children and hide meters settings. */
      if (window_data->initial_hide_settings & HIDE_CHILDREN) 
	 SetStatusChildrenHidden(comp);
      else 
	 SetStatusChildrenNotHidden(comp);

      if (window_data->initial_hide_settings & HIDE_METERS) 
	 SetStatusMetersHidden(comp);
      else
	 SetStatusMetersNotHidden(comp);
   }      
   else {  /* Accept the changes. Set the component's name. */
      PARAM *parameter;
      Widget textfield;
      String new_value;

      scomponent=comp->scomponent;
      parameter=p_textf->parameter;

      new_value=XmTextFieldGetString(p_textf->textfield);

      if ((0!=strcmp(new_value, p_textf->current_value))&& parameter->p_input) {
	 if (parameter->p_input(parameter, new_value, scomponent)) {
	    ChangeComponentName(comp,new_value);
	 }	
      }
      XtFree(new_value);
   }

   /* Destroy the window. */
   comp->info_window=NULL;
   SetStatusInfoWindowClosed(comp);
   XtUnmanageChild(window_data->dialog_widget);
   XtDestroyWidget(window_data->dialog_widget);
   free(((ParameterTextfield *)(window_data->name))->initial_value);
   free(((ParameterTextfield *)(window_data->name))->current_value);
   free(window_data->name);
   free(window_data);
}

/********************************************* BUILD COMPONENT INFORMATION WINDOW  **********************************************************/
ActionAreaItem comp_info_action1[]={
  {"Parameters",open_param_w_cb, NULL, NULL}
};

ActionAreaItem comp_info_action2[]={
  { "Ok", comp_info_ok_action, NULL, NULL},
  {"Cancel", comp_info_cancel_action, NULL, NULL},
  {"Kill", comp_confirm_kill_cb, NULL, NULL}
};

/**********************************************************************************************************************************************/
Widget BuildComponentInformationWindow(window_data) /* Create a form dialog as follows:

						Label '<type> Information Window'
						Double Separator
						TextInputField(data->name);
						Separator
                                                HideMeters?
                                                HideAllChildren?
						Separator
						Parameters Button
						Separator
						Action Area */
     
     
     CompInfoWinData *window_data;  /* This function reads values from data->comp, window_data->hidesettings, window_data->name
				       and writes a value to window_data->name_textf. */
{
  Widget row_col,comp_info_w,comp_info_w_form,label,sep,name_input,hide_label,toggle,action_area1,action_area2,form,kill_button;
  MComponent *comp;
  Pixel bg;
  XmString str;
  char buf[32];
  int i;
  
  comp=(MComponent *)window_data->comp;
  
  bg=the_environment.component_color[comp->type];

  comp_info_w=XmCreateForm(network_w,"comp_info_w",NULL,0);

  XtVaSetValues(comp_info_w, XmNbackground, bg, NULL);

  /* TITLE LINE */
  sprintf(buf,"%s Information",component_types[comp->type].typename);
  str=XmStringCreateSimple(buf);
  label=XtVaCreateManagedWidget("title",xmLabelWidgetClass,comp_info_w,
				XmNbackground, bg,
				XmNlabelString, str,
				XmNtopAttachment, XmATTACH_FORM,
				XmNrightAttachment, XmATTACH_FORM,
				XmNleftAttachment, XmATTACH_FORM,
				XmNtranslations, the_environment.move_window_translations,
                                XmNuserData,comp_info_w,
				NULL);
  XmStringFree(str);

  sep=CreateSeparator(comp_info_w, XmDOUBLE_LINE, label);

/* NAME TEXTFIELD */
  {
    COMPONENT *scomponent;
    PARAM *parameter;
    ParameterTextfield *p_textf;
    String initial_name;
    Widget textfield;

    scomponent=window_data->comp->scomponent;
    parameter=(PARAM *)scomponent->co_params->q_head;
    p_textf=(ParameterTextfield *)sim_malloc(sizeof(ParameterTextfield));
    initial_name=parameter->p_make_text(scomponent,parameter);

    p_textf->parameter=parameter;
    p_textf->initial_value=new_string(initial_name);
    p_textf->current_value=new_string(initial_name);
    name_input=CreateTextInput(comp_info_w, "Name:", initial_name, &textfield);
    p_textf->textfield=textfield;
    window_data->name=p_textf;

    SetWidgetUserData(textfield,p_textf);
  
    XtVaSetValues(name_input,
		  XmNtopAttachment, XmATTACH_WIDGET,
		  XmNtopWidget, sep,
		  XmNleftAttachment, XmATTACH_FORM,
		  NULL);

    if (parameter->p_flags & (InputMask | ModifyMask))
       XtAddCallback(textfield, XmNactivateCallback, change_parameter_cb, window_data);
    else
       XtVaSetValues(textfield, XmNeditable,False,NULL);

  }    
  
  sep=CreateSeparator(comp_info_w, XmSINGLE_LINE, name_input);
  
  /* HIDE COMPONENTS PART */
  form=XtVaCreateWidget("hide_children_form", xmFormWidgetClass, comp_info_w,
			XmNbackground, bg,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, sep,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL);
  
  
  hide_label=XtVaCreateManagedWidget("label", xmLabelWidgetClass, form,
				     XmNbackground, bg,
				     XmNleftAttachment, XmATTACH_FORM,
				     XmNtopAttachment, XmATTACH_FORM,
				     XmNbottomAttachment, XmATTACH_FORM,
				     NULL);
  
  row_col=XtVaCreateWidget("toggle_button_row_column", xmRowColumnWidgetClass, form,
			   XmNbackground, bg,
			   XmNleftAttachment, XmATTACH_WIDGET,
			   XmNleftWidget, hide_label,
			   XmNtopAttachment, XmATTACH_FORM,
			   XmNbottomAttachment, XmATTACH_FORM,
			   XmNrightAttachment, XmATTACH_FORM,
			   NULL);
     
/* CHILDREN TOGGLE */
  toggle=XtVaCreateManagedWidget("children_toggle_button", xmToggleButtonWidgetClass, row_col,
				 XmNbackground, bg,
				 XmNset, (window_data->initial_hide_settings & HIDE_CHILDREN)? True : False,
				 NULL);
  XtAddCallback(toggle, XmNvalueChangedCallback, comp_info_children_toggle_cb, window_data); 


/* METER TOGGLE */
  toggle=XtVaCreateManagedWidget("meters_toggle_button", xmToggleButtonWidgetClass, row_col,
				 XmNbackground, bg,
				 XmNset, (window_data->initial_hide_settings & HIDE_METERS)? True : False,
				 NULL);
  XtAddCallback(toggle, XmNvalueChangedCallback, comp_info_meter_toggle_cb, window_data); 

  XtManageChild(row_col);
  XtManageChild(form);
  
  sep=CreateSeparator(comp_info_w, XmSINGLE_LINE, form);
  
  /* PARAMETERS BUTTON */
  for (i=0;i<XtNumber(comp_info_action1);i++) comp_info_action1[i].data=window_data;
  action_area1=CreateActionArea(comp_info_w,comp_info_action1,XtNumber(comp_info_action1),-1);
  XtVaSetValues(action_area1,
		XmNleftAttachment,XmATTACH_FORM,
		XmNrightAttachment,XmATTACH_FORM,
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget, sep,
		NULL);

  sep=CreateSeparator(comp_info_w, XmSINGLE_LINE, action_area1);
  
  
  /* ACTION_AREA */
  for (i=0;i<XtNumber(comp_info_action2);i++) comp_info_action2[i].data=window_data;
  comp_info_action2[i-1].callback=the_environment.killwarn? comp_confirm_kill_cb : comp_info_kill_action;
  action_area2=CreateActionArea(comp_info_w,comp_info_action2,XtNumber(comp_info_action2),0);
  XtVaSetValues(action_area2,
		XmNleftAttachment,XmATTACH_FORM,
		XmNbottomAttachment,XmATTACH_FORM,
		XmNrightAttachment,XmATTACH_FORM,
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget, sep,
		NULL);
  return(comp_info_w);
}


/****************************************Component Information Window Callback Functions ********************************************************/
void close_confirm_kill_comp_dialog(w,window_data) 
     Widget w;
     CompInfoWinData *window_data;
{
  XtRemoveCallback(w,XmNokCallback, comp_info_kill_action, window_data);
  XtRemoveCallback(w,XmNokCallback, close_confirm_kill_comp_dialog, window_data);
  XtRemoveCallback(w,XmNcancelCallback, close_confirm_kill_comp_dialog, window_data);    
  XtUnmanageChild(w);
}

/*******************************************************************************************************************************************/
void comp_confirm_kill_cb(w,window_data,cbs) 
     Widget w;
     CompInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
/* We could make this dialog static, but remember, the window_data changes each time it is called. */
  static Widget question_dialog=NULL;
  CoordinatePair pos;

  if (!question_dialog) {
    question_dialog=XmCreateQuestionDialog(main_w,"confirm_kill_component_dialog",NULL,0);
  }
  
  XtAddCallback(question_dialog,XmNokCallback, comp_info_kill_action, window_data);
  XtAddCallback(question_dialog,XmNokCallback, close_confirm_kill_comp_dialog, window_data);
  XtAddCallback(question_dialog,XmNcancelCallback, close_confirm_kill_comp_dialog, window_data);    

  pos.x=((XButtonEvent *)cbs->event)->x_root;
  pos.y=((XButtonEvent *)cbs->event)->y_root;
  XtAddCallback(XtParent(question_dialog), XmNpopupCallback, SetShellPosition, &pos);
  XtManageChild(question_dialog);
  XtRemoveCallback(XtParent(question_dialog), XmNpopupCallback, SetShellPosition, &pos);
}

/**********************************************************************************************************************************************/
void comp_info_kill_action(w,window_data,cbs) 
     Widget w;
     CompInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
  MComponent *comp=window_data->comp;
  
  if (CompSelected(comp)) 
     UnselectComponent(comp);
  DestroyComponent(comp);
}

/**********************************************************************************************************************************************/
void comp_info_cancel_action(w,window_data,cbs)
     Widget w;
     CompInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
   MComponent *comp=window_data->comp;
   window_data->cancel_changes=1;
   CloseComponentInformationWindow(comp);
}

/**********************************************************************************************************************************************/
void comp_info_ok_action(w,window_data,cbs)
     Widget w;
     CompInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
  MComponent *comp=window_data->comp;
  window_data->cancel_changes=0;
  CloseComponentInformationWindow(comp);
}


/**********************************************************************************************************************************************/
void comp_info_children_toggle_cb(w,window_data,cbs) 
     Widget w;
     CompInfoWinData *window_data;
     XmToggleButtonCallbackStruct *cbs;
{
   MComponent *comp=window_data->comp;

   if (cbs->set) { /* Hide children */
      SetStatusChildrenHidden(comp);
      l_traverse(comp->children,MakeChildrenInvisible);
   }
   else {  /* Unhide children */
      SetStatusChildrenNotHidden(comp);
      l_traverse(comp->children,MakeChildrenVisible);
   }
}

/**********************************************************************************************************************************************/
void comp_info_meter_toggle_cb(w,window_data,cbs)							
     Widget w;
     CompInfoWinData *window_data;
     XmToggleButtonCallbackStruct *cbs;
{
   MComponent *comp=window_data->comp;

   if (cbs->set) { /* Hide meters */
      SetStatusMetersHidden(comp);
      l_traverse(comp->children,MakeMetersInvisible);
   }
   else {  /* Unhide meters */
      SetStatusMetersNotHidden(comp);
      l_traverse(comp->children,MakeMetersVisible);
   }
}


/**********************************************************************************************************************************************/
void SaveVisibility(MComponent *comp)  /* Note: Hiding attached components only affects the direct children, so we don't have to 
                                        recurse on the children of this component. */ 
     
{
  if (CompVisible(comp)) SetStatusWasVisible(comp);
  else SetStatusWasNotVisible(comp);
}

/**********************************************************************************************************************************************/
void RestoreVisibility(MComponent *comp)
{
  if (CompWasVisible(comp)!=CompVisible(comp)) {
    if (CompWasVisible(comp)) MakeVisible(comp);
    else {
      MakeInvisible(comp);
    }
  }
}

/**********************************************************************************************************************************************/
void MakeChildrenVisible(MComponent *comp)  /* Note: Hiding attached components only affect the direct children of the component, so 
						we don't have recurse on this child's children. */
{
  if (comp->type!=TYPE_METER && !CompVisible(comp)) MakeVisible(comp);
}

/**********************************************************************************************************************************************/
void MakeChildrenInvisible(MComponent *comp)
{
  if (comp->type!=TYPE_METER && CompVisible(comp)) {
    MakeInvisible(comp);
  }
}
/**********************************************************************************************************************************************/
void MakeMetersVisible(MComponent *comp)  /* Note: Hiding attached components only affects the direct children of the component, so 
						 we don't have recurse on this child's children. */
{
  if (comp->type==TYPE_METER && !CompVisible(comp)) {
    MakeVisible(comp);
  }
}


/**********************************************************************************************************************************************/
void MakeMetersInvisible(MComponent *comp)
{
  if (comp->type==TYPE_METER && CompVisible(comp)) {
    MakeInvisible(comp);
  }
}


/*=============================================================================================================================================
                                                 GROUP INFORMATION WINDOW FUNCTIONS
 ==============================================================================================================================================*/
void initialize_open_close_button(Widget w, GroupInfoWinData *window_data) {
   window_data->open_close_button=w;
   if (CompGroupOpen((MComponent *)window_data->comp)) {
      SetLabelString(w, "Close");
   }
   else {
      SetLabelString(w, "Open");
   }
}

/**********************************************************************************************************************************************/
void SetGroupOpenCloseButton(MComponent *group) {
   char *s;
   
   s=CompGroupOpen(group)? "Close" : "Open";
   SetLabelString(((GroupInfoWinData *)group->info_window)->open_close_button, s);
}
      
/**********************************************************************************************************************************************/
void OpenGroupInformationWindow(MComponent *comp, int given_x, int given_y, int valid_coords) {
   Widget group_info_w;

   GroupInfoWinData *window_data;

   if (CompInfoWindowOpen(comp)) {
      WARNING("Attempting to open an already open window");
      return;
   }
  
   window_data=(GroupInfoWinData *)sim_malloc(sizeof(GroupInfoWinData));
   window_data->comp=comp;
   window_data->initial_name=new_string(comp->name);
   window_data->current_name=new_string(comp->name);
   window_data->group_open=CompGroupOpen(comp);
   window_data->dialog_widget=group_info_w=BuildGroupInformationWindow(window_data);
   window_data->cancel_changes=0;

/* Set the shell position. */
   if (valid_coords)  {
      if (given_x<0) 
	 given_x=0;	
      if (given_y<0) 	
	 given_y=0;

      SetWidgetCoordinates(group_info_w, given_x, given_y);
   }
   else {
      SetWidgetCoordinates(group_info_w, comp->x, comp->y);
   }

   window_data->comp->info_window=(InfoWinData *)window_data;

   XtManageChild(group_info_w);
   RaiseWidget(group_info_w);
   SetStatusInfoWindowOpen(comp);
}

/**********************************************************************************************************************************************/
void CloseGroupInformationWindow(MComponent *comp) 
{
   GroupInfoWinData *window_data=comp->info_window;

   if (!CompInfoWindowOpen(comp)) {
      WARNING("Attempting to close an already closed window");
      return;
   }
  
   if (window_data->cancel_changes) {
      if (strcmp(window_data->initial_name,window_data->current_name)!=0) {  /* Restore the group's name. */
	 ChangeComponentName(comp,window_data->initial_name);
      }
      if (CompGroupOpen(comp)!=window_data->group_open) {  /* Restore the group's open/closed status. */
	 if (window_data->group_open) OpenGroup(comp);
	 else {
	    UnselectGroupMembers(comp);
	    CloseGroup(comp);
	 }
      }
   }
   else {
      char *new_name;
      new_name=XmTextFieldGetString(window_data->name_textf);  /* Set the new name if the textfield has changed. */
      if (0!=strcmp(new_name, window_data->initial_name)) {
	 ChangeComponentName(comp,new_name);
      }
      XtFree(new_name);
   }

   /* Destroy the window. */
   SetStatusInfoWindowClosed(comp); 
   comp->info_window=NULL;
   XtUnmanageChild(window_data->dialog_widget);
   XtDestroyWidget(window_data->dialog_widget);
   free(window_data->initial_name);
   free(window_data->current_name);
   free(window_data);
}

/**********************************************************************************************************************************************/
void change_group_name_cb(Widget w, GroupInfoWinData *window_data, XmAnyCallbackStruct *cbs) {
  String new_name;
  Widget textfield;

  textfield=window_data->name_textf;

  new_name=XmTextFieldGetString(textfield);
  if (0!=strcmp(new_name, window_data->current_name)) {
    free(window_data->current_name);
    window_data->current_name=new_string(new_name);
    ChangeComponentName(window_data->comp,window_data->current_name);
  }
  else {
    XmTextFieldSetString(textfield, window_data->current_name);
  }
  
  XtFree(new_name);
}

/********************************************* BUILD GROUP INFORMATION WINDOW  ***************************************************************/
ActionAreaItem group_info_action1[]={
  { "Ungroup", group_info_ungroup_action, NULL, NULL},
  { "Open", group_info_open_action, NULL, initialize_open_close_button}
};

ActionAreaItem group_info_action2[]={
  { "Ok", group_info_ok_action, NULL, NULL},
  { "Cancel", group_info_cancel_action, NULL, NULL},
  { "Kill", group_confirm_kill_cb, NULL, NULL}
};

/**********************************************************************************************************************************************/
Widget BuildGroupInformationWindow(GroupInfoWinData *window_data) /* Create a form dialog as follows: 
								     
								     Label '<type> Information Window'
								     Double Separator
								     TextInputField(data->name);
								     Separator
								     Action Area1
								     Separator
								     ActionArea2 */
     
{
  Widget group_info_w,label,sep,name_input,action_area1,action_area2;
  char *initial_values[1];
  char *prompts[1];
  XmString str;
  int i;
  Pixel bg;

  bg= the_environment.group_color;

  group_info_w=XmCreateForm(network_w,"group_info_w",NULL,0);

  XtVaSetValues(group_info_w, XmNbackground, bg, NULL);
  
/* TITLE LINE */
  str=XmStringCreateSimple("Group Information");
  label=XtVaCreateManagedWidget("title",xmLabelWidgetClass,group_info_w,
				XmNbackground, bg,
				XmNlabelString, str,
				XmNtopAttachment, XmATTACH_FORM,
				XmNrightAttachment, XmATTACH_FORM,
				XmNleftAttachment, XmATTACH_FORM,
				XmNtranslations, the_environment.move_window_translations,
				XmNuserData, group_info_w,
				NULL);
  sep=CreateSeparator(group_info_w,XmDOUBLE_LINE,label);
  XmStringFree(str);

/* NAME TEXTFIELD */
  {
    Widget textfield;
    
    name_input=CreateTextInput(group_info_w, "Name:", window_data->initial_name, &textfield);
    window_data->name_textf=textfield;
    
    XtVaSetValues(name_input,
		  XmNtopAttachment, XmATTACH_WIDGET,
		  XmNtopWidget, sep,
		  XmNleftAttachment, XmATTACH_FORM,
		  NULL);
    
    XtAddCallback(textfield,XmNactivateCallback, change_group_name_cb, window_data);
  }

  sep=CreateSeparator(group_info_w,XmSINGLE_LINE,name_input);
  
  /* ACTION AREA (UNGROUP, OPEN/CLOSE) */
  for (i=0;i<XtNumber(group_info_action1);i++) group_info_action1[i].data=window_data;
  action_area1=CreateActionArea(group_info_w,group_info_action1,XtNumber(group_info_action1),-1);
  XtVaSetValues(action_area1,
		XmNleftAttachment,XmATTACH_FORM,
		XmNrightAttachment,XmATTACH_FORM,
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget, sep,
		NULL);

  sep=CreateSeparator(group_info_w,XmSINGLE_LINE,action_area1);

/* ACTION AREA (OK, CANCEL) */
  for (i=0;i<XtNumber(group_info_action2);i++) group_info_action2[i].data=window_data;
  group_info_action2[i-1].callback=the_environment.killwarn? group_confirm_kill_cb : group_info_kill_action;
  action_area2=CreateActionArea(group_info_w,group_info_action2,XtNumber(group_info_action2),0);
  XtVaSetValues(action_area2,
		XmNleftAttachment,XmATTACH_FORM,
		XmNbottomAttachment,XmATTACH_FORM,
		XmNrightAttachment,XmATTACH_FORM,
		XmNtopAttachment,XmATTACH_WIDGET,
		XmNtopWidget, sep,
		NULL);

  return(group_info_w);
}

/************************************************ Group Information Window Callback Functions **************************************************/
void group_info_ok_action(w,window_data,cbs)
     Widget w;
     GroupInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
  MComponent *comp=window_data->comp;

  window_data->cancel_changes=0;
  CloseGroupInformationWindow(comp);
}

/***********************************************************************************************************************************************/
void group_info_cancel_action(w,window_data,cbs) 
     Widget w;
     GroupInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
   MComponent *comp=window_data->comp;

   window_data->cancel_changes=1;
   CloseGroupInformationWindow(comp);
}

/*******************************************************************************************************************************************/
void close_confirm_kill_group_dialog(w,window_data) 
     Widget w;
     CompInfoWinData *window_data;
{
  XtRemoveCallback(w,XmNokCallback, group_info_kill_action, window_data);
  XtRemoveCallback(w,XmNokCallback, close_confirm_kill_group_dialog, window_data);
  XtRemoveCallback(w,XmNcancelCallback, close_confirm_kill_group_dialog, window_data);    
  XtUnmanageChild(w);
}

/*******************************************************************************************************************************************/
void group_confirm_kill_cb(w,window_data,cbs)
     Widget w;
     GroupInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
  static Widget question_dialog=NULL;
  CoordinatePair pos;

  if (!question_dialog) {
    question_dialog=XmCreateQuestionDialog(main_w,"confirm_kill_group_dialog",NULL,0);
  }
  XtAddCallback(question_dialog, XmNokCallback, group_info_kill_action, window_data);
  XtAddCallback(question_dialog, XmNokCallback, close_confirm_kill_group_dialog, window_data);
  XtAddCallback(question_dialog, XmNcancelCallback, close_confirm_kill_group_dialog, window_data);

  pos.x=((XButtonEvent *)cbs->event)->x_root;
  pos.y=((XButtonEvent *)cbs->event)->y_root;
  XtAddCallback(XtParent(question_dialog), XmNpopupCallback, SetShellPosition, &pos);
  XtManageChild(question_dialog);
  XtRemoveCallback(XtParent(question_dialog), XmNpopupCallback, SetShellPosition, &pos);
}

/**********************************************************************************************************************************************/
void group_info_kill_action(w,window_data,cbs)
     Widget w;
     GroupInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
   Widget question_dialog;

   UnselectGroupMembers(window_data->comp);
   DestroyComponent(window_data->comp);
}

/**********************************************************************************************************************************************/
void group_info_open_action(w,window_data,cbs)  /* if the group closed, open it.
                                                     if the group's icon is selected, we should unselect it so that we don't have unseen
                                                       selected components. 
						     Change "Open Group" button to "Close Group".
                                                   else close it.
						     Unselect and selected group members.
						     Change "Close Group" button to "Open Group". */
     Widget w;
     GroupInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
  MComponent *comp=window_data->comp;

  if (!CompGroupOpen(comp)) {
    if (CompSelected(comp)) UnselectComponent(comp);
    OpenGroup(comp);
    SetLabelString(w,"Close");
  }
  else {
    UnselectGroupMembers(comp);
    CloseGroup(comp);
    SetLabelString(w,"Open");
  }
}

/**********************************************************************************************************************************************/
void group_info_ungroup_action(w,window_data,cbs)  /* Call Ungroup
						      Unselect the group icon, if it is selected. 
						      Close the group's information window. */
     Widget w;
     GroupInfoWinData *window_data;
     XmAnyCallbackStruct *cbs;
{
  MComponent *comp=window_data->comp;
  if (CompSelected(comp)) UnselectComponent(comp);
  CloseGroupInformationWindow(comp);
  Ungroup(comp);
}
