/* $Id: lines.c,v 10.1 92/10/06 23:03:23 ca Exp $ */
/*
 * MaRS Maryland Routing Simulator
 * Copyright (c) 1991 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Eric Bull
 *          Systems Design and Analysis Group
 *          Department of Computer Science 
 *          University of Maryland at College Park. 
 */
#include <sys/types.h>
#include <stdio.h>
#include "xm-lines.h"

/* This file contains functions to manage the line segments drawn between components.

   SegmentList *create_segmentlist();                                           Create a segment list.
   void destroy_segmentlist(SegmentList *sl);                                   Destroy a segment list.
   int add_segment(SegmentList *sl, MComponent *comp1, *comp2);                 Add a segment to a segment list.
   int delete_segment(SegmentList *sl, MComponent *comp1, *comp2);              Delete a segment from a segment list.
   int find_segment(SegmentList *sl, MComponent *comp1, *comp2);                Find a segment in a segment list.
   void draw_segments(SegmentList *sl, Display *display, Window window, GC gc); Draw the segment in a segment list.
   void add_segments(SegmentList *sl, SegmentList *new_segments);               Add segments to a segment list.
   void delete_segments(SegmentList *sl, SegmentList *old_segments);            Delete segments from a segment list.

*/

SegmentList *network_segmentlist;  /* The list of line segments between components. */

SegmentList *create_segmentlist()
{
   SegmentList *sl;
   sl=(SegmentList *)sim_malloc(sizeof(SegmentList));
   sl->line_array = (XSegment *) sim_calloc(1, sizeof(XSegment));
   sl->comp_pairs = (ComponentPair *)sim_calloc(1, sizeof(ComponentPair));
   sl->num_lines = 0;
   sl->memory = 1;
   return(sl);
}

void destroy_segmentlist(SegmentList *sl) {
   free(sl->line_array);
   free(sl->comp_pairs);
   free(sl);
}

int add_segment(sl,comp1,comp2)  /* Returns TRUE if the segment was successfully added. */
     SegmentList *sl;
     MComponent *comp1, *comp2;
{
   XSegment *line_array=sl->line_array;
   int number_of_lines=sl->num_lines;

   if (find_segment(sl, comp1, comp2) == -1) {
      line_array[number_of_lines].x1 = comp1->x;
      line_array[number_of_lines].x2 = comp2->x;
      line_array[number_of_lines].y1 = comp1->y;
      line_array[number_of_lines].y2 = comp2->y;
      sl->comp_pairs[number_of_lines].comp1=comp1;
      sl->comp_pairs[number_of_lines].comp2=comp2;

      sl->num_lines= ++number_of_lines;

      if (number_of_lines >= sl->memory) {
	 sl->memory += 10;
	 sl->line_array = (XSegment *) sim_realloc((char *) line_array, (unsigned) (sl->memory * sizeof(XSegment)));	 
	 sl->comp_pairs = (ComponentPair *)sim_realloc((char *) sl->comp_pairs, (unsigned) (sl->memory * sizeof(ComponentPair)));
      }
   }
   else 
      return(FALSE);
   return(TRUE);
}	

int delete_segment(sl,comp1, comp2)
     SegmentList *sl;
     MComponent *comp1, *comp2;
{
   register short i;
   int counter, end;
   int found;
   XSegment *temp_array, *line_array;
   ComponentPair *temp_pairs, *comp_pairs;

   line_array = sl->line_array;
   comp_pairs = sl->comp_pairs;
   temp_array = (XSegment *) sim_calloc(sl->memory, sizeof(XSegment));
   temp_pairs = (ComponentPair *)sim_calloc(sl->memory, sizeof(ComponentPair));

   end = sl->num_lines;

   counter = 0;
   found = 0;
   for (i = 0; i != end; ++i)  {

      if ((comp_pairs[i].comp1 == comp1) &&
	  (comp_pairs[i].comp2 == comp2)) {
	 sl->num_lines--;
	 found=1;
	 break; /* line found, now just copy the rest. */
      }
      if ((comp_pairs[i].comp2 == comp1) &&
	  (comp_pairs[i].comp1 == comp2)) {
	 sl->num_lines--;
	 found=1;
	 break; /* line found, now just copy the rest. */
      }
      temp_array[counter].x1 = line_array[i].x1;
      temp_array[counter].x2 = line_array[i].x2;
      temp_array[counter].y1 = line_array[i].y1;
      temp_array[counter].y2 = line_array[i].y2;
      temp_pairs[counter].comp1 = comp_pairs[i].comp1;
      temp_pairs[counter].comp2 = comp_pairs[i].comp2;
      counter++;  
   }
   for (i = i + 1; i<end; ++i) {
      temp_array[counter].x1 = line_array[i].x1;
      temp_array[counter].x2 = line_array[i].x2;
      temp_array[counter].y1 = line_array[i].y1;
      temp_array[counter].y2 = line_array[i].y2;
      temp_pairs[counter].comp1 = comp_pairs[i].comp1;
      temp_pairs[counter].comp2 = comp_pairs[i].comp2;
      counter++;  
   }
   
   free((char *) line_array);
   sl->line_array = (XSegment *) temp_array;
   sl->comp_pairs = (ComponentPair *) temp_pairs;
   return(found);
}

int find_segment(sl, comp1, comp2)
     SegmentList *sl;
     MComponent *comp1, *comp2;
{
   int number_of_lines=sl->num_lines;
   ComponentPair *comp_pairs=sl->comp_pairs;
   register int i;

   if (comp2<comp1) 
      swap((caddr_t)comp1,(caddr_t)comp2);


   for (i = 0; i < number_of_lines; ++i) {
      if ((comp_pairs[i].comp1 == comp1) &&
	  (comp_pairs[i].comp2 == comp2))
	 return (i);
      if ((comp_pairs[i].comp2 == comp1) &&
	  (comp_pairs[i].comp1 == comp2))
	 return (i);
   }
   return (-1);
}

void draw_segments(SegmentList *sl, Display *display, Window window, GC gc) {
   XDrawSegments(display, 
		 window, 
		 gc,
		 sl->line_array,
		 sl->num_lines);
}

/* This could be done a lot faster if the SegmentLists were sorted */
void add_segments(SegmentList *sl, SegmentList *new_segments) {
   register short i;
   register short num_lines=new_segments->num_lines;
   ComponentPair *comp_pairs=new_segments->comp_pairs;

   for(i=0; i<num_lines; i++)
      add_segment(sl, comp_pairs[i].comp1, comp_pairs[i].comp2);
}

void delete_segments(SegmentList *sl, SegmentList *old_segments) {
   register int i;
   register int num_lines=old_segments->num_lines;
   ComponentPair *comp_pairs=old_segments->comp_pairs;
   
   for(i=0; i<num_lines; i++)
      delete_segment(sl, comp_pairs[i].comp1, comp_pairs[i].comp2);
}





