/*
 * MaRS Maryland Routing Simulator
 * Copyright (c) 1991 University of Maryland
 * All Rights Reserved.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of U.M. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  U.M. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * U.M. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL U.M.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Eric Bull
 *          Systems Design and Analysis Group
 *          Department of Computer Science 
 *          University of Maryland at College Park. 
 */
/* $Id: simx.h,v 10.1 92/10/06 23:09:06 ca Exp $ */
#ifndef SIMX_H
#define SIMX_H

#include "simx1.h"
#include "component.h"
#include "list.h"

typedef Component COMPONENT;
typedef Param PARAM;
typedef Neighbor NEIGHBOR;

list *comps;

#ifndef NoX
#ifdef MOTIF
#include <X11/Intrinsic.h>

/*========================================== MComponent and MComponentList data type ====================================*/ 
typedef list *MComponentList;

typedef struct _MComponent {   /* A Component structure will be attached to each component icon. */
  COMPONENT *scomponent;       /* A pointer to the simultator component. */
  int type;                    /* The type of the component. This holds the index of the component in the component_types data structure
                                  else TYPE_GROUP=-1 if it's a group, TYPE_METER=-2 if it's a meter.*/
  char *name;                  /* Only used for group names.  We get component names from parameter->p_make_text(scomponent,
				  (Param *)scomponent->co_params->q_head) whenever we need it. */
  
  Widget icon;                 /* The component's icon. */
  int x,y;                     /* The coordinates of the component's widget. */
  int rel_x,rel_y;             /* The component's position relative to it's parent. Whenever a component is moved, all its children move with it */  
  int min_x,min_y,max_x,max_y; /* The min and max coordinates of the component's children. These are used to prevent the user from dragging
				  component's off the edge of the network window. */

  unsigned short status;       /* bit 0: selected       Is the component selected?
				  bit 1: visible        This indicates whether or not the component should actually be drawn when a call to 
				                           DrawComponent is made.
					  
                                  bit 2: group_open     Is the group open? (Only useful if the component type is TYPE_GROUP)
                                  bit 3: widget_drawn   This indicated whether or not the components' widget is currently managed.
                                  bit 4: info_w_open    Is the component's information window open?
                                  bit 5: param_w_open   Is the component's parameter window open?
				  bit 6: was_visible    Store the previous visibility setting of the component so it can be restored when
				                        a 'hide' operation is canceled. 
                                  bit 7: hide_children  The status of the hide_children toggle button.
                                  bit 8: hide_meters    The status of the hide meters toggle button. */

  void *info_window;           /* A pointer to the component information window. */
  void *param_window;          /* A pointer to the component parameters window. */
  
  int num_meters;              /* The number of meters the component has in the global 'm_list'. */
  
  int parent_linetype;         /* Indicates the line type to use when drawing a line to the parent.
                                  children of groups are not connected by lines. (Currently not used)*/
  int sister_linetype;         /* Indicates the line type to use when drawing a line to the sisters.
				  If two sisters each have a different line type, one line type will
				  have to be selected to take precedence. (Currently not used) */
  MComponentList sisters;
  MComponentList children;
  struct _MComponent *parent;
} MComponent;

typedef MComponent XCOMPONENT;

#define TYPE_GROUP -1
#define TYPE_METER -2

#define SELECTED 0x0001
#define VISIBLE 0x0002
#define GROUP_OPEN 0x0004
#define DRAWN 0x0008
#define INFO_WINDOW_OPEN 0x0010
#define PARAM_WINDOW_OPEN 0x0020
#define WAS_VISIBLE 0x0040
#define CHILDREN_HIDDEN 0x0080
#define METERS_HIDDEN 0x0100

#define CompVisible(comp)         (((comp)->status & VISIBLE) != 0)
#define CompSelected(comp)        (((comp)->status & SELECTED) != 0)
#define CompGroupOpen(comp)       (((comp)->status & GROUP_OPEN) != 0)
#define CompDrawn(comp)           (((comp)->status & DRAWN) != 0)
#define CompInfoWindowOpen(comp)  (((comp)->status & INFO_WINDOW_OPEN) != 0)
#define CompParamWindowOpen(comp) (((comp)->status & PARAM_WINDOW_OPEN) != 0)
#define CompWasVisible(comp)      (((comp)->status & WAS_VISIBLE) != 0)
#define CompChildrenHidden(comp)  (((comp)->status & CHILDREN_HIDDEN) != 0)
#define CompMetersHidden(comp)    (((comp)->status & METERS_HIDDEN) != 0)

typedef struct _ENVIRONMENT  {

  Display *the_display; 
  short int monochrome;
  int the_screen;
  char *pr_name;

  Font meter_font;
  XFontStruct *meter_font_info;

  int meter_window_height;  /* The hieght scalar for all meters. */
  int meter_window_width;   /* The width scalar for all meters. */

  int x_center;  	    /* The center of the Main Window (not the scrolling network). */
  int y_center;
  int height;	            /* The dimensions of the Main Window */
  int width;
  int network_width;        /* The dimensions of the scrolling Network Window. */
  int network_height;

  int meter_info_window_entries;

  GC the_gc;
  GC draw_gc;
  GC erase_gc;
  GC redraw_gc;
  GC rubberbox_gc;

  unsigned long back_color;
  unsigned long text_color;
  unsigned long fore_color;

  unsigned long meter_color; 
  unsigned long group_color;

  unsigned long component_color[30];	/* Maximum number of different types of components unfortunately hardwired. */

  unsigned long hide_settings;	        /* Settings of the hide window toggle buttons. */
  unsigned int hide_meters;     	/* Hide meters setting. */
  
  int meter_x_spacing;  /* Spacing between meters in the meter_window. */
  int meter_y_spacing;
  
  int meter_component_spacing;  /* Vertical spacing between meter components in the network window. */

  int meter_component_x_offset; /* offset of meter components from their parent. */
  int meter_component_y_offset; 

  XtTranslations component_translations;
  XtTranslations network_w_translations;
  XtTranslations move_window_translations;
  XtTranslations create_component_translations;
  XtTranslations meter_window_translations;

  Boolean meter_window_open;  /* Boolean flag which tells whether or not the meter window is open */
  Boolean quitwarn;
  Boolean killwarn;
  Boolean draw_meter_component;

  short delta;

  /** Has it ever been run? */
  int never_ran;
  /**** Single stepping or not? */
  int single_step;
  /** Is the simulator iconified? */
  int iconified;
  /* What is the display like? */
  int reverse_video;		/* this is only relevant to
				   monochrome displays. */
} ENVIRONMENT, *ENVIRONMENT_PTR;

#define GlobalHidden(type) (((type)!=TYPE_METER)? ((the_environment.hide_settings>>(type)) & 0x1L) : the_environment.hide_meters)

#define EVENT_STEP_MODE 1
#define TIME_STEP_MODE 2
#define CONTINUOUS_MODE 3

#else /* MOTIF */  


#define RightButton Button3
#define LeftButton Button1
#define MiddleButton Button2


typedef struct {
  short button;
  short key;
} OPTION;

typedef struct {
 OPTION select;
 OPTION move;
 OPTION resize;
 OPTION create;
 OPTION connection;
 OPTION peer;
 OPTION raise;
 OPTION lower;
} KEY_BINDING;

typedef struct _XCOMPONENT {
  COMPONENT *scomponent;     /* pointer to object's component */
  Window comp_window;  /* window id of component window */
  Window info_window;   /* window id of information window */
  GC back_gc;			/* Pointer to gc to clear infowindow */
  short which_one;         /* COMP_WINDOW or INFO_WINDOW*/
  int hposition;
  int wposition;
  int x,y;
  int user_x,user_y;
  int width, height;
  Param **p_list;     /* Array of pointers to parameters
			in info windows (IN ORDER) */
  int num_params;
  Param **m_list;     /* Array of params that have meters */
  int num_meters;
} XCOMPONENT;

typedef struct _ENVIRONMENT  {

  Display *the_display;
  short int monochrome;
  int the_screen;
  char *pr_name;

  Window back_window;
  Window clock_window;
  Window icon_window;

  Font info_font;
  Font comp_font;
  Font control_font;
  Font edit_font;
  Font meter_font;
  Font text_font;
  Font clock_font;

  XFontStruct *info_font_info;
  XFontStruct *comp_font_info;
  XFontStruct *control_font_info;
  XFontStruct *edit_font_info;
  XFontStruct *meter_font_info;
  XFontStruct *text_font_info;
  XFontStruct *clock_font_info;

  int meter_indicator_width;

  int component_window_height;
  int component_window_width;

  int info_window_width;
  int info_window_height;
  int info_window_vertical_padding;
  int info_window_horizontal_padding;

  int meter_window_height;
  int meter_window_width;
  int meter_info_window_entries;

  int control_window_height;
  int control_window_width;

  int text_window_height;
  int text_window_width;

  int border_width;
  int x_center;
  int y_center;
  int x;
  int y;
  int height;
  int width;

  GC the_gc;
  GC meter_or_log_indicator_gc;
  GC clock_erase_gc;
  GC clock_draw_gc;
  GC line_gc;
  GC back_gc;
  GC xor_gc;

  unsigned long clock_color;
  unsigned long edit_color;
  unsigned long back_color;
  unsigned long text_color;
  unsigned long fore_color;

  unsigned long component_color[30];	/* Maximum number of different
     types of components unfortunately hardwired. */
  GC component_back_gc[30];	/* To use to clear parts of infowindows. */

  short delta;

  /** Has it ever been run? */
  int never_ran;
  /**** Single stepping or not? */
  int single_step;
  /** Is the simulator iconified? */
  int iconified;
  /* What is the display like? */
  int reverse_video;		/* this is only relevant to
				   monochrome displays. */
} ENVIRONMENT;

extern KEY_BINDING user_bindings;

#endif /* MOTIF */
#else /* NoX */

typedef struct _ENVIRONMENT  {

#ifndef NoX
  Display *the_display;
  short int monochrome;
  int the_screen;
#endif /* NoX */
  char *pr_name;

#ifndef NoX
  Window back_window;
  Window clock_window;
  Window icon_window;

  Font info_font;
  Font comp_font;
  Font control_font;
  Font edit_font;
  Font meter_font;
  Font text_font;
  Font clock_font;

  XFontStruct *info_font_info;
  XFontStruct *comp_font_info;
  XFontStruct *control_font_info;
  XFontStruct *edit_font_info;
  XFontStruct *meter_font_info;
  XFontStruct *text_font_info;
  XFontStruct *clock_font_info;

  int meter_indicator_width;

  int component_window_height;
  int component_window_width;

  int info_window_width;
  int info_window_height;
  int info_window_vertical_padding;
  int info_window_horizontal_padding;

  int meter_window_height;
  int meter_window_width;
  int meter_info_window_entries;

  int control_window_height;
  int control_window_width;

  int text_window_height;
  int text_window_width;

  int border_width;
  int x_center;
  int y_center;
  int x;
  int y;
  int height;
  int width;

  GC the_gc;
  GC meter_or_log_indicator_gc;
  GC clock_erase_gc;
  GC clock_draw_gc;
  GC line_gc;
  GC back_gc;
  GC xor_gc;

  unsigned long clock_color;
  unsigned long edit_color;
  unsigned long back_color;
  unsigned long text_color;
  unsigned long fore_color;

  unsigned long component_color[30];	/* Maximum number of different
     types of components unfortunately hardwired. */
  GC component_back_gc[30];	/* To use to clear parts of infowindows. */
#endif /* NoX */

  short delta;

  /** Has it ever been run? */
  int never_ran;
  /**** Single stepping or not? */
  int single_step;
#ifndef NoX
  /** Is the simulator iconified? */
  int iconified;
  /* What is the display like? */
  int reverse_video;		/* this is only relevant to
				   monochrome displays. */
#endif /* NoX */
} ENVIRONMENT;

#endif /* NoX */

extern ENVIRONMENT the_environment; 

#endif /* SIMX_H */
