/* PTViewer	-	Interactive Viewer for Panoramic Images
   Copyright (C) 2000 - Helmut Dersch  der@fh-furtwangen.de
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*------------------------------------------------------------*/


import java.applet.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.net.*;
import java.net.URLConnection;
import java.util.*;
import java.text.*;
import java.io.*;
import java.lang.reflect.*;
import java.lang.*;


public class ptviewer extends Applet implements Runnable{
	static final boolean debug 	= false;  	// Issue debug messages
	static final double HFOV_MIN 	= 10.5;
	static final double HFOV_MAX 	= 165.0;
	static final long TIME_PER_FRAME= 10;//40;	// Minimum frame time in auto-mode
	static final long ETERNITY 	= 100000000;	// Number of frames which will never be rendered
		
	int quality = 3;				// Interpolators: 
							// 0 - always nn
							// 1 - nn for panning & autopanning, bil else
							// 2 - nn for panning, bil else
							// 3 - always bil

	boolean inited			= false;	// init() has been called
 	Color 	bgcolor 		= null;		// backgroundcolor

	long 	waittime 		= 0;		// Wait image is displayed at least that long
	boolean WaitDisplayed 		= false;	// Has wait image been displayed?
	
	Image 	view 			= null,		// Panorama Viewport
		dwait 			= null, 	// Display during download
		frame			= null,		// frame image
		offImage 		= null;		// offscreen image
		
	Graphics offGraphics 		= null;
	int 	offwidth		= 0, 
		offheight		= 0;
	MemoryImageSource source 	= null;		// View is calculated here before display
	
	int 	awidth 			= 320, 		// Dimension of applet window 
		aheight 		= 200; 		// just a guess, will be set later
	
	public int vwidth		= 0, 
		vheight			= 0;		// Dimension of viewer window 	
	boolean vset 			= false;	// is vwidth/height fixed?
	int 	vx			= 0, 
		vy			= 0;		// Position of viewer window
	int 	pwidth			= 0, 
		pheight			= 0;		// Dimension of panoramic image
	
   	int[]   vdata 			= null;		// RGB data viewer window
   	byte[]  hs_vdata 		= null;		// hotspot viewer window
   	int[][] pdata 			= null;		// RGB data panorama image
   	boolean show_pdata 		= true;
   
   
   			
	boolean ready			= false;	// ready to render and display images
	boolean hsready			= false;	// ready to render and display hotspotimages
	boolean PanoIsLoaded		= false;	// panoramic image loaded and set up
   	boolean fatal			= false;	// a fatal error has occured
   	boolean mouseInWindow 		= true;		// true if mouse is in applet window
   	boolean mouseInViewer 		= true;		// true if mouse is in viewer window
   
   	boolean panning 		= false;	// mousebutton has been pressed
   	public boolean dirty 		= true; 	// Image needs recalculation
   	boolean showhs			= false;	// Show Hotspot images?
   	
   	boolean showCoordinates		= false;	// Display mouse coordinates for editing
   	 
   	int 	oldx			= 0,
		oldy			= 0,
		newx			= 0,
		newy			= 0;		// Mouse coordinates	
   	
   	int 	ptcursor		= Frame.DEFAULT_CURSOR; // Cursor used in PTWindow for panning

   
 
   	public double yaw 		= 0.0;	 	// Current Pan Angle 
   	public double hfov 		= 70.0;	  	// Current horizontal field of view 			
   	public double hfov_min 		= HFOV_MIN;	// maximum horizontal field of view 			
	public double hfov_max 		= HFOV_MAX;	// minimum horizontal field of view			
   	public double pitch 		= 0.0; 		// tilt angle	
   	public double pitch_max		= 90.0;		// maximum tilt angle
   	public double pitch_min		= -90.0; 	// minimum tilt angle
   	public double yaw_max 		= 180.0; 	// maximum pan angle
   	public double yaw_min		= -180.0; 	// minimum pan angle
	public double phfov		= 360.0;	// Panorama Horizontal Field of View
	
	double MASS 			= 0.0;
	double oldspeedx 		= 0.0;
	double oldspeedy 		= 0.0;		

	double autopan			= 0.0;		// pan increment
	double autotilt			= 0.0;		// tilt increment
	double zoom			= 1.0;		// Zoomfactor
	
	public double pan_steps		= 20.0;		// Number of steps for panning  one hfov

   	String filename 		= null;		// Name of panoramic image
   	String inits			= null;		// Commands to be executed on load.
	String MouseOverHS 		= null;		// Javascript function called by the applet when over hotspot
												// Syntax: MouseOverHS( int numhs );
	String GetView			= null;		// Javascript function called by the applet when view changes
												// Syntax: GetView( double pan, double tilt, double fov );

	
	int	click_x 		= -1, 
		click_y 		= -1;
	
	long 	frames			= 0;		// total number of rendered frames
	long 	lastframe		= 0;		// Stop autopanning after rendering this frame
	long 	ptimer			= 0;
	
   	Thread 	loadPano 		= null;
   	Thread 	ptviewerScript 		= null;
   	
   	String 	PTScript		= null;
 	
 	String 	PTViewer_Properties 	= null;


	
	// Panoramic image descriptions
	
	int	CurrentPano		= -1;		// Currently loaded pano

   
    
    
    // Sender - Receiver stuff
    
	Hashtable sender 		= null;

	// Private image cache
	
	Thread	preloadthread		= null ;
	String 	preload			= null;

	// Tile order in QTVR-file
	String 	order=null;
	int[] cube_order 		= {0,1,2,3,4,5};

	// Antialiasing

	boolean antialias 		= false;
	Vector scaledPanos 		= null;
	double max_oversampling		= 1.5;
	
	
	// Constructors
	
	public ptviewer(){}
	
	public ptviewer(int[][] pd){
		pdata = pd;
		PanoIsLoaded = true;
		math_updateLookUp( (int) (pdata[0].length * 360.0 / phfov) );
		filename = "Pano";
	}
		 	    
    
	
	void initialize(){
		numhs 		= 0;
		curhs 		= -1;
		curshs 		= -1;
		
		yaw 		= 0.0;		// Pan angle
		hfov 		= 70.0;		// horizontal field of view
		hfov_min 	= HFOV_MIN;	// maximum horizontal field of view
		hfov_max 	= HFOV_MAX;	// minimum horizontal field of view
		pitch 		= 0.0;		// tilt angle
		pitch_max	= 90.0;		// maximum tilt
		pitch_min	= -90.0;	// minimum tilt
		yaw_max 	= 180.0;     	// maximum pan
		yaw_min		= -180.0;	// minimum pan
		phfov		= 360.0;	// Panorama hfov

		autopan		= 0.0;		// pan increment
		autotilt	= 0.0;		// tilt increment
		zoom		= 1.0;		// Zoomfactor
		
		pwidth		= 0;
		pheight		= 0;
		stopPan();
		lastframe	= 0;
		dirty 		= true; 
		showhs		= false;
		showCoordinates	= false;
		MouseOverHS	= null;
		GetView		= null;
		
		WaitDisplayed 	= false;
		pan_steps	= 20;
		order		= null;
	}
				
   
	public void init(){
     		fatal = false;
		preloadthread 	= null;
		preload 	= null;
		ptcursor	= Frame.DEFAULT_CURSOR;
		

		file_init();
		math_init();
		pb_init();
		app_init();
		snd_init();
		shs_init();
		hs_init();
		
		sender = new Hashtable();

   		inited = true;
   		repaint();
		
		// Load property file
		
		
		byte[] bp = file_read( "PTDefault.html", null );
		if( bp != null ){
			PTViewer_Properties = new String( bp );
			bp = null;
		}
		

		initialize();
		if( PTViewer_Properties != null )
			ReadParameters( PTViewer_Properties );
		ReadParameters( null );
		if( filename != null && filename.startsWith("ptviewer:") ){
			int n = Integer.parseInt( filename.substring( filename.indexOf(':')+1 ) );
			String p = myGetParameter(null, "pano" + n);
			if( p != null ){
				filename = null; // To reset PTViewer
				ReadParameters( myGetParameter(null, "pano" + n) );
			}
		}		
    }
	
	
 
   	public String getAppletInfo(){
      		return "PTViewer v. 2.6   H. Dersch, der@fh-furtwangen.de";
   	}

   	public void start(){
   		if(loadPano == null)
   		{
      		loadPano = new Thread(this);
      		loadPano.start();
      	}
   	}

	public synchronized void stop(){
		stopThread(preloadthread); 	
		preloadthread 	= null;
		stopThread(loadPano); 		
		loadPano 	= null;

      		stopAutoPan();
		stopPan();
      	
      		stopApplets(0);
      	 
  		ready = false;
 		hsready = false;
     		
		vdata 		= null; 
		hs_vdata	= null;
		view		= null;
    		if( !vset ){
   			vwidth 	= 0; 
   			vheight = 0;
   		}
       		offImage 	= null;
		scaledPanos 	= null; 	
    	}


	synchronized void PV_reset(){
    		
   		ready 		= false;
   		hsready		= false; 		
			    
		hs_dispose();

		PanoIsLoaded 	= false;
 
 		filename 	= null;
		MouseOverHS 	= null;
		GetView		= null;
		
		
		pb_reset();
  		
 		inits = null;
 		order = null;
      		System.gc();
	}

	public synchronized void destroy(){
 		stopThread( ptviewerScript ); 
		ptviewerScript = null;
  		
		PV_reset();  
		
		if(sender != null ){
			sender.clear();
			sender = null;
		}		


		vdata 	= null;
		hs_vdata= null;
	    	source 	= null;
	    	frame 	= null;
		view 	= null;
		dwait 	= null;
		pdata   = null;
	    
		math_dispose();
		shs_dispose();
		snd_dispose();
		
    		System.gc();
    	}


		
 	/**
 	* Run threads to load and display the panoramic images and
 	* run ptviewer:commands.
 	*/
   	public void run(){
 
 		if( Thread.currentThread() == preloadthread && preload != null ){
			int n = getNumArgs( preload, ',' );

 			if( n > 0 ){
  				for(int i=0; i<n; i++){
  					String fname = getArg( i, preload, ',' ) ;
  					
  					if( fname != null && file_cachefiles && file_Cache != null &&
  						file_Cache.get( fname ) == null && fname != filename ){
  						file_read( fname, null);
  					}
  				}
  			}
   			return;
   		}
		
 
 		if( Thread.currentThread() == ptviewerScript ){
 			if( PTScript != null ){
 				PTViewerScript( PTScript );
 			}
 			return;
 		}
 				
  		
 		// Current thread is loadPano
 		
		
		ResetCursor();
 		 				
		// Load panoramic image, if not done
        
		if( !PanoIsLoaded ){
        		show_pdata = true;
         		if(filename == null){
         			if(pwidth != 0) 
         				filename = "_PT_Grid"; // Display grid
         			else
         				show_pdata = false;
         		}
         		if( filename != null && filename.toLowerCase().endsWith(".mov")){ // Don't load qtvr files here
         			pdata = im_loadPano(null, pdata, pwidth, pheight);
         		}else
        			pdata = im_loadPano(filename, pdata, pwidth, pheight);
     			System.gc();
		}
		
		if( pdata == null ){
			fatal = true; repaint();
			return;
		}
  
		// Panorama is loaded now
		
		// Load QTVR panorama if there is one

		if( filename != null && filename.toLowerCase().endsWith(".mov")){
			try{
				String p = " {file=" + filename + "} ";
				if( order != null ) p = p + "{order=" + order +"} ";
				if( antialias ){
				 	p = p + "{antialias=true} ";
					p = p + "{oversampling=" + max_oversampling + "} ";
				}
				Class c = Class.forName( "ptmviewer" );
	    		Constructor ac = c.getConstructor( new Class[]{ Class.forName("ptviewer"), String.class });
	    		Applet qtvr = (Applet) ac.newInstance( new Object[]{ this, p });
	    		qtvr.init();
	    		qtvr.start();
	    		qtvr = null;
	    		System.gc();
			}catch(Exception e){}
		}

        	pheight = pdata.length;
		pwidth  = pdata[0].length;

  		// Set maximum and minimum tilt angles
		double f = phfov * (double)pheight / (double)pwidth;
		if( f < 180.0 ){ // Limit tilt angles
			f /= 2;
       			if( pitch_max > f ) pitch_max = f;
        		if( pitch_min < -f) pitch_min = -f;  
        	}
			
		// Limit hfov of viewer window

        	if( hfov > yaw_max - yaw_min ) hfov = yaw_max - yaw_min;
 		if( !PanoIsLoaded ) math_updateLookUp( (int) (pdata[0].length * 360.0 / phfov) );

 		finishInit( PanoIsLoaded );
 
   	}
 
 	void finishInit(boolean p){
		if(!p) shs_setup(); // Set up static hotspots
               
   		ready = true;
		requestFocus();
		ResetCursor();
  		
  		repaint();
  		paint(getGraphics()); // Extra kick for IE	

   		
  		if(!PanoIsLoaded)	
			hs_setup(pdata);
				
		hsready = true;
		
		PanoIsLoaded = true;
  		
   		if(autopan != 0.0)
  			lastframe = frames + ETERNITY;

  		if( inits!=null){
			int index = inits.indexOf('*');
			if( index == -1 )
  				JumpToLink( inits, null );
			else{ // Target set
				JumpToLink( inits.substring(0, index), inits.substring(index+1) );
                    	}
  		}
		
		repaint();
       	
       		SetupSounds();
       	
       		// Panorama is displayed now; start 
       		// download of more images in the background
     		if(preload != null && preloadthread == null){
      			preloadthread = new Thread(this);
      			try{
      				preloadthread.setPriority(Thread.MIN_PRIORITY);
      			}catch( SecurityException se ){
      			} 
      			preloadthread.start();
      		}

	} 		  	
 
   	public boolean mouseDown(Event evt, int x, int y){
   		if( x>=vx && x<vx+vwidth && y>=vy && y<vy+vheight ){
   			if( lastframe > frames ){
				stopThread( ptviewerScript ); ptviewerScript = null;
   				stopAutoPan();
     			oldx =  x;
      			oldy =  y;
   				return true;
   			}
   			if( showCoordinates ){
   				showStatus(DisplayHSCoordinates( x-vx, y-vy ));
   				showCoordinates = false;
   				return true;
   			}
   		}
 		
 		if( !panning && mouseInViewer ){
      		oldx =  x;
      		oldy =  y;
      		if( curhs<0 ){
      			panning 	= true;
      			if( evt.shiftDown() )
      				zoom = 1.0/1.03;
       			else if( evt.controlDown() )
       				zoom = 1.03;
       			else
      				zoom = 1.0;
      			repaint();
   				PVSetCursor( x,  y);
   			}
   		}

      	newx = x;
      	newy = y;
       	return true;
   	}

    public boolean mouseDrag(Event evt, int x, int y){
 		newx = x;               
      	newy = y;
 		if( mouseInViewer){  
 			panning = true;   
      		if( evt.shiftDown() )
      			zoom = 1.0/1.03;
      		else if( evt.controlDown() )
       			zoom = 1.03;
       		else
      			zoom = 1.0;
      		ResetCursor();
 		}
		repaint();
     	return true;
   	}


   	public boolean mouseUp(Event evt, int x, int y)
   	{
  		newx = x;               
      	newy = y;
		stopPan();
   		zoom = 1.0;
   		if(hsready){
  			if( curshs >= 0 ){
   				int i;
 				for(i=0; i<numshs; i++)
  					if( shs_active[i] )
   						gotoSHS( i );
   			}
   			else if( curhs >= 0 ) {
   				int i;
   				gotoHS( curhs );
   				for(i=curhs+1; i<numhs && curhs != -1; i++)
   					if( hs_link[i] == curhs )
   						gotoHS( i );
   				if( curhs < 0 ) return true;
   			}
   			PVSetCursor( x,  y);
   			
   			click_x = x;
   			click_y = y;
   		}
		return true;
   	}

   	public boolean mouseEnter(Event evt, int x, int y){
       	mouseInWindow = true;
       	mouseInViewer = is_inside_viewer(x,y);
   		PVSetCursor( x,  y);
      	return true;
   	}

    public boolean mouseExit(Event evt, int x, int y){
      	mouseInWindow = mouseInViewer = false;
 	stopPan();
   		zoom = 1.0;
   		ResetCursor();
      	return true;
   	}

   	public boolean keyDown(Event evt, int key)
   	{ 
   		if( !ready ) return true;
   		  		
   		switch( key )
   		{
   			case Event.UP:	panUp();
         					break;
         	case Event.DOWN:panDown();
         					break;
         	case Event.LEFT:panLeft();
         					break;
         	case Event.RIGHT:panRight();
         					break;
         	case '+': 
         	case '>':   	
			case 'A': 
        	case 'a': 
        	case '.': 
        	case '=':		ZoomIn();
 							break;
			case '-':
			case '<':   	
			case 'Z':
			case 'z':   	
			case ',':
			case '_':   	ZoomOut();
     						break;
			case ' ':		toggleHS();
							break;
			case 'i':
			case 'I':		showStatus(getAppletInfo());
							break;
			case 'v':		// Print View information
							showStatus("pan = " + (double)((int)(yaw*100.0))/100 + "deg; tilt = " + (double)((int)(pitch*100.0))/100 + "deg; fov = " +  (double)((int)(hfov*100.0))/100 + "deg");
							break;
			case 'p':		// Print path to document
			case 'P':		showStatus( ptgetPath() );
							break;
			case 'u':		// Print URL to document
			case 'U':		showStatus( getDocumentBase().toString() );
							break;
			case 'h':		// Print Mousecoordinates
							showCoordinates = true;
							showStatus("Click Mouse to display X/Y Coordinates");
							break;
			case '\n':		if(!hsready) break;
							if( curshs >= 0 ) {
								int i;
								for(i=0; i<numshs; i++)
									if( shs_active[i] )
										gotoSHS( i );
							}
							else if( !panning && curhs >= 0 ) {
								int i;
								gotoHS( curhs );
								for(i=curhs+1; i<numhs && curhs != -1; i++)
									if( hs_link[i]==curhs )
										gotoHS( i );
								if( curhs < 0 ) return true;
							}
							break;
     	}
     	return true;
   	}


 	public boolean mouseMove(Event evt, int x, int y){
       	mouseInViewer = is_inside_viewer(x,y);
   		if(mouseInWindow){
			newx = x;               
      		newy = y;
      		
      	}
      	PVSetCursor(x,y);
       	return true;		
   	}

	
	void PVSetCursor(int x, int y){
		int i;
		
		if(!mouseInWindow){ // outside applet window
			ResetCursor();
			return;
		}
		
		if( !ready ){
			i = -1;
		}else
			i = OverStaticHotspot( x, y );
		
		if( i!= curshs ){ // Hotspot status has changed
			curshs = i;
		
			if( curshs >= 0 ){ // We entered a static hotspot
     			try{
					((Frame)getParent()).setCursor( Frame.HAND_CURSOR );
         		}catch(Exception e){}
 				curhs = -1;
 				repaint();
 				return;
 			}
 			else{	// We just left a static hotspot
 				ResetCursor();
 				repaint();
  			}
 		}
 		
 		// Leave if we're over a static hotspot
 		if( curshs >= 0 ) return;

		if( panning || lastframe>frames || !mouseInViewer){
			curhs = -1;
			ResetCursor();
			return;
		}
		
		if( !hsready ){
			i = -1;
		}else
			i = OverHotspot( x-vx, y-vy );
			
		if( i!= curhs ){ // hotspot status has changed
			curhs = i;
      		if( curhs >= 0 ){ // we entered the hotspot
      			try{
					((Frame)getParent()).setCursor( Frame.HAND_CURSOR );
     				if(hsready){
     					showStatus(hs_name[curhs]);
     					hs_exec_popup( curhs );
      					repaint();
       					sendHS();
       				}
       				return;
        		}catch(Exception e){}
     		}
      		else{ // we left the hotspot
      			ResetCursor();
      			repaint();
      			showStatus("");
      			sendHS();
      			return;
      		}
      	}
 		if( curhs >= 0 ) return;

		ResetCursor();
 	}
 	
 	void ResetCursor(){
      	try{
			if(mouseInViewer){
				if( !ready ){
					((Frame)getParent()).setCursor( Frame.WAIT_CURSOR );
					return;
				}
      			if( ((Frame)getParent()).getCursorType() != ptcursor ){
      				((Frame)getParent()).setCursor( ptcursor );	
      			}
      		}else{
       			if( ((Frame)getParent()).getCursorType() != Frame.DEFAULT_CURSOR  ){
      				((Frame)getParent()).setCursor( Frame.DEFAULT_CURSOR );	
      			}
      		}
      	}
 		catch(Exception e){}
 	}
 			

	void sendView()
	{
		if(GetView != null && ready && loadPano!=null) // && !panning && !(lastframe>frames) 
			executeJavascriptCommand( GetView + "(" + yaw + "," + pitch + "," + hfov + ")");
	}
	
	void sendHS()
	{
      	if( MouseOverHS != null && ready && loadPano!=null)
      		executeJavascriptCommand( MouseOverHS + "(" + curhs + ")" );
  	}
 


	public void update(Graphics g) {	paint(g);	}


	public synchronized void paint(Graphics g){
 		if( !inited ) return;
  		
       		if(fatal){
         		setBackground(Color.red);
         		g.clearRect(0,0,size().width,size().height);
         		return;
      		}
      	
      	
 		if( offImage == null ){
 			awidth = size().width; aheight = size().height;
 			if( !vset || offwidth == 0 ){
       				offwidth = size().width; offheight = size().height;
       			}
  			offImage 	 = createImage(offwidth, offheight);
      			offGraphics  = offImage.getGraphics();
 		}
		

		
		if( !ready || System.currentTimeMillis() < ptimer){
			if(dwait != null){
				if( bgcolor != null && !WaitDisplayed){
					setBackground( bgcolor );
					offGraphics.clearRect(0, 0, offwidth, offheight );
				}
				if( !WaitDisplayed ){
					if( waittime != 0 )
						ptimer = System.currentTimeMillis() + waittime;
					WaitDisplayed = true;
				}
				offGraphics.drawImage(dwait,
							(offwidth  - dwait.getWidth(null))/2,
							(offheight - dwait.getHeight(null))/2,this);
				
				pb_draw( offGraphics, offwidth, offheight);
				
				g.drawImage(offImage,0,0,this);
				
				if( ready ){
					try{ Thread.sleep(20);}				
					catch (InterruptedException e) {return;}
					repaint();
				}
			}else{
				if( bgcolor != null ){
					setBackground( bgcolor );
				}
        			g.clearRect(0, 0, size().width, size().height );
        			if( percent != null && percent[0] > 0 )
        				g.drawString("Loading Image..." + percent[0] + "% complete", 30, size().height/2);
        			else
        				g.drawString("Loading Image...", 30, size().height/2);
         		}
        		return;
        	} 
        
        	// At this point the panoramic image is loaded

      		// Initialize viewer window
     	
        	if( vdata == null){
        		if( vwidth == 0 ) 	vwidth  = size().width;
        		if( vheight == 0 ) 	vheight = size().height;
        	
        		// Set some vwidth/vheight related data
  
        		while( math_fovy(hfov, vwidth, vheight) > pitch_max - pitch_min ){
        			hfov /= 1.03;
        		}
 			double fovy2 = math_fovy(hfov, vwidth, vheight)/2.0;
        
        		if( pitch > pitch_max - fovy2 && pitch_max != 90.0) pitch = 0.0;
        		if( pitch < pitch_min + fovy2 && pitch_min != -90.0) pitch = 0.0;

        		vdata 	= new int[vwidth * vheight ];
        		hs_vdata= new byte[vwidth * vheight];
        	
        		if( filename != null && filename.toLowerCase().endsWith(".mov") )
				for(int i=0; i<hs_vdata.length; i++) hs_vdata[i] = (byte)0;
			else
				for(int i=0; i<hs_vdata.length; i++) hs_vdata[i] = (byte)0xff;
				
        		dirty	= true;
       			source 	= new MemoryImageSource(vwidth, vheight, vdata, 0, vwidth);
        		source.setAnimated(true);
   		
 			if( view == null ){
   				view 	= createImage(source);
  			}
			// initialize antialiasing arrays

			if( antialias && pdata != null){
				scaledPanos = new Vector();
				scaledPanos.addElement( pdata );

				int [][] pd = pdata;

				double sf = hfov_max / ( (double)vwidth * phfov * max_oversampling );
				int i = 0;
				while( pd != null && (double)pd[0].length * sf > 1.0 ){
					pd = im_halfsize(pd);
					scaledPanos.addElement( pd ); i++;
				}
		//		System.out.println("Created " + i + " Resolutions.");
			}		
		}   		

      	

			
    	if( panning ){
   			double scale = 1.0 / 2000.0 * hfov / 70.0 * 320.0 / vwidth;
			double speed_x = (newx - oldx)*(newx - oldx)* (newx > oldx? 1.0 : -1.0); 
			speed_x = (speed_x + MASS * oldspeedx) / ( 1.0 + MASS );
			oldspeedx = speed_x;
			double speed_y = (oldy - newy)*(oldy - newy)* (oldy > newy? 1.0 : -1.0);
			speed_y = (speed_y + MASS * oldspeedy) / ( 1.0 + MASS );
			oldspeedy = speed_y;

			gotoView( yaw 	+ scale * speed_x  , 
			          pitch + scale * speed_y  , 
					  hfov 	* zoom );
		}
    	
    	if( lastframe>frames ){
			gotoView( yaw + autopan, 
						  pitch + autotilt, 
						  hfov * zoom );
		}
		
		if(hsready){
			if( hs_drawWarpedImages(pdata, curhs, showhs) ){
				dirty = true;
			}
		}
        	
	if( dirty ){// image needs recalculation
     
		// Clear image
		for(int i = 0;  i < vdata.length; i++) vdata[i] = 0;
 		
 		// Call extension applets if any

		// if cube, optimize order
		
		if( app_properties.size() == 6 && filename != null && filename.toLowerCase().endsWith(".mov") ){
			get_cube_order( (int)yaw, (int)pitch, cube_order );
			//System.out.println( "Order: " + cube_order[0] );
			for(int i=0; i<6; i++){
				Applet a = (Applet)applets.get( app_properties.elementAt(cube_order[i]) );
 				if( a != null && sender != null && 
 		   	 		sender.get( a ) != null ){ //Applet is registered as sender
					String s = a.getAppletInfo();
					if( dirty && s != null && s.equals("topFrame")  ){ // Paint image into viewer window
						a.paint(null);
					}
				}
			}
		}else{
			for(int i=0; i<app_properties.size(); i++){
 				Applet a = (Applet)applets.get( app_properties.elementAt(i) );
 				if( a != null && sender != null && 
 		   	 	sender.get( a ) != null ){ //Applet is registered as sender
					String s = a.getAppletInfo();
					if( dirty && s != null && s.equals("topFrame")  ){ // Paint image into viewer window
						a.paint(null);
					}
				}
			}
		}

  		if( dirty && show_pdata ){
				
			int[][] p = pdata;
		
			if( antialias && scaledPanos != null ){
				double sf = hfov / ( (double)vwidth * phfov * max_oversampling );
				int i,pw;
				for(i=0, pw = pdata[0].length; (double) pw * sf > 1.0; i++, pw/=2);
				
				if( scaledPanos.elementAt(i)  != null ){
					p =  (int[][])scaledPanos.elementAt(i);
					// System.out.println("Using Resolution " + i);
					math_updateLookUp( (int) (p[0].length * 360.0 / phfov) );
				}
			}
			switch( quality ){
				case 0: math_extractview(p, vdata, hs_vdata, vwidth, hfov, yaw, pitch, false);
					dirty = false;
					break;
				case 1: if( panning  || lastframe>frames)
						math_extractview(p, vdata, hs_vdata, vwidth, hfov, yaw, pitch, false);
					else{
						math_extractview(p, vdata, hs_vdata, vwidth, hfov, yaw, pitch, true);
						System.gc();
						dirty = false;
					}
					break;
				case 2: if( panning )
						math_extractview(p, vdata, hs_vdata, vwidth, hfov, yaw, pitch, false);
					else{
						math_extractview(p, vdata, hs_vdata, vwidth, hfov, yaw, pitch, true);
						System.gc();
						dirty = false;
					}
					break;
				case 3: math_extractview(p, vdata, hs_vdata, vwidth, hfov, yaw, pitch, true);
					dirty = false;
					break;
				}
			}				
					
		
		hs_setCoordinates(vwidth, vheight, pwidth, pheight, 
							yaw, pitch, hfov);
		sendView();
		frames++;
             
       		source.newPixels();
         }
       	
      	// draw image
 		
 		// Set cursor if image has moved
 		if( panning || lastframe>frames )	
      			PVSetCursor(newx, newy);
 
  //   	g.drawImage(view,0,0,this);	return;	
      	offGraphics.drawImage(view,vx,vy,this);
      	
      	// Hotspot images
      	
       	if(hsready)
       		hs_draw(offGraphics, vx, vy, vwidth, vheight, curhs, showhs);
 
       		
 		if( frame != null )      	
      		offGraphics.drawImage(frame,
      							offwidth  - frame.getWidth(null),
      							offheight - frame.getHeight(null),
      							this);
		
		if( ready ){
			shs_draw( offGraphics );
		}
		
		{
			Enumeration en = sender.elements();
			while( en.hasMoreElements() ){
				try{
					Applet ext = (Applet) en.nextElement();
					if( ext.getAppletInfo() != "topFrame"){
						ext.paint(offGraphics);
					}
				}catch( Exception e1){}
			}
		}
		
    	g.drawImage(offImage,0,0,this);
    	
    }
 	
 
 
	
	
	
	String DisplayHSCoordinates( int xv, int yv ){
		double[] x = math_view2pano( xv, yv, vwidth, vheight,
                            	   pwidth, pheight,
                            	   yaw, pitch, hfov);
		
		x[0] =  ((double) Math.rint( x[0] * 100000.0 / (double)pwidth ))/ 1000.0;
		x[1] =  ((double) Math.rint( x[1] * 100000.0 / (double)pheight))/ 1000.0;
		
		return ("X = " + x[0] + "; Y = " + x[1]);		
	}
	
	
	

	// Return number of hotspot under x/y coordinate
	// or -1, if there is none
	
	int OverHotspot( int x, int y ){
		int i;
		int[] cp;
							
		if( !hsready  || x < 0 || x >= vwidth || y < 0 || y >= vheight )
			return -1;
		
		/*
		cp = math_int_view2pano( x, y, vwidth, vheight,
                             pwidth, pheight,
                             yaw, pitch, hfov);

		i=( pdata[cp[1]][cp[0]] >>> 24 );
		cp = null;
		*/
		i = hs_vdata[y*vwidth + x] & 0xff;
		
		if( filename != null && filename.toLowerCase().endsWith(".mov")){ // qtvr, use other syntax
			if( i == 0 )
				return -1;
			else
				return i-1;
		}
		
		
		if( i!=0xff && i < numhs ){
			// System.out.println("HS " + i);
			return i;
		}
		
		if( hs_image != null )
			return -1;
			

		for( i = 0; i < numhs; i++){
			if( hs_visible[i] && hs_mask[i] == null   &&
								 hs_link[i] == -1	  &&
								 hs_up[i] == NO_UV	  &&
								 hs_vp[i] == NO_UV	  &&
								 x < hs_xv[i] + HSIZE && 
								 x > hs_xv[i] - HSIZE &&
								 y < hs_yv[i] + HSIZE &&
								 y > hs_yv[i] - HSIZE ){
				return i;
			}
		}
		return -1;
	}


	


	/** Wait while autopanning */
	public void waitWhilePanning(){
		while( lastframe>frames ){
			try{
				Thread.sleep( 200 );
			}catch(Exception e){
				return;
			}
		}
		return;
	}
	
	// Some public functions to script the applet
	/** Zoom in 3% */
	public void ZoomIn()		{ gotoView( yaw, 	pitch, 			 hfov/1.03 	); } 
	/** Zoom out 3% */
	public void ZoomOut()		{ gotoView( yaw, 	pitch, 			 hfov*1.03 	); } 
	/** Tilt up 5 degrees */
	public void panUp()			{ gotoView( yaw, 	pitch+hfov/pan_steps, hfov 		); } 
 	/** Tilt down 5 degrees */
 	public void panDown()		{ gotoView( yaw, 	pitch-hfov/pan_steps, hfov 		); } 
	/** Pan left 5 degrees */
	public void panLeft()		{ gotoView( yaw-hfov/pan_steps,pitch, 	 hfov 		); } 
 	/** Pan right 5 degrees */
 	public void panRight()		{ gotoView( yaw+hfov/pan_steps,pitch, 	 hfov 		); } 

 	/** Show Hotspot Images */
 	public void showHS()		{ showhs = true; 	repaint(); } 
 	/** Hide Hotspot Images */
 	public void hideHS()		{ showhs = false;	repaint(); } 
 	/** Toggle Visibility of Hotspot Images */
 	public void toggleHS()		{ showhs = !showhs;	repaint(); } 
 	/** Are Hotspot Images visible? */
 	public boolean isVisibleHS(){ return showhs; } 

 	/** Return current pan angle */
 	public double pan()			{ return yaw; }		
 	/** Return current tilt angle */
 	public double tilt()		{ return pitch; }	
 	/** Return current field of view angle */
 	public double fov()			{ return hfov; }	
	/** Set quality */
 	public void setQuality(int n){
		if( n >=0 && n <=3 ) {
			quality = n;
			dirty = true;
			repaint();
		}
	}
 	
 	/**
 	* Moves from a specific position to another position using a specified amount of frames
    * @param p0 Pan angle of starting view
    * @param p1 Pan angle of target view
    * @param t0 Tilt angle of starting view
    * @param t1 Tilt angle of target view
    * @param f0 Field of View angle of starting view
    * @param f1 Field of View of target view
    * @param nframes the number of frames
    */
	public void moveFromTo( double p0, double p1, double t0, double t1, double f0, double f1, int nframes )
	{
		double dp = 0.0; 
		double dt = (t1-t0)/(double)nframes;
		double z = Math.pow(f1/f0, 1.0/(double)nframes);
		double df = (f1-f0)/(double)nframes;
		
		if( Math.abs(p1-p0) < 180.0 || yaw_max != 180.0 || yaw_min != -180.0)
			dp = (p1-p0)/(double)nframes;
		else if( p1 > p0 )
			dp = (p1-p0-360.0)/(double)nframes;
		else if( p1 < p0 )
			dp = (p1-p0+360.0)/(double)nframes;
			 
		gotoView( p0, t0, f0 );
		lastframe = frames + nframes;
		startAutoPan( dp, dt, z );
		
	}

 	/**
 	* Moves from the current position to another position using a specified amount of frames
    * @param pan Pan angle of target view
    * @param tilt Tilt angle of target view
    * @param fov Field of View of target view
    * @param nframes the number of frames
    */
	public void moveTo( double pan, double tilt, double fov, int nframes )
	{
		moveFromTo( yaw, pan, pitch, tilt, hfov, fov, nframes );
	}

	/**
 	* Starts autopanning.
    * @param p Pan angle increment per frame
    * @param t Tilt angle increment per frame
    * @param z Field of View angle factor per frame
    */
	public void startAutoPan( double p, double t, double z )
	{
		autopan = p; autotilt = t; zoom = z; 
		if( lastframe <= frames )
			lastframe = frames + ETERNITY;
		repaint();
	}

	/**
 	* Stops autopanning. Also stops ongoing <CODE>moveTo()</CODE>
 	* or <CODE>moveFromTo()</CODE> processes.
	*/
	public void stopAutoPan()
	{
		lastframe = 0; autopan = 0.0; autotilt = 0.0; zoom = 1.0; 
	}

	void stopPan(){
		panning = false; oldspeedx = 0.0; oldspeedy = 0.0;
	}	
	
	/**
 	* Returns true if autopanning, or ongoing <CODE>moveTo()</CODE>
 	* or <CODE>moveFromTo()</CODE> processes.
	*/
	public boolean getAutoPan(){ return lastframe > frames; }
	
	
	
	/**
 	* Jump to specific position
    * @param pan Pan angle 
    * @param tilt Tilt angle
    * @param fov Field of View angle
	*/	
	public void gotoView( double pan, double tilt, double fov ){
		if( pan == yaw && tilt == pitch && fov == hfov ) return;

		while( pan > 180.0 ) pan -= 360.0;
		while( pan <-180.0 ) pan += 360.0;
		
		
		double f  = math_fovy( fov,  vwidth, vheight)/2.0;  

		if(tilt > pitch_max - f && pitch_max != 90.0) 
			tilt =  pitch_max - f;
		else if(tilt > pitch_max)
			tilt = pitch_max;
		else if( tilt < pitch_min + f && pitch_min != -90.0) 
			tilt =  pitch_min + f;
		else if(tilt < pitch_min)
			tilt = pitch_min;
		
		// Check and correct for yaw_max/min
		
		if(yaw_max != 180.0 || yaw_min != -180.0){
			double x[]; 
			double xl, xr;
			
			// check left edge
			
			x = math_view2pano( 0, ( pitch > 0.0 ? 0 : vheight-1), vwidth, vheight,
                            	 pwidth, pheight,
                            	 pan, tilt, fov);
			xl = x[0];

			x = math_view2pano( vwidth-1, ( pitch > 0.0 ? 0 : vheight-1), vwidth, vheight,
                            	 pwidth, pheight,
                            	 pan, tilt, fov);
			xr = x[0];
			
			x = null;
			
		//	if( (xr - xl) > (yaw_max-yaw_min)/360.0 * (double)pwidth ) return;
			if( (xr - xl) > (yaw_max-yaw_min)/phfov * (double)pwidth ) return;

		//	if( xl < (yaw_min + 180.0) / 360.0 * (double) pwidth ){ // too far to the right
			if( xl < (yaw_min + phfov/2.0) / phfov * (double) pwidth ){ // too far to the right
				if( lastframe > frames ){ // autopanning
					autopan *= -1.0;
				}
		//		pan +=  yaw_min - ( xl/(double)pwidth * 360.0 - 180.0);
				pan +=  yaw_min - ( xl/(double)pwidth * phfov - phfov/2.0);
			}
				
		//	if( xr > (yaw_max + 180.0) / 360.0 * (double) pwidth ){
			if( xr > (yaw_max + phfov/2.0) / phfov * (double) pwidth ){
				if(lastframe > frames ){
					autopan *= -1.0;
				}
		//		pan -= ( xr/(double)pwidth * 360.0 - 180.0) - yaw_max ;
				pan -= ( xr/(double)pwidth * phfov - phfov/2.0) - yaw_max ;
			}
		}
			
		if( 2.0 * f <=  pitch_max - pitch_min 
			&& fov <= hfov_max
			&& fov >= hfov_min
			&& fov <= yaw_max - yaw_min
			&& tilt <= pitch_max
			&& tilt >= pitch_min
			&& pan  <= yaw_max
			&& pan  >= yaw_min){
						
			if( pan != yaw || tilt != pitch || fov != hfov){
				yaw = pan; pitch = tilt; hfov = fov;
				dirty = true;
				repaint();
				return;
			}
		}
		
		// If we reach this point, then there is no change
		// We have probably reached the end of an autopan
		stopAutoPan();
		
	}

	/**
	* Jump to the url-link specified in a hotspot
    * @param n The list number of the hotspot
	*/
	public void gotoHS( int n ){
		if( n < 0 || n >= numhs )
			return;
			
		JumpToLink( hs_url[n], hs_target[n] );
	}
	
	/**
	* Jump to the url-link specified in a static hotspot
    * @param n The list number of the static hotspot
	*/
	void gotoSHS( int n ){
		if( n < 0 || n >= numshs )
			return;
			
		JumpToLink( shs_url[n], shs_target[n] );
	}
	
	void JumpToLink( String url, String target )
	{
		URL u;
		
		if(url == null )
			return;
		
		if( url.startsWith("ptviewer:") ){
			executePTViewerCommand( url.substring( url.indexOf(':')+1 ));
			return;
		}
		
		if( url.startsWith("javascript:") ){
			executeJavascriptCommand( url.substring( url.indexOf(':')+1 ));
			return;
		}
				
		try{
			u = new URL(getDocumentBase(),url );
		}
		catch(MalformedURLException mue){
			System.err.println("URL " + url + " ill-formed");
			return;
		}
		
		if( target == null )
			getAppletContext().showDocument(u);
		else
			getAppletContext().showDocument(u, target );
		 
	}

	
	/** Load a new panoramic image from a list.
    * @param k The list number of the panorama
    * @param pan Pan angle
    * @param tilt Tilt angle
    * @param fov Field of view angle
	*/
	public synchronized void newPanoFromList( int k, double pan, double tilt, double fov ){
		loadPanoFromList(  k );	
		yaw = pan; pitch = tilt; hfov = fov;
		repaint();
		start();
	}

	
	/** Load a new panoramic image from a list.
    * @param k The list number of the panorama
	*/
	public synchronized void newPanoFromList( int k ){
		loadPanoFromList( k );
		repaint();
		start();
	}


	
	void loadPanoFromList( int k ){
		String p = myGetParameter(null, "pano" + k);
 		if( p == null) return;
		stop();
		PV_reset(); 
 		initialize();
		
		CurrentPano = k;
		if( PTViewer_Properties != null )
			ReadParameters( PTViewer_Properties );
 		ReadParameters( p );
	}
	
	public void newPano( String p ){
		stop();
		PV_reset(); 
 		initialize();
		if( PTViewer_Properties != null )
			ReadParameters( PTViewer_Properties );
 		ReadParameters( p );
		repaint();
		start();
	}
	
	// For QTVR compatibility	
	public void SetURL( String url ){
		newPano( "{file=" + url + "}" );
	}
		
	
	
	// Read parameters for panorama "p"
	
	void ReadParameters( String p ){
 		String s;
 		int i,k;
 		
 		s = myGetParameter( p, "bgcolor");	if( s!=null )	bgcolor = new Color( Integer.parseInt(s,16) );
		s = myGetParameter( p, "barcolor");	if( s!=null )	pb_color = new Color( Integer.parseInt(s,16) );
		s = myGetParameter( p, "bar_x");	if( s!=null )	pb_x = Integer.parseInt(s);
		s = myGetParameter( p, "bar_y");	if( s!=null )	pb_y = Integer.parseInt(s);
		s = myGetParameter( p, "bar_width");	if( s!=null )	pb_width = Integer.parseInt(s);
		s = myGetParameter( p, "bar_height");	if( s!=null )	pb_height = Integer.parseInt(s);
		s = myGetParameter( p, "maxarray"); 	if( s!=null )	im_maxarray = Integer.parseInt(s);
		s = myGetParameter( p, "view_width");	if( s!=null )	{ vwidth  = Integer.parseInt(s); vset = true; }
		s = myGetParameter( p, "view_height");	if( s!=null )	{ vheight = Integer.parseInt(s); vset = true; }
		s = myGetParameter( p, "view_x");	if( s!=null )	vx = Integer.parseInt(s);
		s = myGetParameter( p, "view_y");	if( s!=null )	vy = Integer.parseInt(s);
		s = myGetParameter( p, "preload");	if( s!=null )	preload = s;
		s = myGetParameter( p, "cache");	if( s!=null )	{ if( s.equalsIgnoreCase("false") ) file_cachefiles = false; }
		s = myGetParameter( p, "cursor");	if( s!=null )	{ if( s.equalsIgnoreCase("CROSSHAIR") ) ptcursor = Frame.CROSSHAIR_CURSOR;
									  else if( s.equalsIgnoreCase("MOVE"))  ptcursor = Frame.MOVE_CURSOR; }
		s = myGetParameter( p, "grid_bgcolor");	if( s!=null )   grid_bgcolor = Integer.parseInt(s,16);
		s = myGetParameter( p, "grid_fgcolor");	if( s!=null )   grid_fgcolor = Integer.parseInt(s,16);
		s = myGetParameter( p, "mass");		if( s!=null )   { MASS = Double.valueOf(s).doubleValue();}
		s = myGetParameter( p, "antialias");	if( s!=null )	{ antialias = true; }
		s = myGetParameter( p, "quality");	if( s!=null )	{ quality = Integer.parseInt(s); 
									  if( quality < 0 ) quality = 0;
									  if( quality > 3 ) quality = 3;
									}
		s = myGetParameter( p, "pfov");		if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									  if( x > 0.0 && x <= 360.0 ){
										phfov = x;
										yaw_max = x/2.0;
										yaw_min = -yaw_max;
										if(hfov_max > phfov) hfov_max = phfov;
										if(hfov > phfov) hfov = phfov;
									  }
									}
		s = myGetParameter( p, "panmax");	if( s!=null )	yaw_max =  Double.valueOf(s).doubleValue();
		s = myGetParameter( p, "panmin");	if( s!=null )	yaw_min =  Double.valueOf(s).doubleValue();
		s = myGetParameter( p, "fovmax");	if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									if( x <= HFOV_MAX ) hfov_max = x > (yaw_max - yaw_min) ? yaw_max - yaw_min : x; }
		s = myGetParameter( p, "fovmin");	if( s!=null )	hfov_min = Double.valueOf(s).doubleValue();
		s = myGetParameter( p, "fov");		if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									 if( x <= hfov_max && x >= hfov_min ) hfov = x; }
		s = myGetParameter( p, "inits");	if( s!=null )	inits = s;
		s = myGetParameter( p, "tiltmin");	if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									 if( x > -90.0 && x < 0.0 ) pitch_min = x;}
		s = myGetParameter( p, "tiltmax");	if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									 if( x <  90.0 && x > 0.0 ) pitch_max = x;}
		s = myGetParameter( p, "tilt");		if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									if( x >=  pitch_min && x <= pitch_max ) pitch = x;}
		s = myGetParameter( p, "pan");		if( s!=null )	{ double x = Double.valueOf(s).doubleValue();
									if( x>=yaw_min && x<=yaw_max ) yaw = x;}
		s = myGetParameter( p, "wait");		if( s!=null )	{ dwait = null; dwait = loadImage( s ); update(getGraphics()); }
		s = myGetParameter( p, "auto");		if( s!=null )	autopan =  Double.valueOf(s).doubleValue();
		s = myGetParameter( p, "mousehs"); 	if( s!=null )	MouseOverHS = s;
		s = myGetParameter( p, "getview"); 	if( s!=null )	GetView = s;
		s = myGetParameter( p, "frame");	if( s!=null )	{ frame = null; frame = loadImage( s );}
		s = myGetParameter( p, "waittime");	if( s!=null )	waittime = Integer.parseInt(s);
		s = myGetParameter( p, "hsimage");	if( s!=null )	hs_image = s;
		s = myGetParameter( p, "pwidth");	if( s!=null )	pwidth  = Integer.parseInt(s); 
		s = myGetParameter( p, "pheight");	if( s!=null )	pheight = Integer.parseInt(s);
		s = myGetParameter( p, "file"); 	if( s!=null ) 	filename = s;
		s = myGetParameter( p, "order");	if( s!=null )	order = s;
		s = myGetParameter( p, "oversampling");	if( s!=null ) 	max_oversampling = Double.valueOf(s).doubleValue();
		
		for(i=0; i<=hotspots.size(); i++){
			s=myGetParameter( p, "hotspot" + i);
			if( s != null){
				if(i<hotspots.size()){ // Discard hotspots
					hotspots.setSize(i);
				}
				hotspots.addElement(s);
			}
		}
		

		for(i=0; i<=shotspots.size(); i++){
			s=myGetParameter( p, "shotspot" + i);
			if( s != null){
				if(i<shotspots.size()){ // Discard static hotspots
					shotspots.setSize(i);
				}
				shotspots.addElement(s);
			}
		}


		for(i=0; i<=sounds.size(); i++){
			s=myGetParameter( p, "sound" + i);
			if( s != null){
				if(i<sounds.size()){ // Discard sounds
					sounds.setSize(i);
				}
				sounds.addElement(s);
			}
		}
		
		for(i=0; i<=app_properties.size(); i++){
			s=myGetParameter( p, "applet" + i);
			if( s != null){
				if(i<app_properties.size()){ // Discard applets
					stopApplets(i);
					app_properties.setSize(i);
				}
				app_properties.addElement(s);
			}
		}

		

	}
	



					
 	void executeJavascriptCommand( String s ){
 		if( s != null ){
 			try{
 				Class c = Class.forName("netscape.javascript.JSObject");
 				Object win = c.getMethod("getWindow", new Class[]{ Applet.class }).invoke( c, new Object[] {this} );
 				c.getMethod("eval", new Class[]{ String.class }).invoke( win, new Object[]{s} );
   			}catch(Exception e){}
 		}
 	}
 
 	void executePTViewerCommand( String s ){
		stopThread(ptviewerScript); 
 		ptviewerScript = new Thread(this );
 		PTScript = s;
 		ptviewerScript.start();
 	}


 	void PTViewerScript( String s ){
 		int n = getNumArgs( s, ';' );

 		if( n > 0 ){
 			int i;
 			for(i=0; i<n; i++){
 				String c = stripWhiteSpace( getArg( i, s, ';' ) );
 				if( c.equals("loop()") ){
 					i = -1;
 				}else{
 					PTViewerCommand( c );
 				}
 			}
 		}
 	}



 	// Parser to read and execute PTViewer commands
 	void PTViewerCommand( String s )
 	{
  		String args		= s.substring( s.indexOf('(')+1, s.indexOf(')') );
  		
		if		( s.startsWith("ZoomIn"))		ZoomIn();
		else if ( s.startsWith("ZoomOut"))		ZoomOut();
		else if ( s.startsWith("panUp"))		panUp();
		else if ( s.startsWith("panDown"))		panDown();
		else if ( s.startsWith("panLeft"))		panLeft();
		else if ( s.startsWith("panRight"))		panRight();
		else if ( s.startsWith("showHS"))		showHS();
		else if ( s.startsWith("hideHS"))		hideHS();
		else if ( s.startsWith("toggleHS"))		toggleHS();
		else if ( s.startsWith("gotoView"))		{
													if( getNumArgs( args ) != 3 ) return;
													gotoView( Double.valueOf(getArg(0, args)).doubleValue(),
														  	  Double.valueOf(getArg(1, args)).doubleValue(),
														  	  Double.valueOf(getArg(2, args)).doubleValue() );
												}
		else if ( s.startsWith("startAutoPan"))	{
													if( getNumArgs( args ) != 3 ) return;
													startAutoPan( Double.valueOf(getArg(0, args)).doubleValue(),
														  	  Double.valueOf(getArg(1, args)).doubleValue(),
														  	  Double.valueOf(getArg(2, args)).doubleValue() );
												}
		else if ( s.startsWith("stopAutoPan"))	stopAutoPan();
		else if ( s.startsWith("newPanoFromList")){
													if( getNumArgs( args ) == 1 )
														newPanoFromList( Integer.parseInt(args) );
													else if( getNumArgs( args ) == 4 ){
														newPanoFromList( Integer.parseInt(getArg(0, args)),
															Double.valueOf(getArg(1, args)).doubleValue(),
															Double.valueOf(getArg(2, args)).doubleValue(),
															Double.valueOf(getArg(3, args)).doubleValue() );
													}
												}
		else if ( s.startsWith("newPano"))	 	 newPano( args );
		else if ( s.startsWith("SetURL"))	 	 SetURL( args );
		else if ( s.startsWith("PlaySound"))	 PlaySound( Integer.parseInt(args) );
		else if ( s.startsWith("moveFromTo")){
													if( getNumArgs( args ) != 7 ) return;
													moveFromTo( Double.valueOf(getArg(0, args)).doubleValue(),
														  	  Double.valueOf(getArg(1, args)).doubleValue(),
														  	  Double.valueOf(getArg(2, args)).doubleValue(),
														  	  Double.valueOf(getArg(3, args)).doubleValue(),
														  	  Double.valueOf(getArg(4, args)).doubleValue(),
														  	  Double.valueOf(getArg(5, args)).doubleValue(),
														  	  Integer.valueOf(getArg(6, args)).intValue() );
											}
		else if ( s.startsWith("moveTo"))	{												
													if( getNumArgs( args ) != 4 ) return;
													moveTo(   Double.valueOf(getArg(0, args)).doubleValue(),
														  	  Double.valueOf(getArg(1, args)).doubleValue(),
														  	  Double.valueOf(getArg(2, args)).doubleValue(),
														  	  Integer.valueOf(getArg(3, args)).intValue() );
											}
		else if ( s.startsWith("DrawSHSImage"))	 DrawSHSImage( Integer.parseInt(args) );
		else if ( s.startsWith("HideSHSImage"))	 HideSHSImage( Integer.parseInt(args) );
		else if ( s.startsWith("DrawHSImage"))	 DrawHSImage( Integer.parseInt(args) );
		else if ( s.startsWith("HideHSImage"))	 HideHSImage( Integer.parseInt(args) );
		else if ( s.startsWith("ToggleHSImage")) ToggleHSImage( Integer.parseInt(args) );
		else if ( s.startsWith("ToggleSHSImage"))ToggleSHSImage( Integer.parseInt(args) );
		else if ( s.startsWith("waitWhilePanning"))	 waitWhilePanning();
		else if ( s.startsWith("startApplet"))	 startApplet( Integer.parseInt(args) );
		else if ( s.startsWith("stopApplet"))	 stopApplet( Integer.parseInt(args) );
		else if ( s.startsWith("setQuality"))    setQuality(Integer.parseInt(args));
 	}
 
	
	
	/** Draw static hotspot image
    * @param n The list number of the static hotspot
	*/
	public synchronized void DrawSHSImage( int n )
	{
		if( n >= 0 && n < numshs && shs_imode[n] != IMODE_ALWAYS){
			shs_imode[n] = IMODE_ALWAYS;
			repaint();
		}
	}
	
	/** Hide static hotspot image
    * @param n The list number of the static hotspot
	*/
	public synchronized void HideSHSImage( int n )
	{
		if( n >= 0 && n < numshs && shs_imode[n] != IMODE_NORMAL){
			shs_imode[n] = IMODE_NORMAL;
			repaint();
		}
	}

	/** Toggle visibility of static hotspot image
    * @param n The list number of the static hotspot
	*/
	public synchronized void ToggleSHSImage( int n ){
		if( n >= 0 && n < numshs ){
			if( shs_imode[n] != IMODE_NORMAL )
				HideSHSImage( n );
			else if( shs_imode[n] != IMODE_ALWAYS ) 
				DrawSHSImage( n );
		}
	}
		


	/** Draw hotspot image
    * @param n The list number of the hotspot
	*/
	public synchronized void DrawHSImage( int n ){
		if( n >= 0 && n < numhs && (hs_imode[n] & IMODE_ALWAYS) == 0){
			hs_imode[n] |= IMODE_ALWAYS;
			repaint();
		}
	}
	
	/** Hide hotspot image
    * @param n The list number of the hotspot
	*/
	public synchronized void HideHSImage( int n ){
		if( n >= 0 && n < numhs && (hs_imode[n] & IMODE_ALWAYS) != 0){
			hs_imode[n] &= ~IMODE_ALWAYS;
			repaint();
		}
	}
	
	/** Toggle visibility of hotspot image
    * @param n The list number of the hotspot
	*/
	public synchronized void ToggleHSImage( int n ){
		if( n >= 0 && n < numhs ){
			if((hs_imode[n] & IMODE_ALWAYS) != 0)
				 HideHSImage( n );
			else if( (hs_imode[n] & IMODE_ALWAYS) == 0)
				 DrawHSImage( n );
		}
	}
		
	
	/** The current horizontal and relative mouse coordinates 
	* in the panoramic image. 0 - left, 100 - right.
	*/
	public double get_x() { 
	   	int[] cp;
	   	double result = -1.0;
	   	
	   	if( click_x >= 0 && click_y >= 0 ){
	   		cp = math_int_view2pano( click_x - vx, click_y - vy, vwidth, vheight,
                            	 pwidth, pheight,
                            	 yaw, pitch, hfov);
			result =  cp[0]*100.0/(double)pwidth;
		}
		cp = null;
		
		return result;
	}

	/** The current vertical and relative mouse coordinates 
	* in the panoramic image. 0 - top, 100 - bottom.
	*/
	public double get_y() {
	   	int[] cp;
	   	double result = -1.0;
	   	
	   	if( click_x >= 0 && click_y >= 0 ){
	   		cp = math_int_view2pano( click_x - vx, click_y - vy, vwidth, vheight,
                            	 pwidth, pheight,
                            	 yaw, pitch, hfov);
			result =  cp[1]*100.0/(double)pheight;
		}

		cp = null;
		click_x = -1; click_y = -1;
		return result;
	}

	/** The list number of the current panoramic image 	*/
	public int getPanoNumber() { return CurrentPano; }
	
  	
	
	/** Specify an applet to communicate with.
	* This applet's <CODE>paint()</CODE> method will
	* be called whenever the view changes.
    * @param a the applet.
 	*/
 	public  void startCommunicating(Applet a){
  		synchronized(sender){
 			if( a!= null )
 				sender.put( a, a );
			else
				sender.clear();
		}
		dirty = true;
		repaint();
	}
	
	/** Stop communicationg with applet.
    * @param a the applet.
 	*/
 	public void stopCommunicating(Applet a){
 		if( a!= null ){
 			synchronized(sender){
 				sender.remove( a );
 			}
 			dirty = true;
			repaint();
		}
	}


	// Get path to document, but without leading A:/C: etc.
	
	private String ptgetPath(){
		String s = getDocumentBase().getFile();
		int i = s.indexOf(':');
		if(  i != -1  && i+1 < s.length()){
			return s.substring(i+1);
		}
		i = s.indexOf('|');
		if(  i != -1  && i+1 < s.length()){
			return s.substring(i+1);
		}
		
		return s;
	}
	
	void stopThread( Thread t ){
		if(t != null && t.isAlive()){
			try{
				t.checkAccess();
				t.stop();
			}catch( SecurityException  e ){
				t.destroy();
			}
		}
	}

void ptinsertImage( int[][]pd, int xi, int yi, Image im, int ntiles ){
	if( im == null ) return;
	int w = im.getWidth(null), h = im.getHeight(null);
	if(ntiles > h) ntiles = h;
	int ht = (h + ntiles -1) / ntiles;

	int[] idata = new int[ w * ht  ];
	int i, sheight;
	PixelGrabber pg;
	for(i=0; i<ntiles; i++){
			
		sheight = (ht + i * ht > h ? h - i*ht : ht);

		pg = new PixelGrabber( im, 0, i * ht, 
								   w, sheight, idata, 0, w );
        try { pg.grabPixels(); } 
        catch (InterruptedException e) { return; }
			
		im_insertRect( pd, xi, yi + i * ht, idata, w, 0, 0, w, sheight );
		dirty = true;
		repaint();
	}
	idata = null;
}  	
	

boolean is_inside_viewer(int x, int y){
	if( x>= vx && y >= vy && x < vx + vwidth && y < vy + vheight )
		return true;
	return false;
} 	

// determine optimum drawing order for cubic faces	
void get_cube_order( int pan, int tilt, int[] n){
	
	if( tilt > 45 ){
		n[0] = 4; n[5] = 5;
	}else if( tilt < -45 ){
		n[0] = 5; n[5] = 4;
	}else{
		switch( pan/45 ){
			case 0:  n[0] = 2; n[1] = 3; n[3] = 1; n[4] = 0; break;
			case -1: n[0] = 2; n[1] = 1; n[3] = 3; n[4] = 0; break;
			case 1:  n[0] = 3; n[1] = 2; n[3] = 1; n[4] = 0; break;
			case 2:  n[0] = 3; n[1] = 0; n[3] = 1; n[4] = 2; break;
			case 3:  n[0] = 0; n[1] = 3; n[3] = 1; n[4] = 2; break;
			case -2: n[0] = 1; n[1] = 0; n[3] = 3; n[4] = 2; break;
			case -3: n[0] = 1; n[1] = 0; n[3] = 3; n[4] = 2; break;
			default: n[0] = 0; n[1] = 1; n[3] = 3; n[4] = 2; break;
		}
		n[2] = tilt>0?4:5;
		n[5] = tilt>0?5:4; 
		return;
	}

	switch( pan/45 ){
		case 0:  n[1] = 2; n[2] = 3; n[3] = 1; n[4] = 0; break;
		case -1: n[1] = 2; n[2] = 1; n[3] = 3; n[4] = 0; break;
		case 1:  n[1] = 3; n[2] = 2; n[3] = 1; n[4] = 0; break;
		case 2:  n[1] = 3; n[2] = 0; n[3] = 1; n[4] = 2; break;
		case 3:  n[1] = 0; n[2] = 3; n[3] = 1; n[4] = 2; break;
		case -2: n[1] = 1; n[2] = 0; n[3] = 3; n[4] = 2; break;
		case -3: n[1] = 1; n[2] = 0; n[3] = 3; n[4] = 2; break;
		default: n[1] = 0; n[2] = 1; n[3] = 3; n[4] = 2; break;
	}
	return;
}	


// Image loading and manipulating

	int im_maxarray 					=		0x80000;// Max size of linear arrays in Netscape
    // Background Grid
    int  grid_bgcolor					= 0xffffff; // white
    int  grid_fgcolor					= 0;		// black


	/**
	*  Load an image with name <CODE>name</CODE>. This
	* method extends the default <CODE>getImage()</CODE> of
	* Java, and searches the Resources (archive) also.
	* The image is loaded in memory when the function returns,
	* or <CODE>null</CODE> is returned.
	*/
	public Image loadImage( String name ){    		
   		Image image = null;
   		
   		byte[] b = file_read( name, null );
   		
   		if( b != null ){
   			image = bufferToImage( b );
			b = null;
			if(image != null)
				return image;
		}
 
        try{
         	URL url = new URL(getDocumentBase(), name);
          	image = getImage(url);
 			MediaTracker tracker = new MediaTracker(this);
 			tracker.addImage(image, 0);
 			tracker.waitForAll();
 			if( image == null || image.getWidth(null) <= 0 )	// No image
 				return null;
 			else
 				return( image );
        }catch(Exception e){
        	return null; 
        }
	}


    
	// Load an image with download feedback
	Image loadImageProgress( String name ){
   		Image image = null;
   		percent[0] = 0;
   		
   		byte[] b = file_read( name, percent );
   		
   		if( b != null ){
   			image = bufferToImage( b );
			percent[0] = 100; repaint();	// Show 100% when decompressed
			b = null;
			if( image != null )
				return image;
		}
		
		return loadImage( name );
	}
  


	Image bufferToImage( byte[] buf ){
		if(buf==null) return null;
		Image image = Toolkit.getDefaultToolkit().createImage(buf);
		
		MediaTracker tracker = new MediaTracker(this);
		tracker.addImage(image, 0);
		try {
    		tracker.waitForAll();
   		} catch (InterruptedException e) {return null;}
   		return image;
   	}



// Allocate new array only if existing array doesn't fit

int[][] im_allocate_pano(int[][] pd, int pwidth, int pheight){
	if( pd == null ||
	   (pd.length != pheight || pd[0].length != pwidth) ){
		try{
        	int[][]p = new int[pheight][pwidth];
        	return p;
        }catch(Exception e){
 			return null;
 		}
 	}
 	return pd;
}
	   
				
		   	    	

void im_drawGrid( int [][] p, int g_bgcolor, int g_fgcolor ){
	int x,y,i,cy;
	int bg 		= g_bgcolor | 0xff000000;
	int fg 		= g_fgcolor | 0xff000000;
	
	if(p==null) return;
	int height 	= p.length;
	int width 	= p[0].length;
 		
    for(y = 0; y < height; y++)
    	for( x = 0; x < width; x++ ) 
     		p[y][x] = bg;
          	
 	// Horizontal lines
 	for(y = 0,i=18*2*height/width ; i >= 0; i--){
 		cy = y+1;
        for( x = 0; x < width; x++ ){ 
        	p[y] [x] = fg;
        	p[cy][x] = fg;
        }
        if( i!= 0)
        	y += (height-2-y)/i;
   	}
      	
 	// Vertical lines
	for(x = 0,i=36 ; i >= 0; i--){
 		if( x== 0)
         	for( y = 0; y < height; y++ ) 
         		p[y][x] = fg;
        else if( x >= width-1 ){
         	x = width-1; i=0;
          	for( y = 0; y < height; y++ ) 
         		p[y][x] = fg;
        }else{
        	cy = x+1;
        	for( y = 0; y < height; y++ ){ 
         		p[y][x]  = fg;
        		p[y][cy] = fg;
        	}
        }
        if( i!= 0)
        	x += (width-1-x)/i;
	}
}


	// Set alpha channel in rectangular region of
	// two dimensional array p  to 'alpha'
	
	void SetPAlpha( int x0, int y0, int x1, int y1, int alpha, int[][] p){
		int x,y, hmask = (alpha<<24) + 0x00ffffff;
		
		int h = p.length, w = p[0].length;
		
		int ymin = Math.min( y0, y1 ); if(ymin<0)	ymin=0;
		int ymax = Math.max( y0, y1 ); if(ymax>=h) 	ymax=h-1;
		
		if(x0<0)  x0=0;
		if(x0>=w) x0=w-1;
		
		if(x1<0)  x1=0;
		if(x1>=w) x1=w-1;

				
		if(x1 >= x0){
			for(y=ymin; y<=ymax; y++)
				for(x=x0; x<=x1; x++)
					p[y][x] &= hmask;
		}else{
			for(y=ymin; y<=ymax; y++){
				for(x=0; x<=x1; x++)
					p[y][x] &= hmask;
				for(x=x0; x<w; x++)
					p[y][x] &= hmask;
			}
		}
	}
	
// Scale pixel area to pwidth/pheight
// Use same procedure as Panorama Tools

void  scaleImage(int[][] pd, int width, int height){
	if( pd == null ) return;
  	int ph = pd.length, pw = pd[0].length;

	int x,y,xs,ys,xd,yd;
	int dx,dy,xs0, xs1, ys0, ys1;
	int scale = 256 * width / pw;
	int w2  = pw  * 128 - 128;
	int h2  = ph * 128 - 128;
	int sw2 = width   * 128 - 128;
	int sh2 = height  * 128 - 128;
		
	int w3 = - w2 * width / pw + sw2; 
	int w1 = width-1;
		

	for(y=ph-1; y>=0; y--){
		yd = y * 256 - h2;
		ys = yd * width / pw + sh2;
		dy = ys & 0xff; ys = (ys >> 8);
		if(ys<0){
			ys0=ys1=0;
		}else if(ys>=height-1){
			ys0=ys1=height-1;
		}else{
			ys0=ys++;ys1=ys;
		}

		for(x=pw-1; x>=0; x--){
			xs = x * scale + w3; 
			dx = xs & 0xff; xs = (xs >> 8);
			if(xs<0){
				xs0=xs1=0;
			}else if(xs>=w1){
				xs0=xs1=w1;
			}else{
				xs0=xs++;xs1=xs;
			}

			pd[y][x] = bil(pd[ys0][xs0], pd[ys0][xs1], 
						   pd[ys1][xs0], pd[ys1][xs1], dx, dy);
		}
	}
}


	// Load image into 2D-array
	// array must be allocated to hold enough data
	// Uses variable im_maxarray
	void ptImageTo2DArray( int[][]data, Image im ){
		if( im == null || data == null ) return;
		
        int grabHeight = im.getHeight(null);
        if(grabHeight*im.getWidth (null) > im_maxarray) 
        	grabHeight= im_maxarray/im.getWidth(null);
        
        int[] tdata = new int[grabHeight * im.getWidth(null)];

       	// Grab pixels in chunks of im_maxarray size
       	int cy,dy,x,y;
       	PixelGrabber pg;
       	
       	for(cy=0; cy < im.getHeight(null); cy += grabHeight ){
        	int sheight = (grabHeight < im.getHeight(null) - cy ? 
        				   grabHeight : im.getHeight(null) - cy );
        	
        	pg = new PixelGrabber( im, 0, cy,im.getWidth (null), sheight, tdata, 0, im.getWidth (null) );
        	try { pg.grabPixels(); } 
        	catch (InterruptedException e){
            	return;
        	}
        	
        	// Store pixels in 2 dimensional array
        	
        	for(y=0; y<sheight; y++){
        		dy = y*im.getWidth(null);
         		for(x=0; x<im.getWidth(null); x++)
        			data[y+cy][x] = tdata[dy + x] | 0xff000000;
        	}
        }	
  	    tdata = null;
 	    System.gc();
	}
	
		// Set alpha channel of pixel data to image im
	void ptImageToAlpha( int[][]data, Image im){
		if( im==null || data==null ) return;

       	int grabHeight = im.getHeight(null);
        if(grabHeight*im.getWidth (null) > im_maxarray) 
        	grabHeight= im_maxarray/im.getWidth(null);
        
        int[] tdata = new int[grabHeight * im.getWidth(null)];
		
	    int hmask, cy,dy,x,y;
	    PixelGrabber pg;
			
       	for(cy=0; cy < im.getHeight(null); cy += grabHeight ){
        	int sheight = (grabHeight < im.getHeight(null) - cy ? 
        				   grabHeight : im.getHeight(null) - cy );
        	
        	pg = new PixelGrabber( im, 0, cy,im.getWidth (null), sheight, tdata, 0, im.getWidth (null) );
        	try { pg.grabPixels(); } 
        	catch (InterruptedException e){
            	return;
        	}
        	
         	for(y=0; y<sheight; y++){
         		dy = y*im.getWidth(null);
         		for(x=0; x<im.getWidth(null); x++){
         				hmask = ((tdata[dy + x] & 0xff) << 24) + 0x00ffffff;
        				data[y+cy][x] &= hmask;
 	    		}
 	    	}
 	    }
 	    tdata = null;
 	    System.gc();
	}


// Insert a rectangular image into panorama	
// preserve alpha channel in panorama, insert only if alpha channel set
// in insert

void im_insertRect( int[][] pd, int xd, int yd, int[] id, int iwidth, 
					int xs, int ys, int width, int height ){
	int x,y,xp,yp,idx,px;
	
	try{
		for(y = 0, yp = yd; y < height; y++, yp++){
			for(x = 0, idx = (ys + y) * iwidth + xs; x < width; x++, idx++){
				px = id[idx];
				if((px & 0xff000000) != 0 ){ //Non transparent
					xp = x + xd;
					pd[yp][xp] = px & (pd[yp][xp]  | 0x00ffffff);
				}
			}
		}
	}catch(Exception e){
		System.out.println("Insert can't be fit into panorama");
	}
}


				

// Extract a rectangular image from panorama
// Remove alpha channel	

final void im_extractRect( int[][] pd, int xd, int yd, int[] id, int iwidth, 
					int xs, int ys, int width, int height ){
	int x,y,yp,idx,px;
	
	try{
		for(y = 0, yp = yd; y < height; y++, yp++){
			for(x = 0, idx = (ys + y) * iwidth + xs; x < width; x++, idx++){
				id[idx] = pdata[yp][x+xd] | 0xff000000; // remove possible mask
			}
		}
	}catch(Exception e){
		System.out.println("Invalid rectangle");
	}
}
				
// load the panoramic image
final int[][] im_loadPano(String fname, int[][] pd, int pw, int ph){
	int [][] p;
 

	if(fname == null || fname.equals("_PT_Grid")){
		if( pw == 0 ) pw = 100; // Dummy background 

		// Create grid panorama	
		p = im_allocate_pano( pd, pw, (ph == 0 ? pw/2 : ph) );
		im_drawGrid( p, grid_bgcolor, grid_fgcolor );
 		return p;
	}
    	
    	Image pano = loadImageProgress( fname );
			
 	if( pano == null ){
		return null;
	}
 
 	// At this point we have a valid panorama image
 	// Check size:
 		
 	if(pw > pano.getWidth (null)){
 		if(ph == 0) ph = pw/2;
 	}else{
 		pw 	= pano.getWidth (null);
		ph 	= pano.getHeight(null);
	}
		
	// Set up data array for panorama pixels
		
	p = im_allocate_pano( pd, pw, ph);
	if( p == null ) return null;
  
  	ptImageTo2DArray( p, pano );     	
       
        
  	if(pw != pano.getWidth (null)){
        scaleImage( p, pano.getWidth(null), pano.getHeight(null));
	}
	pano = null;
	return p;
}


// Create and return half-sized pixel array
int[][]  im_halfsize(int[][] pd){
	int ph = pd.length, pw = pd[0].length;
	int nh = ph/2, nw = pw/2;

	int[][] nd = new int[nh][nw];
	if( nd == null ) return null;
//	System.out.println("nw = " + nw);

	for(int y=0, y0 = 0, y1 = 1; y<nh ; y++, y0+=2, y1+=2){
		int[] p0 = pd[y0];
		int[] p1 = pd[y1];
		int[] n = nd[y];
		for(int x=0, x0=0, x1 = 1; x<nw ; x++, x0+=2, x1+=2){
			n[x] = im_pixelaverage( p0[x0], p0[x1], p1[x0], p1[x1] );
		}
	}
	return nd;
}

// Create and return half-sized byte array,
// no averaging!
byte[][]  im_halfsize(byte[][] pd){
	int ph = pd.length, pw = pd[0].length;
	int nh = ph/2, nw = pw/2;

	byte[][] nd = new byte[nh][nw];
	if( nd == null ) return null;

	for(int y=0, y0 = 0; y<nh ; y++, y0+=2){
		byte[] p = pd[y0];
		byte[] n = nd[y];
		for(int x=0, x0=0; x<nw ; x++, x0+=2){
			n[x] = p[x0];
		}
	}
	return nd;
}

static final int im_pixelaverage( int x00, int x01, int x10, int x11 ){
	int r = (((x00 >> 16) & 0xff) + ((x01 >> 16) & 0xff) + ((x10 >> 16) & 0xff) + ((x11 >> 16) & 0xff)) / 4;
	if(r<0) r=0; if(r>0xff) r=0xff;
	int g = (((x00 >> 8) & 0xff) + ((x01 >> 8) & 0xff) + ((x10 >> 8) & 0xff) + ((x11 >> 8) & 0xff)) / 4;
	if(g<0) g=0; if(g>0xff) g=0xff;
	int b = ((x00  & 0xff) + (x01  & 0xff) + (x10 & 0xff) + (x11 & 0xff)) / 4;
	if(b<0) b=0; if(b>0xff) b=0xff;

	return (x00 & 0xff000000) + (r << 16) + (g << 8) + b;
}

	
	
// String handling and parser


	// change &quot; to "
	String resolveQuotes( String s ){
		if(s==null) return null;
   		int i=0,length = s.length();

    	if( length < 6 ) return s;

    	StringBuffer w = new StringBuffer(0);
    	
   		for(i=0; i<length-5; i++){
   			if( s.substring(i, i+6).equalsIgnoreCase( "&quot;" ) ){
   				w.append('\"');
   				i+=5;
   			}else{
   				w.append(s.charAt(i));
   			}
   		}
   		w.append(s.substring(i, length));
   		
   		return w.toString();
   	}


 	// Remove White space characters before and after s
 	String stripWhiteSpace( String s ){ 
 		if(s == null) return null;
 		
 		int i=0, l=s.length(), k=l-1;
 		
 		while(i<l && (s.charAt(i) == ' ' || s.charAt(i) == '\r' || s.charAt(i) == '\n' || s.charAt(i) == '\t')) i++;
 		
 		if(i==l) return null;
 		
 		while(k>=0 && (s.charAt(k) == ' ' || s.charAt(k) == '\r' || s.charAt(k) == '\n' || s.charAt(k) == '\t')) k--;
 		
 		if(k<0 || k<i) return null;
 
 		return s.substring( i, k+1 );
  	}		
				 		

// Get size of Textwindow
	
Dimension string_textWindowSize( Graphics g, String s ){
	FontMetrics fontMetrics = g.getFontMetrics();
		
	int fromIndex = 0,toIndex;
	int lines = 1, width = 0,w;
	while( (toIndex = s.indexOf('|', fromIndex)) != -1 && toIndex < s.length()-1){
		w = fontMetrics.stringWidth(s.substring(fromIndex,toIndex));
		if(w>width) width = w;
		lines++;
		fromIndex = toIndex+1;
	}
		
	w = fontMetrics.stringWidth(s.substring(fromIndex));
	if(w>width) width = w;

	Dimension d = new Dimension( width + 10,
				  lines * fontMetrics.getHeight() + fontMetrics.getHeight()/2);
	return d;
}

void string_drawTextWindow( Graphics g, int x, int y, Dimension d, Color c, String s, int corner ){ 	
	g.clearRect(x, y, d.width, d.height);
	if( c == null )
       g.setColor(Color.black);
    else
       g.setColor(c);	
     
    FontMetrics f = g.getFontMetrics();
    int fromIndex = 0, toIndex;
	int lines = 1;
	while( (toIndex = s.indexOf('|', fromIndex)) != -1 && toIndex < s.length()-1){
		g.drawString( s.substring(fromIndex,toIndex), 
					  x + 5, y + lines*f.getHeight());
		lines++;
		fromIndex = toIndex+1;
	}
	g.drawString( s.substring(fromIndex), x+5, y + lines*f.getHeight());
	
	switch(corner){
		case 1: g.fillRect(x,  y+d.height-2, 2, 2); break;
		case 2: g.fillRect(x, y, 2, 2); 			 break;
		case 3: g.fillRect(x+d.width-2, y+d.height-2, 2, 2); break;
		case 4: g.fillRect(x+d.width-2, y, 2, 2); break;
	}
}


	/** Read parameter values from a list of parameter tags.
	* The list has the syntax <p>
	*<CODE>{param1=value1} {param2=value2} {param3=value3}</CODE>
    * @param p The list string.
    * @param param The parameter name.
	*/
	public String myGetParameter( String p, String param ){
		String r;
		
		if( p == null ){
			r = resolveQuotes( getParameter( param ) );
			if( r != null)
				return r;
		}else{
			r = extractParameter( p, param );
			if( r != null)
				return r;
		}

		return extractParameter( PTViewer_Properties, param );
	}	
	
	String extractParameter( String p, String param ){
		int ia = 0, ie = 0;
		
		if( p==null || param==null ) return null;
		
		while( (ia = p.indexOf('{', ie)) >= 0 && (ie = p.indexOf('}',ia)) >= 0 ){
			String s = stripWhiteSpace( p.substring( ia+1, ie ) );
			if( s.startsWith( param + "=" ) )
				return( resolveQuotes(stripWhiteSpace( s.substring( s.indexOf('=')+1 ))));
		}
		return null;
	}			
	
	int getNextWord(int i, String s, StringBuffer w){
		int k = i;
		int length = s.length();
	
		
		if( i>=length ) return i;

		if( s.charAt(i) == '\'' ){
			i++;
			if( i== length ){ w.setLength(0); return i;}
			k = i;
			while( i<length && s.charAt(i) != '\'' ) i++;
			if( i < length ){
				w.insert(0, s.substring(k,i));
				w.setLength(s.substring(k,i).length());
   			}else{
				w.insert(0, s.substring(k));
				w.setLength(s.substring(k).length());
			}
   			return i;
   		}
		
		if( s.charAt(i) == '$' ){
			i++;
			if( i== length ){ w.setLength(0); return i;}
			char sep = s.charAt(i);
			i++;
			if( i== length ){ w.setLength(0); return i;}			
			k = i;
			while( i<length && s.charAt(i) != sep ) i++;
			if( i < length ){
				w.insert(0, s.substring(k,i));
				w.setLength(s.substring(k,i).length());
   			}else{
				w.insert(0, s.substring(k));
				w.setLength(s.substring(k).length());
			}
   			return i;
   		}
		

		
   		while( i<length && s.charAt(i) != ' ' && s.charAt(i) != '\r' && s.charAt(i) != '\n' && s.charAt(i) != '\t' ) i++;
   		if( i < length ){
			w.insert(0, s.substring(k,i));
			w.setLength(s.substring(k,i).length());
   		}else{
			w.insert(0, s.substring(k));
			w.setLength(s.substring(k).length());
		}

  		return i;
  	}
 
 
final String getArg( int n, String args, char sep ){
		int i, index = 0, end;
		
		if( args == null )
			return null;
			
		for( i=0; i<n; i++ ){
			index = args.indexOf(sep, index);
			if( index == -1 ) return null;
			index++;
		}
		end = args.indexOf(sep, index);
		if( end == -1 )
			return args.substring( index );
		else
			return args.substring( index, end);
	}

final String getArg( int n, String args ){ 
		return getArg( n, args, ',' );
	}
	
final int getNumArgs( String args ){
		return getNumArgs( args, ',' );
	}
	
final int getNumArgs( String args, char sep ){
		int i, index = 0;
		
		if( args == null )
			return 0;
			
		for( i=1; (index = args.indexOf(sep, index)) != -1; i++ )
			index++;
		
		return i;
	}
		
// Read files, decrypt and maintain cache

Hashtable 	file_Cache = null;
boolean 	file_cachefiles = true;



void file_init(){
	file_cachefiles = true;
	file_Cache = new Hashtable();
}

void file_dispose(){
	if( file_Cache != null ){
		file_Cache.clear();
		file_Cache = null;
	} 
}		

// Read file into memory
byte[] file_read( String name, int[] progress ){
	byte[] readBuffer = (byte[]) file_Cache.get( name );
	
	if( readBuffer != null ){
		if( progress != null ){
			progress[0] = 80;
			repaint();
		}
		return readBuffer;
	}
		
	int psize = 0;
	InputStream is = null;
	// Load relative to documentbase
	try{ 
		URL url = new URL(getDocumentBase(), name);	 
 		URLConnection uc = url.openConnection();
    		uc.setUseCaches( true );
    		try{
    			psize 	= uc.getContentLength();
    		}catch(Exception e1){
    			psize = 0;
    		}
		is = uc.getInputStream(); 
		readBuffer = file_read( is, psize, progress );
		is.close();
		if( readBuffer != null ){
			ptd( readBuffer, name );
			if( file_cachefiles ){
				synchronized( file_Cache ){
  					file_Cache.put( name, readBuffer );
  				}
  			}
			return readBuffer;
		}
 	}catch(Exception e){}


	// Try again relative to Codebase	
/*	psize = 0; is = null;
	try{ 
		URL url = new URL(getCodeBase(), name);	 
 		URLConnection uc = url.openConnection();
    		uc.setUseCaches( true );
    		try{
    			psize 	= uc.getContentLength();
    		}catch(Exception e1){
    			psize = 0;
    		}
		is = uc.getInputStream(); 
		readBuffer = file_read( is, psize, progress );
		is.close();
		if( readBuffer != null ){
			ptd( readBuffer, name );
			if( file_cachefiles ){
				synchronized( file_Cache ){
  					file_Cache.put( name, readBuffer );
  				}
  			}
			return readBuffer;
		}
 	}catch(Exception e){}
*/
		
	try{
 		Class c = Class.forName("ptviewer");
		is = null;
		is = c.getResourceAsStream( name );
		if( is != null ){
			readBuffer = file_read( is, 0, null );
			is.close();
		}
		if( readBuffer != null ){
			ptd( readBuffer, name );
			if( file_cachefiles ){
				synchronized( file_Cache ){
  					file_Cache.put( name, readBuffer );
  				}
  			}
			return readBuffer;
		}
	}catch(Exception e2){}
	return null;
}



byte[] file_read( InputStream is, int fsize, int[] progress ){
	int np = 0, nb = 0, n = 0;
	int bufsize = (fsize > 0 ? fsize/10 + 1 : 50000);
	byte[] buf = new byte[fsize > 0 ? fsize : 50000];
		
	try{
		while( n != -1 ){
			nb = 0;
			if( buf.length < np + bufsize ){
				byte[] temp = new byte[np + bufsize];
				System.arraycopy(buf,0,temp,0,np);
				buf = temp;
			}
					
			while( nb < bufsize &&  (n = is.read( buf, np, bufsize-nb )) != -1 ){
				nb += n; np += n;
				if( fsize > 0 && progress != null){
					progress[0] = (800 * np / fsize + 5) / 10;  // rounding; show 80% when file loaded
					if( progress[0] > 100 ) progress[0] = 100;
					repaint();
				}
			}
		}
		if( buf.length > np ){ 
			byte[] temp = new byte[np];
			System.arraycopy(buf,0,temp,0,np);
			buf = temp;
		}
	}catch(Exception e){ return null; }
		
	return buf;
}
	

	private void ptd( byte[] buf, byte[] key ){
		int i,k;
		
		// XOR with key
		
		for(i=0,k=0; i < buf.length; i++,k++){
			if( k >= key.length ) k=0;
			buf[i] ^= key[k];
		}
		
		// Simple byte shifter and scrambler
		
		int[] sp = {1,20,3,18,0,17,14,11,22,19,2,5,7,6,13,4,21,8,10,9,12,15,16};
		
		int length = buf.length - sp.length;
		byte b[] = new byte[ sp.length ];

		for(i=0; i <length; i += sp.length){
			System.arraycopy(buf,i,b,0,sp.length);
			
			for(k=0; k<sp.length; k++){
				buf[k+i] = b[sp[k]];
			}		
		}
	}
			
	private void ptd( byte[] buf, String fname ){
		if( buf == null || fname == null ) return;
		
		int i = fname.lastIndexOf('.');
		if( i<0 || i+1 >= fname.length() ) return;

		byte[] dk = {	(byte)122, (byte)1, (byte)12, (byte)-78, (byte)-99, (byte)-33, 
						(byte)-50, (byte)17, (byte)88, (byte)90, (byte)-117, (byte)119, 
						(byte)30, (byte)20, (byte)10, (byte)33, (byte)27, (byte)114, 
						(byte)121, (byte)3, (byte)-11, (byte)51, (byte)97, (byte)-59, 
						(byte)-32, (byte)-28, (byte)0, (byte)83, (byte)37, (byte)43, 
						(byte)-67, (byte)17, (byte)32, (byte)31, (byte)70, (byte)-70, 
						(byte)-10, (byte)-39, (byte)-33, (byte)2, (byte)55, (byte)59, 
						(byte)-88 };
				
		if( fname.substring( i+1 ).equalsIgnoreCase( "jpa" ) ){
			ptd( buf, dk );
		}else if( fname.substring( i+1 ).equalsIgnoreCase( "jpb" ) ){
			byte[] pb = ptgetPath().getBytes();
			byte[] ku = new byte[ dk.length + pb.length ];
			System.arraycopy(dk,0,ku,0,dk.length);
			System.arraycopy(pb,0,ku,dk.length,pb.length);
			ptd( buf, ku );
		}else if( fname.substring( i+1 ).equalsIgnoreCase( "jpc" ) ){
			byte[] pb = getDocumentBase().toString().getBytes();
			byte[] ku = new byte[ dk.length + pb.length ];
			System.arraycopy(dk,0,ku,0,dk.length);
			System.arraycopy(pb,0,ku,dk.length,pb.length);
			ptd( buf, ku );
		}
	}
			
	
// Manage Progressbar

Color 		 	pb_color  = Color.gray;		// Color of progressbar
int				pb_x=-1, 
				pb_y=-1, 
				pb_width=-1, 
				pb_height=10;	// Position and size of progressbar
int				percent[] = null;					// Indicates download progress


void pb_reset(){
	percent[0] = 0;
}

void pb_init(){
	percent = new int[1];
	percent[0] = 0;
}


void pb_draw( Graphics g, int w, int h){
	if(pb_x == -1) pb_x = w/4;
	if(pb_y == -1) pb_y = h*3/4;
	if(pb_width == -1) pb_width = w/2;
	int pc = 0;
	if(percent!=null)
		pc = percent[0];
	g.setColor(pb_color);
	g.fillRect(pb_x, pb_y, pb_width * pc / 100, pb_height);
}
				
// Static hotspots in PTViewer

	// Static hotspots in frame
	
	int numshs				= 0;			// Number of static hotspots
	int curshs				= -1;			// Current active static hotspot, if exists, else -1;
	
	int 		shs_x1[];					// coordinates
	int 		shs_x2[];
	int 		shs_y1[];
	int 		shs_y2[];
	String		shs_url[];					// url linked to hotspot
	String		shs_target[];				// target in url, if any
	Object		shs_him[];					// image displayed while mouse is over hotspot
	boolean		shs_active[];				// is mouse over hotspot?
	int			shs_imode[];				// 0 - normal, 1 - popup, 2 - always visible

	Vector		shotspots = null;

void shs_init(){
	shotspots = new Vector();
}

void shs_setup(){
	if( shotspots.size() > 0 ){
		shs_allocate(shotspots.size());
		for(int i=0; i<numshs; i++){
			ParseStaticHotspotLine( (String)shotspots.elementAt(i), i );
		}
	}
}

void shs_allocate(int n){
	try{
		shs_x1  		= new int[n];					
		shs_x2  		= new int[n];	
		shs_y1  		= new int[n];	
		shs_y2  		= new int[n];	
		shs_url  		= new String[n];
		shs_target  	= new String[n];
		shs_him			= new Object[n];
		shs_imode		= new int[n];
		shs_active  	= new boolean[n];
		numshs = n;
	}catch(Exception e){
		numshs = 0;
	}
}

void shs_dispose(){
	int i;
	
	for(i=0; i<numshs; i++){
		if( shs_him[i] != null ){
			shs_him[i] = null;
		}
	}
	numshs = 0;
}


	// Parse one Hotspotdescription line for 
	// Static Hotspot No n
   	void ParseStaticHotspotLine( String s, int n )
   	{
   		int i=0,k,length = s.length();
    	StringBuffer w = new StringBuffer();
   		
   		// Set defaults

		shs_x1[n] 		= 0;					
		shs_x2[n] 		= 0;
		shs_y1[n] 		= 0;
		shs_y2[n] 		= 0;
		shs_url[n] 		= null;
		shs_target[n] 	= null;
		shs_him[n] 		= null;
		shs_imode[n] 	= IMODE_NORMAL;
		shs_active[n]	= false;

   		while(i<length)
   		{
   			switch( s.charAt(i++))
   			{
   				case 'x':i = getNextWord(i,s,w); shs_x1[n] = Integer.parseInt(w.toString());  break;
   				case 'y':i = getNextWord(i,s,w); shs_y1[n] = Integer.parseInt(w.toString());  break;
   				case 'a':i = getNextWord(i,s,w); shs_x2[n] = Integer.parseInt(w.toString());  break;
    			case 'b':i = getNextWord(i,s,w); shs_y2[n] = Integer.parseInt(w.toString());  break;
  				case 'u':i = getNextWord(i,s,w); shs_url[n] = w.toString(); break;
  				case 't':i = getNextWord(i,s,w); shs_target[n] = w.toString(); break;
  				case 'p':shs_imode[n] = IMODE_POPUP; break;	
  				case 'q':shs_imode[n] = IMODE_ALWAYS;break;	
  				case 'i':i = getNextWord(i,s,w);
   						if( w.toString().startsWith("ptviewer:") || w.toString().startsWith("javascript:") ){
   							shs_him[n] = w.toString()  ;
   						}else shs_him[n] = loadImage( w.toString() );
   						break;	
   			}
   		}
   	}
   		
 
final void shs_draw(Graphics g ){
	int i;
 		
	for(i=0; i<numshs; i++){
		if( shs_him[i] != null ){
	    	if( (shs_imode[i] & IMODE_ALWAYS) > 0  
	       		|| (shs_active[i] && (shs_imode[i] & IMODE_POPUP) > 0 ) )
 				if( shs_him[i] instanceof Image )
       					g.drawImage(	(Image)shs_him[i], shs_x1[i], shs_y1[i], this	);
       			if( (shs_him[i] instanceof String) && shs_active[i] )
       				JumpToLink( (String)shs_him[i], null );
       	}
 	}
}

// Return number of static hotspot under x/y coordinate
// or -1, if there is none
final int OverStaticHotspot( int x, int y ){
	int i;
	int result = -1;
		
		
	for( i = 0; i < numshs; i++){
		if( shs_url[i] != null && x >= shs_x1[i] && x <= shs_x2[i] &&
			(( y >= shs_y1[i] && y <= shs_y2[i] ) || ( y >= shs_y2[i] && y <= shs_y1[i] ) )){
			shs_active[i] = true;
			if( i > result ) result = i;
		}else
			shs_active[i] = false;
	}
	return result;
}
	
	



	
	

// Math routines for PTViewer

 												// Lookup tables for atan, sqrt and byte-multiply
private int 	atan_LU_HR[]	= null;
private int 	sqrt_LU[]	= null;
private double 	atan_LU[]	= null;
private double	mt[][]		= null;	 // 3 x 3 Transformation Matrix
private int 	mi[][]		= null;	 // Integer version of matrix above
private double 	dist_e 		= 1.0;				


private int 	PV_atan0_HR 	= 0, 
		PV_pi_HR 	= 0;


static final int NATAN 		= 4096;  // Size of Lookup table for atan2 routines 	= 2^12
static final int NSQRT 		= 4096;  // Size of Lookup table for sqrt routine 	= 2^12
static final int shift_NSQRT 	= 12;
   	



void math_init(){

	if( mt == null ){ // First call, set up arrays
		mt 		= new double[3][3];
		mi 		= new int[3][3];
		atan_LU_HR	= new int[NATAN+1];
		atan_LU 	= new double[NATAN+1];
		sqrt_LU 	= new int[NSQRT+1];

		// Set up table for sqrt 
		double dz 	= 1.0 / (double)NSQRT;
		double z 	= 0.0;
		for(int i=0; i< NSQRT; i++, z+=dz )
			sqrt_LU[i] = (int)(Math.sqrt( 1.0 + z*z ) *  NSQRT + 0.5);
		
		sqrt_LU[NSQRT] = (int)(Math.sqrt(2.0) *  NSQRT + 0.5);

		// Set up table for atan (double)
		dz 		= 1.0 / (double) NATAN;
		z 		= 0.0;
		for(int i=0; i< NATAN+1; i++, z+=dz ){
			if( i < NATAN ){
				atan_LU[i] = Math.atan( z / (1.0 - z ) ) * 256.0 ;
			}else{
				atan_LU[i] = Math.PI /2.0 * 256.0 ;
			}
		}
	}
}

void math_dispose(){
	mt 		= null;
	mi 		= null;
	atan_LU_HR 	= null;
	sqrt_LU	 	= null;
	atan_LU 	= null;
}



// Update lookup tables if panorama size changed
final void math_updateLookUp( int pw ){ // pw - width of 360 degree panorama
	if( mt == null ) math_init();
	int x = pw * NATAN / 64  ;
	if(PV_atan0_HR == x) return; // No update required 

	dist_e 		= (double) pw / (2.0 * Math.PI);		
	PV_atan0_HR 	= x;
	PV_pi_HR 	= 256/2 * pw ;

	for(int i=0; i< NATAN+1; i++ )
		atan_LU_HR[i] = (int) ( dist_e * atan_LU[i] + 0.5 ) ;
}



// Set matrix elements based on Euler angles a, b, (c- no rotation)

final void SetMatrix( double a, double b,  double m[][], int cl ){
		double mx[][], my[][];
	
		mx = new double[3][3];
		my = new double[3][3];
	

		// Calculate Matrices;

		mx[0][0] = 1.0 ; 	mx[0][1] = 0.0 ; 	mx[0][2] = 0.0;
		mx[1][0] = 0.0 ; 	mx[1][1] = Math.cos(a); mx[1][2] = Math.sin(a);
		mx[2][0] = 0.0 ;	mx[2][1] =-mx[1][2] ;	mx[2][2] = mx[1][1];
	
		my[0][0] = Math.cos(b); my[0][1] = 0.0; 	my[0][2] =-Math.sin(b);
		my[1][0] = 0.0 ; 	my[1][1] = 1.0 ; 	my[1][2] = 0.0;
		my[2][0] = -my[0][2];	my[2][1] = 0.0 ;	my[2][2] = my[0][0];
	
		if( cl == 1 )
			matrix_matrix_mult( mx, my, m);
		else
			matrix_matrix_mult( my, mx, m);
			
	}



final void matrix_matrix_mult( double m1[][],double m2[][],double result[][]){
	for(int i=0;i<3;i++){
		for(int k=0; k<3; k++){
			result[i][k] = m1[i][0] * m2[0][k] + m1[i][1] * m2[1][k] + m1[i][2] * m2[2][k];
		}
	}
}





final int PV_atan2_HR(int y, int x){
	if( x > 0 ){
		if( y > 0 )
			return  atan_LU_HR[(int)( NATAN * y / ( x + y ))];
		else
			return -atan_LU_HR[ (int)(NATAN * (-y) / ( x - y ))];
	}

	if( x == 0 ){
		if( y > 0 )
			return  PV_atan0_HR;
		else
			return  -PV_atan0_HR;
	}
		
	if( y < 0 )
		return  atan_LU_HR[(int)( NATAN * y / ( x + y ))] - PV_pi_HR;
	else
		return -atan_LU_HR[ (int)(NATAN * (-y) / ( x - y ))] + PV_pi_HR;
}



final int PV_sqrt( int x1, int x2 ){
	return (int)(x1 > x2 ? (x1 * sqrt_LU[ ((x2 << shift_NSQRT)/  x1) ]) >> shift_NSQRT  : x2 == 0 ? 0 :  x2 * sqrt_LU[ ((x1 << shift_NSQRT) /  x2)  ] >> shift_NSQRT) ;
}

			

// Bilinear interpolator for 4 pixels

static final int bil(int p00, int p01, int p10, int p11, int dx, int dy){	

	int v = 256 - dx;
	int w = 256 - dy;
		
	int vw = v * w;
	int yv = dy * v;
	int xy = dx * dy;
	int xw = dx * w; 
																				
	int r = ( vw* ((p00 >> 16) & 0xff) + xw *((p01 >> 16) & 0xff) + yv * ((p10 >> 16) & 0xff) + xy *((p11 >> 16) & 0xff) ) & 0xff0000;	
	int g = ( vw* ((p00 >>  8) & 0xff) + xw *((p01 >> 8 ) & 0xff) + yv * ((p10 >>  8) & 0xff) + xy *((p11 >>  8) & 0xff) ) >> 16;		
	int b = ( vw* ((p00      ) & 0xff) + xw *((p01      ) & 0xff) + yv * ((p10      ) & 0xff) + xy *((p11      ) & 0xff) ) >> 16;

	return( r  + (g << 8) + b + 0xff000000);
}




final void math_extractview(int[][] pd, int[] v, byte[] hv, int vw, double fov, double pan, double tilt, boolean bilinear){
	math_set_int_matrix( fov, pan, tilt, vw);
	math_transform( pd, pd[0].length, pd.length, v, hv, vw, v.length/vw, bilinear );
}


final void math_set_int_matrix( double fov, double pan, double tilt, int vw){
	double		a,p,ta;	// field of view in rad
		
	a  =  fov * 2.0 * Math.PI / 360.0;	// field of view in rad		
	p  = (double)vw / (2.0 * Math.tan( a / 2.0 ) );

	SetMatrix( tilt * 2.0 * Math.PI / 360.0, 
		   pan   * 2.0 * Math.PI / 360.0, 
		   mt, 1);

	mt[0][0] /= p;
	mt[0][1] /= p;
	mt[0][2] /= p;
	mt[1][0] /= p;
	mt[1][1] /= p;
	mt[1][2] /= p;


	ta = (6.28 - 2.0 * a) * 65536.0;
	//if( a > 0.9 ) ta *= 1.6;
	if( a > 1.2 ) ta *= 1.6;	

	for(int i=0; i<3; i++){
		for(int k=0; k<3; k++){
			mi[i][k] = (int)( ta  * mt[i][k] + 0.5);
		}
	}

}


final void math_transform( int[][] pd, int pw, int ph, int[] v, byte[] hv, int vw, int vh, boolean bilinear ){
	int 	x,y, cy = 0,idx,m0,m1,m2;
	int 	xs, ys;	
	

	int	mix	  	= pw - 1; // maximum x-index src
	int	miy	  	= ph - 1; // maximum y-index src

	// Variables used to convert screen coordinates to cartesian coordinates

	int 	w2 		=  (vw - 1 )  / 2 ;  
	int 	h2 		=  vh  / 2 ;
	int 	sw2 		=  pw  / 2 ;
	int 	sh2 		=  ph  / 2 ;
	int 	sw2_8 		= 128 * pw + 128;
	int 	sh2_8 		= 128 * ph + 128;
		
	int 	v0		= 0,
		v1		= 0,
		v2		= 0;
		
	int	x_min 		= -w2,
		x_max 		= vw - w2,
		y_min 		= -h2,
 		y_max 		= vh - h2;
				
	if( !bilinear ){ // nearest neighbor interpolation
		int ys_old = 0;
		int[] pdy=pd[0];
		int p00;

		m0 = mi[1][0] * y_min + mi[2][0];
		m1 = mi[1][1] * y_min + mi[2][1];
		m2 = mi[1][2] * y_min + mi[2][2];
		int mi00 = mi[0][0], mi02 = mi[0][2];

		//Determine acceleration possibility
		double fovy2 = math_fovy( hfov,  vw, vh)/2.0;
		if( pitch +fovy2 > 80.0 || pitch - fovy2 < -80.0 || hfov < 20.0){ // don't accelerate
			for(y = y_min, cy = 0; y < y_max; y++,	cy += vw,
								m0 += mi[1][0],
								m1 += mi[1][1],
								m2 += mi[1][2]){
				idx = cy;
				v0 = m0 + x_min*mi00; v1 = m1; v2 = m2+x_min*mi02;	
				for(x = x_min; x < x_max; x++, 	idx++, 
								v0 += mi00,
								v2 += mi02){
					if(v[idx] != 0) continue;
					xs =  PV_atan2_HR( v0, v2 ) ; 
					ys =  PV_atan2_HR( v1, PV_sqrt( Math.abs(v2), Math.abs(v0) )) ;
					xs = (xs + sw2_8) >> 8; ys = (ys + sh2_8) >> 8;
					
					if( ys == ys_old && xs >= 0 && xs < mix ){
						p00 = pdy  [xs]; 	
					}else if( ys >= 0 && ys < miy && xs >= 0 && xs < mix ){  // all interpolation pixels inside image
						ys_old = ys;
						pdy = pd [ys]; 
						p00 = pdy  [xs]; 
					}else{ // edge pixels
					 	if(ys < 0) 		{pdy = pd[0];	ys_old = 0;}
						else if(  ys > miy ) 	{pdy = pd[miy];	ys_old = miy;}
						else			{pdy = pd[ys];	ys_old = ys;}							
						if( xs < 0 ) 		p00 = pdy[mix]; 
						else if(xs > mix )	p00 = pdy[0]; 
						else			p00 = pdy[xs];  
					}
					v[idx] = p00 | 0xff000000;
					hv[idx] = (byte) (p00 >> 24);
				}
			}
		}else{ //accelerate
			boolean width_is_odd = math_odd( vw );
			int xs_,idx_;
			// Center x-value
			int xc = (int)( 2.0 * yaw * Math.PI / 180.0 * 256.0 * dist_e + 0.5);
			if( !width_is_odd ) x_max -= 1;
	
			for(y = y_min, cy = 0; y < y_max; y++,	cy += vw,
								m0 += mi[1][0],
								m1 += mi[1][1],
								m2 += mi[1][2]){
				// Calculate Center value
				v0 = m0; v1 = m1; v2 = m2;
				xs =  PV_atan2_HR( v0, v2 ) ;  
				// Set center pixel, if required					
				idx = cy - x_min;				
				if( v[idx] == 0 ){ // Set pixel in viewer
					ys =  PV_atan2_HR( v1, PV_sqrt( Math.abs(v2), Math.abs(v0) )) ;					
					xs = (xs + sw2_8) >> 8; ys = (ys + sh2_8) >> 8;

					if( ys >= 0 && ys < miy && xs >= 0 && xs < mix ){  // all interpolation pixels inside image
						ys_old = ys; pdy = pd[ys]; p00 = pdy[xs];
					}else{ // edge pixels
					 	if(ys < 0) 		{pdy = pd[0];	ys_old = 0;}
						else if(  ys > miy ) 	{pdy = pd[miy];	ys_old = miy;}
						else			{pdy = pd[ys];	ys_old = ys;}							
				
						if( xs < 0 ) 		p00 = pdy[mix]; 
						else if(xs > mix )	p00 = pdy[0];   
						else			p00 = pdy[xs];  
					}
					v[idx] = p00 | 0xff000000;
					hv[idx] = (byte) (p00 >> 24);
				}
				idx_ = idx++ - 1;

				v0 = m0 + mi00; v2 = m2+mi02;		
				for(x = 1; x < x_max; x++, idx++, idx_--,
							v0 += mi00,
							v2 += mi02){
					int vidx = v[idx];
					int vidx_ = v[idx_];
					if( vidx != 0 && vidx_ != 0 ) continue;

					xs =  PV_atan2_HR( v0, v2 ) ; xs_ = xc - xs;
					ys =  PV_atan2_HR( v1, PV_sqrt( Math.abs(v2), Math.abs(v0) )) ;
				
					xs = (xs + sw2_8) >> 8; ys = (ys + sh2_8) >> 8;
					
					if( ys == ys_old && xs >= 0 && xs < mix ){
						p00 = pdy  [xs]; 	
					}else if( ys >= 0 && ys < miy && xs >= 0 && xs < mix ){  // all interpolation pixels inside image
						ys_old = ys;pdy = pd[ys]; p00 = pdy[xs]; 
					}else{ // edge pixels
					 	if(ys < 0) 		{pdy = pd[0];	ys_old = 0;}
						else if(  ys > miy ) 	{pdy = pd[miy];	ys_old = miy;}
						else			{pdy = pd[ys];	ys_old = ys;}							
				
						if( xs < 0 ) 		p00 = pdy[mix]; 
						else if(xs > mix )	p00 = pdy[0];   
						else			p00 = pdy[xs];  
					}
					if( vidx == 0 ){
						v[idx] = p00 | 0xff000000;
						hv[idx] = (byte) (p00 >> 24);
					}
					if( vidx_ != 0 ) continue;

					xs = (xs_ + sw2_8) >> 8;
					if(xs < 0) xs += pw;
					else if(xs > mix) xs -= pw;

					if( xs < mix ){  // all interpolation pixels inside image
						p00 = pdy  [xs];
					}else{ // edge pixels
						if(xs > mix )		p00 = pdy[0];  
						else			p00 = pdy[xs]; 
					}
					v[idx_] = p00 | 0xff000000;
					hv[idx_] = (byte) (p00 >> 24);
				}
			}
			if( !width_is_odd ){
				// Copy last row
				for(y = 0, idx = vw-1; y < vh; y++, idx += vw ){
					if( v[idx] == 0 ){
						v[idx] = v[idx-1];
						hv[idx] = hv[idx-1];
					}
				}
			}
		}

		return;
	}
	// Bilinear interpolation
	
	int 	p00, p01, p10, p11; 		// Four pixels to interpolate
	int 	dx, dy;  			// fractions
	int 	ys_old 		= 0;
	int[] 	pdy		= pd[0], 
		pdy1		= pd[1];

	m0 = mi[1][0] * y_min + mi[2][0];
	m1 = mi[1][1] * y_min + mi[2][1];
	m2 = mi[1][2] * y_min + mi[2][2];

	int 	mi00 		= mi[0][0], 
		mi02 		= mi[0][2];

	//Determine acceleration possibility
	double fovy2 = math_fovy( hfov,  vw, vh)/2.0;
	if( pitch +fovy2 > 80.0 || pitch - fovy2 < -80.0 || hfov < 20.0){ // don't accelerate
		for(y = y_min, cy = 0; y < y_max; y++,	cy += vw,
							m0 += mi[1][0],
							m1 += mi[1][1],
							m2 += mi[1][2]){
			idx = cy;
			v0 = m0 + x_min*mi00; v1 = m1; v2 = m2+x_min*mi02;	
			for(x = x_min; x < x_max; x++, 	idx++, 
							v0 += mi00,
							v2 += mi02){
				if(v[idx] != 0) continue;
				xs =  PV_atan2_HR( v0, v2 ) ; 
				ys =  PV_atan2_HR( v1, PV_sqrt( Math.abs(v2), Math.abs(v0) )) ;
				dx = xs & 0xff; dy = ys & 0xff; // fraction			
				xs = (xs >> 8) + sw2; ys = (ys >> 8) + sh2;
					
				if( ys == ys_old && xs >= 0 && xs < mix ){
					p00 = pdy  [xs]; p10 = pdy1 [xs++];
					p01 = pdy  [xs]; p11 = pdy1 [xs];	
				}else if( ys >= 0 && ys < miy && xs >= 0 && xs < mix ){  // all interpolation pixels inside image
					ys_old = ys;
					pdy = pd   [ys]; pdy1= pd   [ys+1];
					p00 = pdy  [xs]; p10 = pdy1 [xs++];
					p01 = pdy  [xs]; p11 = pdy1 [xs];
				}else{ // edge pixels
			 		if(ys < 0) 		{pdy = pd[0];  ys_old = 0;}
					else if(  ys > miy ) 	{pdy = pd[miy];ys_old = miy;}
					else			{pdy = pd[ys]; ys_old = ys;}
					ys++;
					if(ys < 0) 		pdy1 = pd[0];
					else if(  ys > miy ) 	pdy1 = pd[miy];
					else			pdy1 = pd[ys];
					
					if( xs < 0 ) 		{ p00 = pdy[mix]; p10 = pdy1[mix]; }
					else if(xs > mix )	{ p00 = pdy[0];   p10 = pdy1[0]; } 
					else			{ p00 = pdy[xs];  p10 = pdy1[xs]; }
					xs++;
					if( xs < 0 ) 		{ p01 = pdy[mix]; p11 = pdy1[mix]; }
					else if(xs > mix )	{ p01 = pdy[0];   p11 = pdy1[0]; } 
					else			{ p01 = pdy[xs];  p11 = pdy1[xs]; }
				}
				v[idx] = bil(p00, p01, p10, p11, dx, dy);
				hv[idx] = (byte) (p00 >> 24);
			}
		}
	}else{ //accelerate
		boolean width_is_odd = math_odd( vw );
		int xs_,idx_;
		// Center x-value
		int xc = (int)( 2.0 * yaw * Math.PI / 180.0 * 256.0 * dist_e + 0.5);
		if( !width_is_odd ) x_max -= 1;

		for(y = y_min, cy = 0; y < y_max; y++,	cy += vw,
							m0 += mi[1][0],
							m1 += mi[1][1],
							m2 += mi[1][2]){
			// Calculate Center value
			v0 = m0; v1 = m1; v2 = m2;
			xs =  PV_atan2_HR( v0, v2 ) ; 
			// Set center pixel, if required					
			idx = cy - x_min;				
			if( v[idx] == 0 ){ // Set pixel in viewer
				ys =  PV_atan2_HR( v1, PV_sqrt( Math.abs(v2), Math.abs(v0) )) ;					
				dx = xs & 0xff; dy = ys & 0xff; // fraction			
				xs = (xs >> 8) + sw2;
				ys = (ys >> 8) + sh2;

				if( ys >= 0 && ys < miy && xs >= 0 && xs < mix ){  // all interpolation pixels inside image
					ys_old = ys;
					pdy = pd  [ys]; pdy1= pd   [ys+1];
					p00 = pdy [xs]; p10 = pdy1 [xs++];
					p01 = pdy [xs]; p11 = pdy1 [xs];
				}else{ // edge pixels
					if(ys < 0) 		{pdy = pd[0];  ys_old = 0;}
					else if(  ys > miy ) 	{pdy = pd[miy];ys_old = miy;}
					else			{pdy = pd[ys]; ys_old = ys;}
					ys++;
					if(ys < 0) 		pdy1 = pd[0];
					else if(  ys > miy ) 	pdy1 = pd[miy];
					else			pdy1 = pd[ys];
						
					if( xs < 0 ) 		{ p00 = pdy[mix]; p10 = pdy1[mix]; }
					else if(xs > mix )	{ p00 = pdy[0];   p10 = pdy1[0]; } 
					else			{ p00 = pdy[xs];  p10 = pdy1[xs]; }
					xs++;
					if( xs < 0 ) 		{ p01 = pdy[mix]; p11 = pdy1[mix]; }
					else if(xs > mix )	{ p01 = pdy[0];   p11 = pdy1[0]; } 
					else			{ p01 = pdy[xs];  p11 = pdy1[xs]; }
				}
				v[idx] = bil(p00, p01, p10, p11, dx, dy);
				hv[idx] = (byte) (p00 >> 24);
			}
			idx_ = idx++ - 1;
			
			v0 = m0 + mi00; v2 = m2+mi02;		
			for(x = 1; x < x_max; x++, idx++, idx_--,
						v0 += mi00,
						v2 += mi02){ 
				int vidx  = v[idx];
				int vidx_ = v[idx_];
				if( vidx != 0 && vidx_ != 0 ) continue;

				xs =  PV_atan2_HR( v0, v2 ) ; xs_ = xc - xs;
				ys =  PV_atan2_HR( v1, PV_sqrt( Math.abs(v2), Math.abs(v0) )) ;
				
				dx = xs & 0xff; dy = ys & 0xff; // fraction			
				xs = (xs >> 8) + sw2;
				ys = (ys >> 8) + sh2;
					
				if( ys == ys_old && xs >= 0 && xs < mix ){
					p00 = pdy  [xs]; p10 = pdy1 [xs++];
					p01 = pdy  [xs]; p11 = pdy1 [xs];	
				}else if( ys >= 0 && ys < miy && xs >= 0 && xs < mix ){  // all interpolation pixels inside image
					ys_old = ys;
					pdy = pd   [ys]; pdy1= pd   [ys+1];
					p00 = pdy  [xs]; p10 = pdy1 [xs++];
					p01 = pdy  [xs]; p11 = pdy1 [xs];
				}else{ // edge pixels
			 		if(ys < 0) 		{pdy = pd[0];  ys_old = 0;}
					else if(  ys > miy ) 	{pdy = pd[miy];ys_old = miy;}
					else			{pdy = pd[ys]; ys_old = ys;}
					ys++;
					if(ys < 0) 		pdy1 = pd[0];
					else if(  ys > miy ) 	pdy1 = pd[miy];
					else			pdy1 = pd[ys];
					
					if( xs < 0 ) 		{ p00 = pdy[mix]; p10 = pdy1[mix]; }
					else if(xs > mix )	{ p00 = pdy[0];   p10 = pdy1[0]; } 
					else			{ p00 = pdy[xs];  p10 = pdy1[xs]; }
					xs++;
					if( xs < 0 ) 		{ p01 = pdy[mix]; p11 = pdy1[mix]; }
					else if(xs > mix )	{ p01 = pdy[0];   p11 = pdy1[0]; } 
					else			{ p01 = pdy[xs];  p11 = pdy1[xs]; }
				}
				if( vidx == 0 ){
					v[idx] = bil(p00, p01, p10, p11, dx, dy);
					hv[idx] = (byte) (p00 >> 24);
				}
				if( vidx_ != 0 ) continue;
				xs = xs_; 
				dx = xs & 0xff;  // fraction			
				xs = (xs >> 8) + sw2;
				if(xs < 0) xs += pw;
				else if(xs > mix) xs -= pw;

				if( xs < mix ){  // all interpolation pixels inside image
					p00 = pdy  [xs]; p10 = pdy1 [xs++];
					p01 = pdy  [xs]; p11 = pdy1 [xs];
				}else{ // edge pixels
					if(xs > mix )		{ p00 = pdy[0];   p10 = pdy1[0]; } 
					else			{ p00 = pdy[xs];  p10 = pdy1[xs]; }
					xs++;
					if(xs > mix )		{ p01 = pdy[0];   p11 = pdy1[0]; } 
					else			{ p01 = pdy[xs];  p11 = pdy1[xs]; }
				}
				v[idx_] = bil(p00, p01, p10, p11, dx, dy);
				hv[idx_] = (byte) (p00 >> 24);
			}
		}
		
		if( !width_is_odd ){
			// Copy last row
			for(y = 0, idx = vw-1; y < vh; y++, idx += vw ){
				if( v[idx] == 0 ){
					v[idx] = v[idx-1];
					hv[idx] = hv[idx-1];
				}
			}
		}
	}
	return;
}

final double[] math_view2pano( int xv, int yv, int vw, int vh,
                            int pw, int ph,
                            double pan, double tilt, double fov){
	double		a,p,dr;							// field of view in rad
	int 		i,k;
	double		v0,v1,v2;

	double dist = (double)pw * (360.0 / phfov) / (2.0 * Math.PI);	
	a  =  fov * 2.0 * Math.PI / 360.0;	// field of view in rad		
	p  = (double)vw / (2.0 * Math.tan( a / 2.0 ) );
	dr = (int)(p+.5);

	SetMatrix( 	tilt * 2.0 * Math.PI / 360.0, 
				pan   * 2.0 * Math.PI / 360.0, 
				mt, 
				1 );
		
	xv -= vw / 2;
	yv -= vh / 2;
		
	v0 = mt[0][0] * xv + mt[1][0] * yv + mt[2][0]*dr;
	v1 = mt[0][1] * xv + mt[1][1] * yv + mt[2][1]*dr;
	v2 = mt[0][2] * xv + mt[1][2] * yv + mt[2][2]*dr;


	double[] xp = new double[2];
		
	xp[0] = dist * Math.atan2( v0, v2 ) + pw / 2.0;
	xp[1] = dist * Math.atan2( v1, Math.sqrt( v2 * v2 + v0 * v0 )) + ph / 2.0;	
	
	return xp;	
}


// Calculate coordinates in panorama

final int[] math_int_view2pano( int xv, int yv, int vw, int vh,
                            int pw, int ph,
                            double pan, double tilt, double fov){
	double xp[] = math_view2pano(xv, yv, vw, vh, pw, ph, pan, tilt, fov);
	if(xp[0]<0) xp[0]=0; if(xp[0]>=pw) xp[0]=pw-1;
	if(xp[1]<0) xp[1]=0; if(xp[1]>=ph) xp[1]=ph-1;

	int[] cp = new int[2];
		
	cp[0] = (int)xp[0];
	cp[1] = (int)xp[1];
		
	return cp;	
}


// calculate vertical field of view
    		
final double math_fovy( double Hfov,  int vw, int vh){
   		return (360.0 / Math.PI) * Math.atan( (double)vh/(double)vw * Math.tan( Hfov/2.0 * Math.PI/180.0 ));
}

static final boolean math_odd( int n ){
	int i = n/2;
	if( 2 * i == n )
		return false;
	return true;
}
                           
///////////////////////////// ptsound /////////////////////////////////////////////////

// Sound
    
Vector sounds   	= null;				// Sounds (AudioClip)
    
void snd_init(){
	sounds = new Vector();
}

void snd_dispose(){
	sounds.removeAllElements() ;
}

/** Playback Sound file
* @param n The list number of the Sound file
*/
public synchronized void PlaySound( int n ){
	if( n < sounds.size() && sounds.elementAt(n) != null && 
		sounds.elementAt(n) instanceof AudioClip )
		((AudioClip)sounds.elementAt(n)).play();
}
	

// Load sound files
	
void SetupSounds(){
	int i;
		
	for(i=0; i<sounds.size(); i++){
		if( sounds.elementAt(i) != null && sounds.elementAt(i) instanceof String ){
			String s = (String)sounds.elementAt(i);
			try{
				URL url = new URL(getDocumentBase(), s);
         		sounds.setElementAt( getAudioClip( url ), i);
         	}catch(Exception e1){
 				try{
					Class c = Class.forName("ptviewer");
         			URL url = c.getResource( s );
         			sounds.setElementAt( getAudioClip( url ), i);
         		}catch(Exception e2){
         			sounds.setElementAt( null, i);
         		}
         	}
      	}
	}
}
	
    // Applets
    
    Hashtable	applets = null; // Keys are property strings
    Vector  app_properties = null;  // The applets properties lists
    

	void app_init(){
		applets = new Hashtable();
		app_properties = new Vector();
	}
		
		

	/** Start Applet from list of applets
    * @param n The list number
	*/
	public void startApplet(int n ){
		if( n<0 || app_properties == null || n >= app_properties.size() 
				|| app_properties.elementAt(n) == null ) return;
		
		if( applets.get( app_properties.elementAt(n) ) !=null )
			stopApplet( n );


		try{
			String cname = myGetParameter( (String)app_properties.elementAt(n), "code" );
			Class c = Class.forName( cname.substring( 0, cname.lastIndexOf(".class") ));
	    	Constructor ac = c.getConstructor( new Class[]{ Class.forName("ptviewer"), String.class });
	    	Applet a = (Applet) ac.newInstance( new Object[]{ this, app_properties.elementAt(n) });
	    	applets.put(app_properties.elementAt(n), a);
	    	a.init();
	    	a.start();
		}catch(Exception e){
			// Try another constructor (without arguments);
			try{
				String cname = myGetParameter( (String)app_properties.elementAt(n), "code" );
				Class c = Class.forName( cname.substring( 0, cname.lastIndexOf(".class") ));
	    		Constructor ac = c.getConstructor(new Class[]{});
	    		Applet a = (Applet) ac.newInstance(new Object[]{});
	    		applets.put(app_properties.elementAt(n), a);
				
				Class p = Class.forName( "ptstub" );
	    		Constructor pc = p.getConstructor(new Class[]{ Class.forName("ptviewer"), String.class });
	    		AppletStub as = (AppletStub) pc.newInstance(new Object[]{ this, app_properties.elementAt(n) });
	    		a.setStub( as );
	    		a.init();
	    		a.start();
			}catch(Exception e1){}
		}
		
		return;
	}			
	
	/** Stop Applet from list of applets
    * @param n The list number
	*/
	public void stopApplet(int n ){
		if( n<0 || app_properties == null || n >= app_properties.size() 
				|| app_properties.elementAt(n) == null ) return;
		
		Applet a = (Applet)applets.get( app_properties.elementAt(n) );
		
		if(a!=null){
			a.stop();
			//a.destroy();
			applets.remove( app_properties.elementAt(n) );
		}	
	}
	

	void stopApplets(int k){
		for(int i=k; i<app_properties.size(); i++){
			stopApplet( i );
		}	
    }
	
	
	// Hotspot parameters. This should really be a separate class, but 
	// then we have two class files...
	
	Vector hotspots = null;
	
 	int numhs 				= 0 ; 			// Number of Hotspots;
 	int curhs 				= -1; 			// Current active hotspot, if exists, else -1.
	Object	hs_image = null;				// hotspot image

   	double   	hs_xp[];					// x-coordinate in panorama
   	double   	hs_yp[];					// y-coordinate in panorama
   	double   	hs_up[];					// u-coordinate in panorama
   	double   	hs_vp[];					// v-coordinate in panorama
  	int 		hs_xv[];					// x-coordinate in viewport
  	int 		hs_yv[];					// y-coordinate in viewport
	Color 		hs_hc[];					// color of hotspot marker
	String		hs_name[];					// name displayed in status bar
	String		hs_url[];					// url linked to hotspot
	String		hs_target[];				// target in url, if any
	Object		hs_him[];					// image displayed while mouse is over hotspot
	String		hs_mask[];					// name of maskimage for hotspotshape
	boolean		hs_visible[];				// is hotspot inside viewport?
	int			hs_imode[];					// 0 - normal, 1 - popup, 2 - always visible
	int			hs_link[];					// Same hotspot location as?

	static final double NO_UV 			= -200.0;	// impossible u/v coordinate
	static final int HSIZE = 12; 				// Size of Hotspot
	
	static final int IMODE_NORMAL = 0;
	static final int IMODE_POPUP  = 1;
	static final int IMODE_ALWAYS = 2;
	static final int IMODE_WARP   = 4;
	static final int IMODE_WHS 	  = 8;
	static final int IMODE_TEXT   = 16;		// Text image

	void hs_init(){
		hotspots = new Vector();
	}

	void hs_allocate(int n){
		try{
	    	hs_xp 		= new double	[n];
        	hs_yp 		= new double	[n];
	    	hs_up 		= new double	[n];
        	hs_vp 		= new double	[n];
			hs_xv 		= new int		[n];
			hs_yv 		= new int		[n];
			hs_hc 		= new Color		[n];
			hs_name 	= new String	[n];
			hs_url 		= new String	[n];
			hs_target	= new String	[n];
			hs_him 		= new Object	[n];
			hs_visible	= new boolean	[n];
			hs_imode	= new int		[n];
			hs_mask		= new String	[n];
			hs_link		= new int		[n];
			numhs = n;
		}catch(Exception e){
			numhs = 0;
		}
		
	}
	
	void hs_dispose(){
		int i;
		
		for(i=0; i<numhs; i++){
	    	if(hs_him[i] != null){
	    		hs_him[i] = null;
	    	}
			hs_hc[i] 		= null;
			hs_name[i]		= null;
			hs_url[i] 		= null;
			hs_target[i]	= null;
			hs_mask[i]		= null;
	    }
	    numhs 		= 0;
		hotspots.removeAllElements();

        hs_xp 		= null;
        hs_yp 		= null;
        hs_up 		= null;
        hs_vp 		= null;
		hs_xv 		= null;
		hs_yv 		= null;
		hs_hc 		= null;
		hs_name 	= null;
		hs_url 		= null;
		hs_him 		= null;
		hs_visible	= null;
		hs_target	= null;
		hs_mask		= null;
		hs_imode	= null;
		hs_link		= null;
		
		hs_image = null;	

	}

	// Parse one Hotspotdescription line for 
	// Hotspot No n
	   	
   	void ParseHotspotLine( String s, int n )
   	{
   		int i=0,length = s.length();
   		StringBuffer w = new StringBuffer();
   		
   		// Set defaults
   		
   		hs_xp[n] 	= 0.0;
        hs_yp[n] 	= 0.0;
   		hs_up[n] 	= NO_UV;
        hs_vp[n] 	= NO_UV;
		hs_xv[n] 	= 0;
		hs_yv[n] 	= 0;
		hs_hc[n] 	= null;
		hs_name[n] 	= null;
		hs_url[n] 	= null;
		hs_target[n]= null;
		hs_him[n] 	= null;
		hs_visible[n] = false;
		hs_imode[n] = IMODE_NORMAL;
		hs_mask[n]	= null;
		hs_link[n]	= -1;

   		
   		while(i<length)
   		{
   			switch( s.charAt(i++))
   			{
   				case 'x': i = getNextWord(i,s,w); hs_xp[n] = Double.valueOf(w.toString()).doubleValue();  break;
    			case 'X': i = getNextWord(i,s,w); hs_xp[n] = -Double.valueOf(w.toString()).doubleValue(); break;
   				case 'y': i = getNextWord(i,s,w); hs_yp[n] = Double.valueOf(w.toString()).doubleValue();  break;	
   				case 'Y': i = getNextWord(i,s,w); hs_yp[n] = -Double.valueOf(w.toString()).doubleValue(); break;	
   				case 'a': i = getNextWord(i,s,w); hs_up[n] = Double.valueOf(w.toString()).doubleValue();  break;	
    			case 'A': i = getNextWord(i,s,w); hs_up[n] = -Double.valueOf(w.toString()).doubleValue(); break;	
   				case 'b': i = getNextWord(i,s,w); hs_vp[n] = Double.valueOf(w.toString()).doubleValue();  break;	
   				case 'B': i = getNextWord(i,s,w); hs_vp[n] = -Double.valueOf(w.toString()).doubleValue(); break;	
   				case 'c': i = getNextWord(i,s,w); hs_hc[n] = new Color( Integer.parseInt(w.toString(),16) ); break;	
   				case 'n': i = getNextWord(i,s,w); hs_name[n] = w.toString(); break;
  				case 'm': i = getNextWord(i,s,w); hs_mask[n] = w.toString(); break;
  				case 'p': hs_imode[n] |= IMODE_POPUP; break;	
  				case 'q': hs_imode[n] |= IMODE_ALWAYS;break;	
 				case 'w': hs_imode[n] |= IMODE_WARP;  break;	
 				case 'e': hs_imode[n] |= IMODE_TEXT;  break;	
   				case 'u': i = getNextWord(i,s,w); hs_url[n] = w.toString(); break;
   				case 'i': i = getNextWord(i,s,w); hs_him[n] = w.toString(); break;
   				case 't': i = getNextWord(i,s,w); hs_target[n] = w.toString(); break;
   			}
   		}
   	}
  
  	void hs_read(){
 		// Set up hotspot structures
 		if( hotspots.size() == 0 )
 			return;
 		
 		hs_allocate(hotspots.size());
 		for(int i=0; i<numhs; i++){
			ParseHotspotLine( (String)hotspots.elementAt(i), i );
		}
		hs_setLinkedHotspots();
	}  		 		

	void hs_setup(int[][] pd){
		if( pd == null ) return;
  		int ph = pd.length, pw = pd[0].length;
  		PixelGrabber pg;
 		int	i,x,y,cy;
 		
 		hs_read();
		
             
        int[] tdata;

		// Load Hotspotimages, if not done
		
		for(i=0; i<numhs; i++){
			if( hs_him[i] != null && ((hs_imode[i] & IMODE_TEXT) == 0) ){
				String s = (String)hs_him[i];

				if( !(s.startsWith("ptviewer:") || s.startsWith("javascript:"))){
 					hs_him[i] = loadImage( s ) ;
 				}
 			}
 		}	


		hs_rel2abs(pw, ph);
 
		// Process global hotspot image
		
		if( hs_image != null ) hs_image = loadImage( (String)hs_image );
	    if( hs_image != null && 	hs_image instanceof Image &&
	        pw 	== ((Image)hs_image).getWidth (null)  && 
	        ph == ((Image)hs_image).getHeight(null) ){
	        ptImageToAlpha( pd, (Image)hs_image );
 	    }else{
			// Set hotspot masks
		
			for(i=0; i<numhs && i<255 ; i++){ // only 255 indices
				if( hs_link[i] == -1 ){ // Linked Hotspots don't get masks
					if( hs_up[i] != NO_UV && hs_vp[i] != NO_UV){
						SetPAlpha( (int)hs_xp[i], (int)hs_yp[i], 
								   (int)hs_up[i], (int)hs_vp[i], i, pd );
						if(hs_up[i] >= hs_xp[i]){ 
							hs_xp[i] += (hs_up[i] - hs_xp[i])/2;
							hs_up[i] =  hs_up[i] - hs_xp[i];
						}else{
							hs_xp[i] += (hs_up[i] + pw - hs_xp[i])/2;
							hs_up[i] =  hs_up[i] + pw - hs_xp[i];
						}
						hs_yp[i] =  (hs_yp[i] + hs_vp[i])/2;
						hs_vp[i] =  Math.abs(hs_yp[i] - hs_vp[i]);
					}else if((hs_imode[i] &	IMODE_WARP) > 0  && 
						 (hs_him[i] != null) && hs_him[i] instanceof Image &&
						 hs_mask[i]== null){// warped image without mask 
						hs_up[i] = ((Image)hs_him[i]).getWidth(null);
						hs_vp[i] = ((Image)hs_him[i]).getHeight(null);
						SetPAlpha( (int)(hs_xp[i]-hs_up[i]/2.0), (int)(hs_yp[i]-hs_vp[i]/2.0), 
								   (int)(hs_xp[i]+hs_up[i]/2.0), (int)(hs_yp[i]+hs_vp[i]/2.0), i, pd );
					}else if(hs_mask[i] != null){

						Image mim = loadImage(hs_mask[i]);
						if( mim != null ){
							tdata = new int[ mim.getWidth(null)*mim.getHeight(null) ];
							pg = new PixelGrabber( mim, 0, 0, mim.getWidth(null), mim.getHeight(null), tdata, 0, mim.getWidth(null) );
        					try { pg.grabPixels(); } catch (InterruptedException e) { continue; }

        					int hs_y = (int)hs_yp[i], hs_x = (int)hs_xp[i];
 							int hmask = (i<<24) + 0x00ffffff;
        					int k=0;
 
        					for(y=0; y<mim.getHeight(null) && hs_y<ph; y++,hs_y++){
        						cy = y*mim.getWidth(null);
         						for(x=0, hs_x=(int)hs_xp[i]; x<mim.getWidth(null) && hs_x<pw; x++,hs_x++){
         							if( (tdata[cy +x] & 0x00ffffff) == 0x00ffffff ){ // inside mask
         								pd[hs_y][hs_x] &= hmask;k++;
        							}
        						}
        					}
        					hs_yp[i] += mim.getHeight(null)/2;
        					hs_xp[i] += mim.getWidth(null)/2;
        					hs_up[i] =  mim.getWidth(null); // width
 							hs_vp[i] =  mim.getHeight(null); // height
 							mim = null; tdata = null;
       					}
         			}
        		}
        	}
        }
        
		for(i=0; i<numhs; i++)
			if( hs_link[i] != -1 ){
				hs_xp[i] = hs_xp[hs_link[i]];
				hs_yp[i] = hs_yp[hs_link[i]];
        		hs_up[i] = hs_up[hs_link[i]];
 				hs_vp[i] = hs_vp[hs_link[i]];
			}
		        

		// Get and set pixel data for warped hotspots

		for(i=0; i<numhs; i++){
			if( (hs_imode[i] &	IMODE_WARP) > 0  && hs_him[i] != null){
					if( hs_him[i] instanceof Image){
						Image p = (Image)hs_him[i];
		
						int w = p.getWidth(null);
						int h = p.getHeight(null);
						int xp = (int)hs_xp[i] - w/2;
						int yp = (int)hs_yp[i] - h/2;
						
						// System.out.println( xp + " " +yp + " " +w+" "+h);
				
						if( xp>=0 && yp>=0 && w + xp <= pw && h + yp <= ph){
							int[] buf = new int[ w * h * 2 ];
       						pg = new PixelGrabber( p, 0, 0, w, h,  buf, 0, w );
        					try { pg.grabPixels(); } 
        					catch (InterruptedException e){
             					continue;
        					}

        					im_extractRect( pd, xp, yp, buf, w, 0, h, w, h );
        					hs_him[i] = buf;
        					hs_up[i]  = w;
        					hs_vp[i]  = h;
        				}//else
        				//	System.out.println("Image for Hotspot No " + i + " outside main panorama");
        			}
        		
        	}
        }
        					
	}
	

// Insert warped images into panorama pd
// Return true if pd has been changed	
boolean hs_drawWarpedImages(int[][] pd, int chs, boolean shs){
	int i,w,h,xp,yp;
	boolean result = false;
	
	if( pd == null ) return false;
 		
 	for(i=0; i<numhs; i++){
 		if( (hs_imode[i] & IMODE_WARP) > 0 && 
 		     hs_him[i] != null &&
 		     hs_him[i] instanceof int[] ){
				w  =  (int)hs_up[i]; 		h  =  (int)hs_vp[i];
				xp =  (int)hs_xp[i] - w/2;	yp =  (int)hs_yp[i] - h/2;
 				if( shs || (hs_imode[i] & IMODE_ALWAYS) > 0 
	       				|| ( i == chs && (hs_imode[i] & IMODE_POPUP) > 0)
	       			    || ( chs >= 0 && hs_link[i] == chs && (hs_imode[i] & IMODE_POPUP) > 0 )){
	       			if( (hs_imode[i] & IMODE_WHS) == 0 ){
	       				im_insertRect( pd, xp, yp, (int[])hs_him[i], w, 0, 0, w, h );
	       				hs_imode[i] |= IMODE_WHS;
	       				result = true;
	       			}
	       		}else{
	       			if( (hs_imode[i] & IMODE_WHS) > 0 ){
	       				im_insertRect( pd, xp, yp, (int[])hs_him[i], w, 0, h, w, h );
	       				hs_imode[i] &= ~IMODE_WHS;
	       				result = true;
	       			}
	       		}
	       	}
	}
	return result;
}  	


// Convert relative to absolute hotspot coordinates
void hs_rel2abs(int pw, int ph){
	int i;
		
	for(i=0; i<numhs; i++){
		if(hs_xp[i] < 0.0){
			hs_xp[i] = -hs_xp[i] * (double)pw / 100.0;
			if( hs_xp[i] >= pw ) hs_xp[i] = pw - 1;
		}
		if(hs_yp[i] < 0.0){
			hs_yp[i] = -hs_yp[i] * (double)ph / 100.0;
			if( hs_yp[i] >= ph ) hs_yp[i] = ph - 1;
		}
		if(hs_up[i] < 0.0 && hs_up[i] != NO_UV){
			hs_up[i] = -hs_up[i] * (double)pw / 100.0;
			if( hs_up[i] >= pw ) hs_up[i] = pw - 1;
		}
		if(hs_vp[i] < 0.0 && hs_vp[i] != NO_UV){
			hs_vp[i] = -hs_vp[i] * (double)ph / 100.0;
			if( hs_vp[i] >= ph ) hs_vp[i] = ph - 1;
		}
	}
}


// Draw hotspot marker

void hs_draw(Graphics g, int off_x, int off_y, int width, int height, int chs, boolean shs){
	int i;
 		
 	// g.setClip( off_x, off_y, width, height ); // Draw only into viewer window; does not work in NC4.04
 		 		
 	for(i=0; i<numhs; i++){
		if( hs_visible[i] ){
	       	if( shs || (hs_imode[i] & IMODE_ALWAYS) > 0 
	       			|| (i == chs && (hs_imode[i] & IMODE_POPUP) > 0) 
	       			|| (chs >= 0 && hs_link[i] == chs && (hs_imode[i] & IMODE_POPUP) > 0)){
      			if( hs_him[i] == null ){ // default circle
      	 			if( hs_hc[i] == null )
        				g.setColor(Color.red);
        			else
        				g.setColor(hs_hc[i]);
      	 			g.drawOval(hs_xv[i] - 10 + off_x, hs_yv[i] - 10 + off_y,20,20);
      	 			g.fillOval(hs_xv[i] -  5 + off_x, hs_yv[i] -  5 + off_y,10,10);
       			}else if( hs_him[i] instanceof Image ){ // Draw image
       				Image p = (Image)hs_him[i];
        			g.drawImage(	p,
        							hs_xv[i] - p.getWidth(null) /2 + off_x,
        							hs_yv[i] - p.getHeight(null)/2 + off_y,
        							this	);
        		}else if( (hs_imode[i] & IMODE_TEXT) > 0 &&
        				   hs_him[i] instanceof String ){ // Text window
        			String s = (String)hs_him[i];
        			Dimension d = string_textWindowSize( g, s );
        			if( hs_xv[i] >=0 && hs_xv[i] < width &&
        				hs_yv[i] >=0 && hs_yv[i] < height ){
        				int xt=0,yt=0,corner=0;
        				if( hs_xv[i] + d.width < width ){
        					if( hs_yv[i] - d.height > 0 ){
        						xt=hs_xv[i]; yt=hs_yv[i] - d.height;corner=1;
        					}else if( hs_yv[i] + d.height < width ){
        						xt=hs_xv[i]; yt=hs_yv[i];corner=2;
        					}
        				}else if( hs_xv[i] - d.width >= 0 ){
        					if( hs_yv[i] - d.height > 0 ){
        						xt=hs_xv[i]- d.width; yt=hs_yv[i] - d.height;corner=3;
        					}else if( hs_yv[i] + d.height < width ){
        						xt=hs_xv[i]- d.width; yt=hs_yv[i];corner=4;
        					}
        				}
        				if(corner!=0){
        					string_drawTextWindow( g, xt + off_x, yt + off_y, d, hs_hc[i], s, corner);
      					}
      				}
         		}
         	}
        }
  	}
}

final void hs_exec_popup( int chs ){
 	for(int i=0; i<numhs; i++){
		if( hs_visible[i] ){
        	if(  hs_him[i] != null && ( i == chs  || ( chs >= 0 && hs_link[i] == chs)) ){
        		if( hs_him[i] instanceof String && (hs_imode[i] & IMODE_TEXT) == 0 ){
        			JumpToLink( (String)hs_him[i], null );
        		}
        	}
        }
 	}
}



final void hs_setLinkedHotspots(){
  		int i,k=0;
  		
  		for(i=0; i<numhs; i++){
  			for(k=i+1; k<numhs; k++){
  				if( hs_xp[i] == hs_xp[k] && hs_yp[i] == hs_yp[k] && hs_link[i]==-1 )
  					hs_link[k] = i;
  			}
  		}
  	}

final void hs_setCoordinates(int vw, int vh, int pw, int ph, double pan, double tilt, double fov){
		double		a,p;							// field of view in rad
		double		mt[][];
		int 		i,k;
		double 		v0,v1,v2;
		double		x,y,z, theta, phi;
		int 		sw2 =  pw/2;
		int 		sh2 =  ph/2 ;

		mt = new double[3][3];
	
		a  =  fov * 2.0 * Math.PI / 360.0;	// field of view in rad		
		p  = (double)vw / (2.0 * Math.tan( a / 2.0 ) );

		SetMatrix(  -tilt * 2.0 * Math.PI / 360.0, 
					-pan   * 2.0 * Math.PI / 360.0, 
					mt, 
					0 );
		for(i=0; i<numhs; i++)
		{
			x = hs_xp[i] - sw2;
			y = pheight - (hs_yp[i] - sh2);
			
			theta = (x / sw2) * Math.PI;
			phi   = (y / sh2) * Math.PI / 2.0;
			
			if( Math.abs(theta) > Math.PI / 2.0 ) 
				v2 = 1.0;
			else
				v2 = -1.0;
				
			v0 = v2 * Math.tan( theta );
			v1 = Math.sqrt( v0 * v0 + v2 * v2 ) * Math.tan( phi );
			
			x = mt[0][0] * v0 + mt[1][0] * v1 + mt[2][0] * v2;
			y = mt[0][1] * v0 + mt[1][1] * v1 + mt[2][1] * v2;
			z = mt[0][2] * v0 + mt[1][2] * v1 + mt[2][2] * v2;
			
			hs_xv[i] = (int)( x * p/z + vw / 2.0);
			hs_yv[i] = (int)( y * p/z + vh / 2.0 );
         	if(debug)
            	System.out.println("Hotspot Coordinates: x = " + hs_xv[i] + "  y = " + hs_yv[i] + " z = " + z);
            
            int hs_vis_hor = HSIZE;
            int hs_vis_ver = HSIZE;
            
            if( hs_him[i] != null &&  hs_him[i] instanceof Image ){
            	hs_vis_hor = ((Image)hs_him[i]).getWidth(null)/2;
            	hs_vis_ver = ((Image)hs_him[i]).getHeight(null)/2;
            }else if( hs_him[i] != null &&  hs_him[i] instanceof String &&
            		 (hs_imode[i] & IMODE_TEXT) > 0 ){
             	hs_vis_hor = 100; // This should be more intelligent
            	hs_vis_ver = 100;
            }else if( hs_up[i] != NO_UV && hs_vp[i] != NO_UV ){
            	hs_vis_hor = 100; // This should be more intelligent
            	hs_vis_ver = 100;
            }
             	
            
            
            if( hs_xv[i] >= -hs_vis_hor && hs_xv[i] < vwidth +hs_vis_hor && 
            	hs_yv[i] >= -hs_vis_ver && hs_yv[i] < vheight+hs_vis_ver && 
            	z < 0.0){
            	hs_visible[i] = true;
            }else
            	hs_visible[i] = false;
		}
	}


 	


// Final brace for ptviewer.class
}
