/*
 * Copyright (C) 2004  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#import "PopplerPage.h"
#import "PopplerDocument.h"
#import "NSObject+PopplerObject.h"

#include "bindings/poppler_page.h"

/**
 * Non-Public methods.
 */
@interface PopplerPage (Private)
@end


@implementation PopplerPage

- (id) initWithDocument: (PopplerDocument*)aDocument index: (unsigned)anIndex
{
   NSAssert(aDocument, @"no document");
   NSAssert(anIndex > 0 && anIndex <= [aDocument countPages], @"invalid page index");
   
   self = [super init];
   if (self)
   {
      index = anIndex;
      document = aDocument;
      poppler_page = poppler_page_create([aDocument poppler_object], anIndex);
      NSAssert(poppler_page, @"poppler_page_create returned NULL");
   }
   return self;
}

- (void) dealloc
{
   poppler_page_destroy(poppler_page);
   [super dealloc];
}

- (unsigned) index
{
   return index;
}

- (NSSize) size
{
   double width;
   double height;
   
   switch ([self orientation])
   {
      case POPPLER_PAGE_ORIENTATION_PORTRAIT:
      case POPPLER_PAGE_ORIENTATION_UPSIDEDOWN:
      {
         width = poppler_page_get_width(poppler_page);
         height = poppler_page_get_height(poppler_page);
         break;
      }
      case POPPLER_PAGE_ORIENTATION_LANDSCAPE:
      case POPPLER_PAGE_ORIENTATION_SEASCAPE:
      {
         width = poppler_page_get_height(poppler_page);
         height = poppler_page_get_width(poppler_page);
         break;
      }
      default:
      {
         width = height = 0;
         NSAssert(NO, @"unreachable code");
      }
   }

   return NSMakeSize((float)width, (float)height);
}

- (int) rotate
{
   return poppler_page_get_rotate(poppler_page);
}

- (PopplerPageOrientation) orientation
{
   int rotate = poppler_page_get_rotate(poppler_page);
   NSAssert(rotate >= 0, @"got negative rotation factor"); 
   
   PopplerPageOrientation orientation = POPPLER_PAGE_ORIENTATION_UNSET;
   switch (rotate)
   {
      case 90:
         orientation = POPPLER_PAGE_ORIENTATION_LANDSCAPE; break;
      case 180:
         orientation = POPPLER_PAGE_ORIENTATION_UPSIDEDOWN; break;
      case 270:
         orientation = POPPLER_PAGE_ORIENTATION_SEASCAPE; break;
      default:
         orientation = POPPLER_PAGE_ORIENTATION_PORTRAIT;
   }

   return orientation;
}

- (PopplerDocument*) document
{
   return document;
}

- (void*) poppler_object
{
   return poppler_page;
}

@end


/* ----------------------------------------------------- */
/*  Category Private                                     */
/* ----------------------------------------------------- */

@implementation PopplerPage (Private)
@end
