#!/usr/bin/python -OO
# Copyright 2005 Gregor Kaufmann <tdian@users.sourceforge.net>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

__version__ = "0.2.4"
__configversion__ = 15
__queueversion__ = 4
__NAME__ = "sabnzbd"

import os
import logging
import datetime
import tempfile
import cPickle
import zipfile
import re
import random

from threading import RLock, Lock, Condition, Thread

from sabnzbd.assembler import Assembler, PostProcessor
from sabnzbd.downloader import Downloader, BPSMeter
from sabnzbd.nzbqueue import NzbQueue
from sabnzbd.misc import MSGIDGrabber, URLGrabber, DirScanner, synchronized
from sabnzbd.nzbstuff import NzbObject
from sabnzbd.utils.kronos import ThreadedScheduler
from sabnzbd.rss import RSSQueue

import sabnzbd.nzbgrab

LOCK = RLock()
LOCK_NZB = Lock()
CV = Condition(LOCK)

START = datetime.datetime.now()

NZB_QUOTA = None

CFG = None

BYTES_FILE_NAME = 'bytes.sab'
QUEUE_FILE_NAME = 'queue.sab'
RSS_FILE_NAME = 'rss.sab'

FAIL_ON_CRC = False
CREATE_GROUP_FOLDERS = False
CREATE_CAT_FOLDERS = False
CREATE_CAT_SUB = False
DO_FILE_JOIN = False
DO_UNZIP = False
DO_UNRAR = False
DO_SAVE = False
PAR_CLEANUP = False
AUTOSHUTDOWN = False
SEND_GROUP = False

CLEANUP_LIST = []

UMASK = 0755
BANDWITH_LIMIT = 0.0

USERNAME_NEWZBIN = None
PASSWORD_NEWZBIN = None

CACHE_DIR = None
NZB_BACKUP_DIR = None
DOWNLOAD_DIR = None
COMPLETE_DIR = None

POSTPROCESSOR = None
ASSEMBLER = None
DIRSCANNER = None
SCHED = None
NZBQ = None
DOWNLOADER = None
BPSMETER = None
RSS = None

URLGRABBERS = []
MSGIDGRABBERS = []

__INITIALIZED__ = False

################################################################################
# Decorators                                                                   #
################################################################################
def synchronized_CV(func):
    def call_func(*params, **kparams):
        CV.acquire()
        try:
            return func(*params, **kparams)
        finally:
            CV.notifyAll()
            CV.release()
    return call_func

################################################################################
# Signal Handler                                                               #
################################################################################
@synchronized(LOCK)
def sig_handler(signum = None, frame = None):
    if type(signum) != type(None):
        logging.info('[%s] Signal %s caught, saving and exiting...', __NAME__, 
                     signum)
    try:
        save_state()
    finally:
        os._exit(0)
    
################################################################################
# Initializing                                                                 #
################################################################################
@synchronized(LOCK)
def initialize(pause_downloader = False):
    global __INITIALIZED__, FAIL_ON_CRC, CREATE_GROUP_FOLDERS,  DO_FILE_JOIN, \
           DO_UNZIP, DO_UNRAR, DO_SAVE, PAR_CLEANUP, CLEANUP_LIST, \
           USERNAME_NEWZBIN, PASSWORD_NEWZBIN, POSTPROCESSOR, ASSEMBLER, \
           DIRSCANNER, SCHED, NZBQ, DOWNLOADER, NZB_BACKUP_DIR, DOWNLOAD_DIR, \
           COMPLETE_DIR, CACHE_DIR, UMASK, SEND_GROUP, CREATE_CAT_FOLDERS, \
           CREATE_CAT_SUB, BPSMETER, BANDWITH_LIMIT
           
    if __INITIALIZED__:
        return False
           
    logging.info("Initializing SABnzbd v%s", __version__)
    
    ###########################
    ## CONFIG Initialization ##
    ###########################
    
    USERNAME_NEWZBIN = CFG['newzbin']['username']
    PASSWORD_NEWZBIN = CFG['newzbin']['password']
    
    FAIL_ON_CRC = bool(int(CFG['misc']['fail_on_crc']))
    logging.debug("FAIL_ON_CRC -> %s", FAIL_ON_CRC)
    
    CREATE_GROUP_FOLDERS = bool(int(CFG['misc']['create_group_folders']))
    logging.debug("CREATE_GROUP_FOLDERS -> %s", CREATE_GROUP_FOLDERS)
    
    DO_FILE_JOIN = bool(int(CFG['misc']['enable_filejoin']))
    logging.debug("DO_FILE_JOIN -> %s", DO_FILE_JOIN)
    
    DO_UNZIP = bool(int(CFG['misc']['enable_unzip']))
    logging.debug("DO_UNZIP -> %s", DO_UNZIP)
    
    DO_UNRAR = bool(int(CFG['misc']['enable_unrar']))
    logging.debug("DO_UNRAR -> %s", DO_UNRAR)
    
    DO_SAVE = bool(int(CFG['misc']['enable_save']))
    logging.debug("DO_SAVE -> %s", DO_SAVE)
    
    PAR_CLEANUP = bool(int(CFG['misc']['enable_par_cleanup']))
    logging.debug("PAR_CLEANUP -> %s", PAR_CLEANUP)
    
    CLEANUP_LIST = CFG['misc']['cleanup_list']
    if type(CLEANUP_LIST) != type([]):
        CLEANUP_LIST = []
    logging.debug("CLEANUP_LIST -> %s", CLEANUP_LIST)
    
    UMASK = int(CFG['misc']['umask'], 8)
    logging.debug("UMASK -> %s", UMASK)
    
    SEND_GROUP = bool(int(CFG['misc']['send_group']))
    logging.debug("SEND_GROUP -> %s", SEND_GROUP)
    
    CREATE_CAT_FOLDERS = int(CFG['newzbin']['create_category_folders'])
    
    if CREATE_CAT_FOLDERS > 1:
        CREATE_CAT_SUB = True
    CREATE_CAT_FOLDERS = bool(CREATE_CAT_FOLDERS)
    
    logging.debug("CREATE_CAT_FOLDERS -> %s", CREATE_CAT_FOLDERS)
    logging.debug("CREATE_CAT_SUB -> %s", CREATE_CAT_SUB)
    
    if not CFG['misc']['download_dir']:
        logging.error('No DOWNLOAD_DIR defined!')
        return False
    
    DOWNLOAD_DIR = os.path.abspath(CFG['misc']['download_dir'])
    if not os.path.exists(DOWNLOAD_DIR):
        logging.error('Download directory: %s does not exist', DOWNLOAD_DIR)
        return False
    if not os.access(DOWNLOAD_DIR, os.R_OK + os.W_OK):
        logging.error('Download directory: %s error accessing',
                      DOWNLOAD_DIR)
        return False
    logging.info("DOWNLOAD_DIR: %s", DOWNLOAD_DIR)
    
    COMPLETE_DIR = CFG['misc']['complete_dir']
    if COMPLETE_DIR:
        COMPLETE_DIR = os.path.abspath(COMPLETE_DIR)
        if not os.path.exists(COMPLETE_DIR):
            logging.error('Directory: %s does not exist', COMPLETE_DIR)
            return False
        if not os.access(COMPLETE_DIR, os.R_OK + os.W_OK):
            logging.error('Directory: %s error accessing', COMPLETE_DIR)
            return False
    logging.info("COMPLETE_DIR: %s", COMPLETE_DIR)
    
    NZB_BACKUP_DIR = CFG['misc']['nzb_backup_dir']
    if NZB_BACKUP_DIR:
        NZB_BACKUP_DIR = os.path.abspath(NZB_BACKUP_DIR)
        if not os.path.exists(NZB_BACKUP_DIR):
            logging.error('Directory: %s does not exist', NZB_BACKUP_DIR)
            return False
        if not os.access(NZB_BACKUP_DIR, os.R_OK + os.W_OK):
            logging.error('Directory: %s error accessing', NZB_BACKUP_DIR)
            return False
    logging.info("NZB_BACKUP_DIR: %s", NZB_BACKUP_DIR)
    
    if "samefile" in os.path.__dict__:
        if os.path.samefile(DOWNLOAD_DIR, COMPLETE_DIR):
            logging.error('DOWNLOAD_DIR and COMPLETE_DIR cannot be the same!')
            return True
            
    if not CFG['misc']['cache_dir']:
        logging.error('No cache_dir defined!')
        return False
        
    CACHE_DIR = os.path.abspath(CFG['misc']['cache_dir'])
    if not os.path.exists(CACHE_DIR):
        logging.error('Cache directory directory: %s does not exist', CACHE_DIR)
        return False
    if not os.access(CACHE_DIR, os.R_OK + os.W_OK):
        logging.error('Cache directory directory: %s error accessing', CACHE_DIR)
        return False
    logging.info("CACHE_DIR: %s", CACHE_DIR)
    
    dirscan_dir = CFG['misc']['dirscan_dir']
    if dirscan_dir:
        dirscan_dir = os.path.abspath(dirscan_dir)
        if not os.path.exists(dirscan_dir):
            logging.error('Directory: %s does not exist', dirscan_dir)
            return False
        if not os.access(dirscan_dir, os.R_OK + os.W_OK):
            logging.error('Directory: %s error accessing', dirscan_dir)
            return False
    logging.info("dirscan_dir: %s", dirscan_dir)
        
    servers = CFG['servers']
    
    try:
        BANDWITH_LIMIT = float(CFG['misc']['bandwith_limit'])
    except:
        CFG['misc']['bandwith_limit'] = "0.0"
        BANDWITH_LIMIT = 0.0
        
    logging.info("BANDWITH_LIMIT: %s", BANDWITH_LIMIT)
    
    try:
        cache_limit = int(CFG['misc']['cache_limit'])
    except:
        CFG['misc']['cache_limit'] = "0"
        cache_limit = 0
        
    if not CFG['misc']['schedlines']:
        CFG['misc']['schedlines'] = []
        
    schedlines = CFG['misc']['schedlines']
    logging.info("schedlines: %s", schedlines)
    
    dirscan_opts = int(CFG['misc']['dirscan_opts'])
    dirscan_repair, dirscan_unpack, dirscan_delete = pp_to_opts(dirscan_opts)
    logging.info("dirscan_opts: %s", dirscan_opts)
    
    f_mode = bool(int(CFG['misc']['f_mode']))
    logging.info("f_mode: %s", f_mode)
    
    auto_sort = bool(int(CFG['misc']['auto_sort']))
    logging.info("auto_sort: %s", auto_sort)
    
    ############################
    ## Object initializiation ##
    ############################
    
    need_rsstask = init_RSS()
    init_SCHED(schedlines, need_rsstask)
    
    if BPSMETER:
        BPSMETER.reset()
    else:        
        bytes = load_data(BYTES_FILE_NAME, remove = False, do_pickle = False)
        try:
            bytes = int(bytes)
        except:
            bytes = 0
        
        BPSMETER = BPSMeter(bytes)
        
    if NZBQ:
        NZBQ.__init__(f_mode, auto_sort, cache_limit)
    else:
        NZBQ = NzbQueue(f_mode, auto_sort, cache_limit)
        
    if POSTPROCESSOR:
        POSTPROCESSOR.__init__(DOWNLOAD_DIR, COMPLETE_DIR, POSTPROCESSOR.queue)
    else:
        POSTPROCESSOR = PostProcessor(DOWNLOAD_DIR, COMPLETE_DIR)
        NZBQ.__init__stage2__()
        
    if ASSEMBLER:
        ASSEMBLER.__init__(DOWNLOAD_DIR, ASSEMBLER.queue)
    else:
        ASSEMBLER = Assembler(DOWNLOAD_DIR)
        
    if DOWNLOADER:
        DOWNLOADER.__init__(servers, DOWNLOADER.paused)
    else:
        DOWNLOADER = Downloader(servers)
        if pause_downloader:
            DOWNLOADER.paused = True
            
    if dirscan_dir:
        DIRSCANNER = DirScanner(dirscan_dir, dirscan_repair, dirscan_unpack, 
                                dirscan_delete)
                                
    __INITIALIZED__ = True
    return True
    
@synchronized(LOCK)
def start():
    if __INITIALIZED__:
        logging.debug('[%s] Starting postprocessor', __NAME__)
        POSTPROCESSOR.start()
            
        logging.debug('[%s] Starting assembler', __NAME__)
        ASSEMBLER.start()
            
        logging.debug('[%s] Starting downloader', __NAME__)
        DOWNLOADER.start()
            
        if SCHED:
            logging.debug('[%s] Starting scheduler', __NAME__)
            SCHED.start()
            
        if DIRSCANNER:
            logging.debug('[%s] Starting dirscanner', __NAME__)
            DIRSCANNER.start()
            
def halt():
    global __INITIALIZED__, SCHED, DIRSCANNER, RSS
    
    if __INITIALIZED__:
        logging.info('SABnzbd shutting down...')
        
        ## Stop Optional Objects ##
        
        if SCHED:
            logging.debug('Stopping scheduler')
            SCHED.stop()
            SCHED = None
            
        for grabber in URLGRABBERS:
            logging.debug('Stopping grabber {%s}', grabber)
            try:
                grabber.join()
            except:
                logging.exception('[%s] Joining grabber {%s} failed', __NAME__, grabber)
                
        for grabber in MSGIDGRABBERS:
            logging.debug('Stopping grabber {%s}', grabber)
            try:
                grabber.join()
            except:
                logging.exception('[%s] Joining grabber {%s} failed', __NAME__, grabber)
                
        if DIRSCANNER:
            logging.debug('Stopping dirscanner')
            DIRSCANNER.stop()
            try:
                DIRSCANNER.join()
            except:
                pass
            DIRSCANNER = None
            
        ## Stop Required Objects ##
        logging.debug('Stopping downloader')
        CV.acquire()
        try:
            DOWNLOADER.stop()
        finally:
            CV.notifyAll()
            CV.release()
        try:
            DOWNLOADER.join()
        except:
            pass
                
        logging.debug('Stopping assembler')
        ASSEMBLER.stop()
        try:
            ASSEMBLER.join()
        except:
            pass
            
        logging.debug('Stopping postprocessor')
        POSTPROCESSOR.stop()
        try:
            POSTPROCESSOR.join()
        except:
            pass
        
        ## Save State ##
        save_state()
        
        __INITIALIZED__ = False
        
################################################################################
## NZBQ Synchronized Methods                                                  ##
################################################################################
@synchronized(LOCK)
def debug():
    try:
        return NZBQ.debug()
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def move_up_bulk(nzo_id, nzf_ids):
    try:
        NZBQ.move_up_bulk(nzo_id, nzf_ids)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def move_top_bulk(nzo_id, nzf_ids):
    try:
        NZBQ.move_top_bulk(nzo_id, nzf_ids)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def move_down_bulk(nzo_id, nzf_ids):
    try:
        NZBQ.move_down_bulk(nzo_id, nzf_ids)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def move_bottom_bulk(nzo_id, nzf_ids):
    try:
        NZBQ.move_bottom_bulk(nzo_id, nzf_ids)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def load_article(article):
    try:
        return NZBQ.load_article(article)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)

@synchronized(LOCK)
def change_opts(nzo_id, pp):
    try:
        NZBQ.change_opts(nzo_id, pp)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def get_article(host):
    try:
        return NZBQ.get_article(host)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def has_articles():
    try:
        return not NZBQ.is_empty()
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def has_articles_for(server):
    try:
        return not (server in NZBQ.try_list)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def register_article(article, lines = None):
    try:
        return NZBQ.register_article(article, lines)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def remove_from_try_list(server):
    try:
        NZBQ.remove_from_try_list(server)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def reset_try_list():
    try:
        NZBQ.reset_try_list()
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def remove_nzo(nzo_id, add_to_history = True):
    try:
        NZBQ.remove(nzo_id, add_to_history)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def cleanup_nzo(nzo):
    try:
        NZBQ.cleanup_nzo(nzo)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def remove_nzf(nzo_id, nzf_id):
    try:
        NZBQ.remove_nzf(nzo_id, nzf_id)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def sort_by_avg_age():
    try:
        NZBQ.sort_by_avg_age()
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def add_msgid(msgid, pp):
    logging.info('[%s] Fetching msgid %s from www.newzbin.com',
                 __NAME__, msgid)
    msg = "fetching msgid %s from www.newzbin.com" % msgid
    
    repair, unpack, delete = pp_to_opts(pp)
    
    future_nzo = NZBQ.generate_future(msg, repair, unpack, delete)
    
    # Look for a grabber and reinitialize it
    for grabber in MSGIDGRABBERS:
        if not grabber.isAlive():
            grabber.__init__(USERNAME_NEWZBIN, PASSWORD_NEWZBIN, msgid, future_nzo)
            grabber.start()
            return
            
    grabber = MSGIDGrabber(USERNAME_NEWZBIN, PASSWORD_NEWZBIN, msgid, future_nzo)
    grabber.start()
    
@synchronized(LOCK)
def add_url(url, pp):
    logging.info('[%s] Fetching %s', __NAME__, url)
    
    msg = "Trying to fetch .nzb from %s" % url
    
    repair, unpack, delete = pp_to_opts(pp)
    
    future_nzo = NZBQ.generate_future(msg, repair, unpack, delete)
    
    # Look for a grabber and reinitialize it
    for urlgrabber in URLGRABBERS:
        if not urlgrabber.isAlive():
            urlgrabber.__init__(url, future_nzo)
            urlgrabber.start()
            return
            
    urlgrabber = URLGrabber(url, future_nzo)
    urlgrabber.start()
    
    URLGRABBERS.append(urlgrabber)
        
@synchronized(LOCK)
def switch(nzo_id1, nzo_id2):
    try:
        NZBQ.switch(nzo_id1, nzo_id2)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def history_info():
    try:
        return NZBQ.history_info()
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def queue_info(for_cli = False):
    try:
        return NZBQ.queue_info(for_cli = for_cli)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
            
@synchronized(LOCK)
def purge_history():
    try:
        NZBQ.purge()
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized(LOCK)
def save_state():
    try:
        NZBQ.flush_articles()
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
        
    try:
        NZBQ.save()
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
        
    try:
        save_data(str(BPSMETER.get_sum()), BYTES_FILE_NAME, do_pickle = False)
    except NameError:
        logging.exception("[%s] Error accessing BPSMETER?", __NAME__)
    
    if RSS:
        try:
            RSS.save()
        except NameError:
            logging.exception("[%s] Error accessing BPSMETER?", __NAME__)
            
################################################################################
## LOCK_NZB Methods                                                           ##
################################################################################
@synchronized(LOCK_NZB)
def backup_nzb(filename, data):
    if NZB_BACKUP_DIR:
        try:
            path = os.path.join(NZB_BACKUP_DIR, filename)
            logging.info("[%s] Saving %s", __NAME__, path)
            _f = open(path, 'w')
            _f.write(data)
            _f.flush()
            _f.close()
        except:
            logging.exception("[%s] Saving %s failed", __NAME__, path)
            
################################################################################
## CV synchronized (notifys downloader)                                       ##
################################################################################
@synchronized_CV
def add_nzbfile(nzbfile, pp):
    repair, unpack, delete = pp_to_opts(pp)
    
    filename = os.path.basename(nzbfile.filename)
    
    root, ext = os.path.splitext(filename)
    
    logging.info('[%s] Adding %s', __NAME__, filename)
    
    if ext.lower() == '.zip':
        f = tempfile.TemporaryFile()
        f.write(nzbfile.value)
        f.flush()
        try:
            zf = zipfile.ZipFile(f)
            for name in zf.namelist():
                data = zf.read(name)
                name = os.path.basename(name)
                if data:
                    NZBQ.add(NzbObject(name, repair, unpack, delete, data))
        finally:
            f.close()
    else:
        try:
            NZBQ.add(NzbObject(filename, repair, unpack, delete, nzbfile.value))
        except NameError:
            logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized_CV
def add_nzo(nzo, position = -1):
    try:
        NZBQ.add(nzo, position)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized_CV
def insert_future_nzo(future_nzo, filename, data, cat_root = None, cat_tail = None):
    try:
        NZBQ.insert_future(future_nzo, filename, data, cat_root, cat_tail)
    except NameError:
        logging.exception("[%s] Error accessing NZBQ?", __NAME__)
        
@synchronized_CV
def pause_downloader():
    try:
        DOWNLOADER.pause()
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
        
@synchronized_CV
def resume_downloader():
    try:
        DOWNLOADER.resume()
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
         
################################################################################
## Unsynchronized methods                                                     ##
################################################################################
def bps():
    try:
        return BPSMETER.bps
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
        
def reset_bpsmeter():
    try:
        BPSMETER.reset()
    except NameError:
        logging.exception("[%s] Error accessing BPSMETER?", __NAME__)
        
def update_bytes(bytes):
    try:
        BPSMETER.update(bytes)
    except NameError:
        logging.exception("[%s] Error accessing BPSMETER?", __NAME__)
        
def get_bytes():
    try:
        return BPSMETER.get_sum()
    except NameError:
        logging.exception("[%s] Error accessing BPSMETER?", __NAME__)
        return 0
        
def postprocess_nzo(nzo):
    try:
        POSTPROCESSOR.process(nzo)
    except NameError:
        logging.exception("[%s] Error accessing POSTPROCESSOR?", __NAME__)
        
def assemble_nzf(nzf):
    try:
        ASSEMBLER.process(nzf)
    except NameError:
        logging.exception("[%s] Error accessing ASSEMBLER?", __NAME__)
        
def disconnect():
    try:
        DOWNLOADER.disconnect()
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
        
def paused():
    try:
        return DOWNLOADER.paused
    except NameError:
        logging.exception("[%s] Error accessing DOWNLOADER?", __NAME__)
        
################################################################################
# Help functions                                                               #
################################################################################
def get_new_id(prefix):
    try:
        fd, l = tempfile.mkstemp('', 'SABnzbd_%s_' % prefix, CACHE_DIR)
        os.close(fd)
        head, tail = os.path.split(l)
        return tail
    except:
        logging.exception("[%s] Failure in tempfile.mkstemp", __NAME__)
        
def save_data(data, _id, do_pickle = True):
    path = os.path.join(CACHE_DIR, _id)
    logging.info("[%s] Saving data for %s in %s", __NAME__, _id, path)
    
    try:
        _f = open(path, 'wb')
        if do_pickle:
            cPickle.dump(data, _f, 2)
        else:
            _f.write(data)
        _f.flush()
        _f.close()
    except:
        logging.exception("[%s] Saving %s failed", __NAME__, path)
    
def load_data(_id, remove = True, do_pickle = True):
    path = os.path.join(CACHE_DIR, _id)
    logging.info("[%s] Loading data for %s from %s", __NAME__, _id, path)
    
    if not os.path.exists(path):
        logging.info("[%s] %s missing", __NAME__, path)
        return None
    
    data = None
    
    try:
        _f = open(path, 'rb')
        if do_pickle:
            data = cPickle.load(_f)
        else:
            data = _f.read()
        _f.close()
        
        if remove:
            remove_data(_id)
    except:
        logging.exception("[%s] Loading %s failed", __NAME__, path)
        
    return data
    
def remove_data(_id):
    path = os.path.join(CACHE_DIR, _id)
    try:
        os.remove(path)
        logging.info("[%s] %s removed", __NAME__, path)
    except:
        pass
        
def check_for_latest_version():
    try:
        import urllib
        
        fn = urllib.urlretrieve('http://sabnzbd.sourceforge.net/sa')[0]
        
        f = open(fn, 'r')
        data = f.read()
        f.close()
        
        latest = data.split()[0]
        
        return (latest, latest == __version__)
        
    except:
        pass
    
################################################################################
# Misc                                                                         #
################################################################################
def pp_to_opts(pp):
    repair, unpack, delete = (False, False, False)
    if pp > 0:
        repair = True
        if pp > 1:
            unpack = True
            if pp > 2:
                delete = True
    return (repair, unpack, delete)
    
def system_shutdown():
    logging.info("[%s] Performing system shutdown", __NAME__)
    try:
        import win32security
        import win32api
        import ntsecuritycon
        
        flags = ntsecuritycon.TOKEN_ADJUST_PRIVILEGES | ntsecuritycon.TOKEN_QUERY
        htoken = win32security.OpenProcessToken(win32api.GetCurrentProcess(), flags)
        id = win32security.LookupPrivilegeValue(None, ntsecuritycon.SE_SHUTDOWN_NAME)
        newPrivileges = [(id, ntsecuritycon.SE_PRIVILEGE_ENABLED)]
        win32security.AdjustTokenPrivileges(htoken, 0, newPrivileges)
        win32api.InitiateSystemShutdown("", "", 30, 1, 0)
    finally:
        os._exit(0)
        
PROPER_FILENAME_MATCHER = re.compile(r"[a-zA-Z0-9\-_\.+\(\)]")
def fix_filename(filename):
    cst = []
    for i in xrange(len(filename)):
        if PROPER_FILENAME_MATCHER.search(filename[i]):
            cst.append(filename[i])
        else:
            cst.append("_")
    filename = ''.join(cst)
    return filename
    
################################################################################
# SCHED                                                                        #
################################################################################
RSSTASK_MINUTE = random.randint(0, 59)

def init_SCHED(schedlines, need_rsstask = False):
    global SCHED
    
    if schedlines or need_rsstask:
        SCHED = ThreadedScheduler()
        
        for schedule in schedlines:
            m, h, d, action_name = schedule.split()
            m = int(m)
            h = int(h)
            if d == '*':
                d = range(1, 8)
            else:
                d = [int(d)]
                
            if action_name == 'resume':
                action = resume_downloader
            elif action_name == 'pause':
                action = pause_downloader
            else:
                logging.info("[%s] Unknown action: %s", __NAME__, ACTION) 
                
            SCHED.addDaytimeTask(action, '', d, None, (h, m), 
                                 SCHED.PM_SEQUENTIAL, [])
                                 
        if need_rsstask:
            d = range(1, 8)
            
            m = RSSTASK_MINUTE
            logging.debug("[%s] RSSTASK_MINUTE: %s", __NAME__, m)
            
            for h in range(24):
                SCHED.addDaytimeTask(RSS.run, '', d, None, (h, m), 
                                     SCHED.PM_SEQUENTIAL, [])
################################################################################
# RSS                                                                          #
################################################################################
def init_RSS():
    global RSS
    
    need_rsstask = False
    
    if sabnzbd.rss.HAVE_FEEDPARSER:
        tup = load_data(RSS_FILE_NAME, remove = False)
        
        uris = []
        uri_table = {}
        old_entries = {}
        if tup:
            uris, uri_table, old_entries = tup
            
        RSS = RSSQueue(uris, uri_table, old_entries)
        if uris:
            need_rsstask = True
            
    return need_rsstask
            
def add_rss_feed(uri, text_filter, re_filter, unpack_opts, match_multiple):
    if RSS:
        RSS.add_feed(uri, text_filter, re_filter, unpack_opts, match_multiple)
        
def del_rss_feed(uri_id):
    if RSS:
        RSS.del_feed(uri_id)
        
def del_rss_filter(uri_id, filter_id):
    if RSS:
        RSS.del_filter(uri_id, filter_id)
        
def get_rss_info():
    if RSS:
        return RSS.get_info()
