// Description:
//   Block View GL
//
// Copyright (C) 2003 Frank Becker
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation;  either version 2 of the License,  or (at your option) any  later
// version.
//
// This program is distributed in the hope that it will be useful,  but  WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details
//
#include "SDL.h"
#include <gl++.hpp>

#include <Trace.hpp>
#include <Point.hpp>
#include <Config.hpp>
#include <PNG.hpp>

#include "BlockViewGL.hpp"

const int VIDEO_DEFAULT_WIDTH=640;
const int VIDEO_DEFAULT_HEIGHT=480;

BlockViewGL::BlockViewGL( BlockModel &model):
    BlockViewSmooth(model),
    _isFullscreen( false),
    _bpp(0),
    _width( VIDEO_DEFAULT_WIDTH),
    _height( VIDEO_DEFAULT_HEIGHT)
{
    XTRACE();
}

BlockViewGL::~BlockViewGL()
{
    XTRACE();
}

bool BlockViewGL::init( void)
{
    if( SDL_InitSubSystem( SDL_INIT_VIDEO) < 0 )
    {
	LOG_ERROR << "Init Video: failed # " << SDL_GetError() << endl;
	return false;
    }
    LOG_INFO << "Video: OK" << endl;

//    SDL_EnableKeyRepeat(250, 250);

    ConfigS::instance()->getBoolean( "fullscreen", _isFullscreen);

    int videoFlags = SDL_OPENGL;
    if( _isFullscreen)
    {
	LOG_INFO << "Fullscreen request." << endl;
	videoFlags |= SDL_FULLSCREEN;
    }

    ConfigS::instance()->getInteger( "width", _width);
    ConfigS::instance()->getInteger( "height", _height);

    if( ! ::init("libGL.so.1"))
    {
	LOG_ERROR << "SDL Error: " << SDL_GetError() << endl;
	SDL_QuitSubSystem( SDL_INIT_VIDEO);
	return false;
    }

    if( SDL_SetVideoMode( _width, _height, _bpp, videoFlags ) == NULL )
    {
	LOG_ERROR << "Video Mode: failed #" << SDL_GetError() << endl;
	SDL_QuitSubSystem( SDL_INIT_VIDEO);
	return false;
    }

    //set title and icon name
    SDL_WM_SetCaption( "Shaaft OpenGL", "Shaaft GL" );

    SDL_Surface *surf = SDL_GetVideoSurface();

    LOG_INFO << "Video Mode: OK ("
	<< surf->w << "x"
	<< surf->h << "x"
	<< (int)surf->format->BitsPerPixel << ")" << endl;

    LOG_INFO << "OpenGL info follows..." << endl;
    LOG_INFO << "  Vendor  : " << glGetString( GL_VENDOR) << endl;
    LOG_INFO << "  Renderer: " <<  glGetString( GL_RENDERER) << endl;
    LOG_INFO << "  Version : " << glGetString( GL_VERSION) << endl;

    return true;
}

void BlockViewGL::close( void)
{
    SDL_QuitSubSystem( SDL_INIT_VIDEO);
    SDL_Quit();
}

void BlockViewGL::snapshot( void)
{
    static int count = 0;

    char filename[128];
    sprintf( filename, "snap%02d.png", count++);
    SDL_Surface *img;

    img = SDL_CreateRGBSurface( SDL_SWSURFACE, _width, _height, 24,
	    0xFF000000, 0x00FF0000, 0x0000FF00,0);

    glReadPixels( 0, 0, _width, _height, GL_RGB, GL_UNSIGNED_BYTE, img->pixels);

    LOG_INFO << "Writing snapshot: " << filename << endl;
    if( !PNG::Snapshot( img, filename))
    {
	LOG_ERROR << "Failed to save snapshot." << endl;
    }

    SDL_FreeSurface( img);
}
