/* File:      compile.P
** Author(s): Saumya Debray, Kostis Sagonas, Terrance Swift
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: compile.P,v 1.17 2002/06/03 16:27:13 dwarren Exp $
** 
*/


:- compiler_options([sysmod,optimize,spec_repr,xpp_on]).
#include "flag_defs_xsb.h"
#include "extensions_xsb.h"



mc(Modname, Options) :- compile(Modname, Modname, Options, _).

/*======================================================================*/
/* compile(+ModName, +BaseName, +CompilerOptions, -ExportedPredicates)	*/
/*	Compiles a module with the compiler options specified and	*/
/*	returns the list of exported predicates.			*/
/*======================================================================*/

%% For better warnings, we need to know the currently compiled module
:- dynamic(current_compiled_module(_)).

compile(Modname, Basename, Options, ExportList) :-
	telling(OldFile),
	file_names(Basename, PFile, HFile, OFile, AFile, DFile,
		   SpecFile, TIFile, TableFile, OptFile),
	setoptions(Options), initialize,
	%% If --quietload, don't print Compiling/compiled messages
	stat_flag(BANNER_CTL,BannerCtl),
	(   KeepQuiet is BannerCtl mod QUIETLOAD, KeepQuiet =:= 0
	->  true
	;
	    message(('[Compiling ', Basename, ']'))
	),
	cputime(TimeStart),
	retractall(current_compiled_module(_)),
	assert(current_compiled_module(Basename)),
	db_preprocess(PFile, HFile, OptFile, SourceFile),
	parse(Modname, SourceFile, HFile, Module_0),
	( option(profile) -> 
	    time_message(TimeStart, 'Parse finita'), cputime(TimeParse)
	; TimeParse = TimeParse ), % To avoid a silly lamp variable message!
	abort_if_syntax_errors(Modname), !,
	useinfer(Module_0, IsModule, ExportList), !,
	( option(profile) -> 
	    time_message(TimeParse, 'Useinfer finita'), cputime(TimeUse)
	; TimeUse = TimeUse ), % To avoid a silly lamp variable message!
	modeinfer(Modname, Module_0, DFile),
	specialise(Module_0, Module0, SpecFile),
	table_process(Module0, Module, TableFile),
	hack_self_compile,
	get_symtab(Module, SymTab),
	compile1(Modname, Module, SymTab, AFile, OFile, TIFile, IsModule),
	dispose_symtab( SymTab ),
	( option(compile_off) -> print_result(Module) ; true ),
	( conget('error occurred', 1) ->
	    sys_unlink(OFile,_), 
	    message(('The file ', OFile, ' is removed')), abort
	; true
	), !,
	( IsModule =:= 0 -> IsModuleMsg = '' ; IsModuleMsg = 'Module ' ),
	%% If --quietload, don't print Compiling/compiled messages
	stat_flag(BANNER_CTL,BannerCtl1),
	(   KeepQuiet1 is BannerCtl1 mod QUIETLOAD, KeepQuiet1 =:= 0
	->  true
	;   time_message(TimeStart, (IsModuleMsg, Modname, ' compiled'))
	),
	!,
	close_ti_file(TIFile),
	retractall(current_compiled_module(_)),
	tell(OldFile).


file_names(Basename, PFile, HFile, OFile, AFile, DFile,
	   SpecFile, TIFile, TableFile, OptFile) :-
	str_cat(Basename,XSB_SRC_EXTENSION_ATOM,PFile),
	str_cat(Basename,XSB_HDR_EXTENSION_ATOM,HFile),
	str_cat(Basename,XSB_OBJ_EXTENSION_ATOM,OFile),
	str_cat(Basename,XSB_ASM_EXTENSION_ATOM,AFile),
	str_cat(Basename,XSB_DCL_EXTENSION_ATOM,DFile),
	str_cat(Basename, '.spec', SpecFile),
	str_cat(Basename, '.ti', TIFile),
	str_cat(Basename, '.table', TableFile),
	str_cat(PFile, '.opt', OptFile).

initialize :-
	conset('table #', 0),
	conset('syntax errors', 0),
	conset('error occurred', 0),
	conset(ti_all, 0), conset(ti_all_off, 0),
	conset('EDB Dep #', 2),
	prepare.   /* uses globalset here! */

abort_if_syntax_errors(Modname) :-
	conget('syntax errors', N),
	( N =\= 0, Modname \== user -> 
	    %% Forget about syntax errors when compiling module user
	    error(('Aborting compilation of module ', 
		   Modname, ' due to ', N, ' syntax error(s).')), 
	    abort
	;   true
	).

/*======================================================================*/
/* db_preprocess(+PFile, +HFile, +OptFile, -SourceFile)	*/
/*	Calls the database optimizer of XSB as a preprocessor that	*/
/*	produces the file OptFile out of PFile and HFile.		*/
/*======================================================================*/

db_preprocess(PFile, HFile, OptFile, SourceFile) :-
	(option(db_opt)
	 ->	(option(unfold),option(reorder)
		 ->	DB_Opt_Options = [unfold,reorder]
		 ; option(unfold)
		 ->	DB_Opt_Options = [unfold]
		 ; option(reorder)
		 ->	DB_Opt_Options = [reorder]
		 ;	DB_Opt_Options = []
		),
		db_optimize(PFile, HFile, DB_Opt_Options, OptFile),
		SourceFile = OptFile
	 ;	SourceFile = PFile
	).

/*======================================================================*/
/* table_process(+ModuleIn, -ModuleOut, +SFile)				*/
/*	Performs necessary module-level analyses for tabling.		*/
/*======================================================================*/

table_process(ModuleIn, ModuleOut, SFile) :-
	( option(auto_table) -> perform_auto_table(ModuleIn) ; true ),
	( option(suppl_table) ->
		supplementary_tabling(ModuleIn, ModuleMid)
	; ModuleMid = ModuleIn
	),
	conget('table #', Tables),
	( Tables =:= 0 -> ModuleOut = ModuleMid
	; cp_opt(ModuleMid, ModuleOut)
          %%,( option(no_check) -> true ; build_graph(ModuleOut) )
	),
	( option(table_dump) -> dump_module(SFile, ModuleOut) ; true ).

/*======================================================================*/
/* specialise(+ModuleIn, -ModuleOut, +SpecFile)				*/
/*	Performs the specialisation of partially instantiated calls.	*/
/*======================================================================*/

specialise(ModuleIn, ModuleOut, SpecFile) :-
	( option(spec_off) -> ModuleIn = ModuleOut
	; spec(ModuleIn, ModuleOut),
	  ( option(spec_dump) -> dump_module(SpecFile, ModuleOut) ; true )
	).

/*======================================================================*/
/* modeinfer(+ModuleIn)							*/
/*	Infer modes using the mode analyzer.				*/
/*======================================================================*/

modeinfer(ModName, Module, SymTab) :-
	( option(modeinfer) -> modeanalyze(ModName, Module, SymTab)
	; true).

/*======================================================================*/
/*======================================================================*/

myclose(F) :- tell(F), told.

compile1(Modname, Module, SymTab, AFile, OFile, TIFile, IsModule) :-
      ( option(compile_off) ->		%  generate *.A file
	  write_asmheader(AFile, Modname, SymTab, IsModule), myclose(AFile)
      ; conget('error occurred', 1) ->
	  error(('Aborting compilation of module ',Modname,' due to errors')),
	  abort
      ; option(asm_dump) -> 
	  write_asmheader(AFile,Modname,SymTab, IsModule), % generate .A header
	  asm_genheader(OFile, Modname, SymTab, IsModule), % generate OBJ header
	  compile0(Module, 1, AFile, OFile, TIFile, IsModule),
	  myclose(AFile), myclose(OFile)
      ; asm_genheader(OFile, Modname, SymTab, IsModule),   % generate OBJ header
	compile0(Module, 0, AFile, OFile, TIFile, IsModule),
	myclose(OFile)
      ).

/*======================================================================*/
/* compile0(+Module, +GenAsm, +AFile, +OFile, +IsModule)		*/
/*	Compiles a module in the internal form.				*/
/*======================================================================*/

compile0(module(SymTab,_,ClList,[MainPred]),GenAsm,AFile,OFile,TIFile,M) :-
	member(PredBlk, [MainPred|ClList]),
	compile_predicate(PredBlk, GenAsm, AFile, OFile, TIFile, M, SymTab),
	fail.	/* just reclaim space */
compile0(_, _, _, _, _, _).


/*======================================================================*/
/* compile_predicate(+pred(Sym,Clauses,_), +GenAsm,	 		*/
/*		     +AFile, +OFile, +IsModule, +SymTab)		*/
/*	Compile the clauses of a single predicate into an assembly	*/
/*	(*.A) file and/or an object file.				*/
/*======================================================================*/

compile_predicate(PredDef, GenAsm, AFile, OFile, TIFile, IsModule, SymTab) :-
	PredDef = pred(Sym,_,_),
	sym_name(Sym, P, N),
	( option(verbo) -> message(('        ....compiling ', P/N)) ; true ),
	( option(sysmod) -> true
	; standard_symbol(P, N, _) ->
		warning(('Redefining the standard predicate: ', P, '/', N))
	; true
	),
	( sym_prop(im(Mod), Sym) ->
		error(('Cannot redefine ', P, '/', N,
		' (already declared imported from ', Mod, '!)'))
	; true
	),
	preprocess(PredDef, PredBlks, SymTab),
	!,	% needed to cut unnecessary choice points... (bmd & kostis)
	ti_transform(PredBlks, TIPredBlks, TIFile, SymTab),
	!,	% needed to cut unnecessary choice points... (bmd & kostis)
	compile_pred_block(TIPredBlks, PilCode-[], 0, NC),
	!,	% needed to cut unnecessary choice points... (bmd & kostis)
	compile_peephole_opt(PilCode, PilCode1),
	( GenAsm =:= 0 -> true; tell(AFile), write_asm_instructions(PilCode1) ),
	tell(OFile), asm_PIL(Sym, PilCode1, NC, IsModule),
	!.

ti_transform(PredBlks, TIPredBlks, TIFile, SymTab) :-
	ti_pred_block(PredBlks, TIPredBlks, SymTab),
	( option(ti_dump) ->
		telling(F),
		tell(TIFile),
		write_preds(TIPredBlks),
		told,
		tell(F)
	; true
	).

close_ti_file(TIFile) :-
	( option(ti_dump) ->
		telling(F), tell(TIFile), told, tell(F)
	; true
	).

compile_pred_block([], PilCode-PilCode, NCin, NCin).
compile_pred_block([PredDef|PRest], PilCode-PilCodeR, NCin, NCout) :-
	compile_predicate2(PredDef, PilCode-PilCodeM, NClauses),
	NCmid is NCin + NClauses,
	!,
	compile_pred_block(PRest, PilCodeM-PilCodeR, NCmid, NCout).

/*======================================================================*/
/* compile_predicate2(+pred(Sym,Clauses,Prag), -PilCode-PilCodeR, -NC)	*/
/*	Compiles the clauses of a single predicate after preprocessing.	*/
/*	A single predicate in the source level can be transformed by	*/
/*	preprocess/3 into several predicates.  compile_predicate2/2	*/
/*	does the compilation for one of them.  The result is a list of	*/
/*	PIL code in assembler format.  The predicate also returns the	*/
/*	number of clauses NC (length of the Clauses list).		*/
/*======================================================================*/

compile_predicate2(pred(Sym,Clauses,Prag), PilCode-PilCodeR, NC) :-
	gennum(L),
	PilCode = [label(pred,(Sym,L)),test_heap(Arity,Margin)|PilCode1],
	sym_name(Sym,_,Arity), Margin = 2000, % Margin is temporary
	find_if_tabled(Sym, Tabled),
	makeindex(Sym, Clauses, PilCode1, PilCodeM, Prag, Tabled),
	compile_clauses(Clauses, Sym, PilCodeM, PilCodeR, 0, NC).

find_if_tabled(Sym, tabled(Tabind,Ar)):-
	sym_type(Sym, tabled(Tabind,Ar)),
	!. 
find_if_tabled(_, 0). 

/*======================================================================*/

compile_clauses([], _, PilCode, PilCode, Nin, Nin).
compile_clauses([Clause|Clauses], Sym, PilCode, PilCodeR, Nin, Nout) :-
	Nmid is Nin + 1,  
	compile_clause(Sym, Clause, PilCode, PilCodeM, Nmid),
	compile_clauses(Clauses, Sym, PilCodeM, PilCodeR, Nmid, Nout).


/*======================================================================*/
/* hack_self_compile/0							*/
/*	Forces all the modules that are used by the XSB compiler to	*/
/*	be loaded into the system before overwriting the OBJ file when	*/
/*	self-compiling the module (or the whole system).		*/
/*======================================================================*/

hack_self_compile :-
	force_tprog_loaded,
	force_inprog_loaded,
	force_tp_var_loaded,
	force_flatten_loaded,
	force_varproc_loaded,
	force_tp_eval_loaded,
	force_tp_comp_loaded,
	force_tp_cond_loaded,
	force_tp_goal_loaded,
	force_asm_opt_loaded,
	force_tp_index_loaded,
	force_listutil_loaded,
	force_peephole_loaded,
	force_asm_inst_loaded,
	force_asm_pass2_loaded,
	force_preprocess_loaded,
	force_tp_geninline_loaded,
	force_ti_mod_loaded.


