/* File:      varproc.P
** Author(s): Jiyang Xu, Terrance Swift, Kostis Sagonas
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: varproc.P,v 1.4 1999/11/20 06:51:43 kifer Exp $
** 
*/


% :- import write/1, writeln/1, telling/1, told/0, tell/1 from standard.

varproc(VarList, VarTab, PvarCount) :-
	get_lastchunk(VarTab, LastChunk),
	closetail(LastChunk), !,
	mark_occurrence(VarList, VarTab), !,
	mark_type(VarList, VarTab, LastChunk), !,
	alloc_vars(VarList, VarTab, 0, PvarCount).
% ,
% ( option(asm_dump) -> telling(X), tell(user),
% 	write('VarList = '), writeln(VarList),
% 	write('VarTab = '), writeln(VarTab),
% 	write('LastChunk = '), writeln(LastChunk),
% 	told, tell(X)
% ; true
% ).


/*======================================================================*/
/* get_lastchunk(+VarTab, -LastChunks)					*/
/*	Get the chunk id(s) of the last chunk(s).			*/
/*		-LastChunks: is an open ended-list.			*/
/*======================================================================*/

get_lastchunk([One|Rest], LastChunk) :-
	( Rest == [] -> get_lastchunk1(One, LastChunk)
	; get_lastchunk(Rest, LastChunk)
	).

get_lastchunk1(goal(CkId,_,_,_), LastChunk) :- memberchk(CkId, LastChunk).
get_lastchunk1(inlinegoal(CkId,_,_,_), LastChunk) :- memberchk(CkId, LastChunk).
get_lastchunk1(head(_VOList), LastChunk) :- memberchk(1, LastChunk).
get_lastchunk1(branch(A,B), LastChunk) :- 
	get_lastchunk(A, LastChunk),
        get_lastchunk(B, LastChunk).
get_lastchunk1(ifthenelse(_,B,C), LastChunk) :- 
        get_lastchunk(B, LastChunk),
        get_lastchunk(C, LastChunk).


/*======================================================================*/
/* mark_occurrence(+VarList, #VarTab)		        		*/
/*	Fills in occurrence information (f, s, l, or v).		*/
/*	      +VarList is a list of variable(Vid,VPrag) structures	*/
/*		       where Vprag is uninstantiated yet.		*/
/*	      #VarTab  has the occurrence attribute of the		*/
/*		       varocc(Vid,vrec()) structures marked		*/
/*		       after the exit from this predicate.		*/
/*======================================================================*/

mark_occurrence([], _).
mark_occurrence([variable(Vid,_VCPrag)|Rest], VarTab) :-
	mark_occ1(VarTab, 0, _, pr(0,_), LastOcc, Vid),
	mark_last(LastOcc),
	mark_occurrence(Rest, VarTab).

/*======================================================================*/
/* mark_occ1(#VarTab, +CountIn, -CountOut,				*/
/*	     +PrevOcc, -LastOcc, -FirstOcc)			        */
/*	Fills in occurrence information (f, s, l, or v).		*/
/*======================================================================*/

mark_occ1([], N, N, Pre, Pre, _).
mark_occ1([One|Rest], N0, N, Pre, Last, Vid) :-
	mark_occ2(One, N0, N1, Pre, Med, Vid),
	!,	% needed to cut unnecessary choice points... (bmd & kostis)
	mark_occ1(Rest, N1, N, Med, Last, Vid).

mark_occ2(goal(_,_,_,VOList), N0, N, Pre, Last, Vid) :-
	mark_occ3(VOList, N0, N, Pre, Last, Vid).
mark_occ2(head(VOList), N0, N, Pre, Last, Vid) :-
	mark_occ3(VOList, N0, N, Pre, Last, Vid).
mark_occ2(inlinegoal(_,F,_,VOList), N0, N, Pre, Last, Vid) :-
	( F == is ->			% is/2 needs special treatment due to
		VOList = [LHS|RHS],	%      its "reversed" directionality
		mark_occ3(RHS, N0, N1, Pre, L1, Vid),
		mark_occ3([LHS], N1, N, L1, Last, Vid)
	; mark_occ3(VOList, N0, N, Pre, Last, Vid)
	).
mark_occ2(branch(A,B), N0, N, Pre, Last, Vid) :-
	mark_occ1(A, N0, N1, Pre, Last1, Vid),
	mark_occ1(B, N0, N2, Pre, Last2, Vid),
	( N1 > N0, N2 > N0 -> N = N1, Last = br(Last1,Last2)
	; N1 > N0, N2 =:= N0 -> N = N1, 
		( N0 >= 1, Last = Last1
		; N0 <  1, Last = br(Last1, no(Vid))
		)
	; N1 =:= N0, N2 > N0 -> N = N2,
		( N0 >= 1, Last = Last2
		; N0 <  1, Last = br(no(Vid), Last2)
		)
	; % N1=N0, N2=N0,
		N = N0, Last = Pre
	).
mark_occ2(ifthenelse(A,B,C), N0, N, Pre, Last, Vid) :-
	mark_occ1(A, N0, N1, Pre, Med, Vid),
	( C = [inlinegoal(_,fail,0,_)] ->
		mark_occ1(B, N1, N, Med, Last, Vid)
		% fail branch can contain no lamp vars, so this
	; mark_occ2(branch(B,C), N1, N, Med, Last, Vid)
	).

mark_occ3([], N, N, Pre, Pre, _).
mark_occ3([varocc(Vid0,Prag)|Rest], N0, N, Pre, Last, Vid) :-
	( Vid0 = Vid ->
		mark_previous(Pre), N1 is N0 + 1, Med = pr(N1, Prag)
	; N1 = N0, Med = Pre
	),
	mark_occ3(Rest, N1, N, Med, Last, Vid).

mark_previous(pr(OccNum,Prag)) :-
	( OccNum =:= 0 -> true
	; OccNum =:= 1 -> vprag_occ(Prag, f)
	; vprag_occ(Prag, s)
	).
mark_previous(br(Pre1, Pre2)) :-
	mark_previous(Pre1),
	mark_previous(Pre2).
mark_previous(no(_Vid)).

mark_last(pr(OccNum, Prag)) :-
	( OccNum =:= 0 -> true
	; OccNum =:= 1 -> vprag_occ(Prag, v)
	; vprag_occ(Prag, l)
	).
mark_last(br(Pre1, Pre2)) :-
	mark_last(Pre1),
	mark_last(Pre2).
mark_last(no(_Vid)).


/*======================================================================*/
/* mark_type(#VarList, #VarTab, +LastChunk)				*/
/*	Computes and fills in the type of variables.			*/
/*	      #VarList is a list of variable(Vid,VPrag) structures	*/
/*		       where Vprag is uninstantiated on entry (gets	*/
/*		       instantiated by this predicate.			*/
/*	      #VarTab  has the type attribute of the varocc(Vid,vrec())	*/
/*		       structures marked after the exit from this	*/
/*		       predicate in the way described below.		*/
/*									*/
/*   The types of variables filled in can be:				*/
/*	vh - the void variable in the head top level;			*/
/*	 t - when the both following conditions hold:			*/
/*	    * first occur in head, in structure, or in the last chunk	*/
/*	    * does not occur in more than one chunk			*/
/*		(head is considered part of the first chunk)		*/
/*	    I improved the scheme a little bit, now if for every	*/
/*	    indepedant occurrence blocks, above two condition hold, OK.	*/
/*	 p - otherwise							*/
/*======================================================================*/

mark_type([], _, _).
mark_type([variable(Vid,v(Type,_Loc,Use,_NoUse))|Rest], VarTab, LastChunk) :-
	mark_type1(VarTab, Chunks, LastChunk, Safe, Vid),
	length(Chunks, L),
	( L > 1 -> Type = p
	; Type = t
	; true			% Type is vh or p but already determined
	),
	( Type == p -> ( var(Safe) -> Use = d ; Use = u ) ; true ),
	mark_type(Rest, VarTab, LastChunk).

mark_type1([], _, _, _, _).
mark_type1([One|Rest], Chunks, LastChunk, Safe, Vid) :-
	mark_type2(One,  Chunks, LastChunk, Safe, Vid),
	!,	% needed to cut unnecessary choice points... (bmd & kostis)
	mark_type1(Rest, Chunks, LastChunk, Safe, Vid).

mark_type2(head(VOList), Chunks, LastChunk, Safe, Vid) :-
	mark_seen(VOList, 1, Chunks, LastChunk, Safe, Vid).
mark_type2(goal(CkId,_,_,VOList), Chunks, LastChunk, Safe, Vid) :-
	mark_seen(VOList, CkId, Chunks, LastChunk, Safe, Vid).
mark_type2(inlinegoal(CkId,_,_,VOList), Chunks, LastChunk, Safe, Vid) :-
	mark_seen(VOList, CkId, Chunks, LastChunk, Safe, Vid).
mark_type2(branch(A,B), Chunks, LastChunk, Safe, Vid) :-
	mark_type1(A, Chunks, LastChunk, Safe, Vid),
	mark_type1(B, Chunks, LastChunk, Safe, Vid).
mark_type2(ifthenelse(A,B,C), Chunks, LastChunk, Safe, Vid) :-
	mark_type1(A, Chunks, LastChunk, Safe, Vid),
	mark_type1(B, Chunks, LastChunk, Safe, Vid),
	mark_type1(C, Chunks, LastChunk, Safe, Vid).

mark_seen(VOList, CkId, Chunks, LastChunk, Safe, Vid) :-
	mark_seen1(VOList, CkId, Seen, LastChunk, Safe, Vid),
	( var(Seen) -> true ; memberchk(CkId, Chunks) ).

mark_seen1([], _, _, _, _, _).
mark_seen1([varocc(Vid0,Prag)|Rest], CkId, Seen, LastChunk, Safe, Vid) :-
    ( Vid0 = Vid -> 
	Seen = 1,	/* just bind Seen to a constant */
	( member(CkId, LastChunk) -> vprag_lastchunk(Prag, 1)
	; vprag_lastchunk(Prag, 0)
	),
	vprag_occ(Prag, Occ),
	( Occ == f ->
	  % determine if the variable is safe when pval in the last goal
	  ( ( vprag_context(Prag, b) ; vprag_context(Prag, i) ),
	    vprag_level(Prag, t) -> Safe = 0 /* unsafe */
	  ; true
	  ),
	  % determine if the variable must be permanent
	  ( ( vprag_context(Prag, h) 
	    ; vprag_level(Prag,s)
	    ; member(CkId,LastChunk) ) -> true
	  ; vprag_type(Prag, p)
	  )
	; Occ == v ->	% determine if the variable can be typed "vh"
		( vprag_context(Prag, h), vprag_level(Prag, t) -> 
			vprag_type(Prag, vh)
		; true
		)
	; true
	)
    ; true
    ),
    mark_seen1(Rest, CkId, Seen, LastChunk, Safe, Vid).


/*======================================================================*/
/* alloc_vars(#VarList, #VarTab, +PvarCountIn, -PvarCountOut)		*/
/*	Allocates locations for permanent variables, and computes	*/
/*	"use" and "nouse" locations for temporary variables.  The	*/
/*	location attribute of all temporary variables remains		*/
/*	uninstantiated.							*/
/*======================================================================*/

alloc_vars([], _VarTab, PV, PV).
alloc_vars([variable(Vid,v(VT,Loc,Use,NoUse))|VarList], VarTab, PVin, PVout) :-
    ( VT == p -> PVmed is PVin+1, Loc is PVmed+1 /* first pvar has offset 2 */
    ; VT == vh -> PVmed = PVin, Loc = 0
    ; PVmed = PVin,
      search_tvar(VarTab, OccList, LGArity, Vid),
      compute_use1(OccList, HUse0, [], BUse0, [], NoUse0, [], OccInChunk, Vid),
      ( var(OccInChunk) -> ChkArg = 0 ; ChkArg = LGArity ),
      /* Conflict set = ChkArg - BUse0 */
      conflict_avoid(BUse0, HUse0, NoUse0, Use, NoUse, ChkArg)
    ),
    !, % needed to cut unnecessary choice points... (bmd & kostis)
    alloc_vars(VarList, VarTab, PVmed, PVout).


/*======================================================================*/
/*======================================================================*/

search_tvar([One|Rest], OccList, Arity, Vid) :-
	'search tvar'(One, OccList, Arity, Vid, Rest).
search_tvar([_One|Rest], OccList, Arity, Vid) :-
	search_tvar(Rest, OccList, Arity, Vid).

'search tvar'(goal(_,_,Arity,VOList), OccList, Arity, Vid, _) :-
	search_occ(VOList, Vid), OccList = VOList.
'search tvar'(head(VOList), OccList, Arity, Vid, Rest) :-
	search_occ(VOList, Vid), 
	get_OccList(Rest, VOList, OccList, Arity).
'search tvar'(inlinegoal(_,_,_,VOList), OccList, Arity, Vid, Rest) :-
	search_occ(VOList, Vid), 
	get_OccList(Rest, VOList, OccList, Arity).
'search tvar'(ifthenelse(A,B,C), OccList, Arity, Vid, _) :-
	( search_tvar(A, OccList, Arity, Vid)
	; search_tvar(B, OccList, Arity, Vid)
	; search_tvar(C, OccList, Arity, Vid)
	).
'search tvar'(branch(A,B), OccList, Arity, Vid, _) :-
	( search_tvar(A, OccList, Arity, Vid)
	; search_tvar(B, OccList, Arity, Vid)
	).

search_occ([varocc(Vid,_Prag)|_Rest], Vid).
search_occ([_|Rest], Vid) :- search_occ(Rest, Vid).

get_OccList([], VOList, VOList, 0).
get_OccList([One|Rest], VOListIn, VOListOut, Arity) :-
	( One = goal(_,_,Arity,VOList) ->
		append(VOListIn, VOList, VOListOut)
	; One = inlinegoal(_,_,_,VOList) ->
		append(VOListIn, VOList, VOListMed),
		get_OccList(Rest, VOListMed, VOListOut, Arity)
	; One = ifthenelse(Test,ThenGoals,ElseGoals) -> 
		get_OccList(Test, VOListIn, TestVOList, TestArity),
		( TestArity =:= 0 -> % all tests are inlines (or variableless).
		    % In the following the VOLists from the "then" and "else"
		    % parts of the if-then-else should really be union(ed) but
		    % I guess (and hope) that appending them will not do much
		    % harm.
		     get_OccList(ThenGoals, TestVOList, ThenVOList, ThenArity),
		     get_OccList(ElseGoals, ThenVOList, VOListOut, ElseArity),
		     ( ThenArity >= ElseArity -> Arity = ThenArity 
		     ; Arity = ElseArity 
		     )
		; Arity = TestArity, VOListOut = TestVOList
		)
	; VOListOut = VOListIn, Arity = 0
	).


/*======================================================================*/
/* compute_use1(+OccList, -HUse0-HUseR, -BUse0-BUseR,			*/
/*			  -NoUse0-NoUseR, -OccInChunk, +Vid)		*/
/*	Collects information to compute "Use" and "NoUse" for temporary	*/
/*	variables.							*/
/*		-HUse0:  occurs at the top level of head.		*/
/*		-BUse0:  occurs at the top level of body		*/
/*			 (excluding inlines).				*/
/*		-NoUse0: another temporary variable occurring at the	*/
/*			 top level of the body (excluding inlines).	*/
/*======================================================================*/

compute_use1([], HUse0,HUse0, BUse0,BUse0, NoUse0,NoUse0, _, _).
compute_use1([varocc(I,Prag)|Rest],
	     HUse0,HUseR, BUse0,BUseR, NoUse0,NoUseR, OccInChunk, Vid) :-
	( I = Vid -> 
	      NoUse0=NoUseM, vprag_context(Prag, HB),
	      ( HB == b -> OccInChunk = 1 /* just bind it */ ; true ),
	      vprag_level(Prag, Level),
	      ( Level == t ->
		( HB == h ->
			vprag_argno(Prag, Arg), HUse0=[Arg|HUseM], BUse0=BUseM
		; HB == b ->
			vprag_argno(Prag, Arg), BUse0=[Arg|BUseM], HUse0=HUseM
		; HB == i ->
			HUse0=HUseM, BUse0=BUseM
		)
	      ; HUse0=HUseM, BUse0=BUseM
	      )
	; HUse0=HUseM, BUse0=BUseM,
	  ( vprag_context(Prag, b), vprag_level(Prag, t), vprag_type(Prag, t),
	    vprag_argno(Prag, Arg), NoUse0=[Arg|NoUseM]
	  ; NoUse0=NoUseM
	  )
	),
	compute_use1(Rest,HUseM,HUseR,BUseM,BUseR,NoUseM,NoUseR,OccInChunk,Vid).


/*======================================================================*/
/* conflict_avoid(+BUse0, +HUse0, +NoUse0, -Use, -NoUse, +ChkArg)	*/
/*	Computes Use and NoUse from the partial information obtained	*/
/*	by compute_use1/9.						*/
/*======================================================================*/

conflict_avoid(BUse0, HUse0, NoUse0, Use, NoUse, ChkArg) :-
	conf_real_use(HUse0, Use, BUse0, ChkArg),
	conf_nouse1(NoUse0, Use, NoUse),
	conf_nouse2(ChkArg, Use, NoUse).

conf_real_use([], Use, Use, _ChkArg).
conf_real_use([One|More], Use, UseR, ChkArg) :-
	( One =< ChkArg, Use=UseM	% conflict or already in BUse
	; member(One, UseR), Use=UseM	% alread in BUse
	; Use = [One|UseM]		% head par in front
	),
	conf_real_use(More, UseM, UseR, ChkArg).

conf_nouse1([], _, _).
conf_nouse1([One|More], Use, NoUse) :-
	( member(One, Use) ; memberchk(One, NoUse) ),
	conf_nouse1(More, Use, NoUse).

conf_nouse2(N, Use, NoUse) :-
	( N =:= 0 -> closetail(NoUse)
	; ( member2(N, Use) ; memberchk(N, NoUse) ),
	  N1 is N - 1,
	  conf_nouse2(N1, Use, NoUse)
	).


/*======================================================================*/
/*									*/
/* The format of the VPrag record is as follows:			*/
/*									*/
/* vrec(Context,Level,ArgNo,Occurrence,v(Type,Loc,Use,NoUse),LastChunk)	*/
/*									*/
/*======================================================================*/

/*------------------------------------------------------*/
/* The following information is filled by inprog/2.	*/
/*------------------------------------------------------*/

vprag_context(vrec(C,_,_,_,_,_), C).	% h - Head, i - Inline, b - Body
vprag_level(vrec(_,C,_,_,_,_), C).	% t - top level, s - second level
vprag_argno(vrec(_,_,C,_,_,_), C).

/*--------------------------------------------------------------*/
/* The following information is filled by mark_occurrence/2.	*/
/*--------------------------------------------------------------*/

vprag_occ(vrec(_,_,_,C,_,_), C).
	% f - first occ, s - subsequent occ, l - last occ, v - void
vprag_lastchunk(vrec(_,_,_,_,_,C), C).
vprag_common(vrec(_,_,_,_,C,_), C).
vprag_type(vrec(_,_,_,_,v(C,_,_,_),_), C).
	% t - Temp var, p - Permanent var, vh - void var at head

/*------------------------------------------------------*/
/* The following information is filled by alloc_vars/2.	*/
/*------------------------------------------------------*/

vprag_loc(vrec(_,_,_,_,v(_,C,_,_),_), C).
	% for temp var, the register number; for perm var, numbered from 1
vprag_use(vrec(_,_,_,_,v(_,_,C,_),_), C).
	% for temp var, the "use" list; for perm var, "d", "u" or "p"
vprag_nouse(vrec(_,_,_,_,v(_,_,_,C),_), C).
	% for temp var, the "nouse" list; for perm var, not used.

/*----------------------------------------------------------------------*/
/* The non-first occurrence of a permanent variable in the last chunk	*/
/* must be marked by either "u" or "d".  It is marked as "u" if its	*/
/* first occurrence is in the top level of a body (inline) goal; and	*/
/* is marked as "d" otherwise.						*/
/*----------------------------------------------------------------------*/

vprag_type1(Prag, T) :-
	vprag_type(Prag, T0),
	( T0 = p, vprag_lastchunk(Prag, 1),
	  ( vprag_occ(Prag, l) ; vprag_occ(Prag, s) ) -> vprag_use(Prag, T)
	; T = T0
	).


force_varproc_loaded.

