/* File:      aggregs.P
** Author(s): David S. Warren, Swift
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1993-1998
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: aggregs.P,v 1.20 2001/03/14 15:32:36 dwarren Exp $
** 
*/


/*=======================================================================

bagReduce(+UnaryPred,?Result,+BinOp,+OpId) reduces all values that
	(HiLog) UnaryPred is true of using BinOp with identity OpId.
	So, for example, bagReduce(p,Sum,plus,0) binds Sum to the sum
	of all values X such that p(X) is true (assuming a definition
	of: plus(X,Y,Z) :- Z is X+Y. and plus is declared a hilog
	symbol.) bagSum, bagCount, and bagAvg are defined using this
	meta-predicate.

bagPO(+UnaryPred,?Result,+PO) finds all the maximal elements (under
	the partial order PO) that unify with Result and that
	UnaryPred is true of.

filterReduce(?UnaryPred,+BinOp,+OpId,?Result) 
	is copied from bagReduce with two changes.  NOTE CHANGE IN
	ARGUMENT ORDER!!!
	1) it does not use HiLog.  Rather than trying to reduce Hilog
	terms of the form, e.g. p(X,Y)(Value), it assumes that BinOp
	is applied to the last argument of a non-Hilog term,
	e.g. p(X,Y,Value).  The Hilog version is more elegant, surely,
	but is only as robust as XSB's Hilog implementation.

	2) it aggregates answers using a variance rather than a
	unification. If we have two facts

	p(X)(1)
	p(a)(2)
	p(a)(3)

	or equivalently 

	p(X,1).
	p(a,2).
	p(a,3).

	and want to construct the maximum, bagReduce will return

	p(a)(3).

	while filterReduce will return 

	p(X,1).
	p(a,3).

	Clause indexing should be more efficient for filterReduce, but
	indexing of answers within the table should be more efficient for
	bagReduce.

========================================================================*/

bagMin(Call,Var) :- bagReduce(Call,Var,minimum,zzz(zzz)).

bagMax(Call,Var) :- bagReduce(Call,Var,maximum,_).

bagSum(Call,Var) :- bagReduce(Call,Var,sum,0).

bagCount(Call,Var) :- bagReduce(Call,Var,successor,0).

bagAvg(Call,Avg) :- 
	bagReduce(Call,[Sum|Count],sumcount,[0|0]),
	Avg is Sum/Count.


% Keep all lubs in a partial order (HiLog) PO(/2).
bagPO(Call,Var,PO) :- bagPO1(Call,Var,PO), fail.
bagPO(Call,Var,PO) :- bagPO1(Call,Var,PO).
:- table bagPO1/3.
:- use_variant_tabling bagPO1(_,_,_).
bagPO1(Call,Var,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,3,Skel,Cs),
	copy_term(p(Call,Var,Skel),p(Call,Ovar,Oskel)),
	Call(Var),
	((get_returns(Cs,Oskel,Leaf),(Ovar==Var -> true;PO(Var,Ovar)))
	 ->	fail
	 ;	(findall(p(Cs,Ovar,Leaf),
			 (get_returns(Cs,Oskel,Leaf), PO(Ovar,Var)),
			 Rets),
		 sort(Rets, Rets1),
		 member(p(Cs,Ovar,Leaf),Rets1),
		 delete_return(Cs,Leaf),
		 fail
		;
		 true
		)
	).

% Reduce elements using associative Op with identity Id.
bagReduce(Call,_Res,Op,Id) :- bagReduce1(Call,_,Op,Id), fail.
bagReduce(Call,Res,Op,Id) :- bagReduce1(Call,Res0,Op,Id), Res=Res0.
:- table bagReduce1/4.
:- use_variant_tabling bagReduce1(_,_,_,_).
bagReduce1(Call,Res,Op,Id) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
	copy_term(p(Call,Res,Skel),p(Call,Ovar,Oskel)),
	Call(Nvar),
	(get_returns(Cs,Oskel,Leaf)
	 ->	Op(Ovar,Nvar,Res),
		Res \== Ovar,
		delete_return(Cs,Leaf)
	 ;	Op(Id,Nvar,Res)
	).

% Define HiLog bagOf in terms of Prolog bagof. 
bagOf(Call,List) :- bagof(Var,Call(Var),List).

% Reduce elements using associative Op with identity Id.
% Template,Call,Res are like args for setof; Call may have existentials

bagReduce(Template,Call,Res,Op,Id) :-
	table_state(bagReduce1(Template,Call,Res,Op,Id),_PT,_,AnsSetStatus),
	(AnsSetStatus == complete
	 ->	bagReduce1(Template,Call,Res,Op,Id)
	 ; AnsSetStatus == undefined
	 ->	(bagReduce1(Template,Call,Res,Op,Id),fail
		 ;
		 bagReduce(Template,Call,Res,Op,Id)
		)
	 ; AnsSetStatus == incomplete
	 ->	get_ptcp(PTCP),
		get_producer_call(bagReduce1(Template,Call,Res,Op,Id),
		                  SubgPtr,_RetTmplt),
		is_incomplete(SubgPtr,PTCP),
		bagReduce1(Template,Call,Res,Op,Id)
	).

:- table bagReduce1/5.
:- use_variant_tabling bagReduce1(_,_,_,_,_).

bagReduce1(Templ,Call,Res,Op,Id) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,5,Skel,Cs),
	excess_vars(Call,Templ,[],VarList),
	copy_term(p(Call,VarList,Templ,Res,Skel),
		  p(OCall,VarList,OTempl,ORes,Oskel)),
	(atom(Op)
	 ->	Test =.. [Op,Var1,OTempl,Res]
	 ; Op = OpF/3
	 ->	Test =.. [OpF,Var1,OTempl,Res]
	 ; 	copy_term(Op,Test),
		arg(1,Test,Var1),
		arg(2,Test,OTempl),
		arg(3,Test,Res)
	),
	call(OCall),
	(get_returns(Cs,Oskel,Leaf)
	 ->	Var1 = ORes, call_c(Test), %Op(ORes,OTempl,Res),
		Res \== ORes,
		delete_return(Cs,Leaf)
	 ;	Var1 = Id, call_c(Test)   %Op(Id,OTempl,Res)
	).

% Template,Call,Res are like args for setof; Call may have existentials
bagPO(Templ,Call,_Res,PO) :- bagPO1(Templ,Call,_,PO), fail.
bagPO(Templ,Call,Res,PO) :- bagPO1(Templ,Call,Res1,PO), Res=Res1.

:- table bagPO1/4.
:- use_variant_tabling bagPO1(_,_,_,_).
bagPO1(Templ,Call,Res,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
	excess_vars(Call,Templ,[],VarList),
	copy_term(p(Call,VarList,Templ,Res,Skel),
		  p(OCall,VarList,OTempl,Ores,Oskel)),
	(atom(PO)
	 ->	POCA =.. [PO,Ores,OTempl],
		POCB =.. [PO,OTempl,Ores]
	 ; PO = PO1/2
	 ->	POCA =.. [PO1,Ores,OTempl],
		POCB =.. [PO1,OTempl,Ores]
	 ;	copy_term(PO,POCA), copy_term(PO,POCB),
		arg(1,POCA,Ores), arg(2,POCA,OTempl),
		arg(1,POCB,OTempl), arg(2,POCB,Ores)
	),
	call(OCall),
	((get_returns(Cs,Oskel,Leaf),
	  (OTempl==Ores -> true;(call_c(POCA),fail;call_c(POCA))))
	 ->	fail
	 ;	(findall(p(Cs,Leaf),
			 (get_returns(Cs,Oskel,Leaf), call_c(POCB)),
			 Rets),
		 sort(Rets, Rets1),
		 member(p(Cs,Leaf),Rets1),
		 delete_return(Cs,Leaf),
		 fail
		;
		 Res = OTempl
		)
	).


/*=======================================================================*/
/* Need to figure out a way around module system so that 
   the proper call of min is done.

filterMin(Call,Var) :- filterReduce(Call,min,neg_inf,Var).

filterMax(Call,Var) :- filterReduce(Call,max,inf,Var).

max(X,Y,Y):- \+ number(X),!.
max(X,Y,X):- \+ number(Y),!.
max(X,Y,Z):- (X > Y -> Z = X ; Z = Y).

min(X,Y,Y):- \+ number(X),!.
min(X,Y,X):- \+ number(Y),!.
min(One,Two,Min):- One > Two -> Min = Two ; Min = One.
*/

% Reduce elements using associative Op with identity Id.
filterReduce(Call,Op,Id,Res) :- filterReduce1(Call,Op,Id,Res), fail.
filterReduce(Call,Op,Id,Res) :- filterReduce1(Call,Op,Id,Res).
:- table filterReduce1/4.
:- use_variant_tabling filterReduce1(_,_,_,_).
filterReduce1(Call,Op,Id,Res) :-
	'_$savecp'(Breg),
	breg_retskel(Breg,4,Skel,Cs),
		functor(Comp,Op,3),
		functor(Idcomp,Op,3),
	copy_term(Skel,Oskel),
	apply_last(Call,[Nvar],New_call),   % rather than copy_term,
					    % since we want variant.
		arg(1,Idcomp,Id),arg(2,Idcomp,Nvar),arg(3,Idcomp,Res),
		arg(1,Comp,Oopt),arg(2,Comp,Nvar),arg(3,Comp,Res),
	get_non_opt(Skel,Sargs),	        % divide into opt arg and rest
	New_call,
%	writeln(New_call),
	(get_returns(Cs,Oskel,Leaf),
%	 writeln(get_returns(Cs,Oskel,Leaf)),
 	 get_opt_non_opt(Oskel,Osargs,Oopt),	
						% divide into opt arg and rest
%	 writeln((Sargs,Osargs)),
	 variant(Sargs,Osargs)
	 ->	
		Comp,
%		writeln(p(Comp,Res,Oopt)),
		Res \== Oopt,
		delete_return(Cs,Leaf)
	 ;	
		Idcomp	
	).

% Keep all lubs in a partial order (HiLog) PO(/2).
filterPO(Call,Var,PO) :- filterPO1(Call,Var,PO), fail.
filterPO(Call,Var,PO) :- filterPO1(Call,Var,PO).
:- table filterPO1/3.
:- use_variant_tabling filterPO1(_,_,_).
filterPO1(Call,Var,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,3,Skel,Cs),
	copy_term(p(Call,Var,Skel),p(Call,Ovar,Oskel)),
	apply_last(Call,[Var],Newcall),
	call(Newcall),
	((get_returns(Cs,Oskel,Leaf),
	    (Ovar==Var 
	    ->  true
            ;   apply_last(PO,[Ovar,Var],PO_call),
%	        writeln(p(2,PO_call)),
		call(PO_call)))
	 ->	fail
	 ;	(findall(p(Cs,Ovar,Leaf),
			 (get_returns(Cs,Oskel,Leaf), 
			  apply_last(PO,[Var,Ovar],PO_call_1),
%		          writeln(p(32,PO_call_1)),
			  call(PO_call_1)
%			  writeln(p(33,PO_call_1))
			  ),
			 Rets),
		 sort(Rets,Rets1),
		 member(p(Cs,Ovar,Leaf),Rets1),
%		 writeln(deleting(Ovar)),
		 delete_return(Cs,Leaf),
		 fail
		;
		 true
		)
	).

apply_last(Call,Var,Newcall):-
	Call =.. List,
	append(List,Var,Newlist),
	Newcall =.. Newlist.

get_opt_non_opt(Call,Args,Opt):- 
	Call =.. [_Functor|List],
	get_opt_non_opt_1(List,Args,Opt).

get_non_opt(Call,Args):- 
	Call =.. [_Functor|List],
	get_non_opt_1(List,Args).

get_non_opt_1([_H],[]):-!.
get_non_opt_1([H|T],[H|T1]):-
	get_non_opt_1(T,T1).

get_opt_non_opt_1([H],[],H):-!.
get_opt_non_opt_1([H|T],[H|T1],O):-
	get_opt_non_opt_1(T,T1,O).

%-----------------

filterPO(Call,PO) :- filterPO1(Call,PO), fail.
filterPO(Call,PO) :- filterPO1(Call,PO).

:- table filterPO1/2.
:- use_variant_tabling filterPO1(_,_).
filterPO1(Call,PO) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,2,Skel,Cs),
	copy_term(p(Call,Skel),p(Ocall,Oskel)),
	call(Call),
	((get_returns(Cs,Oskel,Leaf),
          apply_last(PO,[Ocall,Call],PO_call),
	  call(PO_call))
	 ->	fail
	 ;	(findall(p(Cs,Ocall,Leaf),
			 (get_returns(Cs,Oskel,Leaf), 
			  apply_last(PO,[Call,Ocall],PO_call_1),
			  call(PO_call_1)
			  ),
			 Rets),
		 sort(Rets,Rets1),
		 member(p(Cs,Ocall,Leaf),Rets1),
		 delete_return(Cs,Leaf),
		 fail
		;
		 true
		)
	).

%-----------------
:- table table_once/1.
:- use_variant_tabling table_once(_).
table_once(Call) :- 
	'_$savecp'(Breg),
	breg_retskel(Breg,1,Skel,Cs),
	copy_term(Skel,Oskel),
	call(Call),
	(get_returns(Cs,Oskel,_Leaf) -> fail ; true).

/*
** A built-in used in the implementation of aggregates.                   
** Takes: Breg (where the choice point was saved), 
**    	  Arity of the skeleton term to return.
** Returns: Subgoal skeleton, Ptr to a subgoal structure.
**          Subgoal skeleton is of the form ret(X,Y,Z), where X,Y,Z are
**    	    all the free variables in the subgoal.
*/

breg_retskel(_BregOffset,_Arity,_RetTerm,_SubgoalPtr) :- '_$builtin'(154).



