/* File:      storage.P  -- Primitives for backtrackable updates
** Author(s): Prasad Rao, Hasan Davulcu, kifer
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1998-2000
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: storage.P,v 1.10 2002/03/19 05:03:32 kifer Exp $
** 
*/


:- compiler_options([xpp_on,sysmod,optimize]).


#include "storage_xsb_defs.h"
#include "builtin.h"


storage_builtin(_BuiltinNumber,_StorageName,_Handle,_Snapshot,_Changed) :-
	'_$builtin'(STORAGE_BUILTIN).


%% Inserts facts. On backtracking, the fact is deleted.
storage_insert_fact_bt(StorageName,Fact,Inserted) :-	
	triehandle_for_storage(StorageName,H,Snapshot),
	trie_intern(Fact, H, Leaf, New, _),
	(New == 0
	->  Inserted=1,  % new fact inserted
	    mark_storage_changed_bt(StorageName),
	    (  true
	    ; %% On backtracking
		triehandle_for_storage(StorageName,_,NewSnapshot),
		(NewSnapshot =< Snapshot
		-> trie_unintern_nr(H, Leaf), fail 
		)
	    )
	; Inserted=0	 % fact was already there: no action
	).


%% Nonbacktrackable insert
storage_insert_fact(StorageName,Fact,Inserted) :-	
	triehandle_for_storage(StorageName,H,_),
	trie_intern(Fact, H, _Leaf, New, _),
	(New == 0
	->  Inserted=1   % new fact inserted
	; Inserted=0	 % fact was already there: no action
	).


%% Backtrackable delete.
%% Doesn't remove anything, but instead "marks" for deletion.
%% On backtracking: unmarks facts that are marked for deletion

storage_delete_fact_bt(StorageName,Fact,Deleted) :-
	triehandle_for_storage(StorageName,H,Snapshot),
	(trie_interned(Fact, H, Leaf, _)
	->  Deleted=1,   % existing fact deleted
	    mark_storage_changed_bt(StorageName),
	    ( trie_unintern_nr(H, Leaf)
	    ; %% On backtracking
		triehandle_for_storage(StorageName,_,NewSnapshot),
		(NewSnapshot =< Snapshot
		-> unmark_uninterned_nr(H, Leaf), fail
		)
	    )
	;   Deleted=0    % non-existing fact: no action
	).

%% Nonbacktrackable delete
storage_delete_fact(StorageName,Fact,Deleted) :-
	triehandle_for_storage(StorageName,H,_),
	(trie_interned(Fact, H, Leaf, _)
	->  Deleted=1,   % existing fact deleted
	    trie_unintern_nr(H, Leaf)
	;   Deleted=0    % non-existing fact: no action
	).

%% deletes the whole trie
storage_delete_all(StorageName) :- 
	triehandle_for_storage(StorageName,H,_),
	storage_builtin(DESTROY_STORAGE_HANDLE,StorageName,_,_,_),
	delete_trie(H).


%% Find fact in storage
storage_find_fact(StorageName,Fact) :-
	triehandle_for_storage(StorageName,H,_),
	trie_interned(Fact, H, _, _).



%% The folowing two functions insert key-value pairs
%% If inserting an already existing keypair, then return 0
%% If inserting a non-existing keypair with an existing key, then return -1.
%% In both cases don't insert anything.
%% If keypair is new, return 1 and insert pair(Key,Val)
storage_insert_keypair_bt(StorageName,Key, Value, Inserted):-
	triehandle_for_storage(StorageName,H,Snapshot),
	%% If this key already exists, then don't insert and return -1
	(trie_interned(pair(Key,Val), H, _Leaf, _)
	-> (Val==Value -> Inserted=0 ; Inserted = -1)
	;
	    %% Key doesn't exist
	    trie_intern(pair(Key,Value), H, Leaf_Pair, _New, _ ),
	    Inserted = 1,	% new fact: insert it
	    mark_storage_changed_bt(StorageName),
	    ( true
	    ; %% On backtracking
		triehandle_for_storage(StorageName,_,NewSnapshot),
		(NewSnapshot =< Snapshot
		->  trie_unintern_nr(H, Leaf_Pair), fail
		)
	    )
	).



%% If key exists, then delete the pair and return 1; otherwise, return 0
storage_delete_keypair_bt(StorageName,Key, Deleted):-
	triehandle_for_storage(StorageName,H,Snapshot),
	(trie_interned(pair(Key,_Value), H, Leaf, _)
	->  Deleted = 1,  % this is an existing fact: delete it
	    mark_storage_changed_bt(StorageName),
	    ( trie_unintern_nr(H, Leaf) 
	    ; %% On backtracking
		triehandle_for_storage(StorageName,_,NewSnapshot),
		(NewSnapshot =< Snapshot
		-> unmark_uninterned_nr(H, Leaf), fail 
		)
	    )
	;   Deleted = 0   % no such fact -- no action
	).



%% Like keypair_insert_bt, keypair_delete_bt, 
%% but these are not backtrackable.
storage_insert_keypair(StorageName,Key, Value, Inserted):-
	triehandle_for_storage(StorageName,H,_),
	%% If this key already exists, then don't insert and return -1
	( trie_interned(pair(Key,Val), H, _Leaf, _)
	-> (Val==Value -> Inserted=0 ; Inserted = -1)
	;
	    %% Key doesn't exist
	    trie_intern(pair(Key,Value), H, _Leaf_Pair, _New, _ ),
	    Inserted = 1	% new fact: insert it
	).


%% If key exists, then delete the pair and return 1; otherwise, return 0
storage_delete_keypair(StorageName,Key, Deleted):-
	triehandle_for_storage(StorageName,H,_),
	(trie_interned(pair(Key,_Value), H, Leaf, _)
	->  Deleted = 1,  % this is an existing fact: delete it
	    trie_unintern_nr(H, Leaf) 
	;   Deleted = 0   % no such fact -- no action
	).


storage_find_keypair(StorageName,Key,Value) :-
	triehandle_for_storage(StorageName,H,_),
	trie_interned(pair(Key,Value),H,_,_).


%% Commit changes to the storage trie associated with StorageName
%% (only if storage has been changed)
storage_commit(StorageName) :-
	( storage_builtin(INCREMENT_STORAGE_SNAPSHOT,StorageName,_,_,_),
	    !
	%% don't backtrack over it
	; fail
	).

%% Reclaims space by removing nodes from the backtrackable insert/keypair trie
%% which were marked for deletion. This should be done only at the top 
%% level of a query.
storage_reclaim_space(StorageName) :-
	triehandle_for_storage(StorageName,H,_),
	trie_reclaim_uninterned_nr(H).


%% Create a new trie or use an existing one 
%% that is already saved as a property of StorageName
triehandle_for_storage(StorageName,Handle,Snapshot) :-
	storage_builtin(GET_STORAGE_HANDLE,StorageName,Handle,Snapshot,_).


%% This one seems no longer used. Check!!!
mark_storage_changed_bt(StorageName) :-
	storage_builtin(MARK_STORAGE_CHANGED,StorageName,_,_,_).

storage_show_table_state :-
	storage_builtin(SHOW_TABLE_STATE,_StoreName,_Handle,_Snapshot,_Change).
