/* File:      wrapping.P
** Author(s): Luis Castro
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: wrapping.P,v 1.4 2003/02/21 17:57:12 lfcastro Exp $
** 
*/

wrap(InFile,OutFile) :-
	see(InFile),
	tell(OutFile),
	write_header,
	repeat,
	read(Line),
	(Line = end_of_file ->
	    seen, told;
	    process_directive(Line),
	    fail).

%test(File) :-
%	see(File),
%	write_header,
%	repeat,
%	read(Line),
%	(Line = end_of_file -> 
%	    seen;
%	    write(Line),
%	    process_directive(Line),
%	    fail).

write_header :-
	write('/*'),nl,
	write('** This program has been automatically generated by the'),nl,
	write('** second-level foreign language interface of the'),nl,
	write('** XSB System'),nl,
	write('*/'),nl,nl,
	write('#include <stdio.h>'),nl,
	xsb_configuration(install_dir,InstallDir),
	write('#include "'),
	write(InstallDir),
	write('/emu/cinterf.h"'),nl.

process_directive(':-'(foreign_pred(PredDecl))) :- !,
%	write(PredDecl),
	process_from(PredDecl,_PredName).

process_from(from(PDecl,CDecl),/(PredName,Arity)) :-
	write('/* New Definition: '),
	process_pdecl(PDecl,PredName,Arity,NArgUseList),
	write(' from '),
	process_cdecl(CDecl,FuncName,ArgTypeList,RetType),
	linearize_arguse(NArgUseList,ArgUseList),
	write('. */'),
	nl,
	create_ctypelist(ArgTypeList,ArgUseList,CTypeList),
	define_ctype(RetType,'+',RRetType),
	% create prototype:
	write(RRetType),
	write(' '),
	write(FuncName),
	write('('),
	write_cargs(CTypeList),
	write(');'),
	nl,
	% check for need to deal with return value
	( select(arguse(retval,_),ArgUseList,_) ->
	    process_from1(PredName,FuncName,ArgTypeList,NArgUseList,
	                                      ArgUseList,RetType);
	    process_from2(PredName,FuncName,ArgTypeList,NArgUseList,
	                                      ArgUseList,RetType)).

process_from1(PredName,FuncName,ArgTypeList,NArgUseList,ArgUseList,RetType) :-
	% function definition using return value
	out_funchead(PredName),
	out_all_variables([argtype(retval,RetType)|ArgTypeList],ArgUseList),
	out_all_precall([argtype(retval,RetType)|ArgTypeList],
	                                ArgUseList,NArgUseList),
	write('   retval ='),
	out_ccall(FuncName,ArgTypeList,ArgUseList),
	out_all_postcall([argtype(retval,RetType)|ArgTypeList],ArgUseList),
	out_functail,nl.

process_from2(PredName,FuncName,ArgTypeList,NArgUseList,ArgUseList,_) :-
	% function definition 
	out_funchead(PredName),
	out_all_variables(ArgTypeList,ArgUseList),
	out_all_precall(ArgTypeList,ArgUseList,NArgUseList),
	out_ccall(FuncName,ArgTypeList,ArgUseList),
	out_all_postcall(ArgTypeList,ArgUseList),
	out_functail,nl.
	

%check_for_retval(ArgUseList,RVUse) :-
%	select(arguse(retval,RVUse),ArgUseList,_), !.
%check_for_retval(_,_).

out_all_variables([],_) :- !.
out_all_variables([argtype(Var,Type)|ArgType],ArgUse) :-
	select(arguse(Var,Use),ArgUse,NArgUse),
	out_all_variables(ArgType,NArgUse),
	(Var = retval ->
	    (mod_type(Type,MType), out_variables(Var,Use,MType));
	    out_variables(Var,Use,Type)).

mod_type(Type,MType) :-
	atom_codes(Type,LType),
	append(LType,"ptr",LLType),
	atom_codes(MType,LLType).

out_all_precall([],_,_) :- !.
out_all_precall([argtype(Var,Type)|ArgType],ArgUse,UseOrder) :-
	select(arguse(Var,Use),ArgUse,NArgUse),
	out_all_precall(ArgType,NArgUse,UseOrder),
	out_all_precall0(Var,Use,Type,UseOrder).

out_all_precall0(Var,'+',Type,UseOrder) :-
	calc_position(arguse(Var,'+'),UseOrder,InUse),
	out_precall(Var,'+',Type,InUse,0).
out_all_precall0(Var,'-',Type,UseOrder) :-
	calc_position(arguse(Var,'-'),UseOrder,OutUse),
	(Var = retval ->
	    (mod_type(Type,MType),out_precall(Var,'-',MType,0,OutUse));
	    out_precall(Var,'-',Type,0,OutUse)).
out_all_precall0(Var,'+-',Type,UseOrder) :-
	calc_position(arguse(Var,'+'),UseOrder,InUse),
	calc_position(arguse(Var,'-'),UseOrder,OutUse),
	out_precall(Var,'+-',Type,InUse,OutUse).

calc_position(_,[],0) :- !.
calc_position(Elem,[Elem|Rest],Pos) :-
	!, length([Elem|Rest],Pos).
calc_position(Elem,[_|Rest],Pos) :-
	calc_position(Elem,Rest,Pos).

out_all_postcall([],_) :- !.
out_all_postcall([argtype(Var,Type)|ArgType],ArgUse) :-
	select(arguse(Var,Use),ArgUse,NArgUse),
	out_all_postcall(ArgType,NArgUse),
	(Var = retval ->
	    (mod_type(Type,MType), out_postcall(Var,Use,MType));
	out_postcall(Var,Use,Type)).

write_cargs([]) :- !,write(void).
write_cargs([carg(Var,Type)|Rest]) :-
	write_cargs0(Rest),
	write(Type),
	write(' '),
	write(Var).

write_cargs0([]) :- !.
write_cargs0([carg(Var,Type)|Rest]) :-
	write_cargs0(Rest),
	write(Type),
	write(' '),
	write(Var),
	write(', ').
	
create_ctypelist([],_,[]) :- !.
create_ctypelist([argtype(Var,Type)|Rest],ArgUse,[carg(Var,CType)|CRest]) :-
	select(arguse(Var,Use),ArgUse,NArgUse),
	define_ctype(Type,Use,CType),
	create_ctypelist(Rest,NArgUse,CRest).

process_cdecl(':'(CFunc,CRetType),Functor,ArgTypeList,CRetType) :- !,
	write(CRetType),
	write(' '),
	process_cfunc(CFunc,Functor,ArgTypeList).

process_cfunc(CFunc,Functor,LArgType) :-
	functor(CFunc,Functor,Arity),
	write(Functor),
	write('/'),
	write(Arity),
	write('('),
	cargs_list(CFunc,Arity,LArgType),
	write(')').

cargs_list(_,0,[]) :- !.
cargs_list(CFunc,1,[]) :-
	arg(1,CFunc,void).
cargs_list(CFunc,Arity,[ArgType|LArgType]) :-
	NArity is Arity - 1,
	cargs_list(CFunc,NArity,LArgType),
	arg(Arity,CFunc,H),
	(Arity  =\= 1 -> write(','); true),
        process_carg(H,ArgType).

process_carg(':'(Arg,Type),argtype(Arg,Type)) :- !,
	write(Type),
	write(' '),
	write(Arg).

process_pdecl(PDecl,Functor,Arity,LArgUse) :-
	functor(PDecl,Functor,Arity),
	write(Functor),
	write('/'),
	write(Arity),
	(Arity =\= 0 ->
	    write('('),
	    pargs_list(PDecl,Arity,LArgUse),
	    write(')')
	; true).


linearize_arguse([],[]) :- !.
linearize_arguse([arguse(Var,'+')|Rest],[arguse(Var,'+-')|LinearRest]) :-
	member(arguse(Var,'-'),Rest),!,
	select(arguse(Var,'-'),Rest,TempRest),
	linearize_arguse(TempRest,LinearRest).
linearize_arguse([arguse(Var,'-')|Rest],[arguse(Var,'+-')|LinearRest]) :-
	member(arguse(Var,'+'),Rest),!,
	select(arguse(Var,'+'),Rest,TempRest),
	linearize_arguse(TempRest,LinearRest).
linearize_arguse([ArgUse|Rest],[ArgUse|LRest]) :-
	linearize_arguse(Rest,LRest).

pargs_list(_,0,[]) :- !.
pargs_list(PDecl,Arity,[ArgUse|LArgUse]) :-
	NArity is Arity - 1,
	pargs_list(PDecl,NArity,LArgUse),
	arg(Arity,PDecl,H),
	(Arity =\= 1 -> write(','); true),
	process_parg(H,ArgUse).

process_parg('+'(Var),arguse(Var,'+')) :-
	write(Var),
	write(':'),
	write('input').
process_parg('-'(Var),arguse(Var,'-')) :-
	write(Var),
	write(':'),
	write('output').

define_ctype(void,'+',void). % for return type
define_ctype(int,'-',int).
define_ctype(float,'-',double).
define_ctype(atom,'-','unsigned long').
define_ctype(chars,'-','char *').
define_ctype(string,'-','char *').
define_ctype(term,'-',prolog_term).

define_ctype(int,'+',int).
define_ctype(float,'+',double).
define_ctype(atom,'+','unsigned long').
define_ctype(chars,'+','char *').
define_ctype(chars(_),'+','char *').
define_ctype(string,'+','char *').
define_ctype(string(_),'+','char *').
define_ctype(term,'+',prolog_term).

define_ctype(intptr,_,'int *').
define_ctype(floatptr,_,'double *').
define_ctype(atomptr,_,'unsigned long *').
define_ctype(charsptr,_,'char **').
define_ctype(stringptr,_,'char **').
define_ctype(termptr,_,'prolog_term *').
define_ctype(chars(_),_,'char *').
define_ctype(string(_),_,'char *').

define_ccall(int,'+',Var,Var).
define_ccall(float,'+',Var,Var).
define_ccall(atom,'+',Var,Var).
define_ccall(chars,'+',Var,Var).
define_ccall(chars(_),'+',Var,Var).
define_ccall(string,'+',Var,Var).
define_ccall(string(_),'+',Var,Var).
define_ccall(intptr,'+',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(floatptr,'+',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(atomptr,'+',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(charptr,'+',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(stringptr,'+',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(termptr,'+',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
	
define_ccall(intptr,'-',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(floatptr,'-',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(charsptr,'-',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(stringptr,'-',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(string(_),'-',Var,Var).
define_ccall(chars(_),'-',Var,Var).
define_ccall(termptr,'-',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'
define_ccall(atomptr,'-',Var,CVar) :-
	atom_codes(Var,LVar),
	atom_codes(CVar,[38|LVar]). % 38 = '&'

define_ccall(Type,'+-',Var,CVar) :-
	define_ccall(Type,'-',Var,CVar).

%out_variables(Var,Mode,Type).
out_variables(Var,'+',int) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   prolog_int  %s;',args(Var)),nl.
out_variables(Var,'+',float) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   double %s;',args(Var)),nl.    
out_variables(Var,'+',chars) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   char *%s;',args(Var)),nl,
        fmt_write('   int %ssize;',args(Var)),nl.
out_variables(Var,'+',string) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   char *%s;',args(Var)),nl.
out_variables(Var,'+',chars(Size)) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   char %s[%d+1];',args(Var,Size)),nl,
        fmt_write('   char %ssize;',args(Var)),nl.
out_variables(Var,'+',string(Size)) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   char %s[%d+1];',args(Var,Size)),nl,
        fmt_write('   char *%sTemp;',args(Var,Size)),nl.
out_variables(Var,'+',intptr) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
	fmt_write('   prolog_int %s;',args(Var)),nl.
out_variables(Var,'+',floatptr) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   double %s;',args(Var)),nl.
out_variables(Var,'+',charsptr) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   int %ssize;',args(Var)),nl,
        fmt_write('   char *%s,*%sTemp',args(Var,Var)),nl.
out_variables(Var,'+',stringptr) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   char *%s;',args(Var)),nl.
out_variables(Var,'+',term) :-
	fmt_write('   prolog_term %s;',args(Var)),nl.
out_variables(Var,'+',termptr) :-
	fmt_write('   prolog_term %s;',args(Var)),nl.
out_variables(Var,'+',atom) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
        fmt_write('   unsigned long %s;',args(Var)),nl.
out_variables(Var,'+',atomptr) :-
	fmt_write('   prolog_term %sIn;',args(Var)),nl,
	fmt_write('   unsigned long %s;',args(Var)),nl.

out_variables(Var,'-',intptr) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   prolog_int %s;',args(Var)),nl.
out_variables(Var,'-',floatptr) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   double %s;',args(Var)),nl.
out_variables(Var,'-',chars(Size)) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   char %s[%d+1];',args(Var,Size)),nl.
out_variables(Var,'-',charsptr) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   char *%s;',args(Var)),nl.
out_variables(Var,'-',string(Size)) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   char %s[%d+1];',args(Var,Size)),nl.
out_variables(Var,'-',stringptr) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   char *%s',args(Var)),nl.
out_variables(Var,'-',termptr) :-
	fmt_write('   prolog_term %s;',args(Var)),nl.
out_variables(Var,'-',atomptr) :-
	fmt_write('   prolog_term %sOut;',args(Var)),nl,
        fmt_write('   unsigned long %s;',args(Var)),nl.

out_variables(Var,'+-',intptr) :-
	fmt_write('   prolog_term %sIn, %sOut;',args(Var,Var)),nl,
        fmt_write('   prolog_int %s;',args(Var)),nl.
out_variables(Var,'+-',floatptr) :-
	fmt_write('   prolog_term %sIn, %sOut;',args(Var,Var)),nl,
        fmt_write('   double %s;',args(Var)),nl.
out_variables(Var,'+-',chars(Size)) :-
	fmt_write('   prolog_term %sIn,%sOut;',args(Var,Var)),nl,
        fmt_write('   char %s[%d+1];',args(Var,Size)),nl,
        fmt_write('   int %ssize;',args(Var)),nl.
out_variables(Var,'+-',string(Size)) :-
	fmt_write('   prolog_term %sIn,%sOut;',args(Var,Var)),nl,
        fmt_write('   char %s[%d+1],*%sTemp;',args(Var,Size,Var)),nl.
out_variables(Var,'+-',charsptr) :-
	fmt_write('   prolog_term %sIn,%sOut;',args(Var,Var)),nl,
        fmt_write('   char *%s,%sTemp;',args(Var,Var)),nl,
        fmt_write('   int %ssize;',args(Var)),nl.
out_variables(Var,'+-',stringptr) :-
	fmt_write('   prolog_term %sIn,%sOut;',args(Var,Var)),nl,
        fmt_write('   char *%s',args(Var)),nl.
out_variables(Var,'+-',termptr) :-
	fmt_write('   prolog_term %s;',args(Var)),nl.
out_variables(Var,'+-',atomptr) :-
	fmt_write('   prolog_term %sIn,%sOut;',args(Var,Var)),nl,
        fmt_write('   unsigned long %s;',args(Var)),nl.


%out_precall(Var,Mode,Type,InPos,OutPos).

out_precall(Var,'+',int,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_int(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_int(%sIn);',args(Var,Var)),nl.
out_precall(Var,'+',float,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_float(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_float(%sIn);',args(Var,Var)),nl.
out_precall(Var,'+',atom,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = (unsigned long) p2c_string(%sIn);',
                                                     args(Var,Var)),nl.
out_precall(Var,'+',chars,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_charlist(%sIn,&%ssize)) return FALSE;',
	                                             args(Var,Var)),nl,
	fmt_write('   %s = (char *) malloc((%ssize+1)*sizeof(char));',
                                                     args(Var,Var)),nl,
	fmt_write('   p2c_chars(%sIn,%s,%ssize);',args(Var,Var,Var)),nl,
	write('   printf("After call to p2c_chars\n");'),nl.
out_precall(Var,'+',string,InPos,_) :- 
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_string(%sIn);',args(Var,Var)),nl.
out_precall(Var,'+',term,InPos,_) :-
	fmt_write('   %s = reg_term(%d);',args(Var,InPos)),nl.
out_precall(Var,'+',chars(Size),InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_charlist(%sIn,&%ssize)) return FALSE;',
	                                             args(Var,Var)),nl,
	fmt_write('   p2c_chars(%sIn,%s,%d);',args(Var,Var,Size)),nl,
	fmt_write('   %s[%d] = (char) NULL;',args(Var,Size)),nl.
out_precall(Var,'+',string(Size),InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %sTemp = p2c_string(%sIn);',args(Var,Var)),nl,
        fmt_write('   strncpy(%s,%sTemp,%d);',args(Var,Var,Size)),nl,
	fmt_write('   %s[%d] = (char) NULL;',args(Var,Size)),nl.
out_precall(Var,'+',intptr,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_int(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_int(%sIn);',args(Var,Var)),nl.
out_precall(Var,'+',floatptr,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_float(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_float(%sIn);',args(Var,Var)),nl.
out_precall(Var,'+',atomptr,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = (unsigned long) p2c_string(%sIn);',
                                                     args(Var,Var)),nl.
out_precall(Var,'+',charsptr,InPos,_) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_charlist(%sIn,&%ssize)) return FALSE;',
	                                             args(Var,Var)),nl,
	fmt_write('   %s=%sTemp=(char *)malloc((%ssize+1)*sizeof(char));',
                                                   args(Var,Var,Var)),nl,
	fmt_write('   p2c_chars(%sIn,%s,%ssize);',args(Var,Var,Var)),nl.
out_precall(Var,'+',stringptr,InPos,_) :- 
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_string(%sIn);',args(Var,Var)),nl.
out_precall(Var,'+',termptr,InPos,_) :-
	fmt_write('   %s = reg_term(%d);',args(Var,InPos)),nl.

	
out_precall(Var,'-',intptr,_,OutPos) :-
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'-',floatptr,_,OutPos) :-
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'-',charsptr,_,OutPos) :-
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'-',stringptr,_,OutPos) :-
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'-',atomptr,_,OutPos) :-
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'-',termptr,_,OutPos) :-
	fmt_write('   %s = reg_term(%d);',args(Var,OutPos)),nl.


out_precall(Var,'+-',chars(Size),InPos,OutPos) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_charlist(%sIn,&%ssize)) return FALSE;',
	                                             args(Var,Var)),nl,
	fmt_write('   p2c_chars(%sIn,%s,%d);',args(Var,Var,Size)),nl,
	fmt_write('   %s[%d] = (char) NULL;',args(Var,Size)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',string(Size),InPos,OutPos) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %sTemp = p2c_string(%sIn);',args(Var,Var)),nl,
        fmt_write('   strncpy(%s,%sTemp,%d);',args(Var,Var,Size)),nl,
	fmt_write('   %s[%d] = (char) NULL;',args(Var,Size)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',intptr,InPos,OutPos) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_int(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_int(%sIn);',args(Var,Var)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',floatptr,InPos,OutPos) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_float(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_float(%sIn);',args(Var,Var)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',atomptr,InPos,OutPos) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = (unsigned long) p2c_string(%sIn);',
                                                     args(Var,Var)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',charsptr,InPos,OutPos) :-
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_charlist(%sIn,&%ssize)) return FALSE;',
	                                             args(Var,Var)),nl,
	fmt_write('   %s=%sTemp=(char *)malloc((%ssize+1)*sizeof(char));',
                                                   args(Var,Var,Var)),nl,
	fmt_write('   p2c_chars(%sIn,%s,%ssize);',args(Var,Var,Var)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',stringptr,InPos,OutPos) :- 
	fmt_write('   %sIn = reg_term(%d);',args(Var,InPos)),nl,
        fmt_write('   if (!is_string(%sIn)) return FALSE;',args(Var)),nl,
	fmt_write('   %s = p2c_string(%sIn);',args(Var,Var)),nl,
	fmt_write('   %sOut = reg_term(%d);',args(Var,OutPos)),nl,
        fmt_write('   if(!is_var(%sOut)) return FALSE;',args(Var)),nl.
out_precall(Var,'+-',termptr,InPos,_) :-
	fmt_write('   %s = reg_term(%d);',args(Var,InPos)),nl.


%out_postcall(Var,Mode,Type).
out_postcall(_,'+',int).
out_postcall(_,'+',float).
out_postcall(Var,'+',chars) :- 
	fmt_write('   free(%s);',args(Var)),nl.
out_postcall(_,'+',string).
out_postcall(_,'+',chars(_)).
out_postcall(_,'+',string(_)).
out_postcall(_,'+',intptr).
out_postcall(_,'+',floatptr).
out_postcall(Var,'+',charsptr) :-
	fmt_write('   free(%sTemp);',args(Var)),nl.
out_postcall(_,'+',stringptr).
out_postcall(_,'+',term).
out_postcall(_,'+',termptr).
out_postcall(_,'+',atom).
out_postcall(_,'+',atomptr).

out_postcall(Var,'-',intptr) :-
	fmt_write('   c2p_int(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'-',floatptr) :-
	fmt_write('   c2p_float(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'-',chars(_)) :-
	fmt_write('   c2p_chars(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'-',string(_)) :-
	fmt_write('   c2p_string(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'-',charsptr) :-
	fmt_write('   c2p_chars(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'-',stringptr) :-
	fmt_write('   c2p_string(%s,%sOut);',args(Var,Var)),nl.
out_postcall(_,'-',termptr).
out_postcall(_,'-',atomptr) :-
	fmt_write('   c2p_string((char *)%s,%sOut);',args(Var,Var)),nl.

out_postcall(Var,'+-',intptr) :-
	fmt_write('   c2p_int(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'+-',floatptr) :-
	fmt_write('   c2p_float(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'+-',chars(_)) :-
	fmt_write('   c2p_chars(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'+-',string(_)) :-
	fmt_write('   c2p_string(%s,%sOut);',args(Var,Var)),nl.
out_postcall(Var,'+-',charsptr) :-
	fmt_write('   c2p_chars(%s,%sOut);',args(Var,Var)),nl,
	fmt_write('   free(%sTemp);',args(Var)),nl.
out_postcall(Var,'+-',stringptr) :-
	fmt_write('   c2p_string(%s,%sOut);',args(Var,Var)),nl.
out_postcall(_,'+-',termptr).
out_postcall(_,'+-',atomptr) :-
	fmt_write('   c2p_string((char *)%s,%sOut);',args(Var,Var)),nl.

out_funchead(FuncName) :-
	fmt_write('DllExport int call_conv %s(void)',args(FuncName)),nl,
        write('{'),nl.

out_functail :- 
	write('   return TRUE;'),nl,
	write('}'),
	nl.

%out_ccall(FuncName,ArgType,ArgUse)
out_ccall(FuncName,ArgType,ArgUse) :-
	write('   '),write(FuncName),
	write('('),
	out_ccall0(ArgType,ArgUse),
	write(');'),nl.

out_ccall0([],_) :- !.
out_ccall0([argtype(Var,Type)|ArgType],ArgUse) :-
	select(arguse(Var,Use),ArgUse,NArgUse),
	out_ccall1(ArgType,NArgUse),
	define_ccall(Type,Use,Var,PVar),
	write(PVar).

out_ccall1([],_) :- !.
out_ccall1([argtype(Var,Type)|ArgType],ArgUse) :-
	select(arguse(Var,Use),ArgUse,NArgUse),
	out_ccall1(ArgType,NArgUse),
	define_ccall(Type,Use,Var,PVar),
	write(PVar),write(', ').


