/* File:      flrlibman.P -- Flora includes and libraries definitions
**
** Author(s): Guizhen Yang
**            Michael Kifer
**
** Contact:   flora-users@lists.sourceforge.net
**
** Copyright (C) The Research Foundation of SUNY, 1999-2001
**
** FLORA-2 is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
**
** FLORA-2 is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
**
** You should have received a copy of the GNU Library General Public License
** along with FLORA-2; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
**
*/


:- compiler_options([xpp_on]).


#include "flora_terms.flh"
#include "flora_exceptions.flh"

?- bootstrap_flora.

/**********************************************************************
  flora_register_syslib(+LibBaseFile)
**********************************************************************/
flora_register_syslib(LibBaseFile) :-
	assert(flora_syslib_loaded(LibBaseFile)).


/**********************************************************************
  flora_load_library(+List)
  flora_load_library(+LibID)

  Note: This predicate always succeeds.
**********************************************************************/
flora_load_library(LibID) :-
	var(LibID),
	!,
	flora_error_line('Uninstantiated FLORA system library name! Loading failed!').

flora_load_library([]) :- !.

flora_load_library([H|T]) :-
	!,
	flora_load_library(H),
	flora_load_library(T).

flora_load_library(LibID) :-
	( flora_library_file(LibID,BaseFileName,SubDir) ->
	    ( flora_syslib_loaded(BaseFileName) -> true
	    ;
	      flora_configuration(installdir,FloraDir),
	      flora_slash(S),
	      flora_concat_atoms([FloraDir,S,SubDir,S,BaseFileName,'.P'],
				 FullLibPath),
		%% Because LibID implies BaseFilename, we register base
		%% filename rather than the library name
		%% (several LibID can correspond to one file).
	      consult(FullLibPath),
	      flora_register_syslib(BaseFileName)
	    )
	;
	  flora_error_line('Invalid FLORA system library name %S ! Loading failed!',
			   args(LibID))
	),
	!.

flora_load_library(_LibID).


/**********************************************************************
  flora_register_sysmod(+SysModID)
**********************************************************************/
flora_register_sysmod(SysModID) :-
	assert(flora_sysmod_loaded(SysModID)).


/**********************************************************************
  flora_load_system_module(+List)
  flora_load_system_module(+SysModID)

  Note: This predicate always succeeds.
**********************************************************************/
flora_load_system_module(SysModID) :-
	var(SysModID),
	!,
	flora_error_line('Uninstantiated FLORA system module ID! Loading failed!').

flora_load_system_module([]) :- !.

flora_load_system_module([H|T]) :-
	!,
	flora_load_system_module(H),
	flora_load_system_module(T).

flora_load_system_module(SysModID) :-
	( flora_sysmod_loaded(SysModID) -> true
	;
	    %% Register the Flora system module ID.
	    flora_register_sysmod(SysModID),
	    %% Maybe we should unregister on error? Use throw/catch?
	    flora_load_system_module_internal(SysModID)
	),
	!.

flora_load_system_module(_SysModID).


/********************************************************************
  flora_include(+ID,+FilePath)
  indicates which file to be included in the output file. The file
  path contains the subdir relative to the Flora home dir.
********************************************************************/
flora_include(ID,IncFile) :-
	atom(ID),
	flora_slash(S),
	flora_library_file(ID,BaseFilename,SubDir),
	!,
	flora_concat_atoms([SubDir,'inc',S,BaseFilename,'_inc.flh'], IncFile).


/****************************************************************************
  flora_library_file(+LibName,+BaseFileName,+SubDir)

  These libraries implement:
  * Flora aggregation syntax
  * Flora shell commands
  * module specification with variables
  * control structure such as if-then-else and while-loop
*****************************************************************************/
flora_library_file(FLLIBMIN,flraggmin,syslib) :- !.
flora_library_file(FLLIBMAX,flraggmax,syslib) :- !.
flora_library_file(FLLIBSUM,flraggsum,syslib) :- !.
flora_library_file(FLLIBAVG,flraggavg,syslib) :- !.
flora_library_file(FLLIBCOUNT,flraggcount,syslib) :- !.
flora_library_file(FLLIBCOLLECTSET,flraggcolset,syslib) :- !.
flora_library_file(FLLIBCOLLECTBAG,flraggcolbag,syslib) :- !.
flora_library_file(FLLIBLOAD,flrload,syslib) :- !.
flora_library_file(FLLIBANSWER,flranswer,syslib) :- !.
flora_library_file(FLLIBSHELLANS,flranswer,syslib) :- !.
flora_library_file(FLLIBPROGRAMANS,flranswer,syslib) :- !.
flora_library_file(FLLIBDISPLAY,flrdisplay,syslib) :- !.
flora_library_file(FLLIBDBOP,flrdbop,syslib) :- !.
flora_library_file(FLLIBINSERT,flrdbop,syslib) :- !.
flora_library_file(FLLIBINSERTALL,flrdbop,syslib) :- !.
flora_library_file(FLLIBDELETE,flrdbop,syslib) :- !.
flora_library_file(FLLIBDELETEALL,flrdbop,syslib) :- !.
flora_library_file(FLLIBERASE,flrdbop,syslib) :- !.
flora_library_file(FLLIBERASEALL,flrdbop,syslib) :- !.
flora_library_file(FLLIBBTDBOP,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBBTINSERT,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBBTINSERTALL,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBBTDELETE,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBBTDELETEALL,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBBTERASE,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBBTERASEALL,flrbtdbop,syslib) :- !.
flora_library_file(FLLIBREFRESH,flrtables,syslib) :- !.
flora_library_file(FLLIBTABLES,flrtables,syslib) :- !.
flora_library_file(FLLIBSHDIRECT,flrshdirect,syslib) :- !.
flora_library_file(FLLIBDYNMOD,flrdynmod,syslib) :- !.
flora_library_file(FLLIBMODLIT,flrdynmod,syslib) :- !.
flora_library_file(FLLIBMODOBJ,flrdynmod,syslib) :- !.
flora_library_file(FLLIBEQUALITY,flrequality,syslib) :- !.

flora_library_file(FLLIBDYNRULE,flrdynrule,syslib) :- !.
flora_library_file(FLLIBNEWMODULE,flrdynrule,syslib) :- !.
flora_library_file(FLLIBINSERTRULE_A,flrdynrule,syslib) :- !.
flora_library_file(FLLIBINSERTRULE_Z,flrdynrule,syslib) :- !.
flora_library_file(FLLIBDELETERULE_A,flrdynrule,syslib) :- !.
flora_library_file(FLLIBDELETERULE_Z,flrdynrule,syslib) :- !.
flora_library_file(FLLIBDELETERULE,flrdynrule,syslib) :- !.

flora_library_file(FLLIBCONTROL,flrcontrol,syslib) :- !.
flora_library_file(FLLIBIFTHENELSE,flrcontrol,syslib) :- !.
flora_library_file(FLLIBIFTHEN,flrcontrol,syslib) :- !.
flora_library_file(FLLIBUNLESSDO,flrcontrol,syslib) :- !.
flora_library_file(FLLIBWHILEDO,flrcontrol,syslib) :- !.
flora_library_file(FLLIBDOUNTIL,flrcontrol,syslib) :- !.
flora_library_file(FLLIBWHILELOOP,flrcontrol,syslib) :- !.
flora_library_file(FLLIBLOOPUNTIL,flrcontrol,syslib) :- !.

flora_library_file(FLLIBMETAOPS,flrmetaops,syslib) :- !.
flora_library_file(FLLIBUNIVEQFORM,flrmetaops,syslib) :- !.
%% This handles both =.. and ~..
flora_library_file(FLLIBMETAUNIV,flrmetaops,syslib) :- !.

flora_library_file(FLLIBIMPORTEDCALLS,flrimportedcalls,syslib) :- !.

flora_library_file(FLLIBCATCH,flrerrhandler,syslib) :- !.
flora_library_file(FLLIBTHROW,flrerrhandler,syslib) :- !.
flora_library_file(FLLIBERRHANDLER,flrerrhandler,syslib) :- !.


/**************************************************************************
  Flora headers and trailers inclusion files
***************************************************************************/
flora_library_file(FLHEADER,flrheader,header) :- !.
flora_library_file(FLHEADER2,flrheader2,header) :- !.
flora_library_file(FLDEFINITION,flrdefinition,header) :- !.
flora_library_file(FLPATCH,flrpatch,header) :- !.
flora_library_file(FLTRAILER,flrtrailer,header) :- !.
flora_library_file(FLEQLTRAILER,flreqltrailer,header) :- !.
flora_library_file(FLSCALAREQL,flrscalareql,header) :- !.

%% support for undefinedness
flora_library_file(FLPREDDEF,flrpreddef,header) :- !.
flora_library_file(FLINDEX_P,flrindex_P,header) :- !.

%% for dynamically inserted rules
flora_library_file(FLDYNA,flrdyna,header) :- !.
flora_library_file(FLDYNZ,flrdynz,header) :- !.

%% for refreshing tables
flora_library_file(FLREFRESHTABLE,flrrefreshtable,header) :- !.

flora_library_file(FLLIBIMPORTEDCALLS,flrimportedcalls,header) :- !.

/**************************************************************************
  Flora system modules use the system libraries mechanism to include
  their header files, but use different mechanism to load.
***************************************************************************/
flora_library_file(SysModID,BaseFile,SubDir) :-
	flora_sysmod_file(SysModID,BaseFile,SubDir),
	!.


/**************************************************************************
  catch error
***************************************************************************/
flora_library_file(S,F,_) :-
	flora_error_line("Unknown library ID `%S' or file `%S'", args(S,F)),
	flora_abort.


/*****************************************************************************
  flora_sysmod_file(+SysModID,+BaseFileName,+SubDir).

  Flora system modules accessible by the @flora(mod) syntax
*****************************************************************************/
flora_sysmod_file(FLSYSMODPP,flrprettyprint,lib) :- !.
flora_sysmod_file(FLSYSMODIO,flrio,lib) :- !.
flora_sysmod_file(FLSYSMODSYSTEM,flrsystem,lib) :- !.
flora_sysmod_file(FLSYSMODDATABASE,flrstorage,lib) :- !.

flora_system_module(SysModID) :- flora_sysmod_file(SysModID,_BaseFileName,_SubDir).
