/* File:      flrporting.P  -- Non-standard Prolog predicates used in Flora
**
** Author(s): Michael Kifer
**
** Contact:   flora-users@lists.sourceforge.net
**
** Copyright (C) The Research Foundation of SUNY, 2002
**
** FLORA-2 is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
**
** FLORA-2 is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
**
** You should have received a copy of the GNU Library General Public License
** along with FLORA-2; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: flrporting.P,v 1.9 2003/06/18 07:01:36 kifer Exp $
**
*/


:- compiler_options([xpp_on]).

#include "flag_defs_xsb.h"
#include "token_defs_xsb.h"
%% DO NOT include any FLORA .flh files here, or else FLORA won't
%% build correctly when installed outside of the XSB tree


:- import conget/2, conset/2 from gensym.
:- import eval/2 from eval.
:- import
	slash/1, stat_flag/2,
	str_len/2,
	file_puttoken/3, file_writequoted/2
   from machine.
:- import substring/4 from string.
:- import library_directory/1 from usermod.
:- import xsb_configuration/2 from xsb_configuration.
:- import
	path_sysop/2, path_sysop/3, copyIOport/2,
	file_read_line_atom/1, file_read_line_list/1
	from file_io.

:- import banner_control/1 from banner.

:- import flora_abort/1 from flrutils.


:- export
	flora_write_quoted_atom/1,
	flora_write_atom/1,
	flora_slash/1,
	flora_running_under/1,
	flora_file_op/2,
	flora_file_op/3,
	flora_read_line_as_atom/1,
	flora_read_line_as_list/1,
	flora_copy_input/0,
	flora_user_home/1,
	flora_atom_length/2,
	flora_match_substring/3,
	flora_match_substring/5,
	flora_get_substring/4,
	flora_term_to_atom/2,
	flora_concat_items/2,
	flora_concat_atoms/2,
	flora_set_counter/2,
	flora_get_counter/2,
	flora_increment_counter/4,
	flora_cputime/1,
	flora_module_path_get/1,
	flora_module_path_add/1,
	flora_module_path_remove/1,
	flora_banner_control/1.


flora_term_to_atom(Term,Atom) :- fmt_write_string(Atom,'%S',arg(Term)).

%% This concats items or any type - atoms, numbers, etc.
%% Make the most common case fast!
flora_concat_items([Item1,Item2],Atom) :-
	!,
	fmt_write_string(Atom, '%S%S', arg(Item1,Item2)).
flora_concat_items([Item|ItemList],Atom) :-
	fmt_write_string(Atom1, '%S', arg(Item)),
	flora_concat_items(ItemList,Atom2),
	str_cat(Atom1,Atom2,Atom).
flora_concat_items([],'').

%% If all members are atoms, this is more efficient
%% Make the most common case fast!
flora_concat_atoms([Atom1,Atom2],Atom) :- !, str_cat(Atom1,Atom2,Atom).
flora_concat_atoms([Atom1|AtomList],Atom) :-
	flora_concat_atoms(AtomList,Atom2),
	str_cat(Atom1,Atom2,Atom).
flora_concat_atoms([],'').
	


flora_atom_length(Atom,Length) :- str_len(Atom, Length).


flora_get_counter(Counter,Value) :- conget(Counter,Value).
flora_set_counter(Counter,Value) :- conset(Counter,Value).
flora_increment_counter(Counter,Increment,OldValue,NewValue) :-
	eval(Increment,I),
	conget(Counter,OldValue),
	NewValue is OldValue + I,
	conset(Counter,NewValue).

%% Flora wrappers for str_match in XSB's syslib/string.P
%% Checks if Substr matches Str in a given direction and at the right position
flora_match_substring(Substr,Str,Pos) :-
	str_match(Substr,Str,forward,Pos,_).
flora_match_substring(Sub,Str,Direction,Beg,End) :-
	str_match(Sub,Str,Direction,Beg,End).

%% extract substring Subst from String at positions From - To
flora_get_substring(String,From,To,Subst) :- substring(String,From,To,Subst).

flora_slash(Slash) :- slash(Slash).

flora_cputime(X) :- cputime(X).


%% Manipulation of module search path
%% Yap uses path/1, add_to_path/1, remove_from_path/1
flora_module_path_add(Path)    :- assert(library_directory(Path)).
flora_module_path_remove(Path) :- retractall(library_directory(Path)).

flora_module_path_get('.').
flora_module_path_get(Path) :- library_directory(Path).
flora_module_path_get(Path) :-
	xsb_configuration(libdir,LibPath), LibPath = Path.
flora_module_path_get(Path) :-
	xsb_configuration(syslibdir,SyslibPath), SyslibPath=Path.
flora_module_path_get(Path) :-
	xsb_configuration(cmplibdir,CmplibPath), CmplibPath = Path.


%% User Home
flora_user_home(Path) :- xsb_configuration(user_home,Path).

%% File system-related
flora_file_op(exists,File)          :- path_sysop(exists,File).
flora_file_op(mkdir,Dir)            :- path_sysop(mkdir,Dir).
flora_file_op(unlink,File)          :- path_sysop(unlink,File).
flora_file_op(tmpfilename,File)     :- path_sysop(tmpfilename,File).
flora_file_op(isabsolute,File)      :- path_sysop(isabsolute,File).
flora_file_op(rename,File,ToFile)   :- path_sysop(rename,File,ToFile).
flora_file_op(basename,File,Base)   :- path_sysop(basename,File,Base).
flora_file_op(extension,File,Ext)   :- path_sysop(extension,File,Ext).
flora_file_op(expand,File,Expanded) :- path_sysop(expand,File,Expanded).
flora_file_op(newerthan,File,File2) :- path_sysop(newerthan,File,File2).
flora_file_op(dirname,File,Dir)     :- path_sysop(dirname,File,Dir).
flora_file_op(copy,From,To)         :- path_sysop(copy,From,To).


%% Copies stdin to stdout
flora_copy_input :-
	stat_flag(CURRENT_INPUT, StdIn),
	stat_flag(CURRENT_OUTPUT, StdOut),
	copyIOport(StdIn,StdOut).


flora_write_quoted_atom(Atom) :-
	\+atom(Atom),
	flora_abort(['flora_write_quoted_atom: Non-atom argument, ', Atom]).
	
flora_write_quoted_atom(Atom) :-
	stat_flag(CURRENT_OUTPUT, StdOut),
	file_writequoted(StdOut,Atom).
%% Unquoted atom
flora_write_atom(Atom) :-
	stat_flag(CURRENT_OUTPUT, StdOut),
	file_puttoken(StdOut,TK_ATOM,Atom).

flora_read_line_as_atom(Str) :- file_read_line_atom(Str).
flora_read_line_as_list(Str) :- file_read_line_list(Str).


:- table flora_running_under/1.

flora_running_under(cygwin) :-
	xsb_configuration(architecture,A),
	str_sub(cygwin,A),
	!.
flora_running_under(windows) :-
	xsb_configuration(architecture,A),
	str_sub(windows,A),
	!.
flora_running_under(macos) :-
	xsb_configuration(architecture,A),
	str_sub(rhapsody,A),
	!.
flora_running_under(darwin) :-
	xsb_configuration(architecture,A),
	str_sub(rhapsody,A),
	!.
flora_running_under(unix).


flora_banner_control(X) :- banner_control(X).
