/* File:        flrload.P
**
** Author(s): Guizhen Yang
**
** Contact:   flora-users@lists.sourceforge.net
**
** Copyright (C) The Research Foundation of SUNY, 1999-2001
**
** FLORA-2 is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** FLORA-2 is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with FLORA-2; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
** 
** $Id: flrload.P,v 1.13 2003/06/18 07:01:19 kifer Exp $
** 
*/


:- compiler_options([xpp_on]).

#include "flora_terms.flh"
#include "flora_extensions.flh"
#include "flora_exceptions.flh"


:- import
	flora_stderr_nl/0,
	flora_stderr_string/1,
	flora_error_line/1,
	flora_error_line/2,
	flora_error_heading/0
   from flrprint.

:- import
	flora_load_module_internal/1,
	flora_load_module_internal/2,
	flora_locate_file/3
   from flrutils.

:- import flora_display_error/1 from flrdisplay.

:- import
	flora_get_counter/2,
	flora_file_op/3
   from flrporting.

:- import flora_abort/0 from flrutils.



/********************************************************************/
fllibload([],_HostFile,_HostMod,_Rulenum) :- !.

fllibload([T|L],HostFile,HostMod,Location) :-
	!,
	flora_load_file(T,HostFile,HostMod,Location),
	fllibload(L,HostFile,HostMod,Location).


/*********************************************************************
  flora_load_file(+FileName,+HostFile,+HostMod,+Location)
  flora_load_file(FL_RIGHTTO(+FileName,+Workspace),+HostFile,+HostMod,+Location)

  Filename: file being loaded
  Workspace: module into which it is loaded
  HostFile: program file from which this call was made
  HostMod: module into which host program was loaded
  Location: of the form [Line,Char] indicates the line and char
            where the loading literal occurs in the program

  Note: Runtime checking of file and workspace names is needed, since
        the load list can be constructed at runtime.
*********************************************************************/
flora_load_file(T,HostFile,HostMod,Location) :-
	atom(T),
	!,
	flora_check__module_overriding(T,FLORA_DEFAULT_WORKSPACE,HostFile,HostMod,Location),
	flora_load_module_internal(T).

flora_load_file(T,_,_,_) :-
	var(T),
	!,
	flora_stderr_nl,
	flora_error_line('file name unbound at time of loading'),
	fail.

flora_load_file(FL_RIGHTTO(FileName,Workspace),HostFile,HostMod,Location) :-
	!,
	flora_check__module_overriding(FileName,Workspace,HostFile,HostMod,Location),
	( atom(FileName) ->
	    ( atom(Workspace) ->
		flora_load_module_internal(FileName,Workspace)
	    ;
	      flora_stderr_nl,
	      ( var(Workspace) ->
		  flora_error_line('uninstantiated module name at time of loading')
	      ;
	        flora_error_heading,
		flora_stderr_string('invalid module name: '),
		flora_display_error(Workspace),
		flora_stderr_nl
	      ),
	      !,
	      fail
	    )
	;
	  flora_stderr_nl,
	  flora_error_heading,
	  flora_stderr_string('file name '),
	  flora_display_error(FileName),
	  flora_stderr_string(' invalid for loading'),
	  flora_stderr_nl,
	  !,
	  fail
        ).

flora_load_file(X,_,_,_) :-
	flora_stderr_nl,
	flora_error_heading,
	flora_stderr_string('file name '),
	flora_display_error(X),
	flora_stderr_string(' invalid for loading'),
	flora_stderr_nl,
	!,
	fail.

flora_check__module_overriding(_LoadFile,_Workspace,userin,main,_Location) :-
	!.

%% Don't do the check if we are loading a prolog module
%% (i.e., it doesn't have the .flr file and no workspace was given in [...])
flora_check__module_overriding(LoadFile,main,_HostFile,_HostMod,_Location) :-
	not flora_locate_file(LoadFile,FLORA_FILE_EXT,_FlrFile),
	!.


flora_check__module_overriding(LoadFile,Workspace,HostFile,HostMod,[Line,Char]) :-
	HostMod == Workspace,
	!,
	flora_stderr_nl,
	flora_file_op(basename,HostFile,HostFileBasename),
	flora_error_line('~w.flr: Literal [~w >> ~w] near line(~w)/char(~w) overrides the program in which it occurs',
			 [HostFileBasename,LoadFile,HostMod,Line,Char]),
	flora_abort.

flora_check__module_overriding(_LoadFile,_Workspace,_HostFile,_HostMod,_Location).
