/* File:      dbclause.P
** Author(s): David S. Warren, Kostis Sagonas
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: dbclause.P,v 1.21 2003/05/01 18:52:50 dwarren Exp $
** 
*/


:- compiler_options([xpp_on]).
#include "builtin.h"
#include "sig_xsb.h"
#include "flag_defs_xsb.h"

/*======================================================================*/

:- set_inthandler('_$clause_int'(_), MYSIG_CLAUSE).

/************************************************************************/
/* Module dbclause defines routines to step through clauses and 	*/
/* return them, and also to delete clauses for retracting. Space is	*/
/* reclaimed. (See module assert for formats of data structures.	*/
/************************************************************************/


% db_retract(+Head,+Body,+Prref) deletes clauses and reclaims space

db_retract(Head,Body,Prref) :-
	goal_cut_trans_a(Body,Tbody,_),
	db_get_clauses(Prref,0,Head,0,Clref,Typ,EntryP),
	chk_if_trie_assert(Clref,Result,TrieNodeAddr),
	(	Result = trie, 
		call_trie_retract(Head,Body,Clref,TrieNodeAddr)
	;
		Result = nottrie,
		db_clause_match(Head,Tbody,EntryP),
		db_do_retract(Typ,Clref,0)
	).


%% Don''t reclaim space
db_retract_nr(Head,Body,Prref) :-
	goal_cut_trans_a(Body,Tbody,_),
	db_get_clauses(Prref,0,Head,0,Clref,Typ,EntryP),
	chk_if_trie_assert(Clref,Result,TrieNodeAddr),
	(	Result = trie,
		call_trie_retract_nr(Head,Body,TrieNodeAddr)
	;
		Result = nottrie,
		db_clause_match(Head,Tbody,EntryP),
		db_do_retract(Typ,Clref,1)  % 1 -> Don''t reclaim space! *****
	).

db_do_retract(1,_Clref,_) :- !, fail.
db_do_retract(2,_Clref,_) :- abort('Cannot erase compiled code').
db_do_retract(_Typ,Clref,DontReclaim) :- db_retract0(Clref,DontReclaim).

% transform a goal to turn cuts (!) into '_$cutto's. Differs from
% goal_cut_trans in that variables are not wrapped with call/1,
% but are left unchanged.

goal_cut_trans_a(X,X,_) :- var(X),!.
goal_cut_trans_a([X|Y],Z,_) :- !, hack_consult([X|Y],Z). %  a HACK!
goal_cut_trans_a(!,'_$cutto'(Cutpoint),Cutpoint) :- !.
goal_cut_trans_a('_$load_undef'(X),Y,_) :- !,
	(atom(X) -> conname(X,Y1),Y=Y1 ; Y=X).
goal_cut_trans_a((A,B),(Na,Nb),Cutpoint) :- !,
	goal_cut_trans_a(A,Na,Cutpoint),
	goal_cut_trans_a(B,Nb,Cutpoint).
goal_cut_trans_a((A;B),(Na;Nb),Cutpoint) :- !,
	goal_cut_trans_a(A,Na,Cutpoint),
	goal_cut_trans_a(B,Nb,Cutpoint).
goal_cut_trans_a((A->B),(A->Nb),Cutpoint) :- !,
	goal_cut_trans_a(B,Nb,Cutpoint).
goal_cut_trans_a(X,X,_).

hack_consult([X],consult(X)) :- !.
hack_consult([X|Y],(consult(X),Z)) :- !,hack_consult(Y,Z).


% db_retractall(+Call,+Prref) deletes all clauses matching a pattern

db_retractall(Call,Prref) :-
	( is_most_general_term(Call) ->	db_retract_abol(Prref)
	;   db_retract(Call,_,Prref)
	),
	fail.
db_retractall(_,_).

%%:- import writeln/1 from standard.
db_retract_abol(Prref1) :-
	buff_byte(Prref1,0,Op1),
	((Op1 =:= 197 ; Op1 =:= 192)   % tabletryfail or tabletrysingle
	 ->	buff_word(Prref1,24,Prref)
	 ;	Prref = Prref1
	),
	buff_byte(Prref,0,Op),
	(Op =\= 248	% fail	
	 ->	(Op =:= 240  % jump to trie code?
		 ->	buff_word(Prref,4,Buffaddr),
			buff_word(Buffaddr,12,TrieInst),
			buff_byte(TrieInst,0,Inst),
			(Inst =:= 124   %% trie instruction
			 ->	gen_retract_all(Prref)  %% abolish trie
			 ;	abort('retractall: inconsistent code structure')
			)
		 ;	gen_retract_all(Prref)
		)
	 ;	true
	),
	buff_set_byte(Prref,0,248).	% fail


% db_abolish(+Prref) deletes all clauses of a predicate.

db_abolish(Prref) :-
	db_retract_abol(Prref),
        db_remove_prref(Prref).

% db_clause(+Head,?Body,+Prref,-Clref) retrieves a clause.

db_clause(Head,Body,Prref,Clref) :-
	goal_cut_trans_a(Body,Tbody,_),
	db_get_clauses(Prref,0,Head,0,Clref,Typ,EntryP),
	(Typ =:= 2	% compiled code
	 ->	chk_if_trie_assert(Clref,Result,_),
		(Result = trie ->
			Body = true,
			db_clause_match_trassert(Head)
		;
		Result = nottrie ->
			Body = '$compiled'	
		;
		abort(('Strange result type in clause',Result))
		)
	 ;	db_clause_match(Head,Tbody,EntryP)
	).

:- buff_alloc(8,Buff),asserti(db_body_save(Buff),0).

% db_clause_match(+Call,+Body,+Entry)
%	Determines whether Clref contains a clause whose head matches Call.
%	Type indicates the kind	of Clref (indexed or not). 

db_clause_match(Head,Body,Entry) :-
	db_body_save(Buff),
	buff_set_cell(Buff,0,5,true),	% set default
	(db_do_call(Entry,Head)
	 ->	stat_set_flag(CLAUSE_INT,0),
	 	db_body_save(Buff),
		buff_cell(Buff,0,Body0),
		goal_cut_trans_a(Body0, Body1, _),
		(atom(Body1)
		 ->	conname(Body1,Body2),
		 	Body = Body2
		 ;	Body = Body1
		)
	 ;	stat_set_flag(CLAUSE_INT,0),
	 	fail
	).


db_clause_match_trassert(Head) :-
	stat_set_flag(CLAUSE_INT,0),
	call(Head).


% db_do_call(+CodeEntry, +Head)
%	Makes the call to Head on the heap after setting the interrupt flag
%	that will retrieve the code from the heap.  By this time we finally
%	know that Head is "dynamic" (psc_type = T_DYNA) so there is no need to
%	retrieve the type.  If at some later version we need the psc type 
%	the following should appear:
%			conpsc(Head, PSC), psc_type(PSC, Type),
%	BEFORE stat_set_flag/2.  Under NO circumstances, should code be put
%	between stat_set_flag/2 and code_call/3, because the hack of building
%	the body of the call on the heap will not work!!.

db_do_call(Entry, Head) :-
	stat_set_flag(CLAUSE_INT, 1),   % set clause interrupt on (CLAUSE_INT)
	code_call(Entry, Head, 1).	% Type =:= 1 means dynamic predicate.

:- dynamic(db_body_save(_)).

'_$clause_int'(Call) :-
	stat_set_flag(CLAUSE_INT,0),	% turn clause intercept off
 	db_body_save(Buff),
	buff_set_var(Buff,0,8,X),
	X=Call.

%% db_get_clauses(+Prref,+Ind, +Term, +Failed, -Clref,-Typ,Entp) returns Clrefs
%%       for each clause whose head might unify with Term. It uses the SOB 
%%       index blocks to efficiently find them. It may return clauses whose
%%	head won''t unify with Term. (Ind is no longer used.)
%%	Failed means fetch retract(ed)_nr clauses.

db_get_clauses( PrRef, _Index1, Head, Failed, ClRef, Typ, EntryP ) :-
	db_get_clauses0( PrRef, Head, Failed, ClRef, Typ, EntryP ).
	

%% db_get_clauses0 calls the builtin, db_get_clause, to get the next ClRef.  It
%% 	uses db_get_clauses1 to do a one clause look-ahead, so that a 
%%	deterministic retract doesn''t lay down a choice point.  This makes
%%	the semantics of the new C retract more consistent with the old.

db_get_clauses0(PrRef,Head,Failed,ClRef,Typ,EntryP) :-
	db_get_clause(0,0,0,PrRef,Head,Failed,ClRef0,Typ0,EntryP0,NLev,NInd),
	(ClRef0 =:= 0
	 ->	!,fail
	 ;	db_get_clauses1(ClRef0,NLev,NInd,PrRef,Head,Failed,
			ClRef,Typ,EntryP,Typ0,EntryP0)
	).

db_get_clauses1(ClRef0,CurrILvl0,CurrInd0,PrRef,Head,Failed,
			ClRef,Typ,EntryP,Typ0,EntryP0) :-
	db_get_clause(ClRef0,CurrILvl0,CurrInd0,PrRef,Head,Failed,
		      ClRef1,Typ1,EntryP1,NLev,NInd),
	(ClRef1 =:= 0
	 ->	!,
		ClRef = ClRef0, Typ = Typ0, EntryP = EntryP0
	 ;	(ClRef = ClRef0, Typ = Typ0, EntryP = EntryP0
		 ;
		 db_get_clauses1(ClRef1,NLev,NInd,PrRef,Head,Failed,
			ClRef,Typ,EntryP,Typ1,EntryP1)
		)
	).

/************************************************************************/
/* The following routine cleans up by reclaiming space for all clauses	*/
/* that were erased without reclaiming their space.			*/
/************************************************************************/

db_reclaim_space(Head,Prref):-
	db_get_clauses(Prref,0,Head,1,Clref,Typ,_IndLevel),
	db_reclaim0(Clref,Typ),
	fail.
db_reclaim_space(_,_).

%==================================================
% Trie Assert related code
%==================================================

root_is_not_null(NodeAddr) :-
	buff_word(NodeAddr,8,RootOfCall), RootOfCall =\= 0.


chk_if_trie_assert(Y,trie,TrieNodeAddr) :- 
	buff_word(Y, -8, Lenflg),
	Flag is Lenflg /\ 3,
	Flag = 2,
	buff_word(Y,12,TrieNodeAddr),
	buff_byte(TrieNodeAddr,0,Inst),
	Inst = 124,
	!.
chk_if_trie_assert(_,nottrie,_).


call_trie_retract(Head,Body,Clref,NodeAddr) :-
	(
		Body = true,
		repeat,
		(
			root_is_not_null(NodeAddr),
			trie_retract_clref(0), % Make Last Node Save to 0
			call(Head)
			-> trie_retract_clref(Clref)
		;				
			!,fail
		)
	;
	  	\+(Body = true), 			
	  	fail
	).
	


trie_retract_clref(_Clref) :- '_$builtin'(TRIE_RETRACT).
trie_retract_clref_nr(_WhatToDo) :- '_$builtin'(TRIE_RETRACT_SAFE).

% not safe
% if call(Head) results in Backtracking through
% program clauses which may call trie code disaster
% will result.


call_trie_retract_nr(Head,Body,NodeAddr) :-
	%write(call_trie_retract_nr(Head,Body,Prref,Clref)),write(':'),
	(
		Body = true,
	        '_$savecp'(Breg),	
		repeat,
		(
			trie_retract_clref(0), % Zero out Last_Nod_Sav
			root_is_not_null(NodeAddr),call(Head)
			-> (trie_retract_clref_nr(1) -> true
			   ;
			    '_$cutto'(Breg),fail)
		;
			!,fail	
		)
	;
		\+(Body = true),
		fail
	).	
