/* File:      loader.P
** Author(s): Ernie Johnson, Jiyang Xu, Kostis F. Sagonas
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: loader.P,v 1.39 2003/01/06 04:46:36 kifer Exp $
** 
*/



/*
 *  System Startup
 *  --------------
 *  Notice:  We must set the interrupt handler of the dynamic
 *  loader BEFORE any undefined predicate is encountered.
 *  We do initially run without a catcher for aborts etc. but
 *  one could be added early on, if we knew what to do if we
 *  got thrown at that early.
 */

:- compiler_options([sysmod,xpp_on,optimize]).

#include "../prolog_includes/standard.h"
#include "char_defs.h"
#include "flag_defs_xsb.h"
#include "sig_xsb.h"
#include "io_defs_xsb.h"
#include "celltags_xsb.h"
#include "psc_defs.h"
#include "extensions_xsb.h"

%% Provide tests for the respective OS.
%% windows_os_loader is used only here and in parse.
%% All other modules use windows_os imported from xsb_configuration.
%% The *_loader version is needed  here because we can''t import 
%% them from xsb_configuration this early. 

windows_os_loader :-  stat_flag(CONFIG_NAME, ConfigurationC),
	%% convert C string to Prolog
	intern_string(ConfigurationC, Configuration),
	(   str_match('win', Configuration, f, _, _)
	;   str_match('-nt', Configuration, f, _, _)
	;   str_match('w95', Configuration, f, _, _)
	;   str_match('wnt', Configuration, f, _, _) ).

%% we duplicate fmt_write_* here because they are used early on, in libpath()
fmt_write_string(S,Fmt,A) :- formatted_io(FMT_WRITE_STRING, S,Fmt,A,_).
fmt_write(File,Fmt,A) :- formatted_io(FMT_WRITE, File,Fmt,A,_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



:-	set_inthandler('_$load_undef'(_), MYSIG_UNDEF),		% MYSIG_UNDEF
	set_current_module,
	set_current_input,
	set_current_output,
	import_standard_predicates,
	catch(loader_init([]),Ball,default_error_handler(Ball)).  % will print and exit xsb

loader_init(_) :-
	%% Load the xsb_configuration predicate
	stat_flag(CONFIG_FILE, ConfigFileC),
	intern_string(ConfigFileC, ConfigFile), % convert C string to Prolog
	reconsult(ConfigFile),
	%%
	load_library_directories,
	execute_command_module.


/*
 *  Interrupt Handlers are currently stored in flags[32-48] of the
 *  emulator.
 */

set_inthandler(Handler, Number) :-
	Flag is Number + INT_HANDLERS_FLAGS_START,
	term_psc(Handler, PSC),
	stat_set_flag(Flag, PSC).


/*
 *  Set the current module to usermod (a.k.a. global_mod).  Do this by
 *  creating a PSC record for the symbol current_module/0, and
 *  associating the PSC record of usermod/0 with that of
 *  current_module/0 (it's stored in current_module's `ep' field.)
 *  Reflect this setting in the emulator by setting
 *  flags[CURRENT_MODULE] to 0.
 */

set_current_module :-
	conpsc(current_module, CURMOD_PSC),
	conpsc(usermod, UsermodPsc),
	psc_set_prop(CURMOD_PSC, UsermodPsc),
	stat_set_flag(CURRENT_MODULE, 0).


/*
 *  Create PSCs for userin/0 and current_input/0, associate userin/0
 *  with stdin and associate current_input/0 with userin/0.  Set
 *  emulator flag flags[CURRENT_INPUT] to stdin.
 */

set_current_input :-
	conpsc(userin, UserInPSC),
	psc_set_type(UserInPSC, T_FILE),   % ``ep'' field contains file desc
	psc_set_prop(UserInPSC, STDIN),    % ``data'' field set to stdin
	conset(current_input, UserInPSC),
	stat_set_flag(CURRENT_INPUT, STDIN).


/*
 *  Create PSCs for userout/0 and current_output/0, associate
 *  userout/0 with stdout and associate current_output/0 with
 *  userout/0.  Set emulator flag flags[CURRENT_OUTPUT] to stdout.
 */

set_current_output :-
	conpsc(userout, UserOutPSC),
	psc_set_type(UserOutPSC, 5),       % ``ep'' field contains file desc
	psc_set_prop(UserOutPSC, 1),       % ``data'' field set to stdout
	conset(current_output, UserOutPSC),
	stat_set_flag(CURRENT_OUTPUT, 1).


/*
 *  Imports into usermod a bunch of basic-to-operation predicates, those
 *  listed in the file std_xsb.P using the predicate standard_symbol/3.
 */

import_standard_predicates :-
	standard_symbol(P, A, Mod),
	psc_import(P, A, Mod),
	fail.
import_standard_predicates.


/*
 *  Loads syslib/sysinitrc.P and site/siteinitrc.P, .xsb/xsbrc.P,
 *  if they are present.
 *  The emulator flag 50 is set to indicate that library_directory/1
 *  should be used for finding file paths rather than the default.
 */

load_library_directories :-
	slash(Slash),
	%%
	%% Construct system init file name
	xsb_configuration(syslibdir, SyslibDir),
	fmt_write_string(SysFileName, '%s%ssysinitrc', f(SyslibDir, Slash)),
	reconsult(SysFileName),	        % SysFileName is mandatory.
					% It is compiled if a OBJO file does
					% not exist or is obsolete.
	%%
	%% Construct site init file name
	xsb_configuration(site_dir, SiteDir),
	fmt_write_string(SiteFileName,'%s%ssiteinitrc', f(SiteDir,Slash)),
	str_cat(SiteFileName, XSB_SRC_EXTENSION_ATOM, SiteFileNameP),
	str_cat(SiteFileName, XSB_OBJ_EXTENSION_ATOM, SiteFileNameO),
	%% Site init file isn't mandatory
	( (machine_file_exists(SiteFileNameP)
	   ; machine_file_exists(SiteFileNameO)) ->
		reconsult(SiteFileName)
	        ; true
	),
	%%
	%% Construct user''s .xsb/xsbrc.P
	xsb_configuration(user_home, HomeDir),
	fmt_write_string(UsrFileN, '%s%s.xsb%sxsbrc', f(HomeDir,Slash,Slash)),
	str_cat(UsrFileN, XSB_SRC_EXTENSION_ATOM, UserFileNameP),
	str_cat(UsrFileN, XSB_OBJ_EXTENSION_ATOM, UserFileNameO),
	%% User init file isn't mandatory
	( (machine_file_exists(UserFileNameP)
	   ; machine_file_exists(UserFileNameO)) ->
		reconsult(UsrFileN)
		; true
        ).

load_library_directories.


/*****************************************************************************
** In the future, we'll add a modifier "system", so the user could indicate
** more precisely where predicates are imported from. E.g.,
**
** :- import file_write/2 from system(xsb_writ)
**
*****************************************************************************/

/*
 *  Load and execute the module specified on the command line.
 */

execute_command_module :-
	stat_flag(CMD_LOOP_DRIVER, InputFilePtr_INT),
	load(InputFilePtr_INT).




/*************************************************************************

		The XSB Dynamic Loader
		======================

The following code should be self-contained since it would be impossible
to load a predicate on demand if the mechanism by which it is loaded
requires the use of another unloaded predicate.  Therefore we are not
allowed imports and so cannot rely on code defined elsewhere in the
system.

In fact, the only predicates that can be used in this part of the file
are those that are explicitly defined or expanded (in-lined) by the
compiler as calls to C functions.  See the file cmplib/builtin.P for
XSB's in-lined predicates, builtins, and functions.

*************************************************************************/

/*
 *  Undefined Predicate Interrupt Handler
 *  -------------------------------------
 *  This is the starting point for resolving an undefined symbol.
 *  The module containing that symbol is loaded and its directives
 *  are executed.  Processing continues with the re-execution of
 *  the goal which triggered the interrupt.
 */

'_$load_undef'(Term) :-
	term_type(Term, Type),
	( Type =:= XSB_STRUCT -> term_psc(Term, PSC)
	; Type =:= XSB_STRING -> psc_insert(Term, 0, PSC, 0)
	),
	load_pred(PSC),
	call0(Term).


load_pred(PSC) :- \+ (\+ (load_pred0(PSC))).

load_pred0(PSC) :- 
	stat_flag(HITRACE, Hitrace), 	
	stat_set_flag(HITRACE, 0),	  % turn off hitrace
	load_pred1(PSC),
	stat_set_flag(HITRACE, Hitrace).  % resume hitrace status if successful

load_pred1(PSC) :-
	'_$savecp'(B),
	psc_name(PSC, Name),
	psc_arity(PSC, Arity),
	psc_env(PSC, Env), 
	( Env =:= T_UNLOADED
	->  % is "imported"
	    psc_prop(PSC, ModPSC),
	    psc_name(ModPSC, ModName),
	    psc_type(ModPSC, ModType), 
	    (ModType =:= T_MODU
	     ->	    true   % module loaded
	     ;	    load(ModName)	% might abort here 
	    ),
	    (	psc_type(PSC, NType),
		(   NType =:= T_PRED; NType =:= T_FORN; NType =:= T_DYNA ), 
		%% module loaded, and pred/foreign-pred is there
		'_$cutto'(B)
		;
		throw(error(undefined_predicate(Name,Arity,ModName),
			    ['The predicate: ', Name, '/', Arity,
			     ' is not defined in module ', ModName]))
	    /*
		call_undef_pred_handlers(PSC, ['The predicate: ',
					       Name, '/', Arity,
					       ' is not defined in module ',
					       ModName])
	    */
	    )
	;
	    throw(error(undefined_predicate(Name,Arity,usermod),
			['Undefined predicate: ', Name, '/', Arity]))
	/*
	    call_undef_pred_handlers(PSC, ['Undefined predicate: ',
					   Name, '/', Arity])
	*/
	).


/*
%% Invoke undefined pred handlers until one succeeds.
%% If the successful handler sets the 4th arg to true, then succeed
%% else fail.
:- import conget/2, conset/2 from gensym.
call_undef_pred_handlers(PSC,AbortMsg) :-
	%% the _$_$_undefhook trick guards against Undefined preds in 
	%% the hook themselves; prevents going into an infinite loop
	( not_handling_undef_pred
	; abort(['[In undefined predicate handler] ' | AbortMsg])
	),
	set_undef_pred_handling,
	!,
	( call_xsb_hook(xsb_undefined_predicate_hook(_),
			[PSC,RetCode],
			until_true),
	    reset_undef_pred_handlers
	%% found a handler that agrees to handle this predicate
	->  (   RetCode==true
	    ->  psc_type(PSC, Type1), 
		(  Type1 =< T_ORDI % pred is still undefined
		%% The handler says re-issue the call, but the 
		%% predicate remains undefined after the handler: 
		%% fail the original call
		-> fail
		%% The predicate is defined after the handler did the work
		%% So, then succeed and the call will be made
		;  true
		)
	    %% If RetCode =\= true, then make load_pred1 fail
	    %% and the call will not be made
	    ;   fail
	    )
	%% if all else fails -- abort
	;   reset_undef_pred_handlers,
	    abort(AbortMsg)
	).

reset_undef_pred_handlers :- conset('_$_$_undefhook',0).
set_undef_pred_handling :- conset('_$_$_undefhook',1).
not_handling_undef_pred :- conget('_$_$_undefhook',0).
*/

/*
 *  Loads and executes the directives of a module.  The file containing
 *  the module is constructed using search_file/3, below.
 */

load(ModName) :-
	'_$savecp'(B),
	stat_flag(DEBUG_ON, Debug_on),
	(   Debug_on =:= 1
	->  fmt_write(STDDBG, "[ load module %s ]\n", arg(ModName))
	;   true
	),
	(   search_file(ModName, FileName_O, XSB_OBJ_EXTENSION_ATOM),
	    code_load(FileName_O, Init, 0), % use the C loader
	    (	Init =\= 0	% load succeeded
	    ->	'_$cutto'(B),
		(   Init =\= 4 % prolog byte-code succeeded
		->  ll_code_call(Init, true, 2), % call '_$main'
		    unload_seg(Init) % delete code for '_$main'
		;   true	% load foreign obj code succeeded
		)
	    )
	->  true
	%% If OBJ file doesn't exist or doesn't load,
	%% but SRC file exists somewhere on the load path
	%% then recompile the SRC file and load it.
	;   (   search_file(ModName, FileName_P, XSB_SRC_EXTENSION_ATOM),
		machine_file_exists(FileName_P)
	    ->	reconsult(FileName_P)
	    )
	;   tilde_expand_filename(ModName,ExpandedModName),
	    throw(error(existence_error(module,ExpandedModName),
			['Module ', ExpandedModName,
			 ' is not found in XSB library directories']))
	).

/*
 *  To use code_call/3, it must be the last clause in a rule.
 *  Here we force the issue.
 */

ll_code_call(A,B,C) :- code_call(A,B,C).



/*
 *  Takes a filename (INT or STRING encoded pointer) and returns an
 *  absolute filename (STRING) with Extension
 *  (e.g., XSB_SRC_EXTENSION_ATOM, XSB_OBJ_EXTENSION_ATOM) added to it.
 *  The extension should NOT already be a part of the incoming ModName.
 */

search_file(ModName, AbsFileName, Extension) :- 
	tilde_expand_filename(ModName,ExpandedModName),
	( is_absolute_filename(ExpandedModName)
	->  WholePath = ExpandedModName
	;   libpath(LibPath),
	    str_cat(LibPath, ExpandedModName, WholePath)
	),
	str_cat(WholePath, Extension, AbsFileName).


/*
 *  Used in searching for a file.  Directory paths are returned in the
 *  following manner:
 *
 *      Return in the order specified through the predicate
 *	library_directory/1, followed by the default paths
 *      in this default order:
 *	  lib/   syslib/   cmplib/   ./
 */

libpath(P) :- libraries_loaded, 
	      library_directory(Dir), 
	      expand_filename(Dir, P1),
	      dirname_canonic(P1, P).
libpath(P) :- stat_flag(INSTALL_DIR, InstallDir1),
	      intern_string(InstallDir1, InstallDir),
	      slash(Slash),
	      %%<installdir>/syslib/  (slash type depends on the OS)
	      fmt_write_string(P,'%s%ssyslib%s',f(InstallDir, Slash, Slash)).
libpath(P) :- stat_flag(INSTALL_DIR, InstallDir1),
	      intern_string(InstallDir1, InstallDir),
	      slash(Slash),
	      %% <installdir>/cmplib/  (slash type depends on the OS)
	      fmt_write_string(P,'%s%scmplib%s',f(InstallDir, Slash, Slash)).
libpath(P) :- stat_flag(INSTALL_DIR, InstallDir1),  % get the installation dir
	      intern_string(InstallDir1, InstallDir),
	      slash(Slash),
	      %% <installdir>/lib/  (slash type depends on the OS)
	      fmt_write_string(P, '%s%slib%s', f(InstallDir, Slash, Slash)).
libpath(P) :- slash(Slash), str_cat('.', Slash,P).

/*
 *  Determine whether the ~/.xsbrc module has been loaded.
 */

libraries_loaded :- 
	stat_flag(LIBS_LOADED, Lib_Loaded),
	Lib_Loaded =:= 1.

/*======================================================================*/

