/* File:      shell.P
** Author(s): Kostis F. Sagonas, Jiyang Xu, kifer
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: shell.P,v 1.18 2003/06/09 16:38:15 lfcastro Exp $
** 
*/


:- compiler_options([xpp_on,sysmod]).

#include "celltags_xsb.h"
#include "system_defs_xsb.h"
#include "builtin.h"
#include "syscall_xsb.h"
#include "extensions_xsb.h"


/* Create subprocess. Redirect its standard input to StreamToProc
   and standard output to StreamFromProc. Prolog program can then read 
   from StreamFromProc and write to StreamToProc.
   +CmdAndArglist, -StreamToProc,-StreamFromProc,-StreamFromStderr,-ProcessId
   Fails, if can't create subprocess
*/
spawn_process(Cmd, StreamToProc, StreamFromProc, StreamFromStderr, Pid) :-
	sys_system(SPAWN_PROCESS,
		   Cmd,StreamToProc,StreamFromProc,StreamFromStderr,Pid,_).

/* Like spawn_process, but returns error code instead of Pid.
   XSB doesn't know where the corresponding processes are.
   However, unlike spawn_process, you can pass any shell command, 
   including pipes and redirection. */
shell(Cmd,StreamToProc, StreamFromProc, StreamFromStderr, ErrCode) :-
	sys_system(SHELL,
		   Cmd,StreamToProc,StreamFromProc,StreamFromStderr,ErrCode,_).
shell(Cmd, ErrCode) :- shell(Cmd, none, none, none, ErrCode).
shell(Cmd) :- shell(Cmd, none, none, none, _).

exec(Cmd) :- sys_system(EXEC,Cmd,_,_,_,_,_).

/* Process table is a list [process(Pid,To,From,Stderr,Cmdline), ...]
   that represents all active subprocesses */
get_process_table(ProcessList) :-
	sys_system(GET_PROCESS_TABLE, ProcessList, _,_,_,_,_).

%% obtain process status
process_status(Pid, Status) :- sys_system(PROCESS_STATUS,Pid,Status,_,_,_,_).

%% Send signal to process. Currently wait, kill
process_control(Pid, Signal) :- sys_system(PROCESS_CONTROL,
					   Pid, Signal, _,_,_,_).

%% Sleep for the specified number of seconds
sleep(Seconds) :- sys_system(SLEEP_FOR_SECS, Seconds,_,_,_,_,_).
is_plain_file(File) :- sys_system(IS_PLAIN_FILE, File,_,_,_,_,_).
is_directory(File) :- sys_system(IS_DIRECTORY, File,_,_,_,_,_).

list_directory(Dir,File) :-
	(   (   atom(Dir)
	    ;   is_list(Dir),
		ground(Dir)
	    ),
	    var(File)
	->  (   xsb_configuration(host_os,windows)
	    ->	(   atom(Dir)
		->  DDir = [Dir, '/*'] /* */
		;   append(Dir,'/*',DDir) /* */
		)
	    ;	DDir = Dir
	    ),
	    sys_system(LIST_DIRECTORY,Handle,DDir,FirstFile,_,_,_),
	    (	File = FirstFile
	    ;	list_directory_1(DDir,File,Handle)
	    )
	;   abort('list_directory(+Dir,-File) called with wrong modes.')
	).
list_directory_1(Dir,File,Handle) :-
	(   sys_system(LIST_DIRECTORY,Handle,Dir,NewFile,_,_,_)
	->  (   File = NewFile
	    ;	list_directory_1(Dir,File,Handle)
	    )
	;   fail
	).

cd(Path) :-
	(   var(Path)
	->  abort('Uninstantiated argument 1 of cd/1')
	;   (   expand_filename(Path,Dir)
	    ->	sys_chdir(Dir,Res),
		Res =:= 0
	    ;	abort('Wrong type in argument 1 of cd/1')
	    )
	).
% 	( (atom(Path); is_list(Path)) -> 
% 		expand_filename(Path, Dir), sys_chdir(Dir, Res), Res=:=0
% 	; var(Path) ->
% 		abort('Uninstantiated argument 1 of cd/1')
% 	; abort('Wrong type in argument 1 of cd/1')
% 	).

rename(Old, New) :- sys_rename(Old, New, Res), Res =:= 0.

ls :-
	(   xsb_configuration(host_os,windows)
	->  shell('dir', Res)
	;   shell('ls -F',Res)
	),
	Res =:= 0.

rm(Name) :- sys_unlink(Name, Res), Res =:= 0.

%% current working directory
cwd(Name) :- sys_cwd(Name, Res), Res =:= 0.

edit(File) :- 
	expand_filename(File, FileName), 
	shell(['$EDITOR ', '$EDITOR_OPTIONS ', FileName, XSB_SRC_EXTENSION_ATOM]).


%% sys_exit(-ExitCode) exit subprocess unconditionally with ExitCode
sys_exit(ExitCode) :- sys_syscall(SYS_exit,_,ExitCode,_).

%% Get process id -- Unix only
sys_pid(Pid)     	    :-	sys_syscall(SYS_getpid, Pid, _, _).
%% Link file -- Unix only
sys_link(Source, Dest, Res) :-	sys_syscall(SYS_link, Res, Source, Dest).
sys_unlink(Path, Res) 	    :-	sys_syscall(SYS_unlink, Res, Path, _).
sys_chdir(Dir, Res)	    :- 	sys_syscall(SYS_chdir, Res, Dir, _).
sys_access(File, Mode, Res) :- 	sys_syscall(SYS_access, Res, File, Mode).
sys_stat(Path, Buff, Res)   :-	sys_syscall(SYS_stat, Res, Path, Buff).
sys_rename(Old, New, Res)   :-	sys_syscall(SYS_rename, Res, Old, New).
/* current working dir. Res is 0 on success, 1 on failure */
sys_cwd(Name, Res)   	    :-	sys_syscall(SYS_cwd, Res, Name, _).
sys_mkdir(Name, Res)   	    :-	sys_syscall(SYS_mkdir, Res, Name, _).
sys_rmdir(Name, Res)   	    :-	sys_syscall(SYS_rmdir, Res, Name, _).

