/* File:      string.P
** Author(s): Kostis Sagonas, Jiyang Xu
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: string.P,v 1.25 2002/11/11 22:04:17 dwarren Exp $
** 
*/

/*----------------------------------------------------------------------*/
/* NOTE: This file HAS TO be compiled with the "sysmod" option.		*/
/*----------------------------------------------------------------------*/

:- compiler_options([sysmod]).


%--- 
/* 
TLS 7/99 This version is linear rather than quadratic and seems
much faster than the old one (which I must confess to writing).  Also,
this is well-behaved for memory management since it no longer uses
str_cat or str_len.
*/
 
concat_atom(Atomlist,Atom):-
	atom_to_charlist(Atomlist,Charlist1),
	flatten(Charlist1,Charlist),
	atom_codes(Atom,Charlist).

atom_to_charlist([],[]).
atom_to_charlist([Atom|Rest],[List|Lrest]):- 
	(atom(Atom)
	 ->	atom_codes(Atom,List)
	 ; number(Atom)
	 ->	number_codes(Atom,List)
	),
	atom_to_charlist(Rest,Lrest).

%---
concat_atom(Atomlist,Sep,Atom):-
	atom(Sep),
	atom_codes(Sep,Slist),
	atom_to_charlist(Atomlist,Slist,Charlist1),
	flatten(Charlist1,Charlist),
	atom_codes(Atom,Charlist).

atom_to_charlist([],_,[]).
atom_to_charlist([Atom],_,[List]):- !,
	(atom(Atom)
	 ->	atom_codes(Atom,List)
	 ; number(Atom)
	 ->	number_codes(Atom,List)
	).
atom_to_charlist([Atom|Rest],Slist,[List,Slist|Lrest]):- 
	(atom(Atom)
	 ->	atom_codes(Atom,List)
	 ; number(Atom)
	 ->	number_codes(Atom,List)
	),
	atom_to_charlist(Rest,Slist,Lrest).

%% Get substring of String beginning at from and ending at To.
%% Negative "To" counts from the end. Unbount "To" means use the end of string
substring(String,From,To,Subst) :- substring(String,From,To,Subst).
string_substitute(String,SubstSpec,SubstList,ResultStr) :-
	string_substitute(String,SubstSpec,SubstList,ResultStr).
str_length(Str, Len) :- str_len(Str, Len).

str_cat(S1, S2, S3) :- str_cat(S1, S2, S3).
%% Only checks the first match
str_match(Sub,Str,Direction,Beg,End) :- str_match(Sub,Str,Direction,Beg,End).
str_sub(S1, S2, Pos) :- str_match(S1, S2, forward, Pos, _).
str_sub(S1, S2) :- str_match(S1, S2, forward, _, _).

/***************************************************************/
/*                                                             */
/* term_to_codes (term_to_atom) convert a general XSB term     */
/* into a list of codes (an atom) in canonical term form.      */
/* (It does NOT handle floats.)                                */
/* Terms to be converted should not contain unary function     */
/* symbols named ' _$var$_ '.  				       */
/*                                                             */
/***************************************************************/

term_to_atom(Term,Atom) :-
	copy_term(Term,Term1),  % protect Term's variables
	term_to_codes(Term1,Codes),
	atom_codes(Atom,Codes).

term_to_codes(Term,Codes) :-
	term_to_codes(Term,0,_,Codes,[]).

term_to_codes(Term,Var0,Var,Codes0,Codes) :-
	(var(Term)
	 ->	Term = ' _$var$_ '(Var0),
		Var is Var0 + 1,
		Codes0 = [0'_,0'h|Codes1],
		number_codes(Var0,IntCodes),
		append(IntCodes,Codes,Codes1)
	 ;	Term = ' _$var$_ '(I)
	 ->	Var = Var0,
		Codes0 = [0'_,0'h|Codes1],
		number_codes(I,IntCodes),
		append(IntCodes,Codes,Codes1)
	 ;	integer(Term)
	 ->	Var = Var0,
		number_codes(Term,IntCodes),
		append(IntCodes,Codes,Codes0)
	 ;	atom(Term)
	 ->	Var = Var0,
		extern_atom_codes(Term,Codes0,Codes)
	 ;	float(Term)
	 ->	Var = Var0,
		extern_float_codes(Term,Codes0,Codes)
	 ;	is_list(Term)
	 ->	Term=[Term1|Terms],
		Codes0 = [0'[|Codes1],
		term_to_codes(Term1,Var0,Var1,Codes1,Codes2),
		cvt_termlist_to_codes(Terms,Var1,Var,Codes2,[0']|Codes])
	 ;	structure(Term)
	 ->	Term =.. [Fun,Arg1|Args],
		extern_atom_codes(Fun,Codes0,[0'(|Codes2]),
		term_to_codes(Arg1,Var0,Var1,Codes2,Codes3),
		cvt_termlist_to_codes(Args,Var1,Var,Codes3,[0')|Codes])
	).

cvt_termlist_to_codes([],Var,Var,Codes,Codes).
cvt_termlist_to_codes([Arg|Args],Var0,Var,[0',|Codes0],Codes) :-
	term_to_codes(Arg,Var0,Var1,Codes0,Codes1),
	cvt_termlist_to_codes(Args,Var1,Var,Codes1,Codes).

extern_atom_codes(Atom,Codes0,Codes) :-
	atom_codes(Atom,AtomCodes),
	(if_not_need_quote(AtomCodes)
	 ->	append(AtomCodes,Codes,Codes0)
	 ;	Codes0 = [0''|Codes1],
		copy_double_quote(AtomCodes,Codes1,[0''|Codes])
	).

if_not_need_quote([0'[,0']]) :- !.
if_not_need_quote([C|Cs]) :-
	C >= 0'a, C =< 0'z,
	if_not_need_quote1(Cs).

if_not_need_quote1([]).
if_not_need_quote1([C|Cs]) :-
	(C >= 0'a, C =< 0'z
	 ->	true
	 ; (C >= 0'A, C =< 0'Z)
	 ->	true
	 ; C =:= 0'_
	 ->	true
	 ; (C >= 0'0, C =< 0'9)
	 ->	true
	),
	if_not_need_quote1(Cs).

copy_double_quote([],Cs,Cs).
copy_double_quote([C|Cs],[C|Cs0],Cs1) :-
	(C =:= 0''
	 ->	Cs0 = [0''|Cs0a],
		copy_double_quote(Cs,Cs0a,Cs1)
	 ;	copy_double_quote(Cs,Cs0,Cs1)
	).

extern_float_codes(Float,Codes0,Codes) :-
	Log10 is log(10),
	Exp is floor(log(Float)/Log10-6), % to get 6 or 7 digits, Put . Exp to left from end
	DigitsInt is floor(Float/exp(Exp*Log10)),
	number_codes(DigitsInt,Digits),
	length(Digits,Length),
	Dist is Length+Exp,
	(Dist > 0
	 ->	place_decimal(Digits,Dist,Codes0,Codes)
	 ;	Codes0 = [0'0,0'.|Codes1],
		prefix_decimal(Digits,Dist,Codes1,Codes)
	).

place_decimal([],Dist,Codes0,Codes) :-
	(Dist > 0
	 ->	Dist1 is Dist - 1,
		Codes0 = [0'0|Codes1],
		place_decimal([],Dist1,Codes1,Codes)
	 ;	Codes0 = [0'.,0'0|Codes]
	).
place_decimal([D|Digits],Dist,Codes0,Codes) :-
	(Dist > 0
	 ->	Dist1 is Dist - 1,
		Codes0 = [D|Codes1],
		place_decimal(Digits,Dist1,Codes1,Codes)
	 ; 	append([0'.,D|Digits],Codes,Codes0)
	).
		
prefix_decimal(Digits,Dist,Codes0,Codes) :-
	(Dist =:= 0
	 ->	append(Digits,Codes,Codes0)
	 ;	Dist1 is Dist + 1,
		Codes0 = [0'0|Codes1],
		prefix_decimal(Digits,Dist1,Codes1,Codes)
	).


/***************************************************************/
/*                                                             */
/* codes_to_term (atom_to_term) convert a list of codes (atom) */
/* representing a canonical term into an XSB term.             */
/* (It handles floats only in xx.xxx form.)                    */
/*                                                             */
/***************************************************************/

atom_to_term(Atom,Term) :-
	atom_codes(Atom,AtomC),
	codes_to_term(AtomC,Term).

codes_to_term(Codes,Term) :- codes_to_term(Codes,[],[],_,Term).

codes_to_term(Rest0,Rest,ST0,ST,Term) :-
	scan_blanks(Rest0,Rest1),
	codes_to_term_d(Rest1,Rest,ST0,ST,Term).

codes_to_term_d([0'[|Rest0],Rest,ST0,ST,Term) :- !,
	(Rest0 = [0']|Rest]
	 ->	Term = [],
		ST = ST0
	 ;	codes_to_term(Rest0,Rest1,ST0,ST1,Tok1),
		codes_to_term_list(Rest1,[0']|Rest2],Toks,[],ST1,ST),
		scan_blanks(Rest2,Rest),
		Term = [Tok1|Toks]
	).
codes_to_term_d(Rest0,Rest,ST0,ST,Term) :-
	next_token(Rest0,Rest1,ST0,ST1,Tok0),
	(Rest1 = [0'(|Rest2]
	 ->	codes_to_term(Rest2,Rest3,ST1,ST2,Tok1),
		codes_to_term_list(Rest3,[0')|Rest4],Terms,[],ST2,ST),
		scan_blanks(Rest4,Rest),
		Term =.. [Tok0,Tok1|Terms]
	 ;	Rest = Rest1,
		ST = ST1,
		Term = Tok0
	).

codes_to_term_list(Rest0,Rest,Terms0,Terms,ST0,ST) :-
	Rest0 = [C|Rest1],
	(C =:= 0') 
	 ->	Rest = Rest0,
		Terms = Terms0,
		ST = ST0
	 ;	C =:= 0'] 
	 ->	Rest = Rest0,
		Terms = Terms0,
		ST = ST0
	 ;	C =:= 0',,
	        codes_to_term(Rest1,Rest2,ST0,ST1,Tok),
		Terms0 = [Tok|Terms1],
		codes_to_term_list(Rest2,Rest,Terms1,Terms,ST1,ST)
	).
	
next_token(Rest0,Rest,ST0,ST,Tok) :-
	Rest0 = [C|Rest1],
	(C >= 0'0, C =< 0'9
	 ->	Val is C - 0'0,
		ST = ST0,
		scan_number(Rest1,Rest,Val,Tok)
	 ; (C =:= 0'_ ; (C >= 0'A , C =< 0'Z))
	 ->	scan_variable([C|Rest1],Rest,ST0,ST,Tok)
	 ; C =:= 0''
	 ->	ST = ST0,
		scan_quoted_con(Rest1,Rest,Codes,[]),
		atom_codes(Tok,Codes)
	 ; C =:= 0'.
	 ->	ST = ST0,
		scan_decimal(Rest1,Rest,1.0,0.0,Tok)
	 ; (C =:= 0'- , Rest1 = [C1|Rest2], C1 >= 0'0, C1 =< 0'9)
	 ->	Val is C1 - 0'0,
		scan_number(Rest2,Rest,Val,MTok),
		ST = ST0,
		Tok is -MTok
	 ;	ST = ST0,
		scan_to_delim(Rest0,Rest,Codes,[]),
		atom_codes(Tok,Codes)
	).

scan_number(Rest0,Rest,Val0,Val) :-
	(Rest0 = [D|Rest1]
	 ->	(D >= 0'0, D =< 0'9
		 ->	Val1 is 10*Val0+D-0'0,
			scan_number(Rest1,Rest,Val1,Val)
		 ; D =:= 0'.
		 ->	Val1 is float(Val0),
			scan_decimal(Rest1,Rest,1.0,Val0,Val)
		 ;	Rest = Rest0,
			Val = Val0
		)
	 ;	Rest = Rest0,
		Val = Val0
	).

scan_decimal(Rest0,Rest,Dec,Val0,Val) :-
	(Rest0 = [D|Rest1], D >= 0'0, D =< 0'9
	 ->	Dec1 is Dec / 10,
		Val1 is Val0 + (D - 0'0) * Dec1,
		scan_decimal(Rest1,Rest,Dec1,Val1,Val)
	 ;	Rest = Rest0,
		Val = Val0
	).

scan_variable(Rest0,Rest,ST0,ST,Var) :-
	scan_to_delim(Rest0,Rest,Str,[]),
	(Str == "_"
	 ->	ST = ST0
	 ; 	atom_codes(VName,Str),
		(memberchk(VName=Var,ST0)
		 ->	ST = ST0
		 ;	ST = [VName=Var|ST0]
		)
	).

scan_to_delim(Rest0,Rest,Str0,Str) :-
	(Rest0 = [C|Rest1]
	 ->	((C =:= 0'( ; C =:= 0', ; C =:= 0') ; C =:= 0'])
		 ->	Rest = Rest0,
			Str = Str0
		 ; C =:= 0'  %
		 ->	scan_to_delim(Rest1,Rest,Str0,Str)
		 ;	Str0 = [C|Str1],
			scan_to_delim(Rest1,Rest,Str1,Str)
		)
	 ;	Rest = Rest0,
		Str = Str0
	).

scan_quoted_con([C|Rest0],Rest,Str0,Str) :-
	(C =:= 0''
	 ->	(Rest0 = [0''|Rest1]
		 ->	Str0 = [0''|Str1],
			scan_quoted_con(Rest1,Rest,Str1,Str)
		 ;	Rest = Rest0,
			Str = Str0
		)
	 ;	Str0 = [C|Str1],
		scan_quoted_con(Rest0,Rest,Str1,Str)
	).

scan_blanks(Rest0,Rest) :-
	(Rest0 = [0' |Rest1]
	 ->	scan_blanks(Rest1,Rest)
	 ;	Rest = Rest0
	).

/***************************************************************/
/*                                                             */
/* read_atom_to_term converts an atom containing the           */
/* characters of a valid readable term (without a terminating  */
/* '.') to a term (3-arg returns variable names).  It uses     */
/* file_read_foe, using the current operators.  It fails if    */
/* there is an error.                                          */
/*                                                             */
/***************************************************************/

read_atom_to_term(Atom,Term) :-
	read_atom_to_term(Atom,Term,_).

read_atom_to_term(Atom,Term,Vars) :-
	file_open([Atom,'.'],sr,FileDes),
	(file_read_foe(FileDes,Term,Vars)
	 ->	file_close(FileDes)
	 ;	file_close(FileDes),
		fail
	).
		  

/* ---------------------- end of file string.P ------------------------ */
