/* File:      x_interp.P
** Author(s): David S. Warren, Terrance Swift,
**    	      Ernie Johnson, Jiyang Xu, Kostis Sagonas
** Contact:   xsb-contact@cs.sunysb.edu
** 
** Copyright (C) The Research Foundation of SUNY, 1986, 1993-1998
** Copyright (C) ECRC, Germany, 1990
** 
** XSB is free software; you can redistribute it and/or modify it under the
** terms of the GNU Library General Public License as published by the Free
** Software Foundation; either version 2 of the License, or (at your option)
** any later version.
** 
** XSB is distributed in the hope that it will be useful, but WITHOUT ANY
** WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
** FOR A PARTICULAR PURPOSE.  See the GNU Library General Public License for
** more details.
** 
** You should have received a copy of the GNU Library General Public License
** along with XSB; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** $Id: x_interp.P,v 1.46 2002/11/04 18:09:14 dwarren Exp $
** 
*/

:- compiler_options([xpp_on,sysmod,optimize]).

#include "sig_xsb.h"
#include "flag_defs_xsb.h"
#include "heap_defs_xsb.h"
#include "standard.h"
#include "char_defs.h"

/*======================================================================*/
/*  This file contains the main program of the interpreter.		*/
/*======================================================================*/

:- interpreter.

interpreter :-
        set_inthandler('_$keyboard_int'(_,_), MYSIG_KEYB),
	set_inthandler('_$attv_int'(_,_), MYSIG_ATTV),
	( banner_control(nobanner)
	->  true		% don't print banner
	;   version_message
	),
        prompt('| ?- ',[]),
        prompt('',[],'_$more_prompt'(_)),
        prompt(' ? ',[],'_$debug_prompt'(_)),
        fail.

interpreter :-
	retractall('$$exception_ball'(_)),
	catch(interpreter_cmd_goal([]),Ball,default_error_handler(Ball)),
	fail.
interpreter :-
	retractall('$$exception_ball'(_)),
	x_interp_repeat,
	catch(interpreter_readloop([]),Ball,default_error_handler(Ball)),
	fail.

%% dummy arg to circumvent string flaw with module system, for catch
interpreter_cmd_goal(_) :-
        %% Execute command line goal passed to XSB with the -e option
        xsb_flag(goal, CmdGoal),
        (CmdGoal == '' ->
            true
        ;
	    ( banner_control(quietload)
	    -> true
	    ;  file_nl(STDMSG),
	       file_write(STDMSG, 'Evaluating command line goal:  '),
	       file_nl(STDMSG)
	    ),
	    %% Don't print prompt, if --noprompt specified on cmd line
	    %% Useful when spawning xsb subprocesses, 
	    %% so they won't send junk back to the parent.
	    (	banner_control(noprompt)
	    ->	true
	    ;	file_write(STDFDBK, '| ?- ')
	    ),
	    (  banner_control(quietload)
	    -> true
	    ;  file_write(STDMSG, CmdGoal), file_nl(STDMSG)
	    ),
            file_open(CmdGoal, sr, FileDes),
	    %% promote I/O port to stream
	    ioport2iostream(FileDes,CmdGoalStream),
	    %% make CmdGoalStream the current input stream;s
	    %% push the old stream on cmd_line_argument stack
	    push_stdin(cmd_line_argument, CmdGoalStream),
            file_nl(STDMSG)
        ),
        fail.

%% dummy arg to circumvent string flaw with module system, for catch
interpreter_readloop(_) :-
	%% don't set MYSIG_PSC earlier, because XSB will start calling 
	%% the handler when it starts up, which will cause a lot of grief
	x_interp_repeat,
	trimcore,
	stat_set_flag(HIDE_STATE, 1), %% hide call interceptions.
	stat_set_flag(INVOKE_NUM, 0),
	conset('_$break_level', 0),
	print_debug_state,
	prompt(Prompt,Prompt),
	%% Hide prompt, if --noprompt
	(   banner_control(noprompt)
	->  true
	;   file_write(STDFDBK, Prompt), file_flush(STDFDBK, _)
	),
	stat_flag(CURRENT_INPUT, FileIn),
	file_read(FileIn, X, Vars),
	( nonvar(X), read_in_stop(X), halt, ! %% end of top loop; halt
	; process_query(X, Vars)
	).

read_in_stop(halt) :-          %% quit XSB, regardless of break level
	call_xsb_hook(xsb_exit_hook(_),all),
	halt.
read_in_stop(end_of_file) :-   %% pop a break level, out if at top
	%% if we previously pushed an input stream onto cmd_line_argument stack
	%% then pop it back and make this stream current
	(pop_stdin(cmd_line_argument) ->  fail
	; call_xsb_hook(xsb_exit_hook(_),all)
	).


process_query(end_of_file,_) :- !, file_nl(STDFDBK), fail.
process_query(X, Vars) :-
	exclude__(Vars,PrintableVars), process_query1(X,PrintableVars).

process_query1(X, []) :-	%% no printable variables appeared in the input
	!,
	call_query(X),
	close_open_tables,
	!,
	file_nl(STDFDBK), file_write(STDFDBK, yes), file_nl(STDFDBK),
	fail.
process_query1(X, Vars) :- 
	prompt(MorePrompt,MorePrompt,'_$more_prompt'(_)),
	call_query(X),
	stat_flag(LETTER_VARS, LetterVars),
	( LetterVars =:= 0 -> print_answer(Vars)
	; ( numbervars(Vars), print_answer(Vars), fail ; true )
	),
	file_write(STDFDBK, MorePrompt),
	file_flush(STDFDBK, _),
	no_more,
	close_open_tables,
	!,
	file_nl(STDFDBK), file_write(STDFDBK, yes), file_nl(STDFDBK),
	fail.

no_more :-
	file_get(0, X),
	( X =:= CH_EOF_P /* end_of_file */ ; X =:= CH_NEWLINE ), !.
no_more :- no_more, fail.

call_query(X) :- 
%%	'_$savecp'(C),			%% moved into call/1, affect trace?
	( stat_set_flag(HIDE_STATE, 0)         %% expose tracing
	; stat_set_flag(HIDE_STATE, 1), fail   %% hide again when backtrack
	),
	'_$call'(X), 
	( stat_set_flag(HIDE_STATE, 1)	       %% hide tracing when succeed
	; stat_set_flag(HIDE_STATE, 0), fail   %% expose when backtrack to user
	).
call_query(_) :- 
	file_nl(STDMSG), file_write(STDFDBK, no), file_nl(STDFDBK), fail.

%% hidden version of call
'_$call'(Y) :- 
	'_$savecp'(C), 
	goal_cut_trans(Y,X,C),
	call_expose(X).

%% first argument is an open ended list
exclude__([], []) :- !.
exclude__([V|Vs], PVs) :-
	V = vv(Name,_),
	atom_codes(Name, [Code|_]),
	( Code =:= CH_UNDERSCORE -> PVs = NPVs ; PVs = [V|NPVs] ),
	exclude__(Vs, NPVs).

print_answer([]) :- !.
print_answer([vv(Name,Val)|Tail]) :-
	file_nl(STDFDBK), file_write(STDFDBK, Name), 
	file_write(STDFDBK, ' = '), file_write(STDFDBK, Val),
	print_answer(Tail).


/* === break =========================================================== */

break :-
	seeing(InpFile), telling(OutFile),
	see(userin), tell(userout),
	conget('_$break_level', Blevel),
	Nblevel is Blevel+1, conset('_$break_level', Nblevel),
	file_write(STDMSG, '[ Break (level '), file_write(STDMSG, Nblevel),
	file_write(STDMSG, ') ]'), file_nl(STDMSG),
	readl_brklp1,
	conset('_$break_level', Blevel),
	file_write(STDMSG, '[ End break (level '), file_write(STDMSG, Nblevel),
	file_write(STDMSG, ') ]'), file_nl(STDMSG),
	seen, told,
	see(InpFile), tell(OutFile).

readl_brklp1 :- 
	x_interp_repeat, 
	conget('_$break_level', Blevel),
	file_write(STDFDBK, Blevel),
	file_write(STDFDBK, ': ?- '), file_flush(STDFDBK, _),
	stat_flag(CURRENT_INPUT, FileIn),
	file_read(FileIn, X, Vars),
	(   nonvar(X), read_in_stop(X), !
	%% end read loop; pop a break level, out if X=halt.
	;   process_query(X, Vars)
	).

/* === interrupt handlers ============================================== */

'_$keyboard_int'(Call,_Code) :- 
	is_xwammode(Flag),
	( Flag =:= 1 -> abort ; break, call_c(Call) ).

'_$attv_int'(Intlist, Call) :-
% 	file_write(1, '.... in _$attv_int/2'), file_nl(1),
% 	file_write(1, '.... interrupt chain is : '),
% 	file_write(1, Intlist), file_nl(1),
	handle_interrupts(Intlist),
%	file_write(1, '.... leaving _$attv_int/2'), file_nl(1),
	call_c(Call).

handle_interrupts([]) :- !.
handle_interrupts([[Var|Value]|Ints]) :-
% 	file_write(1, '.... [Var|Value] = '),
% 	file_write(1, [Var|Value]), file_nl(1),
	verify_attributes(Var, Value),
	handle_interrupts(Ints).


/* === version message ================================================= */

version_message :-
	file_nl(STDMSG),
	slash(Slash),
	%% Print the copyright notice
	%%
	%% xsb_configuration(etcdir, EtcDir),
	%% fmt_write_string(CopyingMsg, '%s%scopying.msg', f(EtcDir,Slash)),
	%% print_file_contents(CopyingMsg, STDMSG),
	%% file_nl(STDMSG),
	%%
	%% Do the banner
	xsb_configuration(config_dir, ConfigDir),
	fmt_write_string(ConfMsg, '%s%sbanner.msg', f(ConfigDir,Slash)),
	stat_flag(GARBAGE_COLLECT, GarbageType),
	( GarbageType == COPYING_GC
	->  GC_string = 'copy'
	; GarbageType == SLIDING_GC
	->  GC_string = 'slide'
	; GarbageType == INDIRECTION_SLIDE_GC
	->  GC_string = 'indirection'
	; GC_string = 'none'
	),
	%% Read the banner file
	file_open(ConfMsg, r, Handle),
	file_getbuf_atom(Handle, 500, Str, _),
	file_close(Handle),
	%% Print the banner
	fmt_write(STDMSG, Str, GC_string),
	file_nl(STDMSG).

/* === utilities ======================================================= */

%% The interpreter NEEDS this repeat: do not use the one in standard.
x_interp_repeat :- true ; x_interp_repeat.

prompt(New,Old) :- prompt(New,Old,'_$main_prompt'(_)).

prompt(New,Old,Pstore) :- 
	(Old == []
	 ->	true
	 ;	get_prompt(Pstore,Old)
	),
	(Old \== New
	 ->	set_prompt(Pstore,New)
	 ;	true
	).

get_prompt(Pstore,Prompt) :-
	term_psc(Pstore, Psc),
	psc_prop(Psc, Buff),	
	intern_string(Buff, String),	%% temp solution, since ALIAS not work
	Prompt = String.		%% for double word.

set_prompt(Pstore,Prompt) :-
	term_psc(Pstore, Psc),
	psc_set_type(Psc, 8),			%% T_CSET
	conname(Prompt, String),
	psc_set_prop(Psc, String).

print_debug_state :-
	stat_flag(TRACE,Trace),
	stat_flag(DEBUG_ON,Debug),
	stat_flag(QUASI_SKIPPING,QuasiSkip),
	( (Trace > 0, QuasiSkip =:= 0)
	  ->	file_write(STDDBG,[trace]), file_nl(STDDBG)
	  ; (Debug > 0; (Trace > 0, QuasiSkip > 0))
	  ->	file_write(STDDBG,[debug]), file_nl(STDDBG)
	; true
	).

