/*
 * Copyright(c) 1992 Bell Communications Research, Inc. (Bellcore)
 * Copyright(c) 1995-99 Andrew Lister
 * Copyright  1999, 2000, 2001, 2002, 2003 by the LessTif Developers.
 *
 *                        All rights reserved
 * Permission to use, copy, modify and distribute this material for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of Bellcore not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * Bellcore.
 *
 * BELLCORE MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL BELLCORE OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES RELAT-
 * ING TO THE SOFTWARE.
 *
 * MatrixWidget Author: Andrew Wason, Bellcore, aw@bae.bellcore.com
 *
 * $Id: Utils.c,v 1.61 2004/04/12 05:49:28 dannybackx Exp $
 */

/*
 * Utils.c created by Andrew Lister (7 August, 1995)
 */

#ifdef HAVE_CONFIG_H
#include <XbaeConfig.h>
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <stdlib.h>
#include <assert.h>

#include <Xm/Xm.h>
#include <Xm/ScrollBar.h>

#include <Xbae/MatrixP.h>
#include <Xbae/Macros.h>
#include <Xbae/Utils.h>
#include <Xbae/Actions.h>

#include <XbaeDebug.h>

int  
xbaeCellClip(XbaeMatrixWidget mw, int row, int column)
{
    int clip = CLIP_NONE;
    if (IS_LEADING_FIXED_ROW(mw, row)) {
        clip |= CLIP_FIXED_ROWS;
    } else if (IS_TRAILING_FIXED_ROW(mw, row)) {
        clip |= CLIP_TRAILING_FIXED_ROWS;
    }

    if (IS_LEADING_FIXED_COLUMN(mw,column)) {
        clip |= CLIP_FIXED_COLUMNS;
    } else if (IS_TRAILING_FIXED_COLUMN(mw, column)) {
        clip |= CLIP_TRAILING_FIXED_COLUMNS;
    }
    
    return clip;
}

/*
 * Returns the rectange corresponding to a given location
 * FIXME: for non fixed rows/columns I'm not sure if it's the formula for y2 or x2 that is right 
 */

Boolean 
xbaeSetClipRect(XbaeMatrixWidget mw, Rectangle *rect_p,int clip)
{
    Dimension x1,y1,x2,y2;
    Boolean ret = True;

    if (clip & CLIP_FIXED_ROWS && mw->matrix.fixed_rows) {
        y1 = ROW_LABEL_OFFSET(mw);
        y2 = FIXED_ROW_LABEL_OFFSET(mw) - 1;
    } else if (clip & CLIP_TRAILING_FIXED_ROWS && mw->matrix.trailing_fixed_rows) {
        y1 = TRAILING_FIXED_ROW_LABEL_OFFSET(mw);
        y2 = TRAILING_FIXED_ROW_LABEL_OFFSET(mw) + TRAILING_FIXED_ROW_HEIGHT(mw) - 1;
    } else if (clip & CLIP_VISIBLE_HEIGHT && mw->matrix.rows - mw->matrix.fixed_rows - mw->matrix.trailing_fixed_rows) {
        y1 = FIXED_ROW_LABEL_OFFSET(mw);
        y2 = FIXED_ROW_LABEL_OFFSET(mw) + VISIBLE_HEIGHT(mw) - 1;
    } else if (clip & CLIP_COLUMN_LABELS && mw->matrix.column_labels) {
        y1 = HORIZ_SB_OFFSET(mw);
        y2 = HORIZ_SB_OFFSET(mw) + COLUMN_LABEL_HEIGHT(mw) - 1;
    } else if (clip & CLIP_TRAILING_VERT_FILL) {
        y1 = TRAILING_FIXED_ROW_LABEL_OFFSET(mw) + TRAILING_FIXED_ROW_HEIGHT(mw);
        y2 = TRAILING_FIXED_ROW_LABEL_OFFSET(mw) + TRAILING_FIXED_ROW_HEIGHT(mw) + FILL_VERT_HEIGHT(mw) - 1;
    } else {
        ret = False;
    }

    if (clip & CLIP_FIXED_COLUMNS && mw->matrix.fixed_columns){
        x1 = COLUMN_LABEL_OFFSET(mw);
        x2 = FIXED_COLUMN_LABEL_OFFSET(mw) - 1;
    } else if (clip & CLIP_TRAILING_FIXED_COLUMNS && mw->matrix.trailing_fixed_columns){
        x1 = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw);
        x2 = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) + TRAILING_FIXED_COLUMN_WIDTH(mw) - 1;
    } else if (clip & CLIP_VISIBLE_WIDTH && mw->matrix.columns - mw->matrix.fixed_columns - mw->matrix.trailing_fixed_columns) {
        x1 = FIXED_COLUMN_LABEL_OFFSET(mw);
        x2 = FIXED_COLUMN_LABEL_OFFSET(mw) + VISIBLE_WIDTH(mw) - 1;
    }else if (clip & CLIP_ROW_LABELS && mw->matrix.row_labels){
        x1 = VERT_SB_OFFSET(mw);
        x2 = VERT_SB_OFFSET(mw) + ROW_LABEL_WIDTH(mw) - 1;
    } else if (clip & CLIP_TRAILING_HORIZ_FILL){
        x1 = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) + TRAILING_FIXED_COLUMN_WIDTH(mw);
        x2 = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) + TRAILING_FIXED_COLUMN_WIDTH(mw) + FILL_HORIZ_WIDTH(mw) - 1;
    } else {
        ret = False;
    }
    
    SETRECT(*rect_p, x1, y1, x2, y2);

    return ret;
}

/*
 * Return the top and bottom-most visible non-fixed row
 */
void
xbaeGetVisibleRows(XbaeMatrixWidget mw, int *top_row, int *bottom_row)
{
    *top_row = xbaeYtoRow(mw, FIXED_ROW_HEIGHT(mw) + VERT_ORIGIN(mw));
    *bottom_row = xbaeYtoRow(mw, FIXED_ROW_HEIGHT(mw) + VERT_ORIGIN(mw) +
			       VISIBLE_HEIGHT(mw) - 1);
}

/*
 * Return the left and right-most visible non-fixed column
 */
void
xbaeGetVisibleColumns(XbaeMatrixWidget mw, int *left_column, int *right_column)
{
    *left_column = xbaeXtoCol(mw, FIXED_COLUMN_WIDTH(mw) + HORIZ_ORIGIN(mw));
    *right_column = xbaeXtoCol(mw, FIXED_COLUMN_WIDTH(mw) + HORIZ_ORIGIN(mw) +
			       VISIBLE_WIDTH(mw) - 1);
}

/*
 * Return the top and bottom row and left and right column of
 * the visible non-fixed cells
 */
void
xbaeGetVisibleCells(XbaeMatrixWidget mw,
	int *top_row, int *bottom_row,
	int *left_column, int *right_column)
{
    xbaeGetVisibleRows(mw, top_row, bottom_row);
    xbaeGetVisibleColumns(mw, left_column, right_column);
}

/*
 * Try to make the column specified by the leftColumn resource
 * be the left column. The column is relative to fixed_columns - so 0 would
 * be the first non-fixed column.
 * If we can't make leftColumn the left column, make it as close as possible.
 */
void
xbaeAdjustLeftColumn(XbaeMatrixWidget mw)
{
    int visible_width = VISIBLE_WIDTH(mw);
    int dynamic_columns;

    if (visible_width < 0)	/* will happen on initialisation */
	return;

    /* Adjust the column if it is out of bounds */

    dynamic_columns = mw->matrix.columns - mw->matrix.fixed_columns -
	mw->matrix.trailing_fixed_columns;

    if (mw->matrix.left_column < 0)
	mw->matrix.left_column = 0;
    else if (mw->matrix.left_column > dynamic_columns - 1)
	mw->matrix.left_column =  dynamic_columns - 1;

    /* Find out if there are enough columns to the right of left_column to 
     * fill visible_width. If not try a column further to the left 
     */
    while(  visible_width > COLUMN_POSITION(mw, TRAILING_HORIZ_ORIGIN(mw))
                            - COLUMN_POSITION(mw, mw->matrix.fixed_columns + mw->matrix.left_column)){
	    mw->matrix.left_column--;
    }
    HORIZ_ORIGIN(mw) = COLUMN_POSITION(mw, mw->matrix.left_column);
}

/*
 * Try to make the row specified by the topRow resource (VERT_ORIGIN)
 * be the top row. The row is relative to fixed_rows - so 0 would
 * be the first non-fixed row.
 * If we can't make topRow the top row, make it as close as possible.
 */
void
xbaeAdjustTopRow(XbaeMatrixWidget mw)
{
    int visible_height = VISIBLE_HEIGHT(mw);
    int dynamic_rows;

    if (visible_height < 0)	/* will happen on initialisation */
	return;

    /* Adjust the row if it is out of bounds */

    dynamic_rows = mw->matrix.rows - mw->matrix.fixed_rows -
	mw->matrix.trailing_fixed_rows;

    if (mw->matrix.top_row < 0)
	mw->matrix.top_row = 0;
    else if (mw->matrix.top_row > dynamic_rows - 1)
	mw->matrix.top_row =  dynamic_rows - 1;

    /* Find out if there are enough rows bellow of top_row to 
     * fill visible_height. If not try a row further up 
     */
    while(  visible_height > ROW_POSITION(mw, TRAILING_VERT_ORIGIN(mw))
                             - ROW_POSITION(mw, mw->matrix.fixed_rows + mw->matrix.top_row)){
	    mw->matrix.top_row--;
    }
    VERT_ORIGIN(mw) = ROW_POSITION(mw, mw->matrix.top_row);
}

/*
 * Utility function to clear a cell so we can draw something new in it.
 * Does not generate expose events on the cell.
 * Does not check if the cell is actually visible before clearing it.
 */
void
xbaeClearCell(XbaeMatrixWidget mw, int row, int column)
{
    int x, y;
    Boolean fixed = IS_FIXED(mw, row, column);
    Widget	cw;	/* Clip widget, not used */
    Window win = xbaeGetCellWindow(mw, &cw, row, column);

    if (!win || mw->matrix.disable_redisplay)
	return;

    xbaeRowColToXY(mw, row, column, &x, &y);

    /*
     * Make sure y coord is valid
     */
#if 0
    if ((win == XtWindow(mw)) &&
	((y > (int)(CLIP_VERT_VISIBLE_SPACE(mw) +
		    ROW_LABEL_OFFSET(mw) - 1)) ||
	 (y < (int)ROW_LABEL_OFFSET(mw))))
	return;
#endif

    XClearArea(XtDisplay(mw), win, x, y, COLUMN_WIDTH(mw, column),
	       SOME_ROW_HEIGHT(mw, row), fixed);
}

/*
 * Return True if a row is visible on the screen (not scrolled totally off)
 */
Boolean
xbaeIsRowVisible(XbaeMatrixWidget mw, int row)
{
    /*
     * If we are not in a fixed row or trailing fixed row,
     * see if we are on the screen vertically
     * (fixed rows are always on the screen)
     */
    if (! IS_FIXED_ROW(mw, row))
    {
	    /* SGO: used same method as IsColumnVisible */
	    int y;
	    y = ROW_POSITION(mw,row) -
                ROW_POSITION(mw, mw->matrix.fixed_rows) - VERT_ORIGIN(mw);

	    if (y + SOME_ROW_HEIGHT(mw,row) > 0 &&
		y < VISIBLE_HEIGHT(mw))
		    return True;
    }
    else
	return True;

    return False;
}

/*
 * Return True if a column is visible on the screen (not scrolled totally off)
 */
Boolean
xbaeIsColumnVisible(XbaeMatrixWidget mw, int column)
{
    /*
     * If we are not in a fixed column, see if we are on the screen
     * horizontally (fixed columns are always on the screen)
     */
    if (! IS_FIXED_COLUMN(mw, column))
    {
	int x;

	/*
	 * Calculate the x position of the column relative to the clip
	 */
	x = COLUMN_POSITION(mw, column) -
	    COLUMN_POSITION(mw, mw->matrix.fixed_columns) - HORIZ_ORIGIN(mw);

	/*
	 * Check if we are visible horizontally
	 */
	if (x + COLUMN_WIDTH(mw, column) > 0 &&
	    x < VISIBLE_WIDTH(mw))
	    return True;
    }
    else
	return True;

    return False;
}

/*
 * Return True if a cell is visible on the screen (not scrolled totally off)
 */
Boolean
xbaeIsCellVisible(XbaeMatrixWidget mw, int row, int column)
{
    return xbaeIsRowVisible(mw, row) && xbaeIsColumnVisible(mw, column);
}

/*
 * Scroll a row so it is visible on the screen.
 */
void
xbaeMakeRowVisible(XbaeMatrixWidget mw, int row)
{
    int value, slider_size, increment, page_increment, y, vert_value;

    /*
     * If we are in a fixed column, we are already visible.
     */
    if (IS_FIXED_ROW(mw, row))
	return;

    /*
     * Calculate the y position of this row
     */
    y = ROW_POSITION(mw, row) -
	ROW_POSITION(mw, mw->matrix.fixed_rows);

    /*
     * Figure out the new value of the VSB to scroll this cell
     * onto the screen. If the whole cell won't fit, scroll so its
     * top edge is visible.
     */
    if (y < VERT_ORIGIN(mw) || SOME_ROW_HEIGHT(mw, row) >= VISIBLE_HEIGHT(mw)) {
	vert_value = y;
    } else if (y + SOME_ROW_HEIGHT(mw, row) >
	     VISIBLE_HEIGHT(mw) + VERT_ORIGIN(mw))
    {
	vert_value = y + SOME_ROW_HEIGHT(mw, row) - VISIBLE_HEIGHT(mw);
    } else {
	vert_value = VERT_ORIGIN(mw);
    }
    
    /*
     * Give the VSB the new value and pass a flag to make it
     * call our scroll callbacks
     */
    if (vert_value != VERT_ORIGIN(mw))
    {
	XmScrollBarGetValues(VertScrollChild(mw), &value,
			     &slider_size, &increment, &page_increment);
	XmScrollBarSetValues(VertScrollChild(mw), vert_value,
			     slider_size, increment, page_increment, True);
    }
}

/*
 * Scroll a column so it is visible on the screen.
 */
void
xbaeMakeColumnVisible(XbaeMatrixWidget mw, int column)
{
    int value, slider_size, increment, page_increment, x, horiz_value;

    /*
     * If we are in a fixed column, we are already visible.
     */
    if (IS_FIXED_COLUMN(mw, column))
	return;

    /*
     * Calculate the x position of this column
     */
    x = COLUMN_POSITION(mw, column) -
	COLUMN_POSITION(mw, mw->matrix.fixed_columns);

    /*
     * Figure out the new value of the HSB to scroll this cell
     * onto the screen. If the whole cell won't fit, scroll so its
     * left edge is visible.
     */
    if (x < HORIZ_ORIGIN(mw) || COLUMN_WIDTH(mw, column) >= VISIBLE_WIDTH(mw)) {
	horiz_value = x;
    } else if (x + COLUMN_WIDTH(mw, column) >
	     VISIBLE_WIDTH(mw) + HORIZ_ORIGIN(mw))
    {
	horiz_value = x + COLUMN_WIDTH(mw, column) - VISIBLE_WIDTH(mw);
    } else {
	horiz_value = HORIZ_ORIGIN(mw);
    }
    
    /*
     * Give the HSB the new value and pass a flag to make it
     * call our scroll callbacks
     */
    if (horiz_value != HORIZ_ORIGIN(mw))
    {
	XmScrollBarGetValues(HorizScrollChild(mw), &value,
			     &slider_size, &increment, &page_increment);
	XmScrollBarSetValues(HorizScrollChild(mw), horiz_value,
			     slider_size, increment, page_increment, True);
    }
}

/*
 * Scrolls a fixed or non-fixed cell so it is visible on the screen.
 */
void
xbaeMakeCellVisible(XbaeMatrixWidget mw, int row, int column)
{
    if (!xbaeIsRowVisible(mw, row))
	xbaeMakeRowVisible(mw, row);

    if (!xbaeIsColumnVisible(mw, column))
	xbaeMakeColumnVisible(mw, column);
}

/*
 * Set the clip_mask in our draw and shadow GCs.  This is necessary for
 * drawing non-fixed column labels and fixed rows.
 */
void
xbaeSetClipMask(XbaeMatrixWidget mw, unsigned int clip_reason)
{
    XRectangle r[2];
    int n = 1;
    /*
     * Set new clip reason
     */
    mw->matrix.current_clip = clip_reason;

    /*
     * XRectangle enclosing column labels and fixed rows
     */
    if ((CLIP_FIXED_COLUMNS & clip_reason) && mw->matrix.fixed_columns)
    {
	r[0].x = COLUMN_LABEL_OFFSET(mw);
	r[0].width = FIXED_COLUMN_WIDTH(mw);
    }
    else if ((CLIP_TRAILING_FIXED_COLUMNS & clip_reason) &&
	     (mw->matrix.trailing_fixed_columns || mw->matrix.fill))
    {
	r[0].x = TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw);
	r[0].width = TRAILING_FIXED_COLUMN_WIDTH(mw);
	if (NEED_HORIZ_FILL(mw))
	    r[0].width += FILL_HORIZ_WIDTH(mw);
    }
    else
    {
	r[0].x = FIXED_COLUMN_LABEL_OFFSET(mw);
	r[0].width = ClipChild(mw)->core.width;
	if (NEED_HORIZ_FILL(mw))
	    r[0].width += FILL_HORIZ_WIDTH(mw);
    }

    if (CLIP_VISIBLE_HEIGHT & clip_reason)
    {
	r[0].y = ROW_LABEL_OFFSET(mw);
	r[0].height = ClipChild(mw)->core.height +
	    FIXED_ROW_HEIGHT(mw) + TRAILING_FIXED_ROW_HEIGHT(mw);
	if (NEED_VERT_FILL(mw))
  	    r[0].height += FILL_VERT_HEIGHT(mw);
    }
    else if ((CLIP_TRAILING_FIXED_ROWS & clip_reason) &&
	     (mw->matrix.trailing_fixed_rows || mw->matrix.fill))
    {
 	r[0].y = TRAILING_FIXED_ROW_LABEL_OFFSET(mw);
 	r[0].height = TRAILING_FIXED_ROW_HEIGHT(mw);
  	if (NEED_VERT_FILL(mw))
  	    r[0].height += FILL_VERT_HEIGHT(mw);
    }
    else if (CLIP_BETWEEN_FIXED_ROWS & clip_reason)
    {
	r[0].y = FIXED_ROW_LABEL_OFFSET(mw);
	r[0].height = ClipChild(mw)->core.height;
    }
    else    /* clip fixed rows & clip_reason */
    {
	r[0].y = HORIZ_SB_OFFSET(mw);
	r[0].height = FIXED_ROW_LABEL_OFFSET(mw);
    }
    if (mw->matrix.row_labels)
    {
	r[1].x = VERT_SB_OFFSET(mw);
	r[1].y = FIXED_ROW_LABEL_OFFSET(mw);
	r[1].width = ROW_LABEL_WIDTH(mw);
	r[1].height = ClipChild(mw)->core.height;
	n = 2;
    }
    /*
     * Reset the clip_mask in our clipping GCs
     */
    XSetClipRectangles(XtDisplay(mw), mw->matrix.cell_grid_line_gc,
		       0, 0, r, n, Unsorted);
    XSetClipRectangles(XtDisplay(mw), mw->matrix.cell_top_shadow_clip_gc,
		       0, 0, r, n, Unsorted);
    XSetClipRectangles(XtDisplay(mw), mw->matrix.cell_bottom_shadow_clip_gc,
		       0, 0, r, n, Unsorted);
    XSetClipRectangles(XtDisplay(mw), mw->matrix.label_clip_gc,
		       0, 0, r, n, Unsorted);
}

/*
 * Cache the pixel position of each column
 */
void
xbaeGetColumnPositions(XbaeMatrixWidget mw)
{
    int i, x;

    if(mw->matrix.column_width_in_pixels == True){
        for (i = 0, x = 0;
             i < mw->matrix.columns;
             x += mw->matrix.column_widths[i], i++){
            mw->matrix.column_positions[i] = x;
        }
    }else{
        for (i = 0, x = 0;
             i < mw->matrix.columns;
             x += mw->matrix.column_widths[i] * FONT_WIDTH(mw) + TEXT_WIDTH_OFFSET(mw) * 2, i++){
            mw->matrix.column_positions[i] = x;
        }
    }
    /* Tobias: The column_positions array is one element longer than
    * the number of columns in the matrix. We need to initialze 
    * the last element so wo can calculate widths safely from the 
    * the difference of any two positions
    */
    mw->matrix.column_positions[mw->matrix.columns] = x;
}

int
xbaeCheckColumnPosition(XbaeMatrixWidget mw, int column)
{
    int i, x;

    if(mw->matrix.column_width_in_pixels == True){
        for (i = 0, x = 0;
             i < mw->matrix.columns;
             x += mw->matrix.column_widths[i], i++){
            assert(mw->matrix.column_positions[i] == x);
        }
    }else{
        for (i = 0, x = 0;
             i < mw->matrix.columns;
             x += mw->matrix.column_widths[i] * FONT_WIDTH(mw) + TEXT_WIDTH_OFFSET(mw) * 2, i++){
           assert(mw->matrix.column_positions[i] == x);
        }
    }
    assert(mw->matrix.column_positions[i] == x);

    assert(column >=0 && column <= mw->matrix.columns);

    return mw->matrix.column_positions[column];
}

void
xbaeGetRowPositions(XbaeMatrixWidget mw)
{
    int i, y;

    if (mw->matrix.row_heights) {
        for (i = 0, y = 0; i < mw->matrix.rows; y += mw->matrix.row_heights[i], i++) {
	    mw->matrix.row_positions[i] = y;
        }
    } else {
        for (i = 0, y = 0; i < mw->matrix.rows; y += ROW_HEIGHT(mw), i++) {
            mw->matrix.row_positions[i] = y;
        }
    }
    /* Tobias: The row_positions array is one element longer than
    * the number of rows in the matrix. We need to initialze 
    * the last element so wo can calculate heights safely from the 
    * the difference of any two positions
    */
    mw->matrix.row_positions[mw->matrix.rows] = y;
}

int
xbaeCheckRowPosition(XbaeMatrixWidget mw, int row)
{
    int i, y;

    if (mw->matrix.row_heights) {
        for (i = 0, y = 0; i < mw->matrix.rows; y += mw->matrix.row_heights[i], i++) {
	    assert(mw->matrix.row_positions[i] == y);
        }
    } else {
        for (i = 0, y = 0; i < mw->matrix.rows; y += ROW_HEIGHT(mw), i++) {
	    assert(mw->matrix.row_positions[i] == y);
        }
    }
    assert(mw->matrix.row_positions[mw->matrix.rows] == y);
    
    assert(row >=0 && row <= mw->matrix.rows);
    
    return mw->matrix.row_positions[row];
}

void
xbaeComputeSize(XbaeMatrixWidget mw, Boolean compute_width,
		Boolean compute_height)
{
    unsigned long full_width = NON_FIXED_TOTAL_WIDTH(mw) +
	FIXED_COLUMN_WIDTH(mw) + TRAILING_FIXED_COLUMN_WIDTH(mw) +
	ROW_LABEL_WIDTH(mw) + 2 * mw->manager.shadow_thickness;
    unsigned long full_height = NON_FIXED_TOTAL_HEIGHT(mw) +
	FIXED_ROW_HEIGHT(mw) + TRAILING_FIXED_ROW_HEIGHT(mw) +
	COLUMN_LABEL_HEIGHT(mw) + 2 * mw->manager.shadow_thickness;
    unsigned long width, height;

    /*
     * Calculate our width.
     * If visible_columns is set, then base it on that.
     * Otherwise, if the compute_width flag is set, then we are full width.
     * Otherwise we keep whatever width we are.
     */
    if (mw->matrix.visible_columns)
	width = ROW_LABEL_WIDTH(mw) + 2 * mw->manager.shadow_thickness +
	    COLUMN_POSITION(mw, mw->matrix.fixed_columns +
			    mw->matrix.visible_columns) +
	    TRAILING_FIXED_COLUMN_WIDTH(mw);
    else if (compute_width)
	width = full_width;
    else
	width = mw->core.width;

    /*
     * Calculate our height.
     * If visible_rows is set, then base it on that.
     * Otherwise, if the compute_height flag is set, then we are full height.
     * Otherwise we keep whatever height we are.
     */
    if (mw->matrix.visible_rows) {
	height = COLUMN_LABEL_HEIGHT(mw) + 2 * mw->manager.shadow_thickness +
	    ROW_POSITION(mw, mw->matrix.fixed_rows +
			    mw->matrix.visible_rows) +
	    TRAILING_FIXED_COLUMN_WIDTH(mw);
    } else if (compute_height) {
	height = full_height;
    } else {
	height = mw->core.height;
    }

    /*
     * Store our calculated size.
     */
    mw->core.width = width;
    mw->core.height = height;

    /*
     * If we are less than full width or our horizontal display policy is
     * constant, then we need an HSB, so increment our height by the size
     * of the HSB (if we are allowed to modify our height and we are allowed
     * to have an HSB).
     */
    if (((width < full_width) ||
	 (XmDISPLAY_STATIC == mw->matrix.hsb_display_policy)) &&
	(compute_height || mw->matrix.visible_rows) &&
	(XmDISPLAY_NONE != mw->matrix.hsb_display_policy))
	mw->core.height += HORIZ_SB_HEIGHT(mw);

    /*
     * If we are less than full height or our vertical display policy is
     * constant, then we need a VSB, so increment our width by the size
     * of the VSB (if we are allowed to modify our width and we are allowed
     * to have a VSB).
     */
    if (((height < full_height) ||
	 (XmDISPLAY_STATIC == mw->matrix.vsb_display_policy)) &&
	(compute_width || mw->matrix.visible_columns) &&
	(XmDISPLAY_NONE != mw->matrix.vsb_display_policy))
	mw->core.width += VERT_SB_WIDTH(mw);

    /*
     * Save our calculated size for use in our query_geometry method.
     * This is the size we really want to be (not necessarily the size
     * we will end up being).
     */
    mw->matrix.desired_width = mw->core.width;
    mw->matrix.desired_height = mw->core.height;

    DEBUGOUT(_XbaeDebug(__FILE__, (Widget)mw, "xbaeComputeSize -> w %d h %d\n",
	mw->matrix.desired_width, mw->matrix.desired_height));
}

/*
 * Return the length of the longest row label
 */
short
xbaeMaxRowLabel(XbaeMatrixWidget mw)
{
    int i;
    short max = 0, len;

    /*
     * Determine the length of the longest row label
     */
    for (i = 0; i < mw->matrix.rows; i++)
    {
	len = strlen(mw->matrix.row_labels[i]);
	if (len > max)
	    max = len;
    }
    return max;
}

void
xbaeParseColumnLabel(String label, ColumnLabelLines lines)
{
    char *nl;

    /*
     * First count the number of lines in the label
     */
    lines->lines = 1;
    nl = label;
    while ((nl = strchr(nl, '\n')) != NULL)
    {
	nl++;
	lines->lines++;
    }

    /*
     * Now malloc a lengths array of the correct size.
     */
    lines->lengths = (int *) XtMalloc(lines->lines * sizeof(int));

    /*
     * An entry in the lengths array is the length of that line (substring).
     */

    /*
     * Handle the case of one line (no strchr() needed)
     */
    if (lines->lines == 1)
	lines->lengths[0] = strlen(label);
    else
    {
	int i;

	nl = label;
	i = 0;
	while ((nl = strchr(nl, '\n')) != NULL)
	{
	    lines->lengths[i] = nl - label;
	    nl++;
	    label = nl;
	    i++;
	}
	lines->lengths[i] = strlen(label);
    }
}

/*
 * Convert the coordinates in an event to coordinates relative to the matrix window
 */
/* ARGSUSED */
Boolean 
xbaeEventToMatrixXY(XbaeMatrixWidget mw, XEvent *event, int *x, int *y){
    switch (event->type)
    {
    case ButtonPress:
    case ButtonRelease:
	*x = event->xbutton.x;
	*y = event->xbutton.y;
	break;
    case KeyPress:
    case KeyRelease:
	*x = event->xkey.x;
	*y = event->xkey.y;
	break;
    case MotionNotify:
	*x = event->xmotion.x;
	*y = event->xmotion.y;
	break;
    default:
	return False;
    }

    if (event->xbutton.window == XtWindow(TextChild(mw))){
	Position tx,ty;
        XtVaGetValues(TextChild(mw),
		      XmNx, &tx,
		      XmNy, &ty,
		      NULL);
        *x += tx;
        *y += ty;

        if(mw->matrix.current_parent != (Widget)mw){
	    *x += mw->matrix.current_parent->core.x;
	    *y += mw->matrix.current_parent->core.y;
        }
    } else if (!event->xbutton.window == XtWindow(mw)) {
	DEBUGOUT(_XbaeDebug(__FILE__, (Widget)mw,
		"xbaeEventToMatrixXY: the event window is not the one of the"
                "TextChild or the Matrix"));
    }

    return True;
}

/*
 * Convert a matrix window x position to a column.
 * Return true if x falls in a column or a row label. If so adjust x so that it's
 * relative to the column/label. If not return false and leave x alone
 */
/* ARGSUSED */
Boolean
xbaeMatrixXtoColumn(XbaeMatrixWidget mw,int *x, int *column){
    
    if (*x >= VERT_SB_OFFSET(mw) &&
        *x < VERT_SB_OFFSET(mw) + ROW_LABEL_WIDTH(mw)) {
        /* It's in the row labels */
        *column = -1;
        
        /* Make x relative to the row labels */
       *x -= VERT_SB_OFFSET(mw);
    } else {
        /* Transform x so it's relative to the virtual matrix */
        if (*x >= COLUMN_LABEL_OFFSET(mw) &&
            *x < COLUMN_LABEL_OFFSET(mw) + FIXED_COLUMN_WIDTH(mw)) {
            *x -= COLUMN_LABEL_OFFSET(mw);
        } else if (*x >= TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) && 
                   *x < TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) + TRAILING_FIXED_COLUMN_WIDTH(mw)) {
            *x -= TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw) - COLUMN_POSITION(mw, TRAILING_HORIZ_ORIGIN(mw));
        } else if (*x >= FIXED_COLUMN_LABEL_OFFSET(mw) && 
                   *x < FIXED_COLUMN_LABEL_OFFSET(mw) + VISIBLE_WIDTH(mw)) {
            *x -= FIXED_COLUMN_LABEL_OFFSET(mw) - HORIZ_ORIGIN(mw) - FIXED_COLUMN_WIDTH(mw);
        } else {
            return False;
        }
        /* Get the column it corresponds to */
        *column = xbaeXtoCol(mw,*x);

        /* Make x relative to the that column */
        *x -= COLUMN_POSITION(mw,*column);
    }

    return True;
}

/*
 * Convert a matrix window y position to a row.
 * Return true if y falls in a row or a column label. If so adjust y so that it's 
 * relative to the row/label. If not return false and leave y alone
 */
/* ARGSUSED */
Boolean
xbaeMatrixYtoRow(XbaeMatrixWidget mw,int *y, int *row){

    if (*y >= HORIZ_SB_OFFSET(mw) &&
        *y < HORIZ_SB_OFFSET(mw) + COLUMN_LABEL_HEIGHT(mw)) {
        /* It's in the column labels */
        *row = -1;
        
        /* Make y relative to the column labels */
        *y -= HORIZ_SB_OFFSET(mw);
    } else {
        /* Transform the y coordinate so it's relative to the virtual matrix */
        if (*y >= ROW_LABEL_OFFSET(mw) &&
            *y < ROW_LABEL_OFFSET(mw) + FIXED_ROW_HEIGHT(mw)) {
            *y -= ROW_LABEL_OFFSET(mw);
        } else if (*y >= TRAILING_FIXED_ROW_LABEL_OFFSET(mw) &&
                   *y < TRAILING_FIXED_ROW_LABEL_OFFSET(mw) + TRAILING_FIXED_ROW_HEIGHT(mw)) {
            *y -= TRAILING_FIXED_ROW_LABEL_OFFSET(mw) - ROW_POSITION(mw,TRAILING_VERT_ORIGIN(mw));
        } else if (*y >= FIXED_ROW_LABEL_OFFSET(mw) && 
                   *y < FIXED_ROW_LABEL_OFFSET(mw) + VISIBLE_HEIGHT(mw)) {
            *y -= FIXED_ROW_LABEL_OFFSET(mw) - VERT_ORIGIN(mw) - FIXED_ROW_HEIGHT(mw);
        } else {
            return False;
        }
        /* Get the row it corresponds to */
        *row = xbaeYtoRow(mw,*y);

        /* Make y relative to the that row */
        *y -= ROW_POSITION(mw,*row);
    }
    
    return True;
}

/*
 * Convert a matrix window x, y to a row, column pair.
 * Return true if x, y falls in a cell. If so adjust x and y so they are
 * relative to the cell.
 * Return false if x, y falls in a label. Then one of row/column is -1. If so
 * adjust x and y so they are relative to the labels origin.
 * Return false if x, y doesn't fall in a cell nor a label and set both row and 
 * col to -1 leaving x and y alone.
 */
/* ARGSUSED */
Boolean
xbaeMatrixXYToRowCol(XbaeMatrixWidget mw, int *x, int *y,
		int *row,int *column)
{
    int ret_x = *x;
    int ret_y = *y;

    if (   !xbaeMatrixXtoColumn(mw,&ret_x,column)
        || !xbaeMatrixYtoRow(mw,&ret_y,row)
        || (*row == -1 && *column == -1)){
        /* Not a cell nor a label */
        *row = -1;
        *column = -1;
        return False;
    }

    *x = ret_x;
    *y = ret_y;
    
    return *row != -1 && *column != -1;
}

/*
 * Find where a virtual coordinate falls by (binary) searching the positions array(s)
 */
static int findPosition(int *positions,int start,int end,int pos){
    
    int middle;
    
    /* Tobias: Neither of the conditions should ever be true. If they are there is a bug somewhere 
    * up the call stack. So far I found three. The rest of xbae tries to fix problems instead of failing
    * so we do the same here to keep up with the debugging fun.
    */
    if(pos < positions[start]){
        DEBUGOUT(_XbaeDebug(__FILE__, NULL,
		"pos[start=%d]=%d pos[end=%d]=%d pos=%d\n",
		start,positions[start],end,positions[end],pos));
        return start;
    }else if(pos > positions[end]){
        DEBUGOUT(_XbaeDebug(__FILE__, NULL, "pos[start=%d]=%d pos[end=%d]=%d pos=%d\n",
		start,positions[start],end,positions[end],pos));
        return end - 1;
    }
    
    for(;;){
        middle = (start + end)/2;
        if (positions[middle] > pos) {
            end = middle;
        } else if (positions[middle + 1] < pos){
            start = middle;
        } else {
            break;
        }
    }

    return middle;
}

/*
 * Convert a pixel position to the column it is contained in.
 */
int
xbaeXtoCol(XbaeMatrixWidget mw, int x)
{
#if 1 /* Tobias: use binary search */
    return findPosition(mw->matrix.column_positions,0,mw->matrix.columns,x);
#else
    int i;

    for (i = 0; i < mw->matrix.columns; i++)
	if (COLUMN_POSITION(mw, i) > x)
	    return i - 1;
    /*
     * I have seen cases where this function returned mw->matrix.columns
     * causing a crash as array bounds are read - AL
     */
    if (i > mw->matrix.columns)
	return mw->matrix.columns - 1;

    return i - 1;
#endif
}

/*
 * Danny
 * 3/7/2001 should start counting at VERT_ORIGIN
 */
#if 1
int
xbaeYtoRow(XbaeMatrixWidget mw, int y)
{
#if 1 /* Tobias: use binary search */
    return findPosition(mw->matrix.row_positions,0,mw->matrix.rows,y);
#else
	/* SGO: changed method to same as xbaeXtoCol */

    int i;

    for (i = 0; i < mw->matrix.rows; i++)
	if (ROW_POSITION(mw, i) > y)
	    return i - 1;
    /*
     * I have seen cases where this function returned mw->matrix.columns
     * causing a crash as array bounds are read - AL
     */
    if (i > mw->matrix.rows)
	return mw->matrix.rows - 1;

    return i - 1;
#endif
}
#else
int
xbaeYtoRow(XbaeMatrixWidget mw, int y)
{
	int i;
	int pos=0;

	for (i=0; i<mw->matrix.rows; i++) {
		pos += mw->matrix.row_heights[i];
		if (y <= pos) {
			DEBUGOUT(_XbaeDebug(__FILE__, (Widget)mw,
				"xbaeYtoRow(%d)->%d\n", y, i));
			return i;
		}
	}
	DEBUGOUT(_XbaeDebug(__FILE__, (Widget)mw,
		"xbaeYtoRow(%d)->%d\n", y, mw->matrix.rows));
	return mw->matrix.rows;
}
#endif

/*
 * Convert a pixel position to the trailing
 * fixed column it is contained in
 */
int
xbaeXtoTrailingCol(XbaeMatrixWidget mw, int x)
{
#if 1 /* Tobias: use binary search */
    return findPosition(mw->matrix.column_positions,
        TRAILING_HORIZ_ORIGIN(mw),
        mw->matrix.columns,
        x + COLUMN_POSITION(mw, TRAILING_HORIZ_ORIGIN(mw))
    );
#else
    int i;

    x += COLUMN_POSITION(mw, TRAILING_HORIZ_ORIGIN(mw));
    for (i = TRAILING_HORIZ_ORIGIN(mw); i < mw->matrix.columns; i++)
    {
	if (COLUMN_POSITION(mw, i) > x)
	    return i - 1;
    }

    return i - 1;
#endif
}

/*
 * Convert a pixel position to the trailing
 * fixed row it is contained in
 */
static int
xbaeYtoTrailingRow(XbaeMatrixWidget mw, int y)
{
#if 1 /* Tobias: use binary search */
    return findPosition(mw->matrix.row_positions,
        TRAILING_VERT_ORIGIN(mw),
        mw->matrix.rows,
        y + ROW_POSITION(mw, TRAILING_VERT_ORIGIN(mw))
    );
#else
    int i;

    y += ROW_POSITION(mw, TRAILING_VERT_ORIGIN(mw));
    for (i = TRAILING_VERT_ORIGIN(mw); i < mw->matrix.rows; i++)
    {
	if (ROW_POSITION(mw, i) > y)
	    return i - 1;
    }

    return i - 1;
#endif
}

/*
 * Convert a row/column cell position to the x/y of its upper left corner
 * wrt the window it will be drawn in (either the matrix window for
 * fixed cells, or the clip window for non-fixed).
 */
void
xbaeRowColToXY(XbaeMatrixWidget mw, int row, int column, int *x, int *y)
{
    /*
     * If we are in a fixed cell, calculate x/y relative to Matrixs
     * window (take into account labels etc)
     */

    /*
     * Ignore horiz_origin if we are in a fixed column
     */
    if (IS_LEADING_FIXED_COLUMN(mw, column)) {
        *x = COLUMN_POSITION(mw, column);
        if (IS_FIXED_ROW(mw, row)) {
            /* Totaly fixed cells coordinates are relative to the matrix not a clip */ 
            *x += COLUMN_LABEL_OFFSET(mw);         
        }
    } else if (IS_TRAILING_FIXED_COLUMN(mw, column)) {
        *x = COLUMN_POSITION(mw,column) - COLUMN_POSITION(mw, TRAILING_HORIZ_ORIGIN(mw));
        if (IS_FIXED_ROW(mw, row)) {
            /* Totaly fixed cells coordinates are relative to the matrix not a clip */ 
            *x += TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw);
        }
    } else {
        *x = (COLUMN_POSITION(mw, column)
              - COLUMN_POSITION(mw, mw->matrix.fixed_columns))
              - HORIZ_ORIGIN(mw);
    }

    /*
     * Ignore vert_origin if we are in a fixed row
     */
    if (IS_LEADING_FIXED_ROW(mw, row)) {
        *y = ROW_POSITION(mw,row);
        if (IS_FIXED_COLUMN(mw, column)) {
            /* Totaly fixed cells coordinates are relative to the matrix not a clip */ 
            *y += ROW_LABEL_OFFSET(mw);
	}
    } else if (IS_TRAILING_FIXED_ROW(mw, row)) {
        *y = ROW_POSITION(mw,row) - ROW_POSITION(mw, TRAILING_VERT_ORIGIN(mw));
        if (IS_FIXED_COLUMN(mw, column)){
            /* Totaly fixed cells coordinates are relative to the matrix not a clip */ 
            *y += TRAILING_FIXED_ROW_LABEL_OFFSET(mw);
        }
    } else {
        *y = (ROW_POSITION(mw, row)
              - ROW_POSITION(mw, mw->matrix.fixed_rows))
              - VERT_ORIGIN(mw);
    }
}

/*
 * Returns the window on which a cell is displayed, i.e. the matrix, the clip,
 * or one of the extra clips which handle the fixed row/col cells.
 */
Window
xbaeGetCellWindow(XbaeMatrixWidget mw, Widget *w, int row, int column)
{
    int clip = xbaeCellClip(mw,row,column);
    Window win;

    switch(clip)
    {
    case CLIP_FIXED_ROWS | CLIP_FIXED_COLUMNS:
    case CLIP_FIXED_ROWS | CLIP_TRAILING_FIXED_COLUMNS:
    case CLIP_TRAILING_FIXED_ROWS | CLIP_FIXED_COLUMNS:
    case CLIP_TRAILING_FIXED_ROWS | CLIP_TRAILING_FIXED_COLUMNS:
	/* total fixed cell - on parent matrix window */
	*w = (Widget)mw;
	break;

    case CLIP_NONE:
	/* not fixed at all - on clip child */
	*w = ClipChild(mw);
	break;

    case CLIP_FIXED_COLUMNS:
	/* fixed col only - on left clip */
	*w = LeftClip(mw);
	break;

    case CLIP_TRAILING_FIXED_COLUMNS:
	/* fixed trailing col only - on right clip */
	*w = RightClip(mw);
	break;

    case CLIP_FIXED_ROWS:
	/* fixed row only - on top clip */
	*w = TopClip(mw);
	break;

    case CLIP_TRAILING_FIXED_ROWS:
	/* fixed trailing row only - on bottom clip */
	*w = BottomClip(mw);
	break;
    default:	/* bogus */
	win = (Window)NULL;
	*w = (Widget)NULL;
    }
    return XtWindow(*w);
}

void
xbaeCalcVertFill(XbaeMatrixWidget mw, Window win, int x, int y, int row, int column,
	int *ax, int *ay, int *width, int *height)
{
    *ax = x;
    *width = COLUMN_WIDTH(mw, column);

    if (win == XtWindow(LeftClip(mw)))
    {
	*ax += COLUMN_LABEL_OFFSET(mw);
	*ay = LeftClip(mw)->core.y + LeftClip(mw)->core.height;
    }
    else if (win == XtWindow(RightClip(mw)))
    {
	*ax += TRAILING_FIXED_COLUMN_LABEL_OFFSET(mw);
	*ay = RightClip(mw)->core.y + RightClip(mw)->core.height;
    }
    else if (win == XtWindow(BottomClip(mw)))
    {
	*ax += BottomClip(mw)->core.x;
	*ay = BottomClip(mw)->core.y + BottomClip(mw)->core.height;
    }
    else if (win == XtWindow(ClipChild(mw)))
    {
	if (XtIsManaged(LeftClip(mw)))
	    *ax += FIXED_COLUMN_LABEL_OFFSET(mw);
	else
	    *ax += COLUMN_LABEL_OFFSET(mw);
	*ay = ClipChild(mw)->core.y + ClipChild(mw)->core.height;
	if (*ax < (int)COLUMN_LABEL_OFFSET(mw))
	{
	    *width += *ax - COLUMN_LABEL_OFFSET(mw);
	    *ax = COLUMN_LABEL_OFFSET(mw);
	}
    }
    else			/* must be in a corner */
	*ay = y;

    *height = MATRIX_VERT_VISIBLE_SPACE(mw) + COLUMN_LABEL_HEIGHT(mw) - *ay;
    
    /*
     * Unfortunately, on the filled area, we don't have the luxury
     * of the clip widgets to help us out with the edges of the area.
     * Check our width isn't going to draw outside the left or right clip
     */

    if (! IS_FIXED_COLUMN(mw, column))
    {
	if (XtIsManaged(LeftClip(mw)) &&
	    *ax < (int)FIXED_COLUMN_LABEL_OFFSET(mw))
	{
	    *width -= (FIXED_COLUMN_LABEL_OFFSET(mw) - *ax);
	    *ax = FIXED_COLUMN_LABEL_OFFSET(mw);
	}

	if (XtIsManaged(RightClip(mw)) &&
	    ((*ax + *width) > (int)RightClip(mw)->core.x))
	    *width = RightClip(mw)->core.x - *ax;

	if (win == XtWindow(BottomClip(mw)))
	{
	    if ((*ax + *width) >
		 (int)(BottomClip(mw)->core.x + BottomClip(mw)->core.width))
		*width = BottomClip(mw)->core.width +
		    BottomClip(mw)->core.x - *ax;

	    if (*ax < (int)COLUMN_LABEL_OFFSET(mw))
	    {
		*width += *ax - COLUMN_LABEL_OFFSET(mw);
		*ax = COLUMN_LABEL_OFFSET(mw);
	    }
	}

	if ((win == XtWindow(ClipChild(mw))) &&
	    ((*ax + *width) >
	      (int)(ClipChild(mw)->core.x + ClipChild(mw)->core.width)))
	    *width = ClipChild(mw)->core.width +
		ClipChild(mw)->core.x - *ax;
    }
}

void
xbaeCalcHorizFill(XbaeMatrixWidget mw, Window win, int x, int y, int row, int column,
	int *ax, int *ay, int *width, int *height)
{
    *ay = y;
    *height = SOME_ROW_HEIGHT(mw, row);

    if (win == XtWindow(TopClip(mw)))
    {
	*ax = TopClip(mw)->core.x + TopClip(mw)->core.width;
	*ay += ROW_LABEL_OFFSET(mw);
    } else if (win == XtWindow(BottomClip(mw))) {
	*ax = BottomClip(mw)->core.x + BottomClip(mw)->core.width;
	*ay += TRAILING_FIXED_ROW_LABEL_OFFSET(mw);
    } else if (win == XtWindow(RightClip(mw))) {
	*ax = RightClip(mw)->core.x + RightClip(mw)->core.width;
	*ay += RightClip(mw)->core.y;
    } else if (win == XtWindow(ClipChild(mw))) {
#if 0
	if (XtIsManaged(TopClip(mw)))
#else
	if (row < mw->matrix.fixed_rows)
#endif
	{
	    *ay += FIXED_ROW_LABEL_OFFSET(mw);
	} else {
	    *ay += ROW_LABEL_OFFSET(mw);
	}
	*ax = ClipChild(mw)->core.x + ClipChild(mw)->core.width;
#if 0
	/* We're above the scrolled area.
	 * The caller should figure this out. */
	if (*ay < (int)ROW_LABEL_OFFSET(mw)) {
	    *ay = ROW_LABEL_OFFSET(mw);
	}
#endif
    } else {		/* must be in a corner */
	*ax = x;
    }

    *width = MATRIX_HORIZ_VISIBLE_SPACE(mw) + ROW_LABEL_WIDTH(mw) - *ax;

    /*
     * Unfortunately, on the filled area, we don't have the luxury
     * of the clip widgets to help us out with the edges of the area.
     * Check our width isn't going to draw outside the left or right clip,
     * or out past our matrix region.
     */

    if (! IS_FIXED_ROW(mw, row))
    {
	if (XtIsManaged(LeftClip(mw)) &&
	    (*ay < (int)FIXED_ROW_LABEL_OFFSET(mw)))
	{
	    *height -= (FIXED_ROW_LABEL_OFFSET(mw) - *ay);
	    *ay = FIXED_ROW_LABEL_OFFSET(mw);
	}

	if (XtIsManaged(RightClip(mw)) &&
	    ((*ay + *height) >
	      (int)(RightClip(mw)->core.y + RightClip(mw)->core.height)))
	    *height = RightClip(mw)->core.height +
		RightClip(mw)->core.y - *ay;

	if ((win == XtWindow(ClipChild(mw))) &&
	    ((*ay + *height) >
	      (int)(ClipChild(mw)->core.y + ClipChild(mw)->core.height)))
	    *height = ClipChild(mw)->core.height +
		ClipChild(mw)->core.y - *ay;
    }
}


/*
 * Given a cell row and column, return the required coordinates
 * of the user widget relative to its parent (always the main Xbae widget).
 * Note that xbaeRowColToXY() returns the X and Y coordinates of the widget
 * WINDOW relative to the parent window, which in the case of everything but
 * totally fixed cells is a clip.
 */
Boolean
xbaeRowColToWidgetXY(XbaeMatrixWidget mw, int row, int column,
		       int *widget_x, int *widget_y)
{
    int clip = xbaeCellClip(mw, row, column);
    int x, y;
    Boolean	clipped = False;

    xbaeRowColToXY (mw, row, column, &x, &y);

    switch(clip)
    {
    case CLIP_FIXED_ROWS | CLIP_FIXED_COLUMNS:
    case CLIP_FIXED_ROWS | CLIP_TRAILING_FIXED_COLUMNS:
    case CLIP_TRAILING_FIXED_ROWS | CLIP_FIXED_COLUMNS:
    case CLIP_TRAILING_FIXED_ROWS | CLIP_TRAILING_FIXED_COLUMNS:
	/* total fixed cell - on parent matrix window */
	    *widget_x = x;
	    *widget_y = y;
	break;

    case CLIP_NONE:
	/* not fixed at all - on clip child */
	    *widget_x = x + FIXED_COLUMN_LABEL_OFFSET (mw);
	    *widget_y = y + FIXED_ROW_LABEL_OFFSET (mw);
	if (y < 0)
		clipped = True;
	break;

    case CLIP_FIXED_COLUMNS:
	/* fixed col only - on left clip */
	    *widget_x = x + COLUMN_LABEL_OFFSET (mw);
	    *widget_y = y + FIXED_ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_TRAILING_FIXED_COLUMNS:
	/* fixed trailing col only - on right clip */
	    *widget_x = x + TRAILING_FIXED_COLUMN_LABEL_OFFSET (mw);
	    *widget_y = y + FIXED_ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_FIXED_ROWS:
	/* fixed row only - on top clip */
	    *widget_x = x + FIXED_COLUMN_LABEL_OFFSET (mw);
	    *widget_y = y + ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_TRAILING_FIXED_ROWS:
	/* fixed trailing row only - on bottom clip */
	    *widget_x = x + FIXED_COLUMN_LABEL_OFFSET (mw);
	    *widget_y = y + TRAILING_FIXED_ROW_LABEL_OFFSET (mw);
	break;
    default:	/* bogus */
	    *widget_x = x;
	    *widget_y = y;
	    break;
    }

    return clipped;
}

/*
 * Given the X,Y coordinates of a user widget (assuming this is
 * in relation to the matrix widget), move the widget's window to the
 * correct X, Y coordinates in relation to its parent's window. Parent is
 * one of the clips, or the matrix window for a fixed cell.
 */

void
xbaeMoveWindowFromUserWidgetXY (XbaeMatrixWidget mw, Widget user_widget,
				    int row, int column,
		                    int widget_x, int widget_y)
{
    int clip = xbaeCellClip(mw,row,column);
    int    x, y;

    switch(clip)
    {
    case CLIP_FIXED_ROWS | CLIP_FIXED_COLUMNS:
    case CLIP_FIXED_ROWS | CLIP_TRAILING_FIXED_COLUMNS:
    case CLIP_TRAILING_FIXED_ROWS | CLIP_FIXED_COLUMNS:
    case CLIP_TRAILING_FIXED_ROWS | CLIP_TRAILING_FIXED_COLUMNS:
	/* total fixed cell - no need to reparent */
	    return;

    case CLIP_NONE:
	/* not fixed at all - on clip child */
	    x = widget_x - FIXED_COLUMN_LABEL_OFFSET (mw);
	    y = widget_y - FIXED_ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_FIXED_COLUMNS:
	/* fixed col only - on left clip */
	    x = widget_x - COLUMN_LABEL_OFFSET (mw);
	    y = widget_y - FIXED_ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_TRAILING_FIXED_COLUMNS:
	/* fixed trailing col only - on right clip */
	    x = widget_x - TRAILING_FIXED_COLUMN_LABEL_OFFSET (mw);
	    y = widget_y - FIXED_ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_FIXED_ROWS:
	/* fixed row only - on top clip */
	    x = widget_x - FIXED_COLUMN_LABEL_OFFSET (mw);
	    y = widget_y - ROW_LABEL_OFFSET (mw);
	break;

    case CLIP_TRAILING_FIXED_ROWS:
	/* fixed trailing row only - on bottom clip */
	    x = widget_x - FIXED_COLUMN_LABEL_OFFSET (mw);
	    y = widget_y - TRAILING_FIXED_ROW_LABEL_OFFSET (mw);
	break;
    default:	/* bogus */
	    return;
    }

    /*
     * Only need to reparent if the coordinates differ.
     */
    if ( (x != widget_x) || (y != widget_y)) {
	    XMoveWindow ( XtDisplay ((Widget)mw), XtWindow (user_widget),
			  x, y);
    }
}


/*
 * Below are some functions to deal with a multi-threaded environment.
 * Use these instead of the Xt stuff, to ensure that we can still cope
 * with X11r5 where this stuff didn't exist.
 */
#ifdef  XtSpecificationRelease
#if     XtSpecificationRelease > 5
#define R6plus
#endif
#endif

void xbaeObjectLock(Widget w)
{
#ifdef  R6plus
        if (XmIsGadget(w))
                XtAppLock(XtWidgetToApplicationContext(
                        XtParent(w)));
        else
                XtAppLock(XtWidgetToApplicationContext(w));
#endif
}

void
xbaeObjectUnlock(Widget w)
{
#ifdef  R6plus
        if (XmIsGadget(w))
                XtAppUnlock(XtWidgetToApplicationContext(
                        XtParent(w)));
        else
                XtAppUnlock(XtWidgetToApplicationContext(w));
#endif
}

/* The following 2 functions are here to provide information about the
   version of this library - since we may often build it as a shared/dynamic
   library it's important to have this builtin */
const char *
XbaeGetVersionTxt(void)
{
	return XbaeVersionTxt;
}

int
XbaeGetVersionNum(void)
{
	return XbaeVersion;
}

/* The following 2 functions contain/return info about the
   M*tif/LessTif version this library was built with.
   Building against one toolkit and linking on runtime against
   another is likely to fail! */
const char *
XbaeGetXmVersionTxt(void)
{
        return XmVERSION_STRING;
}

int
XbaeGetXmVersionNum(void)
{
        return XmVersion;
}


/* Some string functions, since strcasecmp()/strncasecmp() are
   not always available */
int
_xbaeStrcasecmp(const char *s1, const char *s2)
{
#ifdef HAVE_STRNCASECMP
    return strcasecmp(s1, s2);
#else
#ifdef HAVE_STRNICMP
    return stricmp(s1, s2);
#else
    int c1, c2;

    while (*s1 && *s2)
    {
	c1 = tolower(*s1);
	c2 = tolower(*s2);
	if (c1 != c2)
	{
	    return (c1 - c2);
	}
	s1++;
	s2++;
    }
    return (int)(*s1 - *s2);
#endif
#endif
}


int
_xbaeStrncasecmp(const char *s1, const char *s2, size_t count)
{
#ifdef HAVE_STRNCASECMP
    return strncasecmp(s1, s2, count);
#else
#ifdef HAVE_STRNICMP
    return strnicmp(s1, s2, count);
#else
    int c1, c2;
    int i = 0;

    while (*s1 && *s2 && i < (int)count)
    {
	c1 = tolower(*s1);
	c2 = tolower(*s2);
	if (c1 != c2)
	{
	    return (c1 - c2);
	}
	s1++;
	s2++;
	i++;
    }
    return (int)(*s1 - *s2);
#endif
#endif
}

void
xbaeScrollRows(XbaeMatrixWidget mw, Boolean Left, int step)
{
	int value, slider_size, increment, page_increment, limit;
	XtVaGetValues(VertScrollChild(mw),
		      (Left ? XmNminimum : XmNmaximum), &limit,
		      NULL);
	XmScrollBarGetValues(VertScrollChild(mw), &value,
			     &slider_size, &increment, &page_increment);
	if (Left)
	XmScrollBarSetValues(VertScrollChild(mw),
			     ((value-step < limit) ? limit : value-step),
			     slider_size, increment, page_increment, True);
	else
	{
	    limit -= slider_size;
	    XmScrollBarSetValues(VertScrollChild(mw),
                             ((value+step > limit) ? limit : value+step),
			     slider_size, increment, page_increment, True);
	}
}

void
xbaeScrollColumns(XbaeMatrixWidget mw, Boolean Up, int step)
{
	int value, slider_size, increment, page_increment, limit;
	XtVaGetValues(HorizScrollChild(mw),
		      (Up ? XmNminimum : XmNmaximum), &limit,
		      NULL);
	XmScrollBarGetValues(HorizScrollChild(mw), &value,
			     &slider_size, &increment, &page_increment);
	if (Up) XmScrollBarSetValues(HorizScrollChild(mw),
                   ((value-step < limit) ? limit : value-step),
                   slider_size, increment, page_increment, True);
	else
	{
	    limit -= slider_size;
	    XmScrollBarSetValues(HorizScrollChild(mw),
                   ((value+step > limit) ? limit : value+step),
                   slider_size, increment, page_increment, True);
	}
}
