
/*
 * ops.c
 *
 * (C) Copyright 1995 Archie L. Cobbs
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "util.h"
#include "error.h"
#include "names.h"
#include "asm.h"

/*
 * DEFINITIONS
 */

	#define	kOpErrBadIns		(-1)
	#define	kOpErrBadAddrMode	(-2)

	OpaqueDecl(OpInfo);
	OpaqueDefn(OpInfo)
	{
		unsigned	id;
		Byte		ops[NUM_ADDRESS_MODES];
	};

/*
 * INTERNAL FUNCTIONS
 */

	static	OpInfo		FindOp(OpInfo table, int size, unsigned iname);

/*
 * Opcode tables
 *
 * NOTE: We use zero to mean "no opcode" and treat BRK as special case.
 * We also rely on a hack which assumes that sizeof(char *) is at most
 * sizeof(unsigned).
 */

	#define	OP(ins,imp,rel,imm,zpg,zpx,zpy,abs,abx,aby,zix,ziy,ind,inx,zin)	\
	{																		\
		(unsigned) #ins,													\
		{																	\
			0x##imp, 0x##rel, 0x##imm, 0x##zpg,								\
			0x##zpx, 0x##zpy, 0x##abs, 0x##abx,								\
			0x##aby, 0x##zix, 0x##ziy, 0x##ind,								\
			0x##inx, 0x##zin												\
		}																	\
	},

/*
 * 6502 opcodes (except BRK)
 */

	static struct OpInfo	gOps6502[] =
	{

/*****  INSTR   Imp Rel Imm ZPg ZPX ZPY Abs AbX AbY ZiX ZiY Ind InX Zin *****/

	OP(  ADC,	 0,  0, 69, 65, 75,  0, 6d, 7d, 79, 61, 71,  0,  0,  0	)
	OP(  AND,	 0,  0, 29, 25, 35,  0, 2d, 3d, 39, 21, 31,  0,  0,  0	)
	OP(  ASL,	0a,  0,  0, 06, 16,  0, 0e, 1e,  0,  0,  0,  0,  0,  0	)
	OP(  BCC,	 0, 90,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BCS,	 0, B0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BEQ,	 0, F0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BGE,	 0, B0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BIT,	 0,  0,  0, 24,  0,  0, 2c,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BLT,	 0, 90,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BMI,	 0, 30,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BNE,	 0, D0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BPL,	 0, 10,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BVC,	 0, 50,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  BVS,	 0, 70,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  CLC,	18,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  CLD,	D8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  CLI,	58,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  CLV,	B8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  CMP,	 0,  0, c9, c5, d5,  0, cd, dd, d9, c1, d1,  0,  0,  0	)
	OP(  CPX,	 0,  0, e0, e4,  0,  0, ec,  0,  0,  0,  0,  0,  0,  0	)
	OP(  CPY,	 0,  0, c0, c4,  0,  0, cc,  0,  0,  0,  0,  0,  0,  0	)
	OP(	 DEC,	 0,  0,  0, c6, d6,  0, ce, de,  0,  0,  0,  0,  0,  0	)
	OP(  DEX,	ca,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  DEY,	88,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  EOR,	 0,  0, 49, 45, 55,  0, 4d, 5d, 59, 41, 51,  0,  0,  0	)
	OP(	 INC,	 0,  0,  0, e6, f6,  0, ee, fe,  0,  0,  0,  0,  0,  0	)
	OP(  INX,	e8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  INY,	c8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  JMP,	 0,  0,  0,  0,  0,  0, 4c,  0,  0,  0,  0, 6c,  0,  0	)
	OP(  JSR,	 0,  0,  0,  0,  0,  0, 20,  0,  0,  0,  0,  0,  0,  0	)
	OP(  LDA,	 0,  0, a9, a5, b5,  0, ad, bd, b9, a1, b1,  0,  0,  0	)
	OP(  LDX,	 0,  0, a2, a6,  0, b6, ae,  0, be,  0,  0,  0,  0,  0	)
	OP(  LDY,	 0,  0, a0, a4, b4,  0, ac, bc,  0,  0,  0,  0,  0,  0	)
	OP(  LSR,	4a,  0,  0, 46, 56,  0, 4e, 5e,  0,  0,  0,  0,  0,  0	)
	OP(  NOP,	EA,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  ORA,	 0,  0, 09, 05, 15,  0, 0d, 1d, 19, 01, 11,  0,  0,  0	)
	OP(  PHA,	48,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PHP,	08,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PLA,	68,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PLP,	28,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  ROL,	2a,  0,  0, 26, 36,  0, 2e, 3e,  0,  0,  0,  0,  0,  0	)
	OP(  ROR,	6a,  0,  0, 66, 76,  0, 6e, 7e,  0,  0,  0,  0,  0,  0	)
	OP(  RTI,	40,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  RTS,	60,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  SBC,	 0,  0, e9, e5, f5,  0, ed, fd, f9, e1, f1,  0,  0,  0	)
	OP(  SEC,	38,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  SED,	F8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  SEI,	78,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  STA,	 0,  0,  0, 85, 95,  0, 8d, 9d, 99, 81, 91,  0,  0,  0	)
	OP(  STX,	 0,  0,  0, 86,  0, 96, 8e,  0,  0,  0,  0,  0,  0,  0	)
	OP(  STY,	 0,  0,  0, 84, 94,  0, 8c,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TAX,	aa,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TAY,	a8,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TSX,	ba,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TXA,	8a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TXS,	9a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TYA,	98,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  XOR,	 0,  0, 49, 45, 55,  0, 4d, 5d, 59, 41, 51,  0,  0,  0	)

	};

/*
 * 65c02 extended opcodes.
 *
 * NB: Instruction names which appear in the above list must
 *     be listed first, and in the same order. To facilitate this,
 *     the above list is sorted alphabetically.
 */

	static struct OpInfo	gOps65c02[] =
	{

/*****  INSTR   Imp Rel Imm ZPg ZPX ZPY Abs AbX AbY ZiX ZiY Ind InX Zin *****/

	/* FIRST: instructions also in the 6502 table, in the same order */

	OP(  ADC,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 72	)
	OP(  AND,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 32	)
	OP(  BIT,	 0,  0, 89,  0, 34,  0,  0, 3c,  0,  0,  0,  0,  0,  0	)
	OP(  CMP,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, d2	)
	OP(  DEC,	3a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  EOR,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 52	)
	OP(  INC,	1a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  JMP,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 7c,  0	)
	OP(  LDA,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, b2	)
	OP(  ORA,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12	)
	OP(  SBC,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, f2	)
	OP(  STA,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 92	)
	OP(  XOR,	 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 52	)

	/* THEN: new instructions */

	OP(  BRA,	 0, 80,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  DEA,	3a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  INA,	1a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PHX,	da,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PHY,	5a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PLX,	fa,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  PLY,	7a,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0	)
	OP(  STZ,	 0,  0,  0, 64, 74,  0, 9c, 9e,  0,  0,  0,  0,  0,  0	)
	OP(  TRB,	 0,  0,  0, 14,  0,  0, 1c,  0,  0,  0,  0,  0,  0,  0	)
	OP(  TSB,	 0,  0,  0, 04,  0,  0, 0c,  0,  0,  0,  0,  0,  0,  0	)

	};

	#define	kNumOps6502		(sizeof(gOps6502) / sizeof(*gOps6502))
	#define	kNumOps65c02	(sizeof(gOps65c02) / sizeof(*gOps65c02))

/*
 * OpcodeInit()
 *
 * Replace all opcode strings with string ID's.
 * Note that ID's will be in increasing order,
 * so we can do a binary search on the table.
 */

void
OpcodeInit(void)
{
	int		k;

/* Verify our assumption */

	assert(sizeof(unsigned) >= sizeof(char *));

/* Convert strings into id's */

	for (k = 0; k < kNumOps6502; k++)
		gOps6502[k].id = NameID(gNames, (char *) gOps6502[k].id);

	for (k = 0; k < kNumOps65c02; k++)
		gOps65c02[k].id = NameID(gNames, (char *) gOps65c02[k].id);
}

/*
 * GetOp()
 *
 * Get operand for instruction, address mode pair.
 * Generate error message if appropriate.
 */

Byte
GetOp(unsigned ins, int mode)
{
	int		opcode;

	switch ((opcode = CheckOp(ins, mode)))
	{
		case kOpErrBadIns:
			uerror("unknown instruction \"%s\"", Name(gNames, ins));
			return(0);
		case kOpErrBadAddrMode:
			uerror("invalid address mode");
			return(0);
		default:
			assert(opcode >= 0);
			return(opcode);
	}
}

/*
 * CheckOp()
 *
 * Get operand for instruction, address mode pair.
 * Returns negative error code if not found.
 */

int
CheckOp(unsigned ins, int mode)
{
	int		insOK = FALSE;
	OpInfo	info;

/* Special case for BRK instruction */

	if ((ins == NameID(gNames, "BRK")) && (mode == mImplied))
		return(0x00);

/* Look in 6502 table */

	if ((info = FindOp(gOps6502, kNumOps6502, ins)))
	{
		insOK = TRUE;
		if (info->ops[mode - 1])
			return(info->ops[mode - 1]);
	}

/* Look in 65c02 table */

	if (g65c02 && (info = FindOp(gOps65c02, kNumOps65c02, ins)))
	{
		insOK = TRUE;
		if (info->ops[mode - 1])
			return(info->ops[mode - 1]);
	}

/* Not found/valid */

	return(insOK ? kOpErrBadAddrMode : kOpErrBadIns);
}

/*
 * FindOp()
 *
 * Find the OpInfo structure for given instruction. Using binary search
 * is probably overkill, but it simplifies *this* code so it's worth it.
 *
 * Note that we rely on the fact that NameID allocates id's in increasing
 * order, so the id's in the table are sorted after OpcodeInit() is called.
 */

static	int
FindCmp(const void *a, const void *b)
{
	return(((OpInfo) a)->id - ((OpInfo) b)->id);
}

static	OpInfo
FindOp(OpInfo table, int size, unsigned iname)
{
	struct OpInfo	key;

	key.id = iname;
	return((OpInfo) bsearch(&key, table, size, sizeof(*table), FindCmp));
}


