# Part of the A-A-P GUI IDE: Aap project Activity class

# Copyright (C) 2002-2003 Stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

import types
import os.path
import string

from Activity import Activity
import Navigator

# TODO: do this properly with gettext().
def _(x):
    return x

start_marker = "#>>> inserted by Agide start <<<\n"
end_marker = "#>>> inserted by Agide end <<<\n"


class AapProject(Activity):
    """A-A-P project Activity."""

    def open(self):
        """Create a navigator for the new Activity, fill it with items from the
           recipe."""
        item = Navigator.ActyItem(self.name, os.getcwd(), self, None)
        nav = Navigator.Navigator("Project", item)
        self.addNav(nav)

        # Try reading the items from the recipe.
        recipe_items, self.topdir = recipe2items(self.name)
        if recipe_items:
            item.addTree(recipe_items, self.topdir, 0)

        self.modified = 0

    def save(self):
        """Save a new version of the project.
           This keeps all the lines in the recipe, except the IDE stuff, which
           are replaced by the items currently in the Activity."""
        # Open the file for reading.
        try:
            fr = open(self.name, "r")
        except StandardError, e:
            print "Cannot open project file '%s': %s" % (self.name, str(e))
            return

        # Search for the start marker.
        found_marker = 0
        while 1:
            line = fr.readline()
            if not line:
                break
            if line == start_marker:
                found_marker = 1
                break

        # Open the new recipe.
        newname = self.name + "_new"
        if os.path.exists(newname):
            print "Aborted recipe update found: '%s'?" % newname
            print "Check the contents, you may want to delete it."
            return
        try:
            fw = open(newname, "w")
        except StandardError, e:
            print "Cannot create new project file for '%s': %s" % (self.name, str(e))
            return

        # Rewind; copy the text up to the start marker to the new recipe.
        fr.seek(0)
        while 1:
            line = fr.readline()
            if not line:
                break
            if found_marker:
                if line == start_marker:
                    break
            else:
                if line[0] != '#':
                    break
            fw.write(line)

        # Skip until the end marker.
        if found_marker:
            while 1:
                line = fr.readline()
                if not line:
                    print "Cannot find end marker!"
                    fw.close()
                    os.remove(newname)
                    return
                if line == end_marker:
                    break
            line = fr.readline()

        # Write the new IDE items with markers.
        if not found_marker:
            fw.write('\n')
        fw.write(start_marker)
        import Dictlist
        for i in self.getTopItem().children.itemlist:
            # Skip empty items and read-only items.
            if i.name and i.name[0] in string.letters:
                l = []
                if i.children:
                    for c in i.children.itemlist:
                        l.append(c.name)
                fw.write("IDE_" + i.name + " = " + Dictlist.list2str(l) + '\n')
        fw.write(end_marker)

        # Copy the rest of the file.
        while line:
            fw.write(line)
            line = fr.readline()
        fr.close()
        fw.close()

        # Rename the original to backup and new file to original.
        bakname = self.name + ".bak"
        try:
            os.remove(bakname)
        except:
            pass
        try:
            os.rename(self.name, bakname)
            os.rename(newname, self.name)
            self.modified = 0
        except StandardError, e:
            print "Cannot move new recipe '%s' into place: %s" % (self.name, str(e))


def getMenuText():
    """Return the name to be used for the "New Activity" menu."""
    return _("A-A-P project"), _("Create a new A-A-P project")


def create(topmodel):
    """Create a new activity: create the recipe."""
    # Get the name to be used for the activity.
    name = topmodel.view.fileDialog(_("Enter name for new A-A-P project"))
    if not name:
        return None
    if len(name) <= 4 or name[-4:] != ".aap":
        name = name + ".aap"

    # Check it doesn't exist yet.
    if os.path.exists(name):
        print "Project '%s' already exists!" % name
        return None

    # Create the file and insert the default contents.
    try:
        f = open(name, "w")
    except StandardError, e:
        print "Cannot create '%s': %s", (name, str(e))
        return None

    try:
        f.write("# A-A-P recipe\n"
                "\n"
                + start_marker +
                "IDE_SOURCE = source_file.c\n"
                "IDE_TARGET = program\n"
                + end_marker +
                "\n"
                ":program $IDE_TARGET : $IDE_SOURCE\n"
                "@if locals().get('IDE_CFLAGS'):\n"
                "  CFLAGS = $IDE_CFLAGS\n"
                "@if locals().get('IDE_CPPFLAGS'):\n"
                "  CPPFLAGS = $IDE_CPPFLAGS\n"
                )
        f.close()
    except StandardError, e:
        print "Cannot write to '%s': %s" % (name, str(e))
        return None

    act = AapProject(name, topmodel)
    act.open()
    return act


def canOpen(name, type):
    """Return TRUE when "name" can be opened for this activity class.
       Only when the file exists and filetype is "aap"."""
    return os.path.isfile(name) and type == "aap"


def recipe2items(name):
    """Read a recipe and return a dictionary with the IDE items in it.
       The dictionary contains nested dictionaries.
       Also returns the directory to which the items are relative to."""
    import Main
    import RecPython

    # Use the recipe executive to read the recipe (and children).
    cwd = os.getcwd()
    nodelist, recdict = Main.get_nodelist(["-f", name])
    # Remember the top directory and go back to where we came from.
    recipedir = os.getcwd()
    os.chdir(cwd)

    res = {}
    for k in recdict.keys():
        dict = {}
        if k[0] == '$' or (not isinstance(recdict[k], types.StringType)
                and not isinstance(recdict[k], types.IntType)):
            continue
        dl = RecPython.var2dictlist(str(recdict[k]))
        for i in dl:
            dict[i["name"]] = {}

        if len(k) > 4 and k[:4] == "IDE_":
            # IDE_ variables go to the toplevel.
            res[k[4:]] = dict
        else:
            # Other variables go below "|read-only".
            if not res.has_key("|read-only"):
                res["|read-only"] = {}
            res["|read-only"][k] = dict

    return res, recipedir



# vim: set sw=4 et sts=4 tw=79 fo+=l:
