# Part of the A-A-P GUI IDE: Activity class and associated stuff

# Copyright (C) 2002-2003 Stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

import os.path

import Filetype
import Util
import Navigator
from glob import glob

# TODO: do this properly with gettext().
def _(x):
    return x

class ActyList:
    """List of toplevel activities: things that the user is currently working
    on in an IDE session."""

    def __init__(self):
        self.list = []          # List of open activities
        self.currentacty = None
        self.view = None        # ActyListView, will be set later
        self.recent = []        # list of recently used Activity names

    def __len__(self):
        return len(self.list)

    def __getitem__(self, i):
        return self.list[i]

    def addActy(self, acty):
        """Add Activity "acty" to the list."""
        self.list.append(acty)
        self.insertRecent(acty.name)
        if self.view:
            self.view.addActy(acty)

    def insertRecent(self, name):
        """Insert "name" in the list of recent activities.  An existing
           entry is moved to the front."""
        for i in range(len(self.recent)):
            if self.recent[i] == name:
                del self.recent[i]
                break
        self.recent.insert(0, name)
        if self.view:
            self.view.updateRecent()

    def getRecentList(self):
        """Return the list of recent activity names."""
        return self.recent

    def getacty(self, idx):
        return self.list[idx]

    def remove(self, acty):
        i = self.list.index(acty)
        self.list.remove(acty)
        if self.view:
            self.view.delActy(i)
        if len(self.list) > 0:
            # Make another activity the current one.
            if i >= len(self.list):
                i = len(self.list) - 1
            self.setCurrentActy(i)
        else:
            # No activities remaining, clear all navigators
            acty.topmodel.view.clearNavs()
            self.currentacty = None

    def setCurrentActy(self, idx):
        acty = self.list[idx]
        if self.currentacty != acty:
            if not acty.topmodel.shutting_down:
                self.currentacty = acty
                acty.foreground()

                # Move this acty to the front of the recent list.
                self.insertRecent(acty.name)

                # mark the entry in the GUI list as selected
                if self.view:
                    self.view.SetSelection(idx)

    def fixActy(self):
        """Called when shutdown won't happen.  Check the current activity is
           valid (was not deleted).  If it's invalid, make any activity the
           current activity."""
        if self.list and not self.currentacty in self.list:
            self.setCurrentActy(0)

    def shutdown(self):
        """Shutdown in progress, stop all activities.
           Return non-zero if shutdown is OK."""
        ok = 1
        for acty in self.list[:]:
            if not acty.close(1):
                ok = 0
            else:
                self.remove(acty)
        return ok

    def name2index(self, name):
        """Find the activity with name "name" in the list.
           Return its index, -1 if not found."""
        i = 0
        for item in self.list:
            if item.name == name:
                return i
            i = i + 1
        return -1


class Activity:
    """Something the user is currently working on in an IDE session."""

    def __init__(self, name, topmodel):
        self.name = name
        self.topmodel = topmodel    # the model (AIDE object)
        self.navlist = []           # List of navigators.
        self.currentnav = None
        self.modified = 0           # non-zero when modified

    def open(self):
        """Open the Activity.  This is to be overruled by derived classes.
           This method simply creates one Navigator with one ActyItem.
           And the tool for the item is started."""
        item = Navigator.ActyItem(self.name, '.', self, None)
        nav = Navigator.Navigator("File", item)
        item.runTool()
        self.addNav(nav)

    def save(self):
        """Save the Activity."""
        print "This Activity can't be saved."

    def addNav(self, nav):
        """Add a navigator to the Activity."""
        self.navlist.append(nav)
        if not self.currentnav:
            self.currentnav = nav

    def findNavByName(self, name):
        """Find the navigator with name "name".  Return the Navigator or
           None."""
        for nav in self.navlist:
            if nav.name == name:
                return nav
        return None

    def getTopItem(self):
        if len(self.navlist) > 0:
            return self.navlist[0].getTopItem()
        return None

    def findItemByName(self, name):
        """Find an item in this activity with the name "name"."""
        for nav in self.navlist:
            item = nav.findItemByName(name)
            if item:
                return item
        return None

    def close(self, shutdown):
        """Close this activity.  Return non-zero if closing is OK."""
        if self.modified:
            res = self.topmodel.doDialog("This activity has changes:\n%s"
                                                        % self.getListString(),
                                            [ "Save", "Discard", "Cancel" ], 2)
            if res == "Save":
                self.save()
                if self.modified:
                    return 0        # saving failed
            elif res == "Discard":
                self.modified = 0
            else:
                return 0
        ok = 1
        for n in self.navlist[:]:
            if not n.close(shutdown):
                ok = 0
            else:
                self.navlist.remove(n)
        return ok

    def getListString(self):
        """Return a string to be displayed in the list of Activities."""
        expname = os.path.expanduser(self.name)
        dir = os.path.dirname(expname)
        if os.path.exists(expname) or (dir and os.path.isdir(dir)):
            return Util.display_name(self.name)
        return self.name

    def foreground(self):
        """Move this activity to the foreground."""
        # Display the Navigators for this activity.
        self.topmodel.view.showNavs(self)

        # Bring the current item in the current Navigator to the foreground.
        if self.currentnav:
            self.currentnav.foreground()

def classlist(topmodel):
    """Return the list of available Activity class names."""
    cwd = os.getcwd()
    os.chdir(topmodel.rootdir)
    l = map(lambda x: x[11:-3], glob("Activities/*.py"))
    l.sort()
    os.chdir(cwd)
    return l

def name2class(name, topmodel):
    """Find an Activity class that can handle the file "name"."""
    type = Filetype.ft_detect(name)
    for c in classlist(topmodel):
        exec "import " + c
        if eval(c + ".canOpen(name, type)"):
            return c
    return None


# vim: set sw=4 et sts=4 tw=79 fo+=l:
