# Part of the A-A-P GUI IDE: Xterm window to execute commands in

# Copyright (C) 2002-2003 Stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

import os
import signal
import string

import Util
import Config

class Xterm:
    def __init__(self, topmodel):
        self.xtermpid = 0
        self.ttyname = ''
        self.topmodel = topmodel

    def open(self, title):
        """Start an xterm to run the program in.  Return the name of the
           tty.  Returns None if it failed."""
        # Most of this comes from DDD.
        # The idea is that the xterm is started to run a shell that prints some
        # info into a temp file and hangs.  GDB will then use the pty of the
        # xterm for the running program.

        # Position the xterm next to the debugger console.
        from GUItop import toplevel_xpos, toplevel_ypos, \
                toplevel_height, border_width, border_height, \
                debug_width

        default_dict = {"Console/program" : "xterm"}
        term = Config.get_conf_key("Console/program", self.topmodel, 
                                                                  default_dict)

        xterm_command = ("%s -T '%s' -geometry +%d+%d -e /bin/sh -c "
                % (term, title, toplevel_xpos + debug_width + border_width,
                    toplevel_ypos + toplevel_height + border_height))
        command = (
                # Set up a temporary file in TMP.
                "tmp=${TMPDIR-/tmp}/ddd$$; export tmp; "

                # Be sure to remove it when exiting...
                "trap \"rm -f $tmp\" 0; "

                # ... or being interrupted.
                "trap 'exit 1' 1 2 15; "

                # Now execute the xterm command
                + xterm_command +

                # which saves TTY, PID, TERM, and WINDOWID in TMP and goes to
                # sleep forever.  Signal 2 (SIGINT) is blocked for two
                # reasons: first, we don't want ^C to kill the tty window;
                # second, later invocations will send us SIGINT to find out
                # whether we're still alive.
                " '"
                "echo `tty` $$ $TERM $WINDOWID >$tmp; "
                "trap \"\" 2; "
                "while true; do sleep 3600; done"
                "' "

                # The whole thing is redirected and in the background such
                # that rsh won't wait for us.
                ">/dev/null </dev/null 2>&1 & "

                # The main file waits for TMP to be created...
                "while test ! -s $tmp; do sleep 1; done; "

                # ...and sends TMP's contents to stdout.
                "cat $tmp"
                )

        if self.xtermpid > 0:
            # We're already running.  Don't start a new tty
            # if the old one is still running.
            command = ("kill -2 %d 2>/dev/null || ( %s )"
                            % (self.xtermpid, command))

        ok, res = Util.redir_system_int({"MESSAGE" : ""}, command, use_tee = 0)
        if ok:
            if res:
                # we currently only use the tty name and the process ID.
                reslist = string.split(res)
                if len(reslist) > 2:
                    self.ttyname = reslist[0]
                    self.xtermpid = int(reslist[1])
                    return self.ttyname
            if self.xtermpid > 0:
                return self.ttyname
        return None

    def close(self):
        """Close the xterm."""
        if self.xtermpid > 0:
            try:
                os.kill(self.xtermpid, signal.SIGKILL)
            except OSError:
                pass        # probably exited already
            self.xtermpid = 0


# vim: set sw=4 et sts=4 tw=79 fo+=l:
