/*
 *    Copyright (c) 1992 Minnesota Supercomputer Center, Inc.
 *    Copyright (c) 1992 Army High Performance Computing Research Center
 *        (AHPCRC), University of Minnesota
 *    Copyright (c) 1995-1999 Laboratory for Computational Science and
 *        Engineering (LCSE), University of Minnesota
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file CONTRIBUTORS for a list of contributors.
 *
 *    Original author(s):
 *      Amir Shinar <amir@lcse.umn.edu>
 *      Grant Erickson <grant@lcse.umn.edu>
 *
 *    This file is maintained by:
 *      Grant Erickson <grant@lcse.umn.edu>
 *
 *    Module name: movie.c
 *
 *    Description:
 *      Key-frame interpolation and movie generation routines.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <limits.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>

#include <Xm/Xm.h>
#include <Xm/PushB.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/Scale.h>
#include <Xm/List.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#include <X11/Xatom.h>
#include <GL/gl.h>
#include <GL/GLwMDrawA.h>

#include "util.h"
#include "skip.h"
#include "xtutil.h"
#include "glutil.h"
#include "draw.h"
#include "vox.h"
#include "bob.h"
#include "setup.h"
#include "movie.h"


/*********************************************************************
**********************************************************************
**
** void Allocate()
**
** Action:
**   Allocates memory for recorded points and interpolated points
**
** 
** Inputs:
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void Allocate()
{
  bob->delta = CallocType(float, NUMBER_OF_POINTS);
  MemCheck(bob->delta);
  
  bob->quat = CallocType(Quat, NUMBER_OF_POINTS);
  MemCheck(bob->quat);
  
  bob->quat_inter = CallocType(Quat, bob->NUMBER_OF_INTER_POINTS+10);
  MemCheck(bob->quat_inter);
  
  bob->trans = CallocType(Trans, NUMBER_OF_POINTS);
  MemCheck(bob->trans);
  
  bob->trans_inter = CallocType(Trans, bob->NUMBER_OF_INTER_POINTS+10);
  MemCheck(bob->trans_inter);
  
  bob->recFrame = CallocType(int, NUMBER_OF_POINTS);
  MemCheck(bob->recFrame);
  
  bob->recFrame_inter = CallocType(int, bob->NUMBER_OF_INTER_POINTS+10);
  MemCheck(bob->recFrame_inter);
  
  bob->keyFrameFlags = CallocType(int, bob->NUMBER_OF_INTER_POINTS+10);
  MemCheck(bob->keyFrameFlags);
}


/*********************************************************************
**********************************************************************
**
** void interpol_trans()
**
** Action:
**   Performs the interpolation of translations between two recorded
**   translations.
** 
** Inputs:
**   q1      - Starting translation to interpolate from
**   q2      - Ending translation to interpolate to
**   delta   - Interpolation "distance"
**  *inter   - Pointer to array in which to insert interpolated 
**             translations
**   index   - Array index to first entry in the segment of the
**             movie
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void interpol_trans(Trans left, Trans right, int delta, Trans *inter, int index)
{

  int i, j;
  int numPoints = delta + 1;

  if ( numPoints == index ) {
    return;
  }

  /* Set the two recorded endpoints of this segment */

  for ( i = 0 ; i < 3 ; i++)
    {
      inter[index][i] = left[i];
      inter[index + numPoints - 1][i] = right[i];
    }

  /* Interpolate the rest in between */

  for ( i = 1; i < numPoints - 1; i++)
    {
      for ( j = 0 ; j < 3 ; j++)
	{
	  inter[index + i][j] = left[j] + (right[j] - left[j])*i/delta;
	}
    }
}


/*********************************************************************
**********************************************************************
**
** void slerp()
**
** Action:
**   Performs spherical linear interpolation between two quaternions
**
** 
** Inputs:
**  *q1    - Pointer to a quaternion to interpolate from
**  *q2    - Pointer to a quaternion to interpolate to
**   alpha - Scale factor for interpolated quaternion
**  *q     - Pointer to a quaternion which is interpolated between
**           q1 and q2
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void slerp(float *q1, float *q2, float alpha, float *q)
{
  double sum;
  double beta1, beta2, theta;
  
  int i;

  sum = q1[3]*q2[3] + q1[0]*q2[0] + q1[1]*q2[1] + q1[2]*q2[2];

  theta = acos(sum);

  if ( theta <= EPSILON ) /* linear interpolation */
    {
      beta1 = 1.0 - alpha;
      beta2 = alpha;
    }
  else                    /* arc interpolation */
    {
      beta1 = sin((1.0-alpha)*theta)/sin(theta);
      beta2 = sin(alpha*theta)/sin(theta);
    }

  for ( i = 0; i < 4; i++ ) 
    {
      q[i] = beta1*q1[i] + beta2*q2[i];
    }

}


/*********************************************************************
**********************************************************************
**
** void interpol_quat()
**
** Action:
**   Performs the interpolation of quaternions between two recorded
**   quaternions.
** 
** Inputs:
**   q1      - Starting quaternion to interpolate from
**   q2      - Ending quaternion to interpolate to
**   delta   - Total number of points to be generated in this
**             segment
**  *inter   - Pointer to array in which to insert interpolated 
**             quaternions
**   index   - Array index to first entry in the segment of the
**             movie
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void interpol_quat(Quat q1, Quat q2, int delta, Quat *inter, int index)
{
  int i;
  int numPoints = delta + 1;

  if ( numPoints == index ) {
    return;
  }
  
  /* Two adjacent points in the path would be the two end
   * points in a segment of the interpolated one. This for loop
   * sets up the endpoints of that segment. */
  
  for ( i = 0; i < 4; i++ )
    {
      inter[index][i] = q1[i];
      inter[index + numPoints - 1][i] = q2[i];
    }

  /* The rest of the points in the segment will be interpolated. */

  for ( i = 1; i < numPoints - 1; i++)
    {
      slerp(q1, q2, (float)i/(float)delta, inter[index + i]);
    }
}


/*********************************************************************
**********************************************************************
**
** void interpol_frame()
**
** Action:
**   Performs the interpolation of data frames between two
**   recorded data frames.
** 
** Inputs:
**   f1      - Starting frame to interpolate from
**   f2      - Ending frame to interpolate to
**   delta   - Total number of points to be generated in this
**             segment
**  *inter   - Pointer to array in which to insert interpolated 
**             frames
**   index   - Array index to first entry in the segment of the
**             movie
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void interpol_frame(int f1, int f2, int delta, int *inter, int index)
{
  int i;
  int numPoints = delta + 1;

  if ( numPoints == index ) {
    return; 
  }

  /* The frames anchoring the end points of the segment to be
   * interpolated will be the recFrame[i] and recFrame[i+1]. */

  inter[index] = f1;
  inter[index + numPoints - 1] = f2;

  /* The rest of the frames will be interpolated */

  for ( i = 1 ; i < numPoints - 1 ; i++)
    {
      inter[index + i] = f1 + i;
    }
}


/*********************************************************************
**********************************************************************
**
** void MarkKeyFrames()
**
** Action:
**   Sets the field bob->keyFrameFlag to TRUE (1) if it is a key 
**   frame, otherwise the field is set to FALSE (0).
** 
** Inputs:
**   delta - Total number of frames in this segment
**  *flags  - Pointer to the array of key frame flags
**   index - Array index to first frame of this segment
**
**********************************************************************
*********************************************************************/

void MarkKeyFrames(int delta, int *flags, int index)
{

  int i;
  int numPoints = delta + 1;

  if (numPoints == index) {
    return;
  }

  /* The frames anchoring the end points of the segment to be
   * interpolated are keyFrames, so set the flag to TRUE (1). */

  flags[index] = 1;
  flags[index + numPoints - 1] = 1;

  /* The rest of the frames are interpolated so set their flag
   * to FALSE (0). */

  for (i = 1; i < numPoints - 1; i++)
    {
      flags[index + i] = 0;
    }
}


/*********************************************************************
**********************************************************************
**
** void DoneRecording()
**
** Action:
**   Coordinates the interpolation of vectors, quaternions, and
**   data frames.
** 
**********************************************************************
*********************************************************************/

void DoneRecording(void)
{

  float  mult_fact;
  int    i, index, temp;

  index = 0;
  temp  = 0;

  /* Check if there has been a large enough movement to interpolate */

  if ( bob->total_delta > MINIMUM_DELTA )
    {
      mult_fact = bob->NUMBER_OF_INTER_POINTS/bob->total_delta;
      
      bob->total_delta = 0.0;
      
      /* The reason for adding the 0.5 is to round delta[i]
       * the closest integer when calling interpol. */

      if (StackSize(bob->frameList) > 1) /* Dynamic data */
	{
	  for ( i = 0; i < bob->num_of_pnts - 1; i++ )
	    {
	      bob->delta[i] = bob->recFrame[i+1] - bob->recFrame[i];
	      temp += (int)bob->delta[i];
	    }
	}
      else                               /* Static data */
	{
	  for ( i = 0 ; i < bob->num_of_pnts - 1 ; i++)
	    {
	      bob->delta[i] = bob->delta[i] * mult_fact + 0.5;
	      temp += (int)bob->delta[i];
	    }

	  /* The total number of points will be close to bob->
	   * NUMBER_OF_INTER_POINTS but not exactly the same due to 
	   * truncating the float to an integer. */
	  
	  bob->delta[i-1] = bob->delta[i-1] + 
	    (float)(bob->NUMBER_OF_INTER_POINTS - temp);

	  /* If the data is static, the last interpolated segment ends up
	   * being one frame too long. This is a terrible hack, but it works */

	  if(!(StackSize(bob->frameList) > 1))
	    {
	      bob->delta[i-1] -= 1;
	    }
	}
      
      /* Interpolate the quaternions, translations vectors, and data
       * frames. In addition, tag the frames that are key frames. */

      for ( i = 0 ; i < bob->num_of_pnts - 1; i++ )
	{
	  interpol_quat(bob->quat[i], bob->quat[i+1], (int)(bob->delta[i]),
			bob->quat_inter, index);

	  interpol_trans(bob->trans[i], bob->trans[i+1], (int)(bob->delta[i]),
			 bob->trans_inter, index);

	  if (StackSize(bob->frameList) > 1) /* Dynamic data */
	    {
	      interpol_frame(bob->recFrame[i], bob->recFrame[i+1], 
			     (int)(bob->delta[i]), bob->recFrame_inter, index);
	    }

	  MarkKeyFrames((int)(bob->delta[i]), bob->keyFrameFlags, index);

	  index += (int)(bob->delta[i]);
	}
      
      /* The total number of points will be close to bob->
       * NUMBER_OF_INTER_POINTS but not exactly the same due to 
       * truncating the float to an integer.
       *
       * Correct by adding one to bob->
       * num_of_inter_pnts, since all the deltas are correct and we
       * can't arbitrarily tack frames onto the end because we can't
       * just "make up" data frames. */

      bob->num_of_inter_pnts = index + 1;
    }
  else
    {
      printf("Not enough recorded points or too short of a distance.\n");
    }

  /* Set the state of user interface elements */
  XtSetSensitive(GetWidget("play"), True);
  XtSetSensitive(GetWidget("stop"), False);
  XtSetSensitive(GetWidget("record"), False);
  XtSetSensitive(GetWidget("savemov"), True);
  XtSetSensitive(GetWidget("savecoord"), True);
  XtSetSensitive(GetWidget("scanfor"), True);
  XtSetSensitive(GetWidget("scanrev"), True);
  XtSetSensitive(GetWidget("skipfor"), True);
  XtSetSensitive(GetWidget("skiprev"), True);
}


/*********************************************************************
**********************************************************************
**
** void total_del()
**
** Action:
**   Computes the Pythagorean vector distance between two
**   volume views.
** 
** Inputs:
**   i - index for arrays of translation vectors and quaternions
**
**********************************************************************
*********************************************************************/

void total_del(int i)
{
  
  float t11, t12, t21, t22, t31, t32;
  float q11, q12, q21, q22, q31, q32, q41, q42;
  
  t11 = bob->trans[i][0];
  t12 = bob->trans[i-1][0];
  t21 = bob->trans[i][1];
  t22 = bob->trans[i-1][1];
  t31 = bob->trans[i][2];
  t32 = bob->trans[i-1][2];
  
  q11 = bob->quat[i][0];
  q12 = bob->quat[i - 1][0];
  q21 = bob->quat[i][1];
  q22 = bob->quat[i-1][1];
  q31 = bob->quat[i][2];
  q32 = bob->quat[i - 1][2];
  q41 = bob->quat[i][3];
  q42 = bob->quat[i - 1][3];
  
  bob->delta[i-1] = sqrt((t11 - t12)*(t11 - t12) + (t21 - t22)*(t21 - t22) +
			 (t31 - t32)*(t31 - t32) + 
			 (q12 - q11)*(q12 - q11) + (q22 - q21)*(q22 - q21) +
			 (q32 - q31)*(q32 - q31) + (q42 - q41)*(q42 - q41));

  bob->total_delta = bob->total_delta + bob->delta[i-1];
}

/*********************************************************************
**********************************************************************
**
** void RecordPoint()
**
** Action:
**   Records a movie frame by extracting the quaternions and
**   translation vectors from the trackball and saving the current
**   data frame if the data is dynamic.
** 
**********************************************************************
*********************************************************************/

void RecordPoint(void)
{
  int   i, j;
  float d[4], ratio, temp_ratio;
  float quat_temp[4];
  int   cross0, cross1, cross2;
  
  /* Stop movie playback if necessary */
  if (bob->playWP)
    {
      XtRemoveWorkProc(bob->playWP);
      bob->playWP = 0;
    }

  /*  Allocate memory for recorded points and the interpolated movie.
   *  Clear out the dimension list. */

  if ( bob->num_of_pnts == 0 )
    {
      Allocate();
      XmListDeleteAllItems(GetWidget("dimlist"));
    }
  else if ((StackSize(bob->frameList) > 1) &&
	   (bob->iframe <= bob->recFrame[bob->num_of_pnts-1]))
    {
      fprintf(stderr, "\007");
      fprintf(stderr, "The frame number for this recorded frame (%d) must\n"
      	      "be greater than the previously recorded frame (%d).\n", 
	      bob->iframe, bob->recFrame[bob->num_of_pnts-1]);
      return;
    }

  /* Stop the volume from spinning if it is. */

  TrackballStopSpinning(bob->vball);
  if (bob->spinWP)
    XtRemoveWorkProc(bob->spinWP);
  bob->spinWP = 0;
  
  bob->anFrame = 0;
  bob->anStep = 0;
  bob->anOffset = 0;
  bob->anFollow = False;

  /* Extract 3-space Cartesian vectors and quaternions from volume view */

  for (i = 0; i < 4; i++)
    {
      bob->quat[bob->num_of_pnts][i] = bob->vball->qrot[i];
    }

  for (i = 0; i < 3; i++)
    {
      bob->trans[bob->num_of_pnts][i] = bob->vball->trans[i];
    }

  /* Store the particular frame that was recorded */

  bob->recFrame[bob->num_of_pnts] = bob->iframe;
  
  /* checking to see if the difference between the bob->quats are
   * more than 1.0 so that the interpolation will be done
   * from 0.x to 0.99999 and from -0.99999 to -0.x instead of
   * from 0.x to 0 and 0 to -0.x */

  if ( bob->num_of_pnts > 0 )
    {
      for ( j = 0 ; j < 4 ; j ++ )
	{
	  d[j] = bob->quat[bob->num_of_pnts][j] - 
	    bob->quat[bob->num_of_pnts-1][j];
	}
      
      cross0 =  ( d[0] > 1.01 ) || ( d[0] < -1.01 );
      cross1 =  ( d[1] > 1.01 ) || ( d[1] < -1.01 );
      cross2 =  ( d[2] > 1.01 ) || ( d[2] < -1.01 );
      
      if ( cross0 )
	{
	  for ( j = 0 ; j < 4 ; j++ )
	    {
	      bob->quat[bob->num_of_pnts][j] = 0.0 - 
		bob->quat[bob->num_of_pnts][j];
	    }
	}
      if ( cross1 && !(cross0))
	{
	  for ( j = 0 ; j < 4 ; j++ )
	    {
	      bob->quat[bob->num_of_pnts][j] = 0.0 - 
		bob->quat[bob->num_of_pnts][j];
	    }
	}
      if ( cross2 && !(cross1) && !(cross0) )
	{ 
	  for ( j = 0 ; j < 4 ; j++ )
	    {
	      bob->quat[bob->num_of_pnts][j] = 0.0 - 
		bob->quat[bob->num_of_pnts][j];
	    }
	}
      
      total_del(bob->num_of_pnts);
      
    }

  bob->num_of_pnts++;

  /* Add recorded view to the dimension list */
  AddDimEntry(NULL);

  /* Set the appropriate state of the user interface controls */
  XtSetSensitive(GetWidget("savemov"), False);
  XtSetSensitive(GetWidget("savecoord"), False);
  XtSetSensitive(GetWidget("play"), False);

  /* If two points have been recorded, allow the user to interpolate */
  if (bob->num_of_pnts >= 2)
    {
      XtSetSensitive(GetWidget("stop"), True);
    }
  else
    {
      XtSetSensitive(GetWidget("stop"), False);
    }
}


/*********************************************************************
**********************************************************************
**
** void StartMovie()
**
** Action:
**   Initializes playback of a generated movie, saving frames to
**   disk if specified. This routine strives to reflect the same 
**   structure as the routine 'StartAnimation' in bob.c. Shinar's
**   original implementation of movie playback didn't preserve the 
**   "feel" and interface of the original Bob well.
** 
** Inputs:
**   save - Boolean indicating whether or not frames should be 
**          saved to disk as they are played.
**
**********************************************************************
*********************************************************************/

void StartMovie(Boolean save)
{

  char       *tf;
  Position   x, y, newx, newy;
  Dimension  shadow;
  Screen     *screen = XtScreen(bob->shell);

  bob->anFrame = 0;
  bob->playFrameInc = 1;

  /* If the movie has reached the end, reset it to the start. */
  if (bob->playFrame > bob->num_of_inter_pnts)
    bob->playFrame = 0;
  else if (bob->playFrame < 0)
    bob->playFrame = bob->num_of_inter_pnts;
  
  if (bob->anFile)
    {
      (void) fclose(bob->anFile);
      bob->anFile = NULL;
    }

  Free(bob->anFileName);
  bob->anFileName = NULL;
  Free(bob->anImage);
  bob->anImage = NULL;

  if (save) /* Save the image frames as they are displayed */
    {
      tf = XmTextFieldGetString(GetWidget("vsize"));
      if (sscanf(tf, "%u%u", &bob->anWidth, &bob->anHeight) == 2) {
	if (bob->anWidth % 4) {
	  bob->anWidth = RoundUp(bob->anWidth, 4);
	  Error("Width must be a multiple of four.\n");
	  Error("Rounding width up to %d\n", bob->anWidth);
	}
	SetValue(XmNwidth, (XtArgVal) bob->anWidth, NULL);	
	SetValue(XmNheight, (XtArgVal) bob->anHeight, GetWidget("vox"));
	GetValue(XmNshadowThickness, &shadow, GetWidget("voxframe"));
	GetValue(XmNx, &x, NULL);
	GetValue(XmNy, &y, bob->shell);
	newx = x;
	newy = y;
	if (x + shadow + bob->vwidth > WidthOfScreen(screen))
	  newx = WidthOfScreen(screen) - shadow - bob->vwidth;
	if (y + shadow + bob->vheight > HeightOfScreen(screen))
	  newy = HeightOfScreen(screen) - shadow - bob->vheight;
	if (newx < -shadow)
	  newx = -shadow;
	if (newy < -shadow)
	  newy = -shadow;
	if (newx != x  ||  newy != y) {
	  SetValue(XmNx, (XtArgVal) newx, NULL);
	  SetValue(XmNy, (XtArgVal) newy, bob->shell);
	}
      }
      XtFree(tf);
      
      bob->anFileName = XmTextFieldGetString(GetWidget("device"));
      
      tf = XmTextFieldGetString(GetWidget("offset"));
      bob->anOffset = ScanLong(tf);
      XtFree(tf);
      
      DrawVoxelWindow(False, True);
    }
  
  /* Set the state of the user interface elements. */
  XtSetSensitive(GetWidget("savemov"), False);
  XtSetSensitive(GetWidget("savecoord"), False);
  XtSetSensitive(GetWidget("stanimate"), False);
  XtSetSensitive(GetWidget("record"), False);
  XtSetSensitive(GetWidget("play"), False);
  XtSetSensitive(GetWidget("stop"), True);
  XtSetSensitive(GetWidget("frame"), False);

  bob->play = 1;
  bob->playWP = XtAppAddWorkProc(bob->app, PlayMovieCB, (XtPointer) NULL);
}


/*********************************************************************
**********************************************************************
**
** void StopMovie()
**
** Action:
**   Stops movie playback
**
**********************************************************************
*********************************************************************/

void StopMovie(void)
{
  /* If the movie is playing, remove the work process and reset global state
   * variables */

  if (bob->play)
    {
      bob->play = 0;
      bob->num_of_pnts = 0;
    }
  if (bob->playWP)
    XtRemoveWorkProc(bob->playWP);

  if (bob->anFile)
    {
      (void) fclose(bob->anFile);
      bob->anFile = NULL;
    }

  if (bob->anImage)
    {
      free(bob->anImage);
      bob->anImage = NULL;
    }

  /* Set state of the user interface elements */
  XtSetSensitive(GetWidget("savemov"), True);
  XtSetSensitive(GetWidget("savecoord"), True);
  XtSetSensitive(GetWidget("stanimate"), True);
  XtSetSensitive(GetWidget("record"), True);
  XtSetSensitive(GetWidget("stop"), False);
  XtSetSensitive(GetWidget("play"), True);
  XtSetSensitive(GetWidget("frame"), True);
}


/*********************************************************************
**********************************************************************
**
** void PlayMovieCB()
**
** Action:
**   XtWorkProcedure which performs the playback of a generated
**   movie. 
** 
** Inputs:
**   closure - Used by X to determine whether this routine should be
**             called again from the event loop.
**
** Outputs:
**
**********************************************************************
*********************************************************************/

Boolean PlayMovieCB(XtPointer closure)
{
  int j;

  /* If the movie is at the end. Reset it to the beginning and 
   * stop playback. Alternatively, if we are playing backwards,
   * stop the movie when it hits the start and leave it there. */

  if (bob->playFrame > bob->num_of_inter_pnts - 1)
    {
      bob->playFrame = bob->num_of_inter_pnts - 1;
      StopMovie();

      return TRUE;
    }
  else if (bob->playFrame < 0)
    {
      bob->playFrame = 0;
      StopMovie();

      return TRUE;
    }
  else /* The end of the movie has not been reached */
    {
      /* Change the volume view to the interpolated view */
      for (j = 0; j < 4; j++) {
	bob->vball->qrot[j] = bob->quat_inter[bob->playFrame][j];
      }
      for (j = 0; j < 3; j++) {
	bob->vball->trans[j] = bob->trans_inter[bob->playFrame][j];
      }
      
      if (StackSize(bob->frameList) > 1)
	{
	  bob->iframe = bob->recFrame_inter[bob->playFrame];
	  if (bob->iframe > StackSize(bob->frameList))
	    bob->iframe = 1;
	  else if (bob->iframe < 1)
	    bob->iframe = StackSize(bob->frameList);
	  
	  XmScaleSetValue(GetWidget("frame"), bob->iframe);
	  ReadData();
	}

      /* Update voxel and finder windows */
      TrackballCopy(bob->vball, bob->fball);
      DrawFinderWindow();
      if (bob->anFileName)
	DrawVoxelWindow(False, True);
      else
	DrawVoxelWindow(False, False);
      
      /* Increment movie to the next frame */
      bob->playFrame += bob->playFrameInc;
      
      return FALSE;
    }
}


/*********************************************************************
**********************************************************************
**
** void ChangeShuttleIncr()
**
** Action:
**   Shuttles the movie forward or backward when the shuttle controls
**   are used. It takes care of the cases when the buttons are single
**   clicked or when the buttons are held down.
** 
** Inputs:
**   incr      - amount by which the frame will be advanced. The routine 
**               is called by ShuttleMovieCB() in bob.c with incr of 1 
**               or -1 for forward and backward respectively.
**   direction - Inidcates which direction the movie should be
**               shuttled, forward or backward.
**   id        - Specifies the ID of the timeout process request.
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void ChangeShuttleIncr(int incr, int direction, XtIntervalId id)
{
  /* Do something */
  if (direction == ButScanRev)
    {
      if ((bob->playFrameInc != -1) && (bob->playFrame != 0))
	{
	  bob->playFrame += -2;
	  bob->playFrameInc = incr;
	}
      if (bob->playFrame == 0)
	{
	  bob->playFrameInc = 0;
	  bob->playFrame = 0;
	}
    }
  else if (direction == ButScanFor)
    {
      if ((bob->playFrameInc != 1) && (bob->playFrame != bob->num_of_inter_pnts - 1))
	{
	  bob->playFrame += 2;
	  bob->playFrameInc = incr;
	}
      if (bob->playFrame == bob->num_of_inter_pnts - 1)
	{
	  bob->playFrameInc = 0;
	  bob->playFrame = bob->num_of_inter_pnts - 1;
	}
    }
  PlayMovieCB((XtPointer)TRUE);

  if (id > 0)
    bob->shuttleTimerID =
      XtAppAddTimeOut(bob->app, 
		      (id == 1) ? 300 : 50, 
		      (XtTimerCallbackProc)ChangeShuttleIncr, 
		      (XtPointer)incr);
}

/* This routine generates a view matrix compatible with Michael
 * Palmer's RendAsunder. */

/*********************************************************************
**********************************************************************
**
** void CreateRendVects()
**
** Action:
**   This routine generates animation path vectors compatible with
**   Michael Palmer's Rend and RendAsunder programs. 
** 
** Inputs:
**   m     - View matrix
**   cdir  - Cartesian 3-space vector specifying camera direction
**   up    - Cartesian 3-space vector specifying up direction
**   eye   - Cartesian 3-space vector specifying eye (look) direction
**   trans - Cartesian 3-space vector extracted from trackball
**
** Outputs:
**
**********************************************************************
*********************************************************************/

void CreateRendVects(GLfloat m[4][4], float cdir[3], float up[3], float eye[3], float trans[3])
{

  /* cdir and up are the transpose matrix of z and the y
   * of the m matrix respectively.
   *
   * The easiest way to think about it is:  The angle of rotation
   * of the original x axis from the z direction is equivalent to
   * the negative angle of rotation that the original z axis moved
   * from the rotated x axis. */

  cdir[0] = -m[0][2];
  cdir[1] = -m[1][2];
  cdir[2] = -m[2][2];
  up[0]   =  m[0][1];
  up[1]   =  m[1][1];
  up[2]   =  m[2][1];
  
  /* bob->eyeDist is the original translation in the -z coord.
   * the + trans[2] is adding the translation in the z coordinate
   * system of the screen (the screen always remain perpendicular
   * to the field). */
  
  eye[0] = (-bob->eyeDist + trans[2])*(cdir[0]) - 
    m[0][0]*trans[0] - m[0][1]*trans[1];
  eye[1] = (-bob->eyeDist + trans[2])*(cdir[1]) - 
    m[1][0]*trans[0] - m[1][1]*trans[1];
  eye[2] = (-bob->eyeDist + trans[2])*(cdir[2]) - 
    m[2][0]*trans[0] - m[2][1]*trans[1];

}


/*********************************************************************
**********************************************************************
**
** void SaveCoordCB()
**
** Action:
**   Writes out animation path coordinates and other information to
**   either a binary or ASCII file.
** 
** Inputs:
**
**********************************************************************
*********************************************************************/

void SaveCoordCB(int flags)
{
  GLfloat   m[4][4];
  int       i, j;
  float     trans[3], eye[3], cdir[3], up[3];
  char      FileName[FILENAME_MAX];
  FILE     *f;
  char     *string;

  /* Get the file name from the coordinates dialog box */

  XtVaGetValues(GetWidget("fileEntry"), XmNvalue, &string, NULL);

  XtFree(string);

  /* Open the coordinates file for writing */

  if ( ( f=fopen(string, "w") ) == NULL )
    {
      fprintf(stderr, "Error opening file \"%s\"! Error #: %d\n", string, errno);
      exit(errno);
    }

  /* Figure out if we should save a binary file or an ASCII file */

  if (!bob->binCoords)  /* ASCII File */
    {
      fprintf(f, "KF ");
      fprintf(f, "Quat[0] Quat[1] Quat[2] Quat[3] Trans[0] Trans[1] Trans[2] ");
      fprintf(f, "Eye[0] Eye[1] Eye[2] CDir[0] CDir[1] CDir[2] Up[0] Up[1] Up[2] ");
      fprintf(f, "FrameNum");
      fprintf(f, "\n");

      for (i = 0; i < bob->num_of_inter_pnts; i++)
	{
	  fprintf(f, "%d ", (bob->keyFrameFlags[i] == 1 ? 1 : 0));
	  fprintf(f, "%f %f %f %f ", 
		  bob->quat_inter[i][0], bob->quat_inter[i][1], 
		  bob->quat_inter[i][2], bob->quat_inter[i][3]);

	  for (j = 0; j < 3; j++)
	    {
	      trans[j] = bob->trans_inter[i][j];
	    }

	  qmatrix(bob->quat_inter[i], m[0]);
	  CreateRendVects(m, cdir, up, eye, trans);

	  fprintf(f, "%f %f %f ", trans[0], trans[1], trans[2]);
	  fprintf(f, "%f %f %f ", eye[0], eye[1], eye[2]);
	  fprintf(f, "%f %f %f ", cdir[0], cdir[1], cdir[2]);
	  fprintf(f, "%f %f %f ", up[0], up[1], up[2]);
	  fprintf(f, "%d\n", i);
	}
	 
      fclose(f);
    }
  else                  /* Binary File */
    {
      printf("Binary Coordinates!\n");
    }
}
