/*
 *    Copyright (c) 1992 Minnesota Supercomputer Center, Inc.
 *    Copyright (c) 1992 Army High Performance Computing Research Center
 *        (AHPCRC), University of Minnesota
 *    Copyright (c) 1995-1999 Laboratory for Computational Science and
 *        Engineering (LCSE), University of Minnesota
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file CONTRIBUTORS for a list of contributors.
 *
 *    Original author(s):
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    This file is maintained by:
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    Module name: vox.c
 *
 *    Description:
 *      Voxel rendering routines.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#ifdef HAVE_ULOCKS_H
#include <ulocks.h>
#endif
#ifdef HAVE_TASK_H
#include <task.h>
#endif

#include <Xm/Xm.h>
#include <Xm/ToggleBG.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/GLwMDrawA.h>

#include "util.h"
#include "skip.h"
#include "xtutil.h"
#include "glutil.h"
#include "draw.h"
#include "vox.h"
#include "bob.h"

/* Preprocessor Defines */

#define VNEAR	0.5
#define VFAR	10.0

#define	ANNOTATE_COLOR	128,  128,  255	/* Slate blue */
#define	BOUNDS_COLOR	255,  128,  0	/* Dark orange */


void SetView(int width, int height)
{
    /* Setup gl viewing from scratch.
     * Use bob->fieldOfView deg of perspective. */

    GLint mm;

    glGetIntegerv(GL_MATRIX_MODE, &mm);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(.1*(bob->fieldOfView),
		   (float) width / (float) height,
		   VNEAR,  VFAR);
    glMatrixMode(mm);

    glViewport(0,  0, width, height);
    glScissor(0,  0, width, height);
    glTranslatef(0.0,  0.0,  -bob->eyeDist);
}


/*static float sFogSetup[] = { 1.0, 0.0, 0.0, 0.0 };*/

void SetupGL(void)
{
    bob->blend = 1; /* blending always supported in OpenGL */
    if (bob->blend) {
	glEnable(GL_LINE_SMOOTH);
	glHint(GL_LINE_SMOOTH_HINT, GL_DONT_CARE);
 	glEnable(GL_POINT_SMOOTH);
        glBlendFunc(GL_SRC_ALPHA,  GL_ONE_MINUS_SRC_ALPHA);
    }
    glEnable(GL_DEPTH_TEST);
    
    bob->fog = 1; /* fog always supported in OpenGL */
    if (bob->fog) {
	/*sFogSetup[0] = bob->fogDensity;*/
        /*glFogf(GL_FOG_DENSITY,  sFogSetup[0]); not necessary -- use default */
        glDisable(GL_FOG);
    }
    
    glMatrixMode(GL_MODELVIEW);
}


static void ParallelPrep(unsigned interpolate)
{
    register unsigned int nx, ny, nz, nxy, dz;
    register unsigned int i, j, k, l, ix, iy, iz;
#ifndef HAVE_M_FORK
    register unsigned int t;
#endif
    register unsigned char *src, *fwd, *dst;
    
    nx = bob->dataDim[0];
    ny = bob->dataDim[1];
    nz = bob->dataDim[2];
    
    /* Transpose x to z */

    src = bob->cdata[0];
    dst = bob->cdata[2];

#ifdef HAVE_M_FORK
    while ((k = m_next()) < nz) {
	l = k*nx*ny;
	for (j = ny; j; --j)
	    for (i = nx; i; --i, l++, k += nz)
		dst[k] = src[l];
    }
    
    m_sync();
#else
    for (t = 0; t < nz; t++) {
	k = t;
	l = k*nx*ny;
	for (j = ny; j; --j)
	    for (i = nx; i; --i, l++, k += nz)
		dst[k] = src[l];
    }
#endif
    
    /* Transpose z to y */

    src = bob->cdata[2];
    dst = bob->cdata[1];

#ifdef HAVE_M_FORK
    while ((k = m_next()) < ny) {
	l = k*nx*nz;
	for (j = nx; j; --j)
	    for (i = nz; i; --i, l++, k += ny)
		dst[k] = src[l];
    }
#else
    for (t = 0; t < ny; t++) {
	k = t;
	l = k*nx*nz;
	for (j = nx; j; --j)
	    for (i = nz; i; --i, l++, k += ny)
		dst[k] = src[l];
    }
#endif
    
    if (interpolate) {
#ifdef HAVE_M_FORK
	m_sync();
	
	while ((ix = m_next()) < 3) {
#else
	for (ix = 0; ix < 3; ix++) {
#endif	    
	    iy = (ix + 1) % 3;
	    iz = (ix + 2) % 3;
	    
	    nx = bob->voxDim[ix][ix];
	    ny = bob->voxDim[ix][iy];
	    nz = bob->voxDim[ix][iz];
	    dz = bob->dataDim[iz];
	    nxy = nx*ny;
	    
	    src = bob->cdata[ix] + (dz-1)*nxy;
	    dst = bob->cdata[ix] + (nz-1)*nxy;
	    for (k = 1; k < dz; k++) {
		memcpy(dst, src, nxy);
		src -= nxy;
		dst -= 2*nxy;
	    }
	    
	    src = bob->cdata[ix];
	    dst = src + nxy;
	    fwd = dst + nxy;
	    for (k = 1; k < dz; k++) {
		for (i = 0; i < nxy; ++i)
		    dst[i] = (src[i] + fwd[i]) / 2;
		src = fwd;
		dst = src + nxy;
		fwd = dst + nxy;
	    }
	}
    }
}
    

static void PrepareVoxel(unsigned interpolate)
{
    DrawFinderSpecial(FinderInterp, 0);
#ifdef HAVE_M_FORK
    if (bob->parked)
	m_rele_procs();

    Verify(m_fork(ParallelPrep, interpolate) == 0,
	   "m_fork ParallelPrep failure");

    if (m_park_procs() == 0)
        bob->parked = True;
#else
    ParallelPrep(interpolate);
#endif
}


static void ParallelMark(unsigned *opaque, unsigned *bittab)
{
    register unsigned int nx, ny, nz, ix, iy, iz, nx8;
    register unsigned int i, j, k, ibit, m, last, optot;
    register unsigned char *face;
    register unsigned char *src;

#ifdef HAVE_M_FORK
    while ((ix = m_next()) < 3) {
	if (m_get_numprocs() == 1)
	    DrawFinderSpecial(FinderScan, 2 + 2*ix);
#else
    for (ix = 0; ix < 3; ix++) {
	DrawFinderSpecial(FinderScan, 2 + 2*ix);
#endif

	iy = (ix + 1) % 3;
	iz = (iy + 1) % 3;

	nx = bob->voxDim[ix][ix];
	ny = bob->voxDim[ix][iy];
	nz = bob->voxDim[ix][iz];
	nx8 = RoundUp(nx, 8) / 8;

	bob->faces[ix] = ReallocType(bob->faces[ix], unsigned char, ny*nz);
	MemCheck(bob->faces[ix]);

	face = bob->faces[ix];
	src = bob->cdata[ix];
	
	optot = 0;

	for (k = 0; k < nz; ++k) {
	    for (j = 0; j < ny; ++j) {
		last = 0;
		face[j] = 0;
		for (i = 0, ibit = 1; i < nx; ibit <<= 1) {
		    if (last)
			face[j] |= ibit;
		    if (opaque[src[i]])
			face[j] |= ibit >> 1;
		    if ((m = i + nx8) > nx)
			m = nx;
		    for (; i < m; ++i)
			if (opaque[src[i]]) {
			    face[j] |= ibit;
			    i = m;
			    break;
			}
		    last = opaque[src[i-1]];
		}
		if (face[j])
		    for (; ibit < 256; ibit <<= 1)
			face[j] |= ibit;
		optot += bittab[face[j]];
		src += nx;
	    }
	    face += ny;
	}

	if (bob->fastRate) {
	    optot *= bob->fastRate * nx8;
	    bob->fastStride[ix] = cbrt((float) optot / bob->polyRate) + 1;
	    if (bob->fastStride[ix] < 2)
		bob->fastStride[ix] = 0;
	} else
	    bob->fastStride[ix] = 0;
    }
}


static void MarkOpaque(void)
{
    unsigned		i, j, opaque[256];
    static unsigned	*bittab = NULL;

    if (!bittab) {
	bittab = CallocType(unsigned, 256);
	MemCheck(bittab);
	for (i = 0; i < 256; ++i) {
	    bittab[i] = 0;
	    for (j = 1; j < 256; j <<= 1)
		if (j & i)
		    ++bittab[i];
	}
    }
    
    /* fill in face bytes */
    for (i = 0; i < 256; ++i)
	opaque[i] = bob->atab[i] > 0;

#ifdef HAVE_M_FORK
    if (bob->parked)
	m_rele_procs();

    if (m_get_numprocs() > 1)
	DrawFinderSpecial(FinderScan, 2);

    Verify(m_fork(ParallelMark, opaque, bittab) == 0, 
	   "m_fork ParallelMark failure");

    if (m_get_numprocs() > 1)
        DrawFinderSpecial(FinderScan, 6);

    if (m_park_procs() == 0)
	bob->parked = True;
#else
    ParallelMark(opaque, bittab);
#endif
}


void ReadAllPoint(void)
{
    int		i, j, tp, pframe;
    int		memfile, shmid;
    char	*pname, *buf;
    
    Free(bob->allpoint);
    bob->allpoint = NULL;
    
    pframe = bob->iframe;
    if (pframe > StackSize(bob->pointList))
	pframe = StackSize(bob->pointList);
    pname = StackSearch(bob->pointList, pframe);
    if (!pname)
	return;
    
    memfile = strncmp(pname, MEMFILE, strlen(MEMFILE)) == 0;
    
    if (memfile) {
	shmid = atoi(pname + strlen(MEMFILE));
	buf = (char *) shmat(shmid, 0, 0);
	if (!bob->nallpoint) {
	    Error("Need to specify number of points\n");
	    return;
	}

    } else {
	FILE	*pfile = fopen(pname, "r");
	int	fsize;

	if (!pfile) {
	    Error("Cannot open point file %s\n", pname);
	    return;
	}

	fsize = FindFileSize(pname);
	bob->nallpoint = fsize / bob->pstride;
	if (!bob->nallpoint) {
	    Error("no points in file %s\n", pname);
	    return;
	}

	buf = CallocType(char, fsize);
	MemCheck(buf);
	if (fread(buf, fsize, 1, pfile) < 1)
	    Error("short read on point file %s\n", pname);
	(void) fclose(pfile);
    }

    tp = bob->nallpoint * 3;
    bob->allpoint = CallocType(float, tp);
    MemCheck(bob->allpoint);

    for (i = 0, j = bob->poffset; i < tp; i += 3, j += bob->pstride)
	memcpy(&bob->allpoint[i], buf + j, 3*sizeof(float));
    
    if (memfile)
	(void) shmdt(buf);
    else
	Free(buf);
}


void ReadPoint(void)
{
    int		i, j, tp, np;
    /*float	hi[3], lo[3], stride;*/
    float	lo[3], stride;
    float	*ap, *pt;
    
    Free(bob->point);
    bob->point = NULL;
    bob->npoint = 0;

    if (StackSize(bob->pointList) == 0)
	return;
    ReadAllPoint();
    if (!bob->allpoint)
	return;

    for (i = 0; i < 3; i++) {
	lo[i] = bob->dataOffset[i];
	/*hi[i] = lo[i] + bob->dataDim[i] * bob->stride;*/
    }
    
    tp = bob->nallpoint * 3;
    ap = bob->allpoint;

    pt = bob->point = CallocType(float, tp);
    np = bob->npoint = 0;

    /*
	if (ap[i+0] >= lo[0]  &&  ap[i+0] <= hi[0]  &&
	    ap[i+1] >= lo[1]  &&  ap[i+1] <= hi[1]  &&
	    ap[i+2] >= lo[2]  &&  ap[i+2] <= hi[2]) {
    */

    j = 0;
    stride = bob->stride;
    for (i = 0; i < tp; i += 3) {
	pt[j+0] = (ap[i+0] - lo[0]) / stride;
	pt[j+1] = (ap[i+1] - lo[1]) / stride;
	pt[j+2] = (ap[i+2] - lo[2]) / stride;
	j += 3;
	np++;
    }
    
    bob->npoint = np;
}


void ReadData(void)
{
    BobFrame		*bframe = StackSearch(bob->frameList, bob->iframe);
    int			dim[3];
    int			center[3];
    int			maxdim, stride;
    unsigned		rdim[3];
    unsigned		*fdim;
    unsigned char	*sd, *cd;
    unsigned		sliceOff, sliceSize, fpos, foff;
    unsigned		block, interpolate;
    register unsigned	i, j, k, l, ix, iy, iz;

    static unsigned char *slice = NULL;
    
    if (!bframe)
	return;
    if (bframe->bobFile != bob->file)
	SetCurrentFile(bframe->bobFile);
    if (!bob->file)
	return;
    
    interpolate = XmToggleButtonGadgetGetState(GetWidget("interpolate"));
    
    bob->useRatio = False;
    GetDimWidgets(dim, &maxdim, center, &stride);
    CheckDimension(dim, &maxdim, center, &stride);
    SetDimWidgets(dim, &maxdim, center, &stride);

    bob->stride = stride;
    for (i = 0; i < 3; ++i) {
	bob->dataDim[i] = dim[i];
	bob->dataOffset[i] = center[i] - (dim[i] * stride) / 2;
	rdim[i] = dim[i] * stride;
    }
    
    for (ix = 0; ix < 3; ++ix) {
	iy = (ix + 1) % 3;
	iz = (iy + 1) % 3;

	bob->voxDim[ix][ix] = dim[ix];
	bob->voxDim[ix][iy] = dim[iy];
	if (interpolate)
	    bob->voxDim[ix][iz] = 2*dim[iz] - 1;
	else
	    bob->voxDim[ix][iz] = dim[iz];

	l = bob->voxDim[ix][0] * bob->voxDim[ix][1] * bob->voxDim[ix][2];
	bob->cdata[ix] = ReallocType(bob->cdata[ix], unsigned char, l);
	MemCheck(bob->cdata[ix]);

	for (i = 0; i < 3; ++i) {
	    bob->voxPos[ix][i] = ReallocType(bob->voxPos[ix][i], float, 
					     bob->voxDim[ix][i]);
	    MemCheck(bob->voxPos[ix][i]);

	    if (interpolate  &&  i == iz)
		for (j = 0; j < bob->voxDim[ix][i]; ++j)
		    bob->voxPos[ix][i][j] = j*bob->dimScale[i]*0.5;
	    else
		for (j = 0; j < bob->voxDim[ix][i]; ++j)
		    bob->voxPos[ix][i][j] = j*bob->dimScale[i];
	}
    }

    fdim = bob->file->dim;
    block = bob->file->blockSize ? bob->file->blockSize : 512;
    
    sliceOff = bob->dataOffset[2]*fdim[0]*fdim[1] +
	fdim[0]*bob->dataOffset[1] + bob->dataOffset[0];

    sliceSize = RoundUp(fdim[0]*(rdim[1] - 1) + rdim[0], block) + block;
    slice = ReallocType(slice, unsigned char, sliceSize);
    MemCheck(slice);

    for (i = 0; i < dim[2]; ++i) {

	DrawFinderSpecial(FinderRead, i);

	if (bob->file->data) {
	    sd = bob->file->data + 
		bframe->seek + sliceOff + i*stride*fdim[0]*fdim[1];

	} else {
	    fpos = bframe->seek + sliceOff + i*stride*fdim[0]*fdim[1];
	    foff = fpos % block;
	    fpos -= foff;
	    
	    if (lseek(bob->file->fd, fpos, SEEK_SET) < 0) {
		Error("Unable to seek on file %s\n", bob->file->fname);
		break;
	    }
	    if (read(bob->file->fd, slice, sliceSize) < 0) {
		Error("Unable to read from file %s\n", bob->file->fname);
		break;
	    }
	    sd = slice + foff;
	}

	cd = bob->cdata[0] + i*dim[0]*dim[1];
	for (j = 0; j < dim[1]; ++j) {
	    if (stride == 1)
		(void) memcpy(cd, sd, dim[0]);
	    else
		for (k = l = 0; l < rdim[0]; ++k, l += stride)
		    cd[k] = sd[l];
	    cd += dim[0];
	    sd += fdim[0] * stride;
	}
    }

    PrepareVoxel(interpolate);
    MarkOpaque();

    ReadPoint();

    finderMode = FinderSync;
}


void PackColor(unsigned char *cmap)
{
    register unsigned	i, j, doAlpha;
    
    doAlpha = XmToggleButtonGadgetGetState(GetWidget("brightness")); 
    
    if (cmap)
	for (i = 0; i < 256; ++i) {
	    j = 3*i;
	    bob->ctab[i][0] = cmap[j]   / 255.0;
	    bob->ctab[i][1] = cmap[j+1] / 255.0;
	    bob->ctab[i][2] = cmap[j+2] / 255.0;
	}

    if (doAlpha  &&  cmap)
	for (i = 0; i < 256; ++i)
	    bob->atab[i] = (0.299 * bob->ctab[i][0] + 
			    0.587 * bob->ctab[i][1] + 
			    0.114 * bob->ctab[i][2]);
    for (i = 0; i < 256; ++i) {
	bob->ctab[i][3] = bob->atab[i] * bob->afact;
	if (bob->ctab[i][3] > 1)
	    bob->ctab[i][3] = 1;
    }

    if (doAlpha  &&  cmap  &&  bob->cdata[0])
	MarkOpaque();
}


void PackAlpha(unsigned char *amap)
{
    register unsigned	i;

    XmToggleButtonGadgetSetState(GetWidget("brightness"), False, False); 
    
    for (i = 0; i < 256; ++i) {
	bob->atab[i] = amap[i] / 255.0;
	bob->ctab[i][3] = bob->atab[i] * bob->afact;
	if (bob->ctab[i][3] > 1)
	    bob->ctab[i][3] = 1;
    }
    if (bob->cdata[0])
	MarkOpaque();
}


static void DrawVoxelScene(unsigned fast)
{
    float	sf;
    float	axis[3];
    int		ix, direction;
    unsigned	i, hi[3], npoly, ktime;
    unsigned	bounds = XmToggleButtonGadgetGetState(GetWidget("bounds"));
    unsigned	annotate = XmToggleButtonGadgetGetState(GetWidget("annotate"));
    unsigned	soft = XmToggleButtonGadgetGetState(GetWidget("softEdge"));
    unsigned	cbar = XmToggleButtonGadgetGetState(GetWidget("colorBar"));
    unsigned	maxval = XmToggleButtonGadgetGetState(GetWidget("maxval"));
    unsigned	dopoint = XmToggleButtonGadgetGetState(GetWidget("point"));
    unsigned	fog = XmToggleButtonGadgetGetState(GetWidget("fog"));
    
    vset(axis,
	 bob->dataDim[0]*bob->dimScale[0],
	 bob->dataDim[1]*bob->dimScale[1],
	 bob->dataDim[2]*bob->dimScale[2]);
    
    /* Turn off lighting */
    glDisable(GL_COLOR_MATERIAL);
    glDisable(GL_LIGHTING);
    
    /* Move to trackball view of scene */
    glPushMatrix();
    TrackballSetMatrix(bob->vball);
    sf = 1.0 / (float) MAX(MAX(axis[0], axis[1]), axis[2]);
    glScalef(sf,  sf,  sf);
    glTranslatef(-0.5*axis[0],  -0.5*axis[1],  -0.5*axis[2]);
    
    if (bob->fog && fog)
	glEnable(GL_FOG);
	
    /* Draw points */
    if (bob->point && dopoint) {
	if (bob->blend)
            glEnable(GL_BLEND);
	DrawPoint(bob->point, bob->npoint, bob->psize, bob->pcolor);
	if (bob->blend)
            glDisable(GL_BLEND);
    }

    /* Draw scene decorations: bounding box and axes annotation */
    if (bounds || annotate) {
	glPushMatrix();
	if (bob->blend)
	    glEnable(GL_BLEND);
	glTranslatef(-0.5 * bob->dimScale[0],
		     -0.5 * bob->dimScale[1],
		     -0.5 * bob->dimScale[2]);
	glColor3ub(BOUNDS_COLOR);
	DrawBounds(axis, bounds);
	if (annotate) {
	    for (i = 0; i < 3; ++i)
		hi[i] = bob->dataOffset[i] + bob->dataDim[i]*bob->stride;
	    glColor3ub(ANNOTATE_COLOR);
	    DrawAnnotatedBox(axis, bob->dataOffset, hi);
	}
	if (bob->blend)
	    glDisable(GL_BLEND);
	glPopMatrix();
    }
    
    if (bob->debug & 0x1) {
	glFinish();
	(void) MarkTime();
    }

    /* Draw the voxels last, blending on top of axis */
    if (maxval) {
	ix = (ViewAxis(&direction) + 1) % 3;
	if (fast && bob->fastStride[ix])
	    DrawStrideMaxVal(ix, direction, bob->voxDim[ix], bob->voxPos[ix], 
			     bob->cdata[ix], bob->ctab, bob->faces[ix],
			     bob->fastStride[ix]);
	else
	    DrawMaxVal(ix, direction, bob->voxDim[ix], bob->voxPos[ix], 
		       bob->cdata[ix], bob->ctab, bob->faces[ix]);
    } else if (bob->blend) {
	ix = (ViewAxis(&direction) + 1) % 3;
	if (fast && bob->fastStride[ix])
	    DrawStrideVoxel(ix, direction, bob->voxDim[ix], bob->voxPos[ix], 
			    bob->cdata[ix], bob->ctab, bob->faces[ix],
			    bob->fastStride[ix]);
	else
	    DrawVoxel(ix, direction, bob->voxDim[ix], bob->voxPos[ix], 
		      bob->cdata[ix], bob->ctab, bob->faces[ix], soft);
    } else
	DrawVoxFaces(bob->voxDim, bob->voxPos, bob->cdata, bob->ctab);

    if (bob->debug & 0x1) {
	glFinish();
	ktime = MarkTime();
	npoly = bob->voxDim[ix][(ix+2)%3] * 
	    (bob->voxDim[ix][(ix+1)%3]-1) * (bob->voxDim[ix][ix]-1);
	(void) fprintf(stderr, "kPoly/sec = %f\n", 
		       (float) npoly / (float) ktime);
    }
    
    if (bob->fog && fog)
	glDisable(GL_FOG);
	
    glPopMatrix();

    if (cbar)
	DrawColorBar(bob->ctab, bob->atab);
}


void DrawVoxelWindow(unsigned fast, unsigned save)
{
    unsigned	useFrontBuffer;
    
    if (!bob->vwin)
	return;

    glXMakeCurrent(bob->display, bob->vwin, bob->vctx);

    if (!fast && !bob->singleBuffer)
	useFrontBuffer = !XmToggleButtonGadgetGetState(GetWidget("dbuffer"));
    else
	useFrontBuffer = 0;

    if (useFrontBuffer) {
	glDrawBuffer(GL_FRONT);
	glReadBuffer(GL_FRONT);
    }

    glClearDepth(1);
    glClearColor(((float)((bob->bgColor)       & 0xff)) / 255.,
		  (float)((bob->bgColor) >>  8 & 0xff)  / 255.,
		  (float)((bob->bgColor) >> 16 & 0xff)  / 255.,
		  (float)((bob->bgColor) >> 24 & 0xff)  / 255. );
    glClear(GL_DEPTH_BUFFER_BIT|GL_COLOR_BUFFER_BIT);
    
    if (bob->cdata[0])
	DrawVoxelScene(fast);
    if (save)
	SaveAnimationFrame();

    if (useFrontBuffer) {
	glDrawBuffer(GL_BACK);
	/* OGLXXX SRC_AUTO not really supported -- see glReadBuffer man page */
	glReadBuffer(GL_BACK);
    } else if (!bob->singleBuffer)
	glXSwapBuffers(bob->display, bob->vwin);
}


void DrawStereoWindow(unsigned fast)
{
    unsigned	useFrontBuffer;
    
    if (!bob->swin)
	return;

    glXMakeCurrent(bob->display, bob->swin, bob->sctx);
    
    if (!fast && !bob->singleBuffer)
	useFrontBuffer = !XmToggleButtonGadgetGetState(GetWidget("dbuffer"));
    else
	useFrontBuffer = 0;
    
    /*if (useFrontBuffer) {
	glDrawBuffer(GL_FRONT);
    }*/
    
    glViewport(0,  0, bob->swidth, bob->sheight);
    /* glScissor(0,  0, bob->swidth, bob->sheight); */
    glClearDepth(1);
    glClearColor(((float)((bob->bgColor)       & 0xff)) / 255.,
                  (float)((bob->bgColor) >>  8 & 0xff)  / 255.,
                  (float)((bob->bgColor) >> 16 & 0xff)  / 255.,
                  (float)((bob->bgColor) >> 24 & 0xff)  / 255.);
    glClear(GL_DEPTH_BUFFER_BIT|GL_COLOR_BUFFER_BIT);
    
    if (bob->cdata[0]) {
	/* Right eye
	 */
        if (useFrontBuffer)
	   glDrawBuffer(GL_FRONT_RIGHT);
        else
	   glDrawBuffer(GL_BACK_RIGHT);
	/*glViewport(0,  0, bob->swidth, YMAXSTEREO+1);
        glScissor(0,  0, bob->swidth, YMAXSTEREO+1);*/
	glLoadIdentity();
	StereoPerspective(bob->fieldOfView, 
			  (float) bob->swidth / (float) bob->sheight,
			  VNEAR, VFAR, bob->eyeDist, bob->eyeSep);
	glTranslatef(0.0,  0.0,  -(bob->eyeDist+0.5));
	DrawVoxelScene(fast);
	
	/* Left eye
	 */
        if (useFrontBuffer)
	   glDrawBuffer(GL_FRONT_LEFT);
        else
	   glDrawBuffer(GL_BACK_LEFT);
	/*glViewport(0,  YOFFSET, bob->swidth, YMAXSTEREO+1);
        glScissor(0,  YOFFSET, bob->swidth, YMAXSTEREO+1);*/
	glLoadIdentity();
	StereoPerspective(bob->fieldOfView,
			  (float) bob->swidth / (float) bob->sheight,
			  VNEAR, VFAR, bob->eyeDist, -bob->eyeSep);
	glTranslatef(0.0,  0.0,  -(bob->eyeDist+0.5));
	DrawVoxelScene(fast);
    }
    
    if (useFrontBuffer) {
	glDrawBuffer(GL_BACK);
    } else if (!bob->singleBuffer)
	glXSwapBuffers(bob->display, bob->swin);
}



void DrawFinderWindow(void)
{
    unsigned	i;
    int		dim[3];
    int		center[3];
    int		maxdim, stride;
    
    /* If there's no window associated with the finder, don't bother. */

    if (!bob->fwin)
	return;

    /* Make the finder window the current rendering context. */

    glXMakeCurrent(bob->display, bob->fwin, bob->fctx);

    glClearDepth(1);
    glClearColor(((float)((bob->bgColor)       & 0xff)) / 255.,
		  (float)((bob->bgColor) >>  8 & 0xff)  / 255.,
		  (float)((bob->bgColor) >> 16 & 0xff)  / 255.,
		  (float)((bob->bgColor) >> 24 & 0xff)  / 255. );
    glClear(GL_DEPTH_BUFFER_BIT|GL_COLOR_BUFFER_BIT);

    if (bob->file) {
	GetDimWidgets(dim, &maxdim, center, &stride);

	for (i = 0; i < 3; ++i)
	    center[i] -= (dim[i] * stride) / 2;

	DrawFinderScene(dim, stride, center, 
			bob->file->dim, bob->dimScale, bob->fball);
    }

    /* The finder window is always double-buffered; swap the buffers. */

    glXSwapBuffers(bob->display, bob->fwin);
}


void DrawFinderSpecial(FinderEnum mode, unsigned plane)
{
    finderMode = mode;
    finderPlane = plane;
    DrawFinderWindow();
}

