/*
 *    Copyright (c) 1992 Minnesota Supercomputer Center, Inc.
 *    Copyright (c) 1992 Army High Performance Computing Research Center
 *        (AHPCRC), University of Minnesota
 *    Copyright (c) 1995-1999 Laboratory for Computational Science and
 *        Engineering (LCSE), University of Minnesota
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file CONTRIBUTORS for a list of contributors.
 *
 *    Original author(s):
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    This file is maintained by:
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    Module name: draw.c
 *
 *    Description:
 *      Voxel screen drawing routines.
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>

#include <X11/Xlib.h>
#include <GL/gl.h>

#include "bob.h"
#include "util.h"
#include "glutil.h"
#include "draw.h"

/* Preprocessor Defines */

#define	SYNC_COLOR	32, 32, 128	/* Midnight blue */
#define	SCAN_COLOR	100, 100, 100	/* 40% gray */
#define	DIFF_COLOR	150, 150, 0	/* Dark yellow */
#define	READ_COLOR	200, 0, 0	/* Medium red */
#define	INTERP_COLOR	READ_COLOR

#define	MATERIAL_LIST	100
#define	LIGHT0_LIST	200
#define	LIGHT1_LIST	201
#define	LIGHTMOD_LIST	300

/* Global Variables */

FinderEnum	finderMode = FinderSync;
unsigned	finderPlane = 0;
static GLuint	fbase = 0;
static GLuint	vbase = 0;


void DrawVoxel(int ix, int direction, unsigned dim[3], float *pos[3],
	       unsigned char *cdata, float cmap[256][4],
	       unsigned char *faces, unsigned soft)
{
    /* Time critical.
     * 
     * This is the core routine for drawing voxels.  The fastest way to
     * draw would be to prepare the colors and locations ahead of
     * time and draw from that array.  This entails a huge overhead in
     * memory (approx 32x), but acheives the fastest quad rate (350 k/sec).
     *
     * Next best is to construct the color and location information
     * on the fly, which is what this routine does.  The maximum quad rate
     * for this technique is about 260 k/sec.
     *
     * The faces array contains one byte per strip, with bits set
     * depending on whether a 1/8 section is opaque.  If all the
     * bits are set, the entire strip is drawn.
     *
     * This routine is passed the correct dimensions, positions,
     * data brick and faces array for the given direction ix.
     *
     * 'soft' indicates that the edges are faded to black.
     * A special color table, btab, is used to color these verticies.
     */
    register int	k, kstop;
    register unsigned	i, j, l, m, ibit, drawing;
    register unsigned	iy, iz, nx, ny, nx1, nx8, ny1;
    register float	**c1, **c2, *v1, *v2;
    register unsigned	c1b, c2b, v1b, v2b;
    register float	*posy, *posz, pp;
    register unsigned char *cd;
    register unsigned char *face;
    float		vnorm[3];
    
    /* Save buffers from pass to pass, 
     * hopefully saving speed on allocing memory.
     */
    static float	*ctab = NULL, *btab = NULL;
    static float	**cbuf = NULL, *vbuf = NULL;
    static unsigned	*qdraw = NULL, *build = NULL;
    
    glEnable(GL_BLEND);

    if (!ctab) {
	ctab = CallocType(float, 1024+4);
	MemCheck(ctab);
	while ((unsigned long)ctab & 0xf)	/* Quad word align */
	    ++ctab;
	btab = CallocType(float, 1024+4);
	MemCheck(btab);
	while ((unsigned long)btab & 0xf)	/* Quad word align */
	    ++btab;
    }
    
    /* Create color table, ctab, and blank table, btab
     */
    for (i = j = 0; j < 256; i += 4, ++j) {
	ctab[i  ] = cmap[j][0];
	ctab[i+1] = cmap[j][1];
	ctab[i+2] = cmap[j][2];
	ctab[i+3] = cmap[j][3];
    }
    if (soft)
	for (i = j = 0; j < 256; i += 4, ++j) {
	    btab[i  ] = cmap[j][0];	/* Same color as ctab */
	    btab[i+1] = cmap[j][1];
	    btab[i+2] = cmap[j][2];
	    btab[i+3] = 0;		/* Clear opacity */
	}

    /* Indecies of the other axes
     */
    iy = (ix + 1) % 3;
    iz = (ix + 2) % 3;

    /* Loop invariants
     */
    nx = dim[ix];
    ny = dim[iy];
    nx1 = nx - 1;
    nx8 = (unsigned int)RoundUp(nx, 8) / 8;
    ny1 = ny - 1;
    
    posy = pos[iy];
    posz = pos[iz];
    
    vnorm[ix] = 0;
    vnorm[iy] = 0;
    vnorm[iz] = direction;
    
    /* Allocate buffers for drawing.
     * vbuf, and cbuf are filled on the fly for each quad strip.
     * They are double length to hold both sides of the quad strip.
     * qdraw and build control drawing.
     */
    vbuf = ReallocType(vbuf, float, 8*nx + 4);	MemCheck(vbuf);
    cbuf = ReallocType(cbuf, float *, 2*nx);	MemCheck(cbuf);
    qdraw = ReallocType(qdraw, unsigned, ny);	MemCheck(qdraw);
    build = ReallocType(build, unsigned, ny);	MemCheck(build);

    /* Set buffer indecies.  Align vbuf to quad boundary.
     */
    c1b = 0;
    c2b = nx;

    v1b = 0;
    while ((unsigned long)(vbuf + v1b) & 0xf)
	++v1b;
    v2b = v1b + 4*nx;

    /* The X postions are always the same, so declare them here.
     * v1 points to the lower edge, v2 the upper edge.
     */
    v1 = vbuf + v1b;
    v2 = vbuf + v2b;
    for (i = 0, l = ix; i < nx; ++i, l += 4)
	v1[l] = v2[l] = pos[ix][i];
    
    /* Increment Z in the proper direction.  For soft edges,
     * Don't display the front & back planes (looks consistent
     * as the volume switched view directions.)
     */
    if (soft)
	if (direction > 0) {
	    k = 1;
	    kstop = dim[iz] - 1;
	} else {
	    k = dim[iz] - 2;
	    kstop = 0;
	}
    else
	if (direction > 0) {
	    k = 0;
	    kstop = dim[iz];
	} else {
	    k = dim[iz] - 1;
	    kstop = -1;
	}
    for (; k != kstop; k += direction) {
	
	/* Select the correct plane of voxel data and the
	 * correct strip of face values.
	 */
	cd = cdata + k*nx*ny;
	face = faces + k*ny;

	/* Construct the build and qdraw flags.  A vertex entry needs
	 * to be built if either it or its neighbors are opaque.
	 * A vertex entry is drawn if it or its lower neighbor are opaque.
	 * Drawing begins with the j=1 strip.
	 */
	for (j = 1; j < ny1; ++j) {
	    qdraw[j] = face[j-1] | face[j];
	    build[j] = qdraw[j]  | face[j+1];
	}
	qdraw[0]   = 0;
	build[0]   = face[0] | face[1];
	qdraw[ny1] = face[ny1-1] | face[ny1];
	build[ny1] = qdraw[ny1];

	/* Fill in the Z location for this plane.
	 */
	v1 = vbuf + v1b;
	v2 = vbuf + v2b;
	pp = posz[k];
	for (i = nx, l = iz; i; --i, l += 4)
	    v1[l] = v2[l] = pp;

	for (j = 0; j < ny; ++j, cd += nx) {

	    /* v1, v2 and c1, c2 swap every strip.
	     */
	    i = c1b; c1b = c2b; c2b = i;
	    i = v1b; v1b = v2b; v2b = i;
	    
	    v1 = vbuf + v1b;
	    v2 = vbuf + v2b;
	    c1 = cbuf + c1b;
	    c2 = cbuf + c2b;
	    pp = posy[j];

	    if (soft)
		/* Soft edges need special treatment at the edges.
		 * The edge color array points into btab, the interior
		 * into ctab.
		 */
		switch (build[j]) {

		  case 0:
		    /* Nothing to build
		     */
		    break;

		  case 0xff:
		    /* Build the whole strip in one run.  
		     * The end strips are allocated all out of btab.
		     */
		    if (j > 0 && j < ny1) {
			for (i = 0, l = iy; i < nx; ++i, l += 4) {
			    c2[i] = ctab + 4*cd[i];
			    v2[l] = pp;
			}
			c2[0]   = btab + 4*cd[0];
			c2[nx1] = btab + 4*cd[nx1];

		    } else
			for (i = 0, l = iy; i < nx; ++i, l += 4) {
			    c2[i] = btab + 4*cd[i];
			    v2[l] = pp;
			}
		    break;

		  default:
		    /* Build each eighth as needed.
		     * The end strips are allocated all out of btab.
		     */
		    if (j > 0 && j < ny1) {
			for (i = 0, l = iy, ibit = 1; i < nx; ibit <<= 1) {
			    if (ibit & build[j]) {
				if ((m = i + nx8) > nx)
				    m = nx;
				for (; i < m; ++i, l += 4) {
				    c2[i] = ctab + 4*cd[i];
				    v2[l] = pp;
				}
			    } else {
				i += nx8;
				l += 4*nx8;
			    }
			}
			c2[0]   = btab + 4*cd[0];
			c2[nx1] = btab + 4*cd[nx1];
			
		    } else
			for (i = 0, l = iy, ibit = 1; i < nx; ibit <<= 1) {
			    if (ibit & build[j]) {
				if ((m = i + nx8) > nx)
				    m = nx;
				for (; i < m; ++i, l += 4) {
				    c2[i] = btab + 4*cd[i];
				    v2[l] = pp;
				}
			    } else {
				i += nx8;
				l += 4*nx8;
			    }
			}
		    break;
		}
	    else
		/* Hard edges are allocated out of ctab only.
		 */
		switch (build[j]) {

		  case 0:
		    /* Nothing to build
		     */
		    break;

		  case 0xff:
		    /* Build the whole strip in one run.
		     */
		    for (i = 0, l = iy; i < nx; ++i, l += 4) {
			c2[i] = ctab + 4*cd[i];
			v2[l] = pp;
		    }
		    break;

		  default:
		    /* Build one eighth at a time.
		     */
		    for (i = 0, l = iy, ibit = 1; i < nx; ibit <<= 1) {
			if (ibit & build[j]) {
			    if ((m = i + nx8) > nx)
				m = nx;
			    for (; i < m; ++i, l += 4) {
				c2[i] = ctab + 4*cd[i];
				v2[l] = pp;
			    }
			} else {
			    i += nx8;
			    l += 4*nx8;
			}
		    }
		    break;
		}

	    switch (qdraw[j]) {

	      case 0:
		/* Nothing to draw.
		 */
		break;

	      case 0xff:
		/* Draw one full strip.
		 */
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		for (i = nx / 4; i; --i) {
		    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
		    glColor4fv(c1[1]); glVertex3fv(v1+ 4); glColor4fv(c2[1]); glVertex3fv(v2+ 4);
		    glColor4fv(c1[2]); glVertex3fv(v1+ 8); glColor4fv(c2[2]); glVertex3fv(v2+ 8);
		    glColor4fv(c1[3]); glVertex3fv(v1+12); glColor4fv(c2[3]); glVertex3fv(v2+12);
		    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
		}
		if (nx & 0x2) {
		    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
		    glColor4fv(c1[1]); glVertex3fv(v1+ 4); glColor4fv(c2[1]); glVertex3fv(v2+ 4);
		    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
		}
		if (nx & 0x1) {
		    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
		}
		glEnd();
		break;

	      default:
		/* Draw a strip one eighth at a time.  Begin and
		 * end the quad strip as needed.
		 */
		drawing = 1;
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		for (i = nx, ibit = 1; i; ibit <<= 1) {
		    l = MIN(nx8, i);
		    if (ibit & qdraw[j]) {
			if (!drawing) {
			    glBegin(GL_QUAD_STRIP);
			    glNormal3fv(vnorm);
			    drawing = 1;
			}
			for (m = l / 4; m; --m) {
			    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
			    glColor4fv(c1[1]); glVertex3fv(v1+ 4); glColor4fv(c2[1]); glVertex3fv(v2+ 4);
			    glColor4fv(c1[2]); glVertex3fv(v1+ 8); glColor4fv(c2[2]); glVertex3fv(v2+ 8);
			    glColor4fv(c1[3]); glVertex3fv(v1+12); glColor4fv(c2[3]); glVertex3fv(v2+12);
			    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
			}
			if (l & 0x2) {
			    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
			    glColor4fv(c1[1]); glVertex3fv(v1+ 4); glColor4fv(c2[1]); glVertex3fv(v2+ 4);
			    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
			}
			if (l & 0x1) {
			    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
			    c1 += 1;    v1 += 4;    c2 += 1;    v2 += 4;
			}
		    } else {
			if (drawing) {
			    glEnd();
			    drawing = 0;
			}
			v1 += l*4;
			v2 += l*4;
			c1 += l;
			c2 += l;
		    }
		    i -= l;
		}
		if (drawing)
		    glEnd();
		break;
	    }
	}
    }
    glDisable(GL_BLEND);
}


void DrawStrideVoxel(int ix, int direction, unsigned dim[3], float *pos[3],
		     unsigned char *cdata, float cmap[256][4],
		     unsigned char *faces, unsigned stride)
{
    /* Time critical.
     * 
     * This is the core routine for drawing voxels.  The fastest way to
     * draw would be to prepare the colors and locations ahead of
     * time and draw from that array.  This entails a huge overhead in
     * memory (approx 32x), but acheives the fastest quad rate (350 k/sec).
     *
     * Next best is to construct the color and location information
     * on the fly, which is what this routine does.  The maximum quad rate
     * for this technique is about 260 k/sec.
     *
     * The faces array contains one byte per strip, with bits set
     * depending on whether a 1/8 section is opaque.  If all the
     * bits are set, the entire strip is drawn.
     *
     * This routine is passed the correct dimensions, positions,
     * data brick and faces array for the given direction ix.
     */
    register int	k, kstop;
    register unsigned	i, j, l, m, n, ibit, drawing;
    register unsigned	iy, iz, nx, ny, nx8, ny1;
    register float	**c1, **c2, *v1, *v2;
    register unsigned	c1b, c2b, v1b, v2b;
    register float	*posy, *posz, pp;
    register unsigned char *cd;
    register unsigned char *face;
    float		vnorm[3];
    
    /* Save buffers from pass to pass, 
     * hopefully saving speed on allocing memory.
     */
    static float	*ctab = NULL;
    static float	**cbuf = NULL, *vbuf = NULL;
    static unsigned	*qdraw = NULL, *build = NULL;
    
    glEnable(GL_BLEND);

    if (!ctab) {
	ctab = CallocType(float, 1024+4);
	MemCheck(ctab);
	while ((unsigned long)ctab & 0xf)	/* Quad word align */
	    ++ctab;
    }
    
    /* Create color table, ctab
     */
    for (i = j = 0; j < 256; i += 4, ++j) {
	ctab[i  ] = cmap[j][0];
	ctab[i+1] = cmap[j][1];
	ctab[i+2] = cmap[j][2];
	ctab[i+3] = cmap[j][3] * (float) stride * 0.8;
	if (ctab[i+3] > 1.0)
	    ctab[i+3] = 1.0;
    }

    /* Indecies of the other axes
     */
    iy = (ix + 1) % 3;
    iz = (ix + 2) % 3;

    /* Loop invariants
     */
    nx = dim[ix];
    ny = dim[iy];
    nx8 = (unsigned int)RoundUp(nx, 8) / 8;
    ny1 = (unsigned int)RoundUp(ny, stride) - stride;
    
    posy = pos[iy];
    posz = pos[iz];
    
    vnorm[ix] = 0;
    vnorm[iy] = 0;
    vnorm[iz] = direction;
    
    /* Allocate buffers for drawing.
     * vbuf, and cbuf are filled on the fly for each quad strip.
     * They are double length to hold both sides of the quad strip.
     * qdraw and build control drawing.
     */
    vbuf = ReallocType(vbuf, float, 8*nx + 4);	MemCheck(vbuf);
    cbuf = ReallocType(cbuf, float *, 2*nx);	MemCheck(cbuf);
    qdraw = ReallocType(qdraw, unsigned, ny);	MemCheck(qdraw);
    build = ReallocType(build, unsigned, ny);	MemCheck(build);

    /* Set buffer indecies.  Align vbuf to quad boundary.
     */
    c1b = 0;
    c2b = nx;

    v1b = 0;
    while ((unsigned long)(vbuf + v1b) & 0xf)
	++v1b;
    v2b = v1b + 4*nx;

    /* The X postions are always the same, so declare them here.
     * v1 points to the lower edge, v2 the upper edge.
     */
    v1 = vbuf + v1b;
    v2 = vbuf + v2b;
    for (i = 0, l = ix; i < nx; i += stride, l += 4)
	v1[l] = v2[l] = pos[ix][i];
    
    /* Increment Z in the proper direction.  
     */
    if (direction > 0) {
	k = 0;
	kstop = (int)RoundUp(dim[iz], stride);
    } else {
	k = (int)RoundUp(dim[iz], stride) - stride;
	kstop = -stride;
    }
    direction *= stride;
    for (; k != kstop; k += direction) {
	
	/* Select the correct plane of voxel data and the
	 * correct strip of face values.
	 */
	cd = cdata + k*nx*ny;
	face = faces + k*ny;

	/* Construct the build and qdraw flags.  A vertex entry needs
	 * to be built if either it or its neighbors are opaque.
	 * A vertex entry is drawn if it or its lower neighbor are opaque.
	 * Drawing begins with the j=1 strip.
	 */
	for (j = stride; j < ny1; j += stride) {
	    qdraw[j] = face[j-stride] | face[j];
	    build[j] = qdraw[j]  | face[j+stride];
	}
	qdraw[0]   = 0;
	build[0]   = face[0] | face[stride];
	qdraw[ny1] = face[ny1-stride] | face[ny1];
	build[ny1] = qdraw[ny1];

	/* Fill in the Z location for this plane.
	 */
	v1 = vbuf + v1b;
	v2 = vbuf + v2b;
	pp = posz[k];
	for (i = 0, l = iz; i < nx; i += stride, l += 4)
	    v1[l] = v2[l] = pp;

	for (j = 0; j < ny; j += stride, cd += nx*stride) {

	    /* v1, v2 and c1, c2 swap every strip.
	     */
	    i = c1b; c1b = c2b; c2b = i;
	    i = v1b; v1b = v2b; v2b = i;
	    
	    v1 = vbuf + v1b;
	    v2 = vbuf + v2b;
	    c1 = cbuf + c1b;
	    c2 = cbuf + c2b;
	    pp = posy[j];
	    
	    switch (build[j]) {
		
	      case 0:
		/* Nothing to build
		 */
		break;
		
	      case 0xff:
		/* Build the whole strip in one run.
		 */
		for (i = 0, l = iy, n = 0; i < nx; i += stride, l += 4, ++n) {
		    c2[n] = ctab + 4*cd[i];
		    v2[l] = pp;
		}
		break;
		
	      default:
		/* Build one eighth at a time.
		 */
		n = 0;
		l = iy;
		for (i = 0, m = nx8, ibit = 1; i < nx; m += nx8, ibit <<= 1) {
		    if (m > nx)
			m = nx;
		    if (ibit & build[j])
			for (; i < m; i += stride, ++n, l += 4) {
			    c2[n] = ctab + 4*cd[i];
			    v2[l] = pp;
			}
		    else
			for (; i < m; i += stride, ++n)
			    l += 4;
		}
		break;
	    }
	    
	    switch (qdraw[j]) {

	      case 0:
		/* Nothing to draw.
		 */
		break;

	      case 0xff:
		/* Draw one full strip.
		 */
		n = nx / stride;
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		for (i = n / 4; i; --i) {
		    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
		    glColor4fv(c1[1]); glVertex3fv(v1+ 4); glColor4fv(c2[1]); glVertex3fv(v2+ 4);
		    glColor4fv(c1[2]); glVertex3fv(v1+ 8); glColor4fv(c2[2]); glVertex3fv(v2+ 8);
		    glColor4fv(c1[3]); glVertex3fv(v1+12); glColor4fv(c2[3]); glVertex3fv(v2+12);
		    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
		}
		if (n & 0x2) {
		    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
		    glColor4fv(c1[1]); glVertex3fv(v1+ 4); glColor4fv(c2[1]); glVertex3fv(v2+ 4);
		    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
		}
		if (n & 0x1) {
		    glColor4fv(c1[0]); glVertex3fv(v1   ); glColor4fv(c2[0]); glVertex3fv(v2   );
		}
		glEnd();
		break;

	      default:
		/* Draw a strip one eighth at a time.  Begin and
		 * end the quad strip as needed.
		 */
		drawing = 1;
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		n = 0;
		l = 0;
		for (i = 0, m = nx8, ibit = 1; i < nx; m += nx8, ibit <<= 1) {
		    if (m > nx)
			m = nx;
		    if (ibit & qdraw[j]) {
			if (!drawing) {
			    drawing = 1;
			    glBegin(GL_QUAD_STRIP);
			    glNormal3fv(vnorm);
			}
			for (; i < m; i += stride, ++n, l += 4) {
			    glColor4fv(c1[n]); glVertex3fv(v1+l); glColor4fv(c2[n]); glVertex3fv(v2+l);
			}
		    } else {
			if (drawing) {
			    drawing = 0;
			    glEnd();
			}
			for (; i < m; i += stride, ++n)
			    l += 4;
		    }
		}
		if (drawing)
		    glEnd();
		break;
	    }
	}
    }
    glDisable(GL_BLEND);
}


void DrawMaxVal(int ix, int direction, unsigned dim[3], float *pos[3],
		unsigned char *cdata, float cmap[256][4],
		unsigned char *faces)
{
    /* Time critical.
     * 
     * This is the core routine for drawing voxels.  The fastest way to
     * draw would be to prepare the colors and locations ahead of
     * time and draw from that array.  This entails a huge overhead in
     * memory (approx 32x), but acheives the fastest quad rate (350 k/sec).
     *
     * Next best is to construct the color and location information
     * on the fly, which is what this routine does.  The maximum quad rate
     * for this technique is about 260 k/sec.
     *
     * The faces array contains one byte per strip, with bits set
     * depending on whether a 1/8 section is opaque.  If all the
     * bits are set, the entire strip is drawn.
     *
     * This routine is passed the correct dimensions, positions,
     * data brick and faces array for the given direction ix.
     */
    register int	k, kstop;
    register unsigned	i, j, l, m, n, ibit, drawing;
    register unsigned	iy, iz, nx, ny, nx8, ny1;
    register float	**c1, **c2, *v1, *v2;
    register unsigned	c1b, c2b, v1b, v2b;
    register float	*posy, *posz, pp;
    register unsigned char *cd;
    register unsigned char *face;
    float		vnorm[3];
    float		trans[3], transy, transz;
    float		zrot[3];
    GLfloat		projMat[16], viewMat[16], saveMat[16], viewInv[16];
    
    /* Save buffers from pass to pass, 
     * hopefully saving speed on allocing memory.
     */
    static float	*ctab = NULL;
    static float	**cbuf = NULL, *vbuf = NULL;
    static unsigned	*qdraw = NULL, *build = NULL;
    
    if (!ctab) {
	ctab = CallocType(float, 1024+4);
	MemCheck(ctab);
	while ((unsigned long)ctab & 0xf)	/* Quad word align */
	    ++ctab;
    }
    
    /* Create color table, ctab
     */
    for (i = j = 0; j < 256; i += 4, ++j) {
	ctab[i  ] = cmap[j][0];
	ctab[i+1] = cmap[j][1];
	ctab[i+2] = cmap[j][2];
	ctab[i+3] = cmap[j][3];
    }

    /* Indecies of the other axes
     */
    iy = (ix + 1) % 3;
    iz = (ix + 2) % 3;

    /* Loop invariants
     */
    nx = dim[ix];
    ny = dim[iy];
    nx8 = (unsigned int)RoundUp(nx, 8) / 8;
    ny1 = ny - 1;
    
    posy = pos[iy];
    posz = pos[iz];
    
    vnorm[ix] = 0;
    vnorm[iy] = 0;
    vnorm[iz] = direction;
    
    /* Fool with view matrix for max value rendering
     */
    glPushMatrix();
    
    glMatrixMode(GL_PROJECTION);
    glGetFloatv(GL_PROJECTION_MATRIX, projMat);
    mcopy(projMat, saveMat);
    projMat[10] = -0.5;
    projMat[14] = 0.2;
    glLoadMatrixf(projMat);

    glMatrixMode(GL_MODELVIEW);
    glGetFloatv(GL_MODELVIEW_MATRIX, viewMat);
    minvert(viewMat, viewInv);
    trans[0] = -viewInv[12];
    trans[1] = -viewInv[13];
    trans[2] = -viewInv[14];
    transy = trans[iy];
    transz = trans[iz];
    zrot[0] = viewMat[2];
    zrot[1] = viewMat[6];
    zrot[2] = viewMat[10];
    viewMat[3] = 0;
    viewMat[7] = 0;
    viewMat[11] = 0;
    viewMat[15] = 1;
    viewMat[12] = 0;
    viewMat[13] = 0;
    viewMat[14] = 0;
    glLoadMatrixf(viewMat);

    /* Allocate buffers for drawing.
     * vbuf, and cbuf are filled on the fly for each quad strip.
     * They are double length to hold both sides of the quad strip.
     * qdraw and build control drawing.
     */
    vbuf = ReallocType(vbuf, float, 8*nx + 4);	MemCheck(vbuf);
    cbuf = ReallocType(cbuf, float *, 2*nx);	MemCheck(cbuf);
    qdraw = ReallocType(qdraw, unsigned, ny);	MemCheck(qdraw);
    build = ReallocType(build, unsigned, ny);	MemCheck(build);

    /* Set buffer indecies.  Align vbuf to quad boundary.
     */
    c1b = 0;
    c2b = nx;

    v1b = 0;
    while ((unsigned long)(vbuf + v1b) & 0xf)
	++v1b;
    v2b = v1b + 4*nx;

    /* The X postions are always the same, so declare them here.
     * v1 points to the lower edge, v2 the upper edge.
     */
    v1 = vbuf + v1b;
    v2 = vbuf + v2b;
    for (i = 0, l = ix; i < nx; ++i, l += 4)
	v1[l] = v2[l] = pos[ix][i] + trans[ix];
    
    /* Increment Z in the proper direction.
     */
    if (direction > 0) {
	k = 0;
	kstop = dim[iz];
    } else {
	k = dim[iz] - 1;
	kstop = -1;
    }
    for (; k != kstop; k += direction) {
	
	/* Select the correct plane of voxel data and the
	 * correct strip of face values.
	 */
	cd = cdata + k*nx*ny;
	face = faces + k*ny;

	/* Construct the build and qdraw flags.  A vertex entry needs
	 * to be built if either it or its neighbors are opaque.
	 * A vertex entry is drawn if it or its lower neighbor are opaque.
	 * Drawing begins with the j=1 strip.
	 */
	for (j = 1; j < ny1; ++j) {
	    qdraw[j] = face[j-1] | face[j];
	    build[j] = qdraw[j]  | face[j+1];
	}
	qdraw[0]   = 0;
	build[0]   = face[0] | face[1];
	qdraw[ny1] = face[ny1-1] | face[ny1];
	build[ny1] = qdraw[ny1];

	/* Fill in the Z location for this plane.
	 */
	v1 = vbuf + v1b;
	v2 = vbuf + v2b;
	pp = posz[k] + transz;
	for (i = nx, l = iz; i; --i, l += 4)
	    v1[l] = v2[l] = pp;

	for (j = 0; j < ny; ++j, cd += nx) {

	    /* v1, v2 and c1, c2 swap every strip.
	     */
	    i = c1b; c1b = c2b; c2b = i;
	    i = v1b; v1b = v2b; v2b = i;
	    
	    v1 = vbuf + v1b;
	    v2 = vbuf + v2b;
	    c1 = cbuf + c1b;
	    c2 = cbuf + c2b;
	    pp = posy[j] + transy;
	    
#define ALPHAZ	c2[i][3]*(zrot[0]*v2[n-3]+zrot[1]*v2[n-2]+zrot[2]*v2[n-1])
	    
	    switch (build[j]) {
		
	      case 0:
		/* Nothing to build
		 */
		break;
		
	      case 0xff:
		/* Build the whole strip in one run.
		 */
		for (i = 0, l = iy, n = 3; i < nx;
		     ++i, l += 4, n += 4) {
		    c2[i] = ctab + 4*cd[i];
		    v2[l] = pp;
		    v2[n] = ALPHAZ;
		}
		break;
		
	      default:
		/* Build one eighth at a time.
		 */
		for (i = 0, l = iy, n = 3, ibit = 1;
		     i < nx; ibit <<= 1) {
		    if (ibit & build[j]) {
			if ((m = i + nx8) > nx)
			    m = nx;
			for (; i < m; ++i, l += 4, n += 4) {
			    c2[i] = ctab + 4*cd[i];
			    v2[l] = pp;
			    v2[n] = ALPHAZ;
			}
		    } else {
			i += nx8;
			l += 4*nx8;
			n += 4*nx8;
		    }
		}
		break;
	    }

#undef ALPHAZ

	    switch (qdraw[j]) {

	      case 0:
		/* Nothing to draw.
		 */
		break;

	      case 0xff:
		/* Draw one full strip.
		 */
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		for (i = nx / 4; i; --i) {
		    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
		    glColor3fv(c1[1]); glVertex4fv(v1+ 4); glColor3fv(c2[1]); glVertex4fv(v2+ 4);
		    glColor3fv(c1[2]); glVertex4fv(v1+ 8); glColor3fv(c2[2]); glVertex4fv(v2+ 8);
		    glColor3fv(c1[3]); glVertex4fv(v1+12); glColor3fv(c2[3]); glVertex4fv(v2+12);
		    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
		}
		if (nx & 0x2) {
		    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
		    glColor3fv(c1[1]); glVertex4fv(v1+ 4); glColor3fv(c2[1]); glVertex4fv(v2+ 4);
		    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
		}
		if (nx & 0x1) {
		    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
		}
		glEnd();
		break;

	      default:
		/* Draw a strip one eighth at a time.  Begin and
		 * end the quad strip as needed.
		 */
		drawing = 1;
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		for (i = nx, ibit = 1; i; ibit <<= 1) {
		    l = MIN(nx8, i);
		    if (ibit & qdraw[j]) {
			if (!drawing) {
			    glBegin(GL_QUAD_STRIP);
			    glNormal3fv(vnorm);
			    drawing = 1;
			}
			for (m = l / 4; m; --m) {
			    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
			    glColor3fv(c1[1]); glVertex4fv(v1+ 4); glColor3fv(c2[1]); glVertex4fv(v2+ 4);
			    glColor3fv(c1[2]); glVertex4fv(v1+ 8); glColor3fv(c2[2]); glVertex4fv(v2+ 8);
			    glColor3fv(c1[3]); glVertex4fv(v1+12); glColor3fv(c2[3]); glVertex4fv(v2+12);
			    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
			}
			if (l & 0x2) {
			    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
			    glColor3fv(c1[1]); glVertex4fv(v1+ 4); glColor3fv(c2[1]); glVertex4fv(v2+ 4);
			    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
			}
			if (l & 0x1) {
			    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
			    c1 += 1;    v1 += 4;    c2 += 1;    v2 += 4;
			}
		    } else {
			if (drawing) {
			    glEnd();
			    drawing = 0;
			}
			v1 += l*4;
			v2 += l*4;
			c1 += l;
			c2 += l;
		    }
		    i -= l;
		}
		if (drawing)
		    glEnd();
		break;
	    }
	}
    }
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glLoadMatrixf(saveMat);
    glMatrixMode(GL_MODELVIEW);
}


void DrawStrideMaxVal(int ix, int direction, unsigned dim[3], float *pos[3],
		      unsigned char *cdata, float cmap[256][4],
		      unsigned char *faces, unsigned stride)
{
    /* Time critical.
     * 
     * This is the core routine for drawing voxels.  The fastest way to
     * draw would be to prepare the colors and locations ahead of
     * time and draw from that array.  This entails a huge overhead in
     * memory (approx 32x), but acheives the fastest quad rate (350 k/sec).
     *
     * Next best is to construct the color and location information
     * on the fly, which is what this routine does.  The maximum quad rate
     * for this technique is about 260 k/sec.
     *
     * The faces array contains one byte per strip, with bits set
     * depending on whether a 1/8 section is opaque.  If all the
     * bits are set, the entire strip is drawn.
     *
     * This routine is passed the correct dimensions, positions,
     * data brick and faces array for the given direction ix.
     */
    register int	k, kstop;
    register unsigned	i, j, l, m, n, q, ibit, drawing;
    register unsigned	iy, iz, nx, ny, nx8, ny1;
    register float	**c1, **c2, *v1, *v2;
    register unsigned	c1b, c2b, v1b, v2b;
    register float	*posy, *posz, pp;
    register unsigned char *cd;
    register unsigned char *face;
    float		vnorm[3];
    float		trans[3], transy, transz;
    float		zrot[3];
    GLfloat		projMat[16], viewMat[16], saveMat[16], viewInv[16];
    
    /* Save buffers from pass to pass, 
     * hopefully saving speed on allocing memory.
     */
    static float	*ctab = NULL;
    static float	**cbuf = NULL, *vbuf = NULL;
    static unsigned	*qdraw = NULL, *build = NULL;
    
    if (!ctab) {
	ctab = CallocType(float, 1024+4);
	MemCheck(ctab);
	while ((unsigned long)ctab & 0xf)	/* Quad word align */
	    ++ctab;
    }
    
    /* Create color table, ctab
     */
    for (i = j = 0; j < 256; i += 4, ++j) {
	ctab[i  ] = cmap[j][0];
	ctab[i+1] = cmap[j][1];
	ctab[i+2] = cmap[j][2];
	ctab[i+3] = cmap[j][3];
    }

    /* Indecies of the other axes
     */
    iy = (ix + 1) % 3;
    iz = (ix + 2) % 3;

    /* Loop invariants
     */
    nx = dim[ix];
    ny = dim[iy];
    nx8 = (unsigned int)RoundUp(nx, 8) / 8;
    ny1 = (unsigned int)RoundUp(ny, stride) - stride;
    
    posy = pos[iy];
    posz = pos[iz];
    
    vnorm[ix] = 0;
    vnorm[iy] = 0;
    vnorm[iz] = direction;
    
    /* Fool with view matrix for max value rendering
     */
    glPushMatrix();
    
    glMatrixMode(GL_PROJECTION);
    glGetFloatv(GL_PROJECTION_MATRIX, projMat);
    mcopy(projMat, saveMat);
    projMat[10] = -0.5;
    projMat[14] = 0.2;
    glLoadMatrixf(projMat);

    glMatrixMode(GL_MODELVIEW);
    glGetFloatv(GL_MODELVIEW_MATRIX, viewMat);
    minvert(viewMat, viewInv);
    trans[0] = -viewInv[12];
    trans[1] = -viewInv[13];
    trans[2] = -viewInv[14];
    transy = trans[iy];
    transz = trans[iz];
    zrot[0] = viewMat[2];
    zrot[1] = viewMat[6];
    zrot[2] = viewMat[10];
    viewMat[3] = 0;
    viewMat[7] = 0;
    viewMat[11] = 0;
    viewMat[15] = 1;
    viewMat[12] = 0;
    viewMat[13] = 0;
    viewMat[14] = 0;
    glLoadMatrixf(viewMat);

    /* Allocate buffers for drawing.
     * vbuf, and cbuf are filled on the fly for each quad strip.
     * They are double length to hold both sides of the quad strip.
     * qdraw and build control drawing.
     */
    vbuf = ReallocType(vbuf, float, 8*nx + 4);	MemCheck(vbuf);
    cbuf = ReallocType(cbuf, float *, 2*nx);	MemCheck(cbuf);
    qdraw = ReallocType(qdraw, unsigned, ny);	MemCheck(qdraw);
    build = ReallocType(build, unsigned, ny);	MemCheck(build);

    /* Set buffer indecies.  Align vbuf to quad boundary.
     */
    c1b = 0;
    c2b = nx;

    v1b = 0;
    while ((unsigned long)(vbuf + v1b) & 0xf)
	++v1b;
    v2b = v1b + 4*nx;

    /* The X postions are always the same, so declare them here.
     * v1 points to the lower edge, v2 the upper edge.
     */
    v1 = vbuf + v1b;
    v2 = vbuf + v2b;
    for (i = 0, l = ix; i < nx; i += stride, l += 4)
	v1[l] = v2[l] = pos[ix][i] + trans[ix];
    
    /* Increment Z in the proper direction.  
     */
    if (direction > 0) {
	k = 0;
	kstop = (int)RoundUp(dim[iz], stride);
    } else {
	k = (int)RoundUp(dim[iz], stride) - stride;
	kstop = -stride;
    }
    direction *= stride;
    for (; k != kstop; k += direction) {
	
	/* Select the correct plane of voxel data and the
	 * correct strip of face values.
	 */
	cd = cdata + k*nx*ny;
	face = faces + k*ny;

	/* Construct the build and qdraw flags.  A vertex entry needs
	 * to be built if either it or its neighbors are opaque.
	 * A vertex entry is drawn if it or its lower neighbor are opaque.
	 * Drawing begins with the j=1 strip.
	 */
	for (j = stride; j < ny1; j += stride) {
	    qdraw[j] = face[j-stride] | face[j];
	    build[j] = qdraw[j]  | face[j+stride];
	}
	qdraw[0]   = 0;
	build[0]   = face[0] | face[stride];
	qdraw[ny1] = face[ny1-stride] | face[ny1];
	build[ny1] = qdraw[ny1];

	/* Fill in the Z location for this plane.
	 */
	v1 = vbuf + v1b;
	v2 = vbuf + v2b;
	pp = posz[k];
	for (i = 0, l = iz; i < nx; i += stride, l += 4)
	    v1[l] = v2[l] = pp + transz;

	for (j = 0; j < ny; j += stride, cd += nx*stride) {

	    /* v1, v2 and c1, c2 swap every strip.
	     */
	    i = c1b; c1b = c2b; c2b = i;
	    i = v1b; v1b = v2b; v2b = i;
	    
	    v1 = vbuf + v1b;
	    v2 = vbuf + v2b;
	    c1 = cbuf + c1b;
	    c2 = cbuf + c2b;
	    pp = posy[j] + transy;
	    
#define ALPHAZ	c2[n][3]*(zrot[0]*v2[q-3]+zrot[1]*v2[q-2]+zrot[2]*v2[q-1])

	    switch (build[j]) {
		
	      case 0:
		/* Nothing to build
		 */
		break;
		
	      case 0xff:
		/* Build the whole strip in one run.
		 */
		for (i = 0, l = iy, n = 0, q = 3; i < nx; 
		     i += stride, l += 4, ++n, q += 4) {
		    c2[n] = ctab + 4*cd[i];
		    v2[l] = pp;
		    v2[q] = ALPHAZ;
		}
		break;
		
	      default:
		/* Build one eighth at a time.
		 */
		n = 0;
		l = iy;
		for (i = 0, m = nx8, q = 3, ibit = 1; 
		     i < nx; m += nx8, ibit <<= 1) {
		    if (m > nx)
			m = nx;
		    if (ibit & build[j])
			for (; i < m; i += stride, ++n, l += 4, q += 4) {
			    c2[n] = ctab + 4*cd[i];
			    v2[l] = pp;
			    v2[q] = ALPHAZ;
			}
		    else
			for (; i < m; i += stride, ++n) {
			    l += 4;
			    q += 4;
			}
		}
		break;
	    }

#undef ALPHAZ
	    
	    switch (qdraw[j]) {

	      case 0:
		/* Nothing to draw.
		 */
		break;

	      case 0xff:
		/* Draw one full strip.
		 */
		n = nx / stride;
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		for (i = n / 4; i; --i) {
		    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
		    glColor3fv(c1[1]); glVertex4fv(v1+ 4); glColor3fv(c2[1]); glVertex4fv(v2+ 4);
		    glColor3fv(c1[2]); glVertex4fv(v1+ 8); glColor3fv(c2[2]); glVertex4fv(v2+ 8);
		    glColor3fv(c1[3]); glVertex4fv(v1+12); glColor3fv(c2[3]); glVertex4fv(v2+12);
		    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
		}
		if (n & 0x2) {
		    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
		    glColor3fv(c1[1]); glVertex4fv(v1+ 4); glColor3fv(c2[1]); glVertex4fv(v2+ 4);
		    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
		}
		if (n & 0x1) {
		    glColor3fv(c1[0]); glVertex4fv(v1   ); glColor3fv(c2[0]); glVertex4fv(v2   );
		}
		glEnd();
		break;

	      default:
		/* Draw a strip one eighth at a time.  Begin and
		 * end the quad strip as needed.
		 */
		drawing = 1;
		glBegin(GL_QUAD_STRIP);
		glNormal3fv(vnorm);
		n = 0;
		l = 0;
		for (i = 0, m = nx8, ibit = 1; i < nx; m += nx8, ibit <<= 1) {
		    if (m > nx)
			m = nx;
		    if (ibit & qdraw[j]) {
			if (!drawing) {
			    drawing = 1;
			    glBegin(GL_QUAD_STRIP);
			    glNormal3fv(vnorm);
			}
			for (; i < m; i += stride, ++n, l += 4) {
			    glColor3fv(c1[n]); glVertex4fv(v1+l); glColor3fv(c2[n]); glVertex4fv(v2+l);
			}
		    } else {
			if (drawing) {
			    drawing = 0;
			    glEnd();
			}
			for (; i < m; i += stride, ++n)
			    l += 4;
		    }
		}
		if (drawing)
		    glEnd();
		break;
	    }
	}
    }
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glLoadMatrixf(saveMat);
    glMatrixMode(GL_MODELVIEW);
}


void DrawVoxFace(int ix, int direction, unsigned dim[3], float *pos[3],
		 unsigned char *cdata, float cmap[256][4])
{
    /* This routine is overkill for just drawing the face of the brick,
     * but it was easy to copy and modify the voxel routine.
     */
    register unsigned	i, j, k, l;
    register unsigned	iy, iz, nx, ny;
    register float	**c1, **c2, *v1, *v2;
    register unsigned	c1b, c2b, v1b, v2b;
    register float	*posy, *posz, pp;
    register unsigned char *cd;
    float		vnorm[3];
    
    /* Save buffers from pass to pass, 
     * hopefully saving speed on allocing memory.
     */
    static float	*ctab = NULL;
    static float	**cbuf = NULL, *vbuf = NULL;
    
    if (!ctab) {
	ctab = CallocType(float, 1024+4);
	MemCheck(ctab);
	while ((unsigned long)ctab & 0xf)	/* Quad word align */
	    ++ctab;
    }
    
    /* Create color table, ctab
     */
    for (i = j = 0; j < 256; i += 4, ++j) {
	ctab[i  ] = cmap[j][0];
	ctab[i+1] = cmap[j][1];
	ctab[i+2] = cmap[j][2];
	ctab[i+3] = cmap[j][3];
    }

    /* Indecies of the other axes
     */
    iy = (ix + 1) % 3;
    iz = (ix + 2) % 3;

    /* Loop invariants
     */
    nx = dim[ix];
    ny = dim[iy];
    
    posy = pos[iy];
    posz = pos[iz];
    
    vnorm[ix] = 0;
    vnorm[iy] = 0;
    vnorm[iz] = direction;
    
    /* Allocate buffers for drawing.
     * vbuf, and cbuf are filled on the fly for each quad strip.
     * They are double length to hold both sides of the quad strip.
     */
    vbuf = ReallocType(vbuf, float, 8*nx + 4);	MemCheck(vbuf);
    cbuf = ReallocType(cbuf, float *, 2*nx);	MemCheck(cbuf);

    /* Set buffer indecies.  Align vbuf to quad boundary.
     */
    c1b = 0;
    c2b = nx;

    v1b = 0;
    while ((unsigned long)(vbuf + v1b) & 0xf)
	++v1b;
    v2b = v1b + 4*nx;

    /* The X postions are always the same, so declare them here.
     * v1 points to the lower edge, v2 the upper edge.
     */
    v1 = vbuf + v1b;
    v2 = vbuf + v2b;
    for (i = 0, l = ix; i < nx; ++i, l += 4)
	v1[l] = v2[l] = pos[ix][i];
    
    /* Set Z to proper plane. */
    if (direction > 0)
	k = dim[iz] - 1;
    else
	k = 0;
    
    /* Select the correct plane of voxel data and the
     * correct strip of face values.
     */
    cd = cdata + k*nx*ny;
    
    /* Fill in the Z location for this plane.
     */
    v1 = vbuf + v1b;
    v2 = vbuf + v2b;
    pp = posz[k];
    for (i = nx, l = iz; i; --i, l += 4)
	v1[l] = v2[l] = pp;
    
    for (j = 0; j < ny; ++j, cd += nx) {
	
	/* v1, v2 and c1, c2 swap every strip.
	 */
	i = c1b; c1b = c2b; c2b = i;
	i = v1b; v1b = v2b; v2b = i;
	
	v1 = vbuf + v1b;
	v2 = vbuf + v2b;
	c1 = cbuf + c1b;
	c2 = cbuf + c2b;
	pp = posy[j];
	
	/* Build the strip.
	 */
	for (i = 0, l = iy; i < nx; ++i, l += 4) {
	    c2[i] = ctab + 4*cd[i];
	    v2[l] = pp;
	}
	if (j == 0)
	    continue;

	/* Draw one full strip.
	 */
	glBegin(GL_QUAD_STRIP);
	glNormal3fv(vnorm);
	for (i = nx / 4; i; --i) {
	    glColor3fv(c1[0]); glVertex3fv(v1   ); glColor3fv(c2[0]); glVertex3fv(v2   );
	    glColor3fv(c1[1]); glVertex3fv(v1+ 4); glColor3fv(c2[1]); glVertex3fv(v2+ 4);
	    glColor3fv(c1[2]); glVertex3fv(v1+ 8); glColor3fv(c2[2]); glVertex3fv(v2+ 8);
	    glColor3fv(c1[3]); glVertex3fv(v1+12); glColor3fv(c2[3]); glVertex3fv(v2+12);
	    c1 += 4;    v1+= 16;    c2 += 4;    v2 += 16;
	}
	if (nx & 0x2) {
	    glColor3fv(c1[0]); glVertex3fv(v1   ); glColor3fv(c2[0]); glVertex3fv(v2   );
	    glColor3fv(c1[1]); glVertex3fv(v1+ 4); glColor3fv(c2[1]); glVertex3fv(v2+ 4);
	    c1 += 2;    v1+= 8;     c2 += 2;    v2 += 8;
	}
	if (nx & 0x1) {
	    glColor3fv(c1[0]); glVertex3fv(v1   ); glColor3fv(c2[0]); glVertex3fv(v2   );
	}
	glEnd();
    }
}


void DrawVoxFaces(unsigned dim[3][3], float *pos[3][3],
		  unsigned char *cdata[3], float cmap[256][4])
{
    register unsigned	i, ix;
    GLfloat		view[16];

    /* The trick is to look down the z column for direction.
     */
    ViewMatrix(view);

    for (i = 0; i < 3; ++i) {
	ix = (i + 1) % 3;
	if (view[i+8] > 0)
	    DrawVoxFace(ix, -1, dim[ix], pos[ix], cdata[ix], cmap);
	else
	    DrawVoxFace(ix,  1, dim[ix], pos[ix], cdata[ix], cmap);
    }
}


void DrawBounds(float dim[3], unsigned asBox)
{
    float	vert[3];

#define V(x,y,z)		vset(vert,x,y,z); glVertex3fv(vert)
#define DoLine(a,b,c,d,e,f)	glBegin(GL_LINE_STRIP); V(a,b,c); V(d,e,f); glEnd();

    /* Draw the primary axes */

    DoLine(0,0,0, dim[0],0,0);
    DoLine(0,0,0, 0,dim[1],0);
    DoLine(0,0,0, 0,0,dim[2]);

    if (!asBox)
	return;

    /* Draw the entire volume bounding box. */

    glBegin(GL_LINE_LOOP);
        V(dim[0],     0,     0);
	V(dim[0],dim[1],     0);
	V(dim[0],dim[1],dim[2]);
	V(dim[0],     0,dim[2]);
    glEnd();
    
    glBegin(GL_LINE_STRIP);
        V(dim[0],dim[1],     0);
	V(     0,dim[1],     0);
	V(     0,dim[1],dim[2]);
	V(dim[0],dim[1],dim[2]);
    glEnd();
    
    glBegin(GL_LINE_STRIP);
       V(dim[0],     0,dim[2]);
       V(     0,     0,dim[2]);
       V(     0,dim[1],dim[2]);
    glEnd();

#undef DoLine
#undef V
}

/*
 * I haven't figured this out but it seems that each context has its own
 * base.  Therefore I must have a separate print function for each window.
 *
 * Wes
 */
void DrawStringV(char *s)
{
    if (!vbase)
       vbase = makeRasterFont();
    glPushAttrib(GL_LIST_BIT);
    glListBase(vbase);
    glCallLists(strlen(s), GL_UNSIGNED_BYTE, (GLubyte *)s);
    glPopAttrib();
}

void DrawStringF(char *s)
{
    if (!fbase)
       fbase = makeRasterFont();
    glPushAttrib(GL_LIST_BIT);
    glListBase(fbase);
    glCallLists(strlen(s), GL_UNSIGNED_BYTE, (GLubyte *)s);
    glPopAttrib();
}

void DrawAnnotatedBox(float dim[3], unsigned lo[3], unsigned hi[3])
{
    float	cv[3];
    float	co[3];
    char	cb[128];
    
#define Extend(x,y,z)	vset(cv,x,y,z); vscale(cv, 1.10); vsub(cv,co,cv)
#define Lower(x,y,z)	vset(cv,x,y,z); vscale(cv, 0.15); vsub(cv,co,cv)
#define Upper(x,y,z)	vset(cv,x,y,z); vscale(cv, 0.95); vsub(cv,co,cv)
#define DoString(s)	glRasterPos3f(cv[0], cv[1], cv[2]); DrawStringV(s)
#define BText(x)	(void) sprintf(cb, "%u", x); DoString(cb)

    vset(co, dim[0], dim[1], dim[2]);
    vscale(co, 0.07);

    Extend(dim[0], 0, 0);
    DoString("x");
    Extend(0,dim[1],0);
    DoString("y");
    Extend(0,0,dim[2]);
    DoString("z");

    Lower(dim[0], 0, 0);
    BText(lo[0]);
    Lower(0,dim[1],0);
    BText(lo[1]);
    Lower(0,0,dim[2]);
    BText(lo[2]);
    
    Upper(dim[0], 0, 0);
    BText(hi[0]);
    Upper(0,dim[1],0);
    BText(hi[1]);
    Upper(0,0,dim[2]);
    BText(hi[2]);

#undef Extend
#undef Lower
#undef Upper
#undef DoString
}


void DrawColorBar(float ctab[256][4], float atab[256])
{
    float	vnorm[3], v1[2], v2[2];
    int		i;

    glPushMatrix();
    glDisable(GL_DEPTH_TEST);
    glTranslatef(0.0,  -0.72,  0.5);
    glScalef(0.0025,  0.06,  1.0);

    /* Draw alpha map line */

    glBegin(GL_LINE_STRIP);
    glColor3ub(255,  255,  255);
    for (i = 0; i < 256; ++i) {
	v1[0] = i - 255;
	v1[1] = MAX(0.02, atab[i]);
	glVertex2fv(v1);
    }
    glEnd();

    glBegin(GL_LINE_STRIP);
    glColor3ub(0, 0, 0);
    
    /* Control the size of the color map gradient */

    vset(vnorm, 0, 0, 1);
    v1[1] = 0;             /* Bottom Y coordinate */
    v2[1] = 1;             /* Top Y coordinate */

    /* Draw color map gradient */

    glBegin(GL_QUAD_STRIP);
    glNormal3fv(vnorm);
    for (i = 0; i < 256; ++i) {
	v1[0] = v2[0] = i;
	glColor3fv(ctab[i]);
	glVertex2fv(v1);
	glVertex2fv(v2);
    }
    glEnd();

    glEnable(GL_DEPTH_TEST);
    glPopMatrix();
}


void DrawAxis(float *axlen)
{
    /* Draw three lines with 'x', 'y' and 'z' at their tips.
     */
    float	zero[3];
    float	tip[3];

    vzero(zero);

    glBegin(GL_LINES);
    vset(tip, axlen[0], 0, 0);
    glVertex3fv(zero);
    glVertex3fv(tip);
    glEnd();
    glRasterPos3fv(tip);
    DrawStringF(" x");

    glBegin(GL_LINES);
    vset(tip, 0, axlen[1], 0);
    glVertex3fv(zero);
    glVertex3fv(tip);
    glEnd();
    glRasterPos3f(tip[0],  tip[1],  tip[2]);
    DrawStringF(" y");

    glBegin(GL_LINES);
    vset(tip, 0, 0, axlen[2]);
    glVertex3fv(zero);
    glVertex3fv(tip);
    glEnd();
    glRasterPos3f(tip[0],  tip[1],  tip[2]);
    DrawStringF(" z");
}


void DrawSetupFinder(void)
{
    static GLfloat mat1d[] = {0.3, 0.3, 0.5, 1.0};
    static GLfloat mat1e[] = {0.0, 0.0, 0.3, 1.0};

    static GLfloat lt1a[] = {0.8, 0.8, 0.8, 1.0};
    static GLfloat lt1s[] = {0.8, 0.8, 0.8, 1.0};
    static GLfloat lt1d[] = {0.8, 0.8, 0.8, 1.0};
    /*static GLfloat lt1p[] = {0.0, 1.0, 1.0, 0.0};*/
    static GLfloat lt1p[] = {-1.0, 1.0, 1.0, 0.0};

/*
 * I think it looks better with only one light -- Wes.
 */
#if 0
    static GLfloat lt2a[] = {0.8, 0.8, 0.8, 1.0};
    static GLfloat lt2s[] = {0.8, 0.8, 0.8, 1.0};
    static GLfloat lt2d[] = {0.8, 0.8, 0.8, 1.0};
    static GLfloat lt2p[] = {-0.5, -1.0, 1.0, 0.0};
#endif

    glNewList(MATERIAL_LIST, GL_COMPILE);
        glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,  mat1d);
        glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, mat1e);
    glEndList();

    glNewList(LIGHT0_LIST, GL_COMPILE);
        glLightfv(GL_LIGHT0, GL_AMBIENT,  lt1a);
        glLightfv(GL_LIGHT0, GL_DIFFUSE,  lt1d);
        glLightfv(GL_LIGHT0, GL_SPECULAR, lt1s);
        glLightfv(GL_LIGHT0, GL_POSITION, lt1p);
    glEndList();

#if 0
    glNewList(LIGHT1_LIST, GL_COMPILE);
        glLightfv(GL_LIGHT1, GL_AMBIENT,  lt2a);
        glLightfv(GL_LIGHT1, GL_DIFFUSE,  lt2d);
        glLightfv(GL_LIGHT1, GL_SPECULAR, lt2s);
        glLightfv(GL_LIGHT1, GL_POSITION, lt2p);
    glEndList();
#endif

    glNewList(LIGHTMOD_LIST, GL_COMPILE);
        glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 0.0);
    glEndList();
}


void DrawFinderScene(int voxdim[3], int stride, int voxoff[3],
		     unsigned filedim[3], float dimScale[3], Trackball *tball)
{
    float		axis[3];
    float		sf;
    GLint		v[3];
    GLfloat		viewmat[16];
    register unsigned	i, j, k, i1, i2;
    
    static float	boxNormal[] = {
	-1.0,  0.0,  0.0,   1.0,  0.0,  0.0,
	 0.0, -1.0,  0.0,   0.0,  1.0,  0.0,
	 0.0,  0.0, -1.0,   0.0,  0.0,  1.0,
    };

    static GLint	boxVert[] = {
	0, 0, 0,  0, 0, 1,  0, 1, 1,  0, 1, 0,
	1, 0, 0,  1, 1, 0,  1, 1, 1,  1, 0, 1,
	0, 0, 0,  1, 0, 0,  1, 0, 1,  0, 0, 1,
	0, 1, 0,  0, 1, 1,  1, 1, 1,  1, 1, 0,
	0, 0, 0,  0, 1, 0,  1, 1, 0,  1, 0, 0,
	0, 0, 1,  1, 0, 1,  1, 1, 1,  0, 1, 1,
    };
    
    glPushMatrix();

    if (bob->blend)
	glEnable(GL_BLEND);
    
    /* Draw the finder cube with lighting */

    glCallList(MATERIAL_LIST);
    /*glEnable(GL_FRONT);*/
    glCallList(LIGHT0_LIST);
    glEnable(GL_LIGHT0);
#if 0
    glCallList(LIGHT1_LIST);
    glEnable(GL_LIGHT1);
#endif
    glCallList(LIGHTMOD_LIST);
    glEnable(GL_LIGHTING);
    
    /*glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE); this is the default*/
    glEnable(GL_COLOR_MATERIAL);

    switch (finderMode) {
      case FinderSync:
	glColor3ub(SYNC_COLOR);
	break;
      case FinderScan:
	glColor3ub(SCAN_COLOR);
	break;
      case FinderDiff:
	glColor3ub(DIFF_COLOR);
	break;
      case FinderRead:
      case FinderInterp:
	glColor3ub(READ_COLOR);
	break;
    }
    
    /* Move to trackball view of scene */
    TrackballSetMatrix(tball);
    sf = 1.0 / MAX(MAX(filedim[0], filedim[1]), filedim[2]);
    glScalef(sf,  sf,  sf);
    glScalef(dimScale[0],  dimScale[1],  dimScale[2]);
    glTranslatef(-0.5*filedim[0],  -0.5*filedim[1],  -0.5*filedim[2]);
    
    /* Augment matrix to expand a unit cube in the right spot */

    glPushMatrix();
    glTranslatef((float) voxoff[0],  (float) voxoff[1],  (float) voxoff[2]);
    glScalef((float) voxdim[0],  (float) voxdim[1],  (float) voxdim[2]);
    glScalef((float) stride,  (float) stride,  (float) stride);

    switch (finderMode) {
      case FinderSync:
      case FinderDiff:
      case FinderInterp:
	finderPlane = 6;
	/* FALLTHRU */
      case FinderScan:
	for (i = 0; i < finderPlane; ++i) {
	    glBegin(GL_POLYGON);
	    glNormal3fv(boxNormal + 3*i);
	    for (j = 0; j < 4; ++j)
		glVertex3iv(boxVert + i*12 + j*3);
	    glEnd();
	}
	break;
      case FinderRead:
	glPushMatrix();
	ViewMatrix(viewmat);
	i = viewmat[10] > 0 ? 4 : 5;
	glTranslatef(0,  0,  (float) finderPlane / (float) voxdim[2]);
	glBegin(GL_POLYGON);
	glNormal3fv(boxNormal + 3*i);
	for (j = 0; j < 4; ++j)
	    glVertex3iv(boxVert + 48 + j*3);
	glEnd();
	glPopMatrix();
        glDisable(GL_LIGHTING);
	glColor3ub(255,  0,  0);
	vset(axis, 1, 1, 1);
	DrawBounds(axis, True);
	break;
    }


    glPopMatrix();	/* shaded cube */
    
    /* Draw lines around file dimensions, no lighting */
    glDisable(GL_LIGHTING);
    glDisable(GL_COLOR_MATERIAL);

    glScalef(0.25*filedim[0],  0.25*filedim[1],  0.25*filedim[2]);

    glColor3ub(0,  64,  0);
    glLineWidth((GLfloat)(1));
    for (i = 0; i < 3; ++i) {
	i1 = (i+1)%3;
	i2 = (i+2)%3;
	for (j = 0; j <=4; ++j) {
	    v[i1] = j;
	    for (k = 0; k <= 4; ++k) {
		v[i2] = k;
		glBegin(GL_LINES);
		v[i] = 0; glVertex3iv(v);
		v[i] = 4; glVertex3iv(v);
		glEnd();
	    }
	}
    }

    vset(axis, 4, 4, 4);
    glColor3ub(255,  128,  0);
    glLineWidth((GLfloat)(2));
    DrawAxis(axis);

    if (bob->blend)
	glDisable(GL_BLEND);

    glPopMatrix();
}


void DrawPoint(float *point, unsigned npoint, int psize, float *pcolor)
{
    unsigned	i, tp;

    glColor3fv(pcolor);
    glPointSize((GLfloat)(psize));
    glDepthFunc(GL_ALWAYS);

    glBegin(GL_POINTS);
    tp = npoint * 3;
    for (i = 0; i < tp; i += 3)
	glVertex3fv(point + i);
    glEnd();

    glDepthFunc(GL_LEQUAL);
}
