/*
 *    Copyright (c) 1992 Minnesota Supercomputer Center, Inc.
 *    Copyright (c) 1992 Army High Performance Computing Research Center
 *        (AHPCRC), University of Minnesota
 *    Copyright (c) 1995-1999 Laboratory for Computational Science and
 *        Engineering (LCSE), University of Minnesota
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file CONTRIBUTORS for a list of contributors.
 *
 *    Original author(s):
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    This file is maintained by:
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    Module name: abar.c
 *
 *    Description:
 *      
 */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "util.h"
#include "xtutil.h"
#include "rgb.h"
#include "cbar.h"
#include "abar.h"

#define	NSTEP	32


void AxisBarResize(AxisBar *abar)
{
    Display	*display;
    Widget	w;
    Dimension	wdim, hdim;
    int		depth;
    Pixmap	map;
    int		i, j;
    int		x, width;

    if (!abar->valid)
	return;

    w = abar->bar[0].w;
    display = XtDisplay(w);

    if (!abar->gc) {
	XGCValues	gcv;
	
	GetValue(XtNbackground, &gcv.background, w);
	abar->gc = XCreateGC(display, XtWindow(w), GCBackground, &gcv);
    }

    GetValue(XtNdepth,  &depth, (Widget) NULL);
    GetValue(XtNwidth,  &wdim,  (Widget) NULL);
    GetValue(XtNheight, &hdim,  w);

    for (i = 0; i < 3; ++i) {
	w = abar->bar[i].w;
	map = XCreatePixmap(display, XtWindow(w), wdim, hdim, depth);

	width = 1 + wdim / NSTEP;
	for (j = 0; j < NSTEP; ++j) {
	    x = (j * wdim) / NSTEP;
	    XSetForeground(display, abar->gc, abar->bar[i].xcolor[j].pixel);
	    XFillRectangle(display, map, abar->gc, x, 0, width, hdim);
	}
	
	XSetWindowBackgroundPixmap(display, XtWindow(w), map);
	XFreePixmap(display, map);
    }
}


void AxisBarSetColors(AxisBar *abar, float cin[3], ColorMode rgbmode, 
		      Colormap cmap)
{
    int		i, j;
    XColor	*xc;
    float	r, g, b;
    float	channel[3];
    float	cscale = 0xffff;

    for (i = 0; i < 3; ++i) {

	for (j = 0; j < 3; ++j)
	    channel[j] = cin[j];
	
	for (j = 0; j < NSTEP; ++j) {
	    channel[i] = ((float) j + 0.5) / (float) NSTEP;
	    switch (rgbmode) {
	      case RGB_MODE:
		r = channel[0];
		g = channel[1];
		b = channel[2];
		break;
	      case HSV_MODE:
		hsv2rgb(channel[0], channel[1], channel[2], &r, &g, &b);
		break;
	      case YUV_MODE:
	      case ALPHA_MODE:
		yuv2rgb(channel[0], channel[1], channel[2], &r, &g, &b);
		break;
	    }
	    
	    xc = abar->bar[i].xcolor + j;
	    xc->red   = r * cscale;
	    xc->green = g * cscale;
	    xc->blue  = b * cscale;
	}
    
	XStoreColors(XtDisplay(abar->bar[i].w), cmap, 
		     abar->bar[i].xcolor, NSTEP);
    }
}


AxisBar *AxisBarNew(Widget *wl)
{
    AxisBar	*abar = MallocType(AxisBar);
    Pixel	holdPix[NSTEP*3];
    Colormap	cmap;
    int		i, j;

    MemCheck(abar);
    ZeroType(abar, AxisBar);
    
    GetValue(XtNcolormap, &cmap, wl[0]);
    
    if (!XAllocColorCells(XtDisplay(wl[0]), cmap, 0, NULL, 0, 
			  holdPix, NSTEP*3))
	return abar;
    
    for (i = 0; i < 3; ++i) {
	abar->bar[i].w = wl[i];
	abar->bar[i].xcolor = CallocType(XColor, NSTEP);
	MemCheck(abar->bar[i].xcolor);
    }

    for (i = 0; i < 3; ++i)
	for (j = 0; j < NSTEP; ++j) {
	    abar->bar[i].xcolor[j].pixel = holdPix[i*NSTEP + j];
	    abar->bar[i].xcolor[j].flags = DoRed | DoGreen | DoBlue;
	}
    
    abar->valid = 1;

    return abar;
}


