/*
 *    Copyright (c) 1992 Minnesota Supercomputer Center, Inc.
 *    Copyright (c) 1992 Army High Performance Computing Research Center
 *        (AHPCRC), University of Minnesota
 *    Copyright (c) 1995-1999 Laboratory for Computational Science and
 *        Engineering (LCSE), University of Minnesota
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file CONTRIBUTORS for a list of contributors.
 *
 *    Original author(s):
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    This file is maintained by:
 *      Ken Chin-Purcell <ken@ahpcrc.umn.edu>
 *
 *    Module name: glutil.h
 *
 *    Description:
 *      Macros, definitions, and data structures and function prototypes
 *      for OpenGL utility routines.
 */

#ifndef	__GLUTIL_H__
#define	__GLUTIL_H__

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "util.h"


#ifdef __cplusplus
extern "C" {
#endif

/* These vector and quaternion macros complement similar
 * routines.
 */

#define vset(a,x,y,z)	(a[0] = x, a[1] = y, a[2] = z)
#define vcopy(a,b)	(b[0] = a[0], b[1] = a[1], b[2] = a[2])
#define vzero(a)	(a[0] = a[1] = a[2] = 0)
#define vadd(a,b,c)	(c[0]=a[0]+b[0], c[1]=a[1]+b[1], c[2]=a[2]+b[2])
#define vsub(a,b,c)	(c[0]=a[0]-b[0], c[1]=a[1]-b[1], c[2]=a[2]-b[2])
#define vdot(a,b)	(a[0]*b[0] + a[1]*b[1] + a[2]*b[2])
#define	vlength(a)	sqrtf(vdot(a,a))
#define vnormal(a)	vscale(a, 1/vlength(a))

#define qset(a,x,y,z,w)	(a[0] = x, a[1] = y, a[2] = z, a[3] = w)
#define qcopy(a,b)	(b[0] = a[0], b[1] = a[1], b[2] = a[2], b[3] = a[3])
#define qzero(a)	(a[0] = a[1] = a[2] = 0, a[3] = 1)
#define qadd(a,b,c)	(vadd(a,b,c), c[3]=a[3]+b[3])

#define	YMAXSTEREO	491
#define	YOFFSET		532

/*extern GLfloat *idmatrix;*/


/* The Trackball package gives that nice 3D rotation interface.
 * A trackball structure is needed for each rotated scene.
 */

typedef struct Trackball {
    float	qrot[4];
    float	qinc[4];
    float	trans[3];
    float	scale[3];
    float	ballsize;
    float	lastx, lasty;
} Trackball;


/* Prototypes */

/* glutil.c */
void	vscale (float *v, float s);
void	vhalf (const float *v1, const float *v2, float *half);
void	vcross (const float *v1, const float *v2, float *cross);
void	vreflect (const float *in, const float *mirror, float *out);
void	vtransform (const float *v, GLfloat *mat, float *vt);
void	vtransform4 (const float *v, GLfloat *mat, float *vt);
void	mcopy (GLfloat *m1, GLfloat *m2);
void	mmult (GLfloat *m1, GLfloat *m2, GLfloat *prod);
void	minvert (GLfloat *mat, GLfloat *result);
void	LinSolve (const float *eqs [], int n, float *x);
void	qnormal (float *q);
void	qmult (const float *q1, const float *q2, float *dest);
void	qmatrix (const float *q, GLfloat *m);
float	ProjectToSphere (float r, float x, float y);
void	CalcRotation (float *q, float newX, float newY, float oldX, float oldY, float ballsize);
float	ScalePoint (long pt, long origin, long size);
void	TrackballReset (Trackball *tball);
Trackball	*TrackballInit (void);
void	TrackballSetMatrix (Trackball *tball);
void	TrackballFlip (Trackball *tball, int axis);
void	TrackballSpin (Trackball *tball);
void	TrackballStopSpinning (Trackball *tball);
int	TrackballSpinning (Trackball *tball);
void	TrackballSetPosition (Trackball *tball, float newx, float newy);
void	TrackballRotate (Trackball *tball, float newx, float newy);
void	TrackballPan (Trackball *tball, float newx, float newy);
void	TrackballZoom (Trackball *tball, float newx, float newy);
void	TrackballCopy (Trackball *src, Trackball *dst);
void	MouseOnTrackball (XEvent *event, unsigned width, unsigned height, Trackball *tball);
void	DumpRLEImage (int ox, int oy, int sx, int sy, char *fname);
void	DumpBinaryImage (int ox, int oy, int sx, int sy, char *fname);
void	ViewMatrix (GLfloat *m);
int	ViewAxis (int *direction);
void	StereoPerspective (int fovy, float aspect, float near, float far, float converge, float eye);

#ifdef __cplusplus
}
#endif

#endif /* __GLUTIL_H__ */
