#include <unistd.h>
#include <netdb.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <errno.h>
#include <fcntl.h>

#include "socket.h"
#include "definition.h"

/*
 * Non-blocking connect
 * nsec is the number of seconds delay before time-out
 */
int
Connect_nonb(int sockfd, const struct sockaddr *saptr, socklen_t salen, int nsec)
{
	int flags, n;
	fd_set rset, wset;
	struct timeval tval;

	flags = fcntl(sockfd, F_GETFL, 0);
	fcntl(sockfd, F_SETFL, flags | O_NONBLOCK);

	if ( (n = connect(sockfd, (struct sockaddr *) saptr, salen)) < 0)
		if (errno != EINPROGRESS) return -1;

	if (n == 0) goto done;

	FD_ZERO(&rset);
	FD_SET(sockfd, &rset);
	wset = rset;
	tval.tv_sec = nsec;
	tval.tv_usec = 0;

	if ( (n = select(sockfd+1,&rset,&wset,NULL, nsec ? &tval : NULL)) == 0)
	{
		errno = ETIMEDOUT;
		return -1;
	}
	
	done:
	fcntl(sockfd, F_SETFL, flags);

	return 0;
}


/*
 * Non-blocking receive
 * nsec is the number of seconds delay before time-out
 */
int
Receive_nonb(int sockfd, char *Buffer, int Size, int nsec)
{
	fd_set rset;
	struct timeval tv;

	tv.tv_sec = nsec;
	tv.tv_usec = 0;
	
	FD_ZERO(&rset);
	FD_SET(sockfd, &rset);

	if (select(sockfd+1,&rset,NULL,NULL,&tv) > 0)
	{
		return recv(sockfd, Buffer, Size, 0);
	}
	else
		return -1;
}


/*
 * To send an arbitrary size of buffer using loop until all buffer is sent
 */
int 
sendall(int sockfd, char *Buffer, int *Length)
{
    int total = 0;
    int bytesleft = *Length;
    int n;

    while(total < *Length) 
    {
        n = send(sockfd, Buffer+total, bytesleft, 0);
        if (n == -1) break;
        total += n;
        bytesleft -= n;
    }

    *Length = total;

    if (n == -1) return -1;
	return 0;
}


/*
 * Sometimes a big packet cannot be received all in one recv. This to make
 * sure a big packet can be received properly.
 * nsec is the number of seconds delay before time-out
 */
int
receiveall(int sockfd, char *Buffer, int nsec)
{
   const int READ_BUFFER_SIZE = 5792;
   
   char ReadBuffer[READ_BUFFER_SIZE];
   int TotalTransferred = 0;
   int NumBytes;
   char *BufferPtr = Buffer;
   
   while ((NumBytes = Receive_nonb(sockfd, ReadBuffer, READ_BUFFER_SIZE, nsec)) > 0)
   {
		TotalTransferred = TotalTransferred + NumBytes;
		memcpy(BufferPtr,ReadBuffer,NumBytes);
		BufferPtr = BufferPtr + NumBytes;
   }
   
   return TotalTransferred;
}


/*
 * Handshake procedure to connect to AnimeNfo server
 */
int
ConnectToServer(char *Buffer, char *Server, int Port)
{
	int    sockfd, numbytes;
	struct hostent *se;
	struct sockaddr_in server_addr;

	/* get server host entity by name */
	if ((se = gethostbyname(Server)) == NULL)
	{ 
		strcpy(Buffer,"Cannot resolve the server name");
		return -1;
	}

	/* create socket */
	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) == -1)
	{
		strcpy(Buffer,"Cannot create socket");
		return -1;
	}

	server_addr.sin_family = AF_INET;
	server_addr.sin_port = htons(Port);
	server_addr.sin_addr = *((struct in_addr *)se->h_addr);
	memset(&(server_addr.sin_zero), '\0', 8);

	/* connect to the server */
	if (Connect_nonb(sockfd, (struct sockaddr *)&server_addr, sizeof(struct sockaddr), 5) == -1)
	{
		close(sockfd);
		strcpy(Buffer,"Cannot connect to the server");
		return -1;
	}

	/* receive server id */
	numbytes = Receive_nonb(sockfd, Buffer, MAXBUFSIZE-1, 5);
	if (numbytes == -1 || numbytes == 0)
	{
		close(sockfd);
		strcpy(Buffer,"Cannot receive from the server");
		return -1;
	}
	Buffer[numbytes] = '\0';

	return sockfd;
}
