/*
 * @(#)Application.cpp 1.00 11 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of Anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

// can only undefine for multithreaded Qt versions
#define SINGLE_THREAD

#include "Application.h"

#include "gadgets/Gadgets.h"
#include "songwindow/SongWindow.h"
#include "dialogues/Panic.h"
#include "dialogues/MiscWindows.h"
#include "dialogues/TSE3Progress.h"
#include "settings/Settings.h"
//#include "gadgets/ARecentFilesAction.h"
#include "misc/KdeMidiScheduler.h"

#include "tse3/TSE3.h"
#include "tse3/plt/Factory.h"
#include "tse3/MidiScheduler.h"
#include "tse3/Transport.h"
#include "tse3/Song.h"
#include "tse3/TSE3MDL.h"
#include "tse3/TSE2MDL.h"
#include "tse3/MidiFile.h"
#include "tse3/Error.h"

#ifndef SINGLE_THREAD
#include "tse3/Mutex.h"
#endif

#include <qtimer.h>
#include <qthread.h>
#include <qdir.h>

#include <kmessagebox.h>
#include <krecentdocument.h>
#include <kstdaction.h>
#include <kaction.h>
#include <kconfig.h>
#include <kcmdlineargs.h>
#include <klibloader.h>

#include <memory>
#include <algorithm>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef VERSION
#define VERSION "0.0.0 (no autoconf)"
#endif

namespace
{
    const std::string anthemName = "Anthem";
    const std::string STATUS     = "Alpha";
    int               MSECS      = 10;
}

#ifndef SINGLE_THREAD

namespace
{

    /**************************************************************************
     * AnthemMutex class
     *************************************************************************/

    class AnthemMutex : public TSE3::Impl::MutexImpl
    {
        public:
            virtual void lock();
            virtual void unlock();
            virtual bool locked();
        private:
            QMutex mutex;
    };


    AnthemMutex::AnthemMutex()
    : mutex(true)
    {
    }


    void AnthemMutex::lock()
    {
        mutex.lock();
    }


    void AnthemMutex::unlock()
    {
        mutex.unlock();
    }


    bool AnthemMutex::locked()
    {
        return mutex.locked();
    }


    /**************************************************************************
     * TransportPollThread class
     *************************************************************************/

    class TransportPollThread : public QThread
    {
        public:
        private:
    };

}

#else
class QThread;
#endif


/******************************************************************************
 * ApplicationImpl class
 *****************************************************************************/

class ApplicationImpl
{
    public:

        ApplicationImpl();

        QTimer             *timer;
        QThread            *thread;
        VuWindow           *vuWindow;
        KeyboardWindow     *keyboardWindow;
        PanicWindow        *panicWindow;
        SettingsWindow     *settingsWindow;
        //ARecentFilesAction *aRecentFilesAction;
        ApplicationPlugins *plugins;

        typedef std::map<TSE3::Song *, SongWindow *> map_type;
        map_type swmap;
};


ApplicationImpl::ApplicationImpl()
: vuWindow(0), keyboardWindow(0), panicWindow(0), settingsWindow(0)
{
}


/******************************************************************************
 * Application class
 *****************************************************************************/

Application *Application::app = 0;

void Application::printStartupInfo()
{
    // Hideous info box
    std::cout << "\n"
              << "+---------------------------------------------+\n"
              << "|          /\\   _  _|_ |_   _   _ _           |\n"
              << "|         /~~\\ | )  |  | ) (/_ | ) )          |\n"
              << "|                                             |\n"
              << "|    O p e n   S o u r c e   E d i t i o n    |\n"
              << "+---------------------------------------------+\n"
              << "\n"
              << "Anthem info\n"
              << "   Version:        " << VERSION << std::endl
              << "   Status:         " << STATUS << std::endl
              << "   Author:         Pete Goodliffe\n"
              << "   Publisher:      A Trax Software product\n"
              << "   Licence:        GPL (see <www.gnu.org>)\n"
              << "   Homepage:       http://anthem.sourceforge.net\n"
              << "\n"
              << "Build info\n"
              << "   Build data:     " << __DATE__ << std::endl
              << "   Build time:     " << __TIME__ << std::endl
              << "   Built for KDE:  " << KDE_VERSION_STRING << std::endl
              << "\n"
              << "TSE3 info\n"
              << "   Anthem incorporates TSE3 technology\n"
              << "   See http://TSE3.sourceforge.net for details\n"
              << "   TSE3 version:   " << TSE3::TSE3_Version() << std::endl
              << "   TSE3 copyright: " << TSE3::TSE3_Copyright() << std::endl
              << "\n";
}


TSE3::MidiSchedulerFactory *factory()
{
    KCmdLineArgs *parsedArgs = KCmdLineArgs::parsedArgs();

    if (parsedArgs->isSet("stream"))
    {
        static KdeMidiSchedulerFactory kde;
        return &kde;
    }
    else
    {
        static TSE3::MidiSchedulerFactory u;
        if (parsedArgs->isSet("alsa"))
        {
            TSE3::Plt::UnixMidiSchedulerFactory::setPreferredPlatform
                (TSE3::Plt::UnixMidiSchedulerFactory::UnixPlatform_Alsa);
        }
        else if (parsedArgs->isSet("oss"))
        {
            TSE3::Plt::UnixMidiSchedulerFactory::setPreferredPlatform
                (TSE3::Plt::UnixMidiSchedulerFactory::UnixPlatform_OSS);
        }
        return &u;
    }
}


Application::Application()
: KApplication(),
  TSE3::App::Application(anthemName, VERSION,
                         factory(), choicesFileName()),
  pimpl(new ApplicationImpl)
{
    app            = this;
    pimpl->plugins = new ApplicationPlugins;

#ifdef SINGLE_THREAD
    pimpl->timer = new QTimer(this);
    pimpl->timer->start(MSECS, false);
    connect(pimpl->timer, SIGNAL(timeout()), SLOT(slotTimeout()));
#else
    // create a new poll thread and use that
#endif

    setSaveChoicesOnDestroy(true);

    // We have one global recent files action that is never displayed that
    // remembers the global recent files.
    //XXX pimpl->aRecentFilesAction = new ARecentFilesAction(this);
    //XXX pimpl->aRecentFilesAction->loadEntries(config());

    // Kick the plugins
    pimpl->plugins->loadChoices(config());
    pimpl->plugins->loadAllPlugins();
}


Application::~Application()
{
    //XXX pimpl->aRecentFilesAction->saveEntries(config());
    //XXX pimpl->plugins->saveChoices(config());

    delete pimpl->vuWindow;
    delete pimpl->keyboardWindow;
    delete pimpl->panicWindow;
    delete pimpl->settingsWindow;
    delete pimpl;
}


std::string Application::choicesFileName()
{
    char *home = getenv("HOME");
    std::string name(home);
    name.append("/.tse3");
    return name;
}


void Application::slotTimeout()
{
#ifdef SINGLE_THREAD
    transport()->poll();
#endif
}


VuWindow *Application::vuWindow()
{
    if (!pimpl->vuWindow)
    {
        pimpl->vuWindow = new VuWindow();
    }
    return pimpl->vuWindow;
}


KeyboardWindow *Application::keyboardWindow()
{
    if (!pimpl->keyboardWindow)
    {
        pimpl->keyboardWindow = new KeyboardWindow();
    }
    return pimpl->keyboardWindow;
}


PanicWindow *Application::panicWindow()
{
    if (!pimpl->panicWindow)
    {
        pimpl->panicWindow = new PanicWindow();
    }
    return pimpl->panicWindow;
}


SettingsWindow *Application::settingsWindow()
{
    if (!pimpl->settingsWindow)
    {
        pimpl->settingsWindow = new SettingsWindow();
    }
    return pimpl->settingsWindow;
}


/* XXX
ARecentFilesAction *Application::aRecentFilesAction()
{
    return pimpl->aRecentFilesAction;
}
*/


void Application::load(const QString &filename, const KURL &url, SongWindow *sw)
{
    TSE3::Song *song = 0;
    if (filename.isEmpty() || url.isEmpty())
    {
        // Create a blank Song
        song = new TSE3::Song();
    }
    else
    {
        TSE3ProgressDialog progress("Loading song", QString::null);

        // Attempt to load the file
        TSE3::FileRecogniser fr(filename.latin1());
        if (fr.type() == TSE3::FileRecogniser::Type_Error)
        {
            QString s;
            s.sprintf("Anthem could not open the file '%s'",
                      filename.latin1());
            KMessageBox::sorry(0, s);
            return;
        }
        else if (fr.type() == TSE3::FileRecogniser::Type_Unknown)
        {
            QString s;
            s.sprintf("Anthem does not recognise the file type of '%s'",
                      filename.latin1());
            KMessageBox::sorry(0, s);
            return;
        }
        else
        {
            try
            {
                song = fr.load(&progress);
            }
            catch (const TSE3::MidiFileImportError &e)
            {
                QString s;
                s.sprintf("Midi file import was unsuccessful: %s",
                          (*e).c_str());
                KMessageBox::sorry(0, s);
                return;
            }
            catch (const TSE3::Error &e)
            {
                std::cerr << "TSE3 exception: Failed to load "
                          << filename.latin1() << " because: "
                          << TSE3::errString(e.reason()) << std::endl;
                QString s("The file couldn't be loaded.\n");
                s.append(TSE3::errString(e.reason()));
                KMessageBox::sorry(0, s);
                return;
            }
        }
        KRecentDocument::add(url.path(), true);
    }
    if (song)
    {
        addSong(song);
        if (!sw)
        {
            SongWindow *sw = new SongWindow(song, filename, url);
            sw->show();
        }
        else
        {
            sw->changeSong(song, filename, url);
        }
    }
    else
    {
        KMessageBox::sorry(0, "The file couldn't be loaded.");
    }
}


SongWindow *Application::songWindow(TSE3::Song *song)
{
    ApplicationImpl::map_type::iterator i = pimpl->swmap.find(song);
    if (i != pimpl->swmap.end())
    {
        return i->second;
    }
    else
    {
        return 0;
    }
}


ApplicationPlugins *Application::plugins() const
{
    return pimpl->plugins;
}


void Application::registerSongWindow(SongWindow *sw, TSE3::Song *song)
{
    deregisterSongWindow(sw);
    pimpl->swmap[song] = sw;
}


void Application::deregisterSongWindow(SongWindow *sw)
{
    ApplicationImpl::map_type::iterator i = pimpl->swmap.begin();
    while (i != pimpl->swmap.end() && i->second != sw) i++;
    if (i != pimpl->swmap.end())
    {
        pimpl->swmap.erase(i);
    }
}


/******************************************************************************
 * Plugins
 *****************************************************************************/

class ApplicationPluginsImpl
{
    public:
        ApplicationPlugins::plugin_dir_list    directories;
        ApplicationPlugins::plugin_loaded_list loaded;
};


ApplicationPlugins::ApplicationPlugins()
: pimpl(new ApplicationPluginsImpl)
{
    //char *home = getenv("HOME");
    //std::string dir(home);
    //dir.append("/.anthem-plugins");
    //pimpl->directories.push_back(dir);
    //XXX
}


ApplicationPlugins::~ApplicationPlugins()
{
}


const ApplicationPlugins::plugin_dir_list &
    ApplicationPlugins::directories() const
{
    return pimpl->directories;
}


const ApplicationPlugins::plugin_loaded_list &
    ApplicationPlugins::loaded() const
{
    return pimpl->loaded;
}


void ApplicationPlugins::saveChoices(KConfig *config)
{
    QStrList list;
    ApplicationPlugins::plugin_dir_list::iterator i
        = pimpl->directories.begin();
    while (i != pimpl->directories.end())
    {
        list.append(i->c_str());
        ++i;
    }
    config->setGroup("Plugins");
    config->writeEntry("Directories", list);
}


void ApplicationPlugins::loadChoices(KConfig *config)
{
    QStrList list;
    config->setGroup("Plugins");
    config->readListEntry("Directories", list);
    pimpl->directories.clear();
    for (unsigned int n = 0; n < list.count(); n++)
    {
        addDirectory(list.at(n), false);
    }
}


void ApplicationPlugins::loadAllPlugins()
{
    std::cout << "Loading plugins\n";
    ApplicationPlugins::plugin_dir_list::iterator i
        = pimpl->directories.begin();
    while (i != pimpl->directories.end())
    {
        std::cout << "  Scanning dir: " << (*i).c_str() << std::endl;
        loadPluginDir(*i);
        ++i;
    }
    std::cout << "  Loading finished\n";
}


void ApplicationPlugins::addDirectory(const std::string &dir, bool doload)
{
    if (std::find(pimpl->directories.begin(),
                  pimpl->directories.end(),
                  dir) == pimpl->directories.end())
    {
        pimpl->directories.push_back(dir);
        if (doload)
        {
            loadPluginDir(dir);
        }
    }
}


void ApplicationPlugins::removeDirectory(const std::string &dir)
{
    plugin_dir_list::iterator i
        = std::find(pimpl->directories.begin(), pimpl->directories.end(), dir);
    if (i != pimpl->directories.end())
    {
        pimpl->directories.erase(i);
    }
}


void ApplicationPlugins::loadPluginDir(const std::string &file)
{
    QDir dir(file.c_str());
    dir.setFilter( QDir::Files | QDir::Hidden | QDir::NoSymLinks );
    dir.setSorting( QDir::Size | QDir::Reversed );

    const QFileInfoList *list = dir.entryInfoList();
    QFileInfoListIterator it( *list );
    QFileInfo *fi;

    while ((fi=it.current()))
    {
        loadPlugin(fi->fileName().data());
        ++it;
    }
}


bool ApplicationPlugins::loadPlugin(const std::string &file)
{
    std::cout << "    Loading plugin: " << file << "\n";
    KLibFactory *factory = KLibLoader::self()->factory(file.c_str());
    if (!factory)
    {
        std::cout << "      Error loading plugin " << file << "\n"
                  << "      This plugin has not registered with Anthem\n";
        return false;
    }
    return true;
}


bool ApplicationPlugins::registerPlugin(const PluginInfo &info)
{
    std::cout << "      Plugin has registered with Anthem:\n"
              << "        Name:     " << info.name     << "\n"
              << "        Author    " << info.author   << "\n"
              << "        Version:  " << info.version  << "\n"
              << "        Date:     " << info.date     << "\n"
              << "        Requires: " << info.requires << "\n";

     bool accept = false;
     if (info.requires == std::string(VERSION))
     {
         accept = true;
     }

     if (accept)
     {
         pimpl->loaded.push_back(info);
     }

     return accept;
}

