/*
 * @(#)Application.h 1.00 11 June 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of Anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_APPLICATION_H
#define ANTHEM_APPLICATION_H

#include <kapp.h>
#include "tse3/app/Application.h"

/******************************************************************************
 * forward declataions
 *****************************************************************************/
class KLibFactory;
class KConfig;

/******************************************************************************
 * Anthem forward declataions
 *****************************************************************************/
class VuWindow;
class KeyboardWindow;
class PanicWindow;
class SongWindow;
class SettingsWindow;
class ARecentFilesAction;

class ApplicationPlugins;

/**
 * The Application object derived from the standard @ref KApplication (which
 * in turn derived from @ref QApplication). It also derived from the
 * @ref TSE3::App::Application class to manage TSE3 components.
 *
 * There can only be one Application object in existance at any given
 * time, it can be retrieved by the static @ref application() method.
 *
 * @short   Anthem application object
 * @author  Pete Goodliffe
 * @version 1.0
 */
class Application : public KApplication,
                    public TSE3::App::Application
{
        Q_OBJECT

    public:

        /**
         * Prints a whole load of 'intersting' startup info about the program
         * to cout.
         */
        static void printStartupInfo();

        /**
         * Creates a Phrase list window containing an entry for each @ref Phrase
         * in the given @ref Song. It will update with the @ref Song.
         *
         * The PhraseListWindow does not automatically open, you must show()
         * it yourself.
         */
        Application();

        virtual ~Application();

        static Application *application() { return app; }

        /**
         * Returns a @ref VuWindow for the application's @ref Transport object.
         *
         * This window is not created until you first call this method.
         */
        VuWindow *vuWindow();

        /**
         * Returns a @ref KeyboardWindow.
         *
         * This window is not created until you first call this method.
         */
        KeyboardWindow *keyboardWindow();

        /**
         * Returns a @ref PanicWindow.
         *
         * This window is not created until you first call this method.
         */
        PanicWindow *panicWindow();

        /**
         * Returns a @ref SetingsWindow.
         *
         * This window is not created until you first call this method.
         */
        SettingsWindow *settingsWindow();

        /**
         * Returns the @ref ARecentFilesAction usedin this application.
         */
        ARecentFilesAction *aRecentFilesAction();

        /**
         * Call this to load a new file, creating a new @ref SongWindow.
         *
         * If filename is empty, a @ref SongWindow is opened with a blank
         * @ref Song.
         *
         * @param filename Filename of new file (local on disk, or downloaded
         *                 if url is remote)
         * @param url      URL of file, may be "file:/" in which case it is
         *                 equivalent to filename, or a remote URL in which
         *                 case the URL is the real remote filename, and the
         *                 @p filename is the local downloaded copy
         * @param sw       If zero, then opens a new @ref SongWindow otherwise
         *                 changes the song in the specified @ref SongWindow
         */
        void load(const QString &filename, const KURL &fileurl,
                  SongWindow *sw = 0);

        /**
         * Returns the @ref SongWindow for the given @ref TSE3::Song, or
         * zero if there is none.
         */
        SongWindow *songWindow(TSE3::Song *song);

        /**
         * Returns the plugins object, which manages any application plugins.
         */
        ApplicationPlugins *plugins() const;

        /**
         * The @ref SongWindow is a friend so that it can call the
         * @ref registerSongWindow and @ref deregisterSongWindow methods
         * only.
         */
        friend class SongWindow;

    private slots:

        void slotTimeout();

    private:

        /**
         * Called by the @ref SongWindow ctor. This updates the internal
         * map of @ref TSE3::Song <-> SongWindow association so that the
         * @ref songWindow() method works. If the @ref SongWindow is already
         * registered, it's accompanying @ref TSE3::Song pointer is updated.
         *
         * @see deregisterSongWindow
         */
        void registerSongWindow(SongWindow *sw, TSE3::Song *);

        /**
         * Called by the @ref SongWindow dtor. This updates the internal
         * map of @ref TSE3::Song <-> SongWindow associations.
         *
         * @see registerSongWindow
         */
        void deregisterSongWindow(SongWindow *sw);

        static std::string choicesFileName();

        static Application *app;

        class ApplicationImpl *pimpl;
};

/**
 * Looks after the application plugins.
 *
 * This is a singleton class, only accessed via the Application object.
 *
 * You can register directories to be scanned for plugins via this interface.
 * For instructions on how to write Anthem plugins, see the documentation for
 * @ref registerPlugin.
 *
 * @short   Anthem plugin manager
 * @author  Pete Goodliffe
 * @version 1.0
 * @see     Application
 */
class ApplicationPlugins
{
    public:

        /**
         * A data structure describing a plugin. Used by @ref registerPlugin.
         *
         * The information contained is:
         * @li name       - The plugin name
         * @li author     - The plugin author
         * @li version    - The plugin version
         * @li date       - The plugin date
         * @li requires   - The minimum version of Anthem you require
         * @li libfactory - The plugin's KLibFactory
         */
        struct PluginInfo
        {
            std::string  name;
            std::string  author;
            std::string  version;
            std::string  date;
            std::string  requires;
            KLibFactory *libfactory;
        };

        typedef std::vector<std::string> plugin_dir_list;
        typedef std::vector<PluginInfo>  plugin_loaded_list;

        /**
         * Saves the current directories list to the specified KConfig
         * in a group "Plugins".
         */
        void saveChoices(KConfig *config);

        /**
         * Loads the current directories list from the specified KConfig.
         */
        void loadChoices(KConfig *config);

        /**
         * Loads all plugins in all added directories now. Any plugins
         * that have already been loaded will not be loaded again.
         */
        void loadAllPlugins();

        /**
         * Returns the current list of plugin directories
         */
        const plugin_dir_list &directories() const;

        /**
         * Returns the current list of loaded plugins
         */
        const plugin_loaded_list &loaded() const;

        /**
         * Adds a directory to the list of plugin directories, and
         * loads all the plugins in it.
         *
         * @param dir    Directory to add
         * @param doload Whether or not to load that dir now
         */
        void addDirectory(const std::string &dir, bool doload);

        /**
         * Removes a directory from the plugin list. However, this can't
         * unload any plugins. You'll have to wait until the next time
         * Anthem is run.
         */
        void removeDirectory(const std::string &dir);

        /**
         * When you write an Anthem plugin you must follow a number of
         * steps:
         *
         * @li Honour the KLibFactory interface (you need to do this to
         *     write KDE in a dynaamically loadable library, anyway)
         * @li In your extern "C" init_XXX function, call this method.
         *     It registers the plugin with Anthem so that it is displayed
         *     in the menu system.
         *     Based on the information you supply to this API, you will be
         *     told whether or not to proceed with the plugin registration.
         * @li Register with the appropriate Anthem plugin API (for example,
         *     register yourself as a PhraseEditor).
         *
         * @param name     The plugin name
         *
         * @return Whether or not your plugin should continue to initialise
         */
        bool registerPlugin(const PluginInfo &info);

        friend class Application;

    private:

        ApplicationPlugins();
        ~ApplicationPlugins();

        /**
         * Loads all the plugins in the specified directory.
         */
        void loadPluginDir(const std::string &file);

        /**
         * Loads the specific plugin
         *
         * @return Whether load was sucessful
         */
        bool loadPlugin(const std::string &file);

        class ApplicationPluginsImpl *pimpl;
};

#endif
