/*
 * @(#)Part.cpp 1.00 5 July 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "dialogues/Part.h"

#include "settings/Part.h"
#include "settings/MidiFilter.h"
#include "settings/MidiParams.h"
#include "settings/DisplayParams.h"

#include <qlayout.h>

#include "tse3/Part.h"
#include "tse3/cmd/CommandHistory.h"
#include "tse3/cmd/Part.h"

/******************************************************************************
 * PartWindow class
 *****************************************************************************/

PartWindow::PartWindow(QWidget *parent, TSE3::Part *part, TSE3::Song *song,
                       TSE3::Cmd::CommandHistory *history)
: KDialogBase(Tabbed, "Part Settings", Cancel|Ok|Apply, Ok, parent,
              "PartWindow", false, true),
  part(part), history(history)
{
    attachTo(part);

    /**************************************************************************
     * Set up tabs
     *************************************************************************/

    partTab   = addPage(QString("General"));
    filterTab = addPage(QString("Filters"));
    paramsTab = addPage(QString("MIDI Params"));
    displayParamsTab = addPage(QString("Display Params"));

    /**************************************************************************
     * Part tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(partTab, spacingHint());
        pwidget = new PartWidget(part, song, partTab);
        layout->add(pwidget);
    }

    /**************************************************************************
     * Filter tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(filterTab, spacingHint());
        mfwidget = new MidiFilterWidget(part->filter(), filterTab);
        layout->add(mfwidget);
    }

    /**************************************************************************
     * Params tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(paramsTab, spacingHint());
        mpwidget = new MidiParamsWidget(part->params(), paramsTab);
        layout->add(mpwidget);
    }

    /**************************************************************************
     * DisplayParams tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(displayParamsTab, spacingHint());
        dpwidget
            = new DisplayParamsWidget(part->displayParams(), true,
                                      displayParamsTab);
        layout->add(dpwidget);
    }

    /**************************************************************************
     * All done
     *************************************************************************/

    show();
}


void PartWindow::show()
{
    updateWidgets();
    KDialogBase::show();
}


PartWindow::~PartWindow()
{
}


void PartWindow::setValues()
{
    /* The old way was to do this:
     *
     * pwidget->accept();
     * mfwidget->accept();
     * mpwidget->accept();
     * dpwidget->accept();
     *
     * but now we do something slightly more complex with commands
     * that allows us to have undo/redo.
     */

    TSE3::MidiFilter    mf;
    TSE3::MidiParams    mp;
    TSE3::DisplayParams dp;

    mfwidget->setValues(mf);
    mpwidget->setValues(mp);
    dpwidget->setValues(dp);

    TSE3::Cmd::Part_SetInfo *command
        = new TSE3::Cmd::Part_SetInfo(part,
                                      pwidget->phrase(),
                                      pwidget->repeat(),
                                      mf, mp, dp);
    command->execute();
    history->add(command);
}


void PartWindow::accept()
{
    setValues();
    done(Accepted);
}


void PartWindow::done(int reason)
{
    delayedDestruct();
}


void PartWindow::slotApply()
{
    setValues();
    updateWidgets();
}


void PartWindow::updateWidgets()
{
    pwidget->updateWidgets();
    mfwidget->updateWidgets();
    mpwidget->updateWidgets();
    dpwidget->updateWidgets();
}


void PartWindow::Notifier_Deleted(TSE3::Part *part)
{
    delete this;
}


void PartWindow::Part_Reparented(TSE3::Part *part)
{
    delete this;
}

