/*
 * @(#)Track.cpp 1.00 22 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#include "dialogues/Track.h"

/*
 * <<< EUCH - works around Qt 2.x 'feature' >>>
 */
#define QT_ALTERNATE_QTSMANIP

#include "settings/Track.h"
#include "settings/MidiFilter.h"
#include "settings/MidiParams.h"
#include "settings/DisplayParams.h"
#include "Application.h"

#include <qlayout.h>
#include <qtextstream.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>

#include "tse3/Track.h"
#include "tse3/cmd/CommandHistory.h"
#include "tse3/cmd/Track.h"
#include "tse3/Song.h"

/******************************************************************************
 * TrackWindow class
 *****************************************************************************/

TrackWindow::TrackWindow(QWidget *parent, TSE3::Track *track,
                         TSE3::Cmd::CommandHistory *history,
                         Tabs firstTab)
: KDialogBase(Tabbed, "Track Settings", Cancel|Ok|Apply, Ok, parent,
              "TrackWindow", false, true),
  track(track), history(history)
{
    attachTo(track);

    /**************************************************************************
     * Set up tabs
     *************************************************************************/

    trackTab         = addPage(QString("General"));
    filterTab        = addPage(QString("Filters"));
    paramsTab        = addPage(QString("MIDI Params"));
    displayParamsTab = addPage(QString("Display Params"));

    showPage(firstTab);

    /**************************************************************************
     * Track tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(trackTab, spacingHint());
        twidget = new TrackWidget(track, trackTab);
        layout->add(twidget);
    }

    /**************************************************************************
     * Filter tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(filterTab, spacingHint());
        mfwidget = new MidiFilterWidget(track->filter(), filterTab);
        layout->add(mfwidget);
    }

    /**************************************************************************
     * Params tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(paramsTab, spacingHint());
        mpwidget = new MidiParamsWidget(track->params(), paramsTab);
        layout->add(mpwidget);
    }

    /**************************************************************************
     * DisplayParams tab
     *************************************************************************/

    {
        QVBoxLayout *layout = new QVBoxLayout(displayParamsTab, spacingHint());
        dpwidget
            = new DisplayParamsWidget(track->displayParams(), false,
                                      displayParamsTab);
        layout->add(dpwidget);
    }

    /**************************************************************************
     * All done
     *************************************************************************/

    show();
}


TrackWindow::~TrackWindow()
{
}


void TrackWindow::show()
{
    updateWidgets();
    KDialogBase::show();
}


void TrackWindow::setValues()
{
    TSE3::MidiFilter    mf;
    TSE3::MidiParams    mp;
    TSE3::DisplayParams dp;

    mfwidget->setValues(mf);
    mpwidget->setValues(mp);
    dpwidget->setValues(dp);

    TSE3::Cmd::Track_SetInfo *command
        = new TSE3::Cmd::Track_SetInfo(track,
                                       twidget->title().latin1(),
                                       mf, mp, dp);
    command->execute();
    history->add(command);
}


void TrackWindow::accept()
{
    setValues();
    done(Accepted);
}


void TrackWindow::done(int reason)
{
    delayedDestruct();
}


void TrackWindow::slotApply()
{
    setValues();
    updateWidgets();
}


void TrackWindow::updateWidgets()
{
    twidget->updateWidgets();
    mfwidget->updateWidgets();
    mpwidget->updateWidgets();
    dpwidget->updateWidgets();
}


void TrackWindow::Notifier_Deleted(TSE3::Track *track)
{
    delete this;
}


void TrackWindow::Track_Reparented(TSE3::Track *track)
{
    delete this;
}


QString paramValToString(int p)
{
    if (p == -2)
    {
        return "None";
    }
    else if (p == -1)
    {
        return "Off";
    }
    else
    {
        QString s;
        s.sprintf("%d", p);
        return s;
    }
}


QString TrackWindow::paramString(TSE3::Track *track)
{
    TSE3::MidiParams *params = track->params();

    QString str;
    QTextOStream os(&str);

    if (params->bankLSB() >= 0 || params->bankLSB() == -2
        || params->bankMSB() >= 0 || params->bankMSB() == -2
        || params->program() >= 0 || params->program() == -2)
    {
        // Look up name
        os << " Prog:"    << paramValToString(params->program())
           << " BankLSB:" << paramValToString(params->bankLSB())
           << " BankMSB:" << paramValToString(params->bankMSB());
    }

    if (params->pan() >= 0 || params->pan() == -2)
    {
        os << " Pan:" << paramValToString(params->pan());
    }

    if (params->reverb() >= 0 || params->reverb() == -2)
    {
        os << " Rev:" << paramValToString(params->reverb());
    }

    if (params->chorus() >= 0 || params->chorus() == -2)
    {
        os << " Ch:" << paramValToString(params->chorus());
    }

    if (params->volume() >= 0 || params->volume() == -2)
    {
        os << " Vol:" << paramValToString(params->volume());
    }

    if (!str.isEmpty())
    {
        // Remove preceding space
        str.remove(0, 1);
    }

    return str;
}


/******************************************************************************
 * TrackSortWindow class
 *****************************************************************************/

TrackSortWindow::TrackSortWindow(QWidget *parent, TSE3::Song *song,
                                 TSE3::App::TrackSelection *ts)
: KDialogBase(Plain, "Sort Tracks", Cancel|Ok|Apply, Ok, parent,
              "TrackSortWindow", false, true),
  song(song), selection(ts)
{
    QVBoxLayout *layout = new QVBoxLayout(plainPage(), spacingHint());

    // Sort by
    QButtonGroup *groupBox
        = new QButtonGroup(1, Horizontal, "Sort by", plainPage());
    layout->addWidget(groupBox);
    groupBox->setExclusive(true);

    byName     = new QCheckBox("By name", groupBox);
    byMuted    = new QCheckBox("By muted", groupBox);
    bySelected = new QCheckBox("By selected", groupBox);
    byPort     = new QCheckBox("By port", groupBox);
    byChannel  = new QCheckBox("By channel", groupBox);
    bySize     = new QCheckBox("By size", groupBox);

    byName->setChecked(true);

    groupBox = new QButtonGroup(1, Horizontal, "Sort order", plainPage());
    layout->addWidget(groupBox);
    groupBox->setExclusive(true);

    inAscending  = new QCheckBox("Ascending order", groupBox);
    inDescending = new QCheckBox("Descending order", groupBox);

    inAscending->setChecked(true);

    attachTo(song);
}


void TrackSortWindow::Notifier_Deleted(TSE3::Song *song)
{
    delete this;
}


void TrackSortWindow::accept()
{
    slotApply();
    done(Accepted);
}


void TrackSortWindow::slotApply()
{
    TSE3::Cmd::Track_Sort::SortBy    by    = TSE3::Cmd::Track_Sort::ByName;
    TSE3::Cmd::Track_Sort::SortOrder order = TSE3::Cmd::Track_Sort::Ascending;

    if (byMuted->isChecked())      by    = TSE3::Cmd::Track_Sort::ByMuted;
    if (bySelected->isChecked())   by    = TSE3::Cmd::Track_Sort::BySelected;
    if (byPort->isChecked())       by    = TSE3::Cmd::Track_Sort::ByPort;
    if (byChannel->isChecked())    by    = TSE3::Cmd::Track_Sort::ByChannel;
    if (bySize->isChecked())       by    = TSE3::Cmd::Track_Sort::BySize;
    if (inAscending->isChecked())  order = TSE3::Cmd::Track_Sort::Ascending;
    if (inDescending->isChecked()) order = TSE3::Cmd::Track_Sort::Descending;

    TSE3::Cmd::Track_Sort *command
        = new TSE3::Cmd::Track_Sort(song, by, order, selection);
    command->execute();
    Application::application()->history(song)->add(command);
}
