/*
 * @(#)Track.h 1.00 30 May 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_TRACK_H
#define ANTHEM_TRACK_H

#include <kdialogbase.h>

#include "tse3/Notifier.h"
#include "tse3/listen/Track.h"
#include "tse3/listen/Song.h"

class TrackWidget;
class MidiFilterWidget;
class MidiParamsWidget;
class DisplayParamsWidget;

namespace TSE3
{
    namespace App
    {
        class TrackSelection;
    }

    namespace Cmd
    {
        class CommandHistory;
    }
}

class QCheckBox;

/**
 * The TrackWindow alows the user to alter a @ref Track's settings.
 *
 * @short   Track settings window.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class TrackWindow : public KDialogBase,
                    public TSE3::Listener<TSE3::TrackListener>
{
        Q_OBJECT

    public:

        /**
         * enum type describing the tab in this window
         */
        enum Tabs
        {
            TrackTab,
            FilterTab,
            ParamsTab,
            DisplayTab
        };

        /**
         * Creates a Track settings window for the given @ref Track.
         * You specify the @ref TSE3::Cmd::CommandHistory that will
         * be used to store the track info set command.
         */
        TrackWindow(QWidget *parent, TSE3::Track *track,
                    TSE3::Cmd::CommandHistory *history,
                    Tabs firstTab = TrackTab);

        /**
         * Dtor - the @ref Track is not deleted.
         */
        virtual ~TrackWindow();

        /**
         * Updates the values before showing the window.
         */
        virtual void show();

        /**
         * @reimplemented
         */
        virtual void Notifier_Deleted(TSE3::Track *track);

        /**
         * @reimplemented
         */
        virtual void Track_Reparented(TSE3::Track *track);

        /**
         * Returns a @ref QString describing the set of Track parameters.
         */
        static QString paramString(TSE3::Track *track);

    public slots:

        /**
         * @reimplemented
         */
        void accept();

        /**
         * @reimplemented
         */
        void done(int reason);

        /**
         * @reimplemented
         */
        void slotApply();

    signals:

        void closed();

    protected:

    private:

        /**
         * Updates all the widgets in the window to reflect the current
         * state of the @ref Track.
         */
        void updateWidgets();

        /**
         * Sets the widget values into the Part.
         */
        void setValues();

        // TSE3 objects
        TSE3::Track               *track;
        TSE3::Cmd::CommandHistory *history;

        // UI objects

        QFrame *trackTab;
        QFrame *filterTab;
        QFrame *paramsTab;
        QFrame *displayParamsTab;

        TrackWidget         *twidget;
        MidiFilterWidget    *mfwidget;
        MidiParamsWidget    *mpwidget;
        DisplayParamsWidget *dpwidget;
};

/**
 * The sort tracks dialogue.
 *
 * @short   Sort Tracks window.
 * @author  Pete Goodliffe
 * @version 1.0
 */
class TrackSortWindow : public KDialogBase,
                        public TSE3::Listener<TSE3::SongListener>
{
        Q_OBJECT

    public:

        TrackSortWindow(QWidget *parent, TSE3::Song *song,
                        TSE3::App::TrackSelection *selection = 0);

        /**
         * @reimplemented
         */
        virtual void Notifier_Deleted(TSE3::Song *song);

    public slots:

        /**
         * @reimplemented
         */
        void accept();

        /**
         * @reimplemented
         */
        void slotApply();

    private:

        TSE3::Song                *song;
        TSE3::App::TrackSelection *selection;
        QCheckBox                 *byName;
        QCheckBox                 *byMuted;
        QCheckBox                 *bySelected;
        QCheckBox                 *byPort;
        QCheckBox                 *byChannel;
        QCheckBox                 *bySize;
        QCheckBox                 *inAscending;
        QCheckBox                 *inDescending;
};

#endif
