/*
 * @(#)DPColourWidget.h 1.00 16 November 2000
 *
 * Copyright (c) Pete Goodliffe 2000 (pete@cthree.org)
 *
 * This file is part of anthem - the TSE3 sequencer.
 *
 * This program is modifiable/redistributable under the terms of the GNU
 * General Public License.
 *
 * You should have recieved a copy of the GNU General Public License along
 * with this program; see the file COPYING. If not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 0211-1307, USA.
 */

#ifndef ANTHEM_DPCOLOURWIDGET_H
#define ANTHEM_DPCOLOURWIDGET_H

#include <qcombobox.h>

namespace TSE3
{
    class DisplayParams;
}

/**
 * This widget provides a @ref QComboBox that alters the colour-related
 * aspects of a @ref TSE3::DisplayParams.
 *
 * That is, it allows you to select the @ref TSE3::DisplayParams::style(), the
 * @ref TSE3::DisplayParams::colour() and
 * @ref TSE3::DisplayParams::presetColour().
 *
 * @short   @ref TSE3::DisplayParams colour value selector
 * @author  Pete Goodliffe
 * @version 1.0
 */
class DPColourWidget : public QComboBox
{
        Q_OBJECT

    public:

        /**
         * Creates a DPColourWidget. If you specify the @p dp parameter,
         * the initial values will be taken from that object.
         *
         * @param dp         DisplayParams object to work on
         * @param useDefault Whether value 0 is "deafult" (true) or "none"
         *                   (false)
         */
        DPColourWidget(TSE3::DisplayParams *dp = 0, bool useDefault = true,
                       QWidget *parent = 0, const char *name = 0);

        virtual ~DPColourWidget();

        /**
         * Returns the style, a value that can be fed to a DisplayParams
         * object.
         */
        int style() const { return _style; }

        /**
         * Returns the currently selected custom colour - this may not
         * be what the style has chosen to display.
         */
        void colour(int &r, int &g, int &b) const { _colour.rgb(&r, &g, &b); }

        /**
         * Returns the currently selected custom colour - this may not
         * be what the style has chosen to display.
         */
        QColor colour() const { return _colour; }

        /**
         * Returns the currently selected preset colour number - this may
         * not be what the style has chosen to display.
         */
        int presetColour() const { return _preset; }

        /**
         * Sets all the colour values to the specified @ref TSE3::DisplayParams
         * object.
         */
        void valuesTo(TSE3::DisplayParams *dp);

        /**
         * Reads all the colour values from the specified
         * @ref TSE3::DisplayParams object.
         */
        void valuesFrom(TSE3::DisplayParams *dp);

    signals:

        /**
         * This signal is emitted when the value of the widget is changed.
         */
        virtual void changed();

    private slots:

        /**
         * Changes the selected style.
         */
        void changeStyle(int s);

        /**
         * Changes to the selected custom colour.
         */
        void changeColour(int r, int g, int b);

        /**
         * Changes to the selected preset colour.
         */
        void changePresetColour(int pc);

    private slots:

        void slotActivated(int index);

    private:

        int    _style;
        QColor _colour;
        int    _preset;
};

#endif
